"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProwlerAudit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// import { Stack, Duration, RemovalPolicy, CustomResource } from 'aws-cdk-lib';
// eslint-disable-next-line no-duplicate-imports
// import { aws_iam as iam, aws_logs as logs, aws_s3 as s3, aws_codebuild as codebuild, aws_lambda as lambda, custom_resources as cr } from 'aws-cdk-lib';
const aws_cdk_lib_1 = require("aws-cdk-lib");
const codebuild = require("aws-cdk-lib/aws-codebuild");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
// eslint-disable-next-line import/no-extraneous-dependencies
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const statement = require("cdk-iam-floyd");
const constructs_1 = require("constructs");
/**
 * Creates a CodeBuild project to audit an AWS account with Prowler and stores the html report in a S3 bucket. This will run onece at the beginning and on a schedule afterwards. Partial contribution from https://github.com/stevecjones
 */
class ProwlerAudit extends constructs_1.Construct {
    constructor(parent, id, props) {
        super(parent, id);
        // defaults
        this.serviceName = props?.serviceName ? props.serviceName : 'prowler';
        this.logsRetentionInDays = props?.logsRetentionInDays
            ? props.logsRetentionInDays
            : logs.RetentionDays.THREE_DAYS;
        this.enableScheduler = props?.enableScheduler
            ? props.enableScheduler
            : false;
        this.prowlerScheduler = props?.prowlerScheduler
            ? props.prowlerScheduler
            : 'cron(0 22 * * ? *)';
        this.prowlerOptions = props?.prowlerOptions
            ? props.prowlerOptions
            : '-M text,junit-xml,html,csv,json';
        this.prowlerVersion = props?.prowlerVersion
            ? props.prowlerVersion
            : '2.10.0';
        const reportBucket = props?.reportBucket ??
            new s3.Bucket(this, 'ReportBucket', {
                //bucketName: `${'123456'}-prowler-reports`,
                autoDeleteObjects: true,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
        const reportGroup = new codebuild.ReportGroup(this, 'reportGroup', {
            /**reportGroupName: 'testReportGroup', */ removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        const preBuildCommands = [];
        if (!!props?.allowlist) {
            const prowlerFilename = 'allowlist.txt';
            if (props.allowlist.isZipArchive) {
                preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} .`);
                preBuildCommands.push(`unzip ${props.allowlist.s3ObjectKey} -d prowler`);
            }
            else {
                preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} prowler/${prowlerFilename}`);
            }
            this.prowlerOptions = this.prowlerOptions + ` -w ${prowlerFilename}`;
        }
        const prowlerBuild = (this.codebuildProject = new codebuild.Project(this, 'prowlerBuild', {
            description: 'Run Prowler assessment',
            timeout: aws_cdk_lib_1.Duration.hours(5),
            environment: {
                environmentVariables: {
                    BUCKET_REPORT: { value: reportBucket.bucketName || '' },
                    BUCKET_PREFIX: { value: props?.reportBucketPrefix ?? '' },
                    ADDITIONAL_S3_ARGS: { value: props?.additionalS3CopyArgs ?? '' },
                    PROWLER_OPTIONS: { value: this.prowlerOptions || '' },
                },
                buildImage: codebuild.LinuxBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-x86_64-standard:3.0'),
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            python: 3.9,
                        },
                        commands: [
                            'echo "Installing Prowler and dependencies..."',
                            'pip3 install detect-secrets',
                            'yum -y install jq',
                            'curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"',
                            'unzip awscliv2.zip',
                            './aws/install',
                            `git clone -b ${this.prowlerVersion} https://github.com/prowler-cloud/prowler`,
                        ],
                    },
                    pre_build: {
                        commands: preBuildCommands,
                    },
                    build: {
                        commands: [
                            `echo "Running Prowler as ./prowler ${this.prowlerOptions} && echo OK || echo FAILED"`,
                            'cd prowler',
                            `./prowler ${this.prowlerOptions} && echo OK || echo FAILED`,
                        ],
                    },
                    post_build: {
                        commands: [
                            'echo "Uploading reports to S3..." ',
                            'aws s3 cp --sse AES256 output/ s3://$BUCKET_REPORT/$BUCKET_PREFIX --recursive $ADDITIONAL_S3_ARGS',
                            'echo "Done!"',
                        ],
                    },
                },
                reports: {
                    [reportGroup.reportGroupName]: {
                        files: ['**/*'],
                        'base-directory': 'prowler/junit-reports',
                        'file-format': 'JunitXml',
                    },
                },
            }),
        }));
        if (!!props?.allowlist) {
            props.allowlist.bucket.grantRead(prowlerBuild);
        }
        prowlerBuild.role?.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('SecurityAudit'));
        prowlerBuild.role?.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('job-function/ViewOnlyAccess'));
        // prowlerBuild.addToRolePolicy(new statement.Dax().allow().to());
        prowlerBuild.addToRolePolicy(new statement.Ds().allow().toListAuthorizedApplications());
        prowlerBuild.addToRolePolicy(new statement.Ec2().allow().toGetEbsEncryptionByDefault());
        prowlerBuild.addToRolePolicy(new statement.Ecr()
            .allow()
            .toDescribeImageScanFindings()
            .toDescribeImages()
            .toDescribeRegistry());
        prowlerBuild.addToRolePolicy(new statement.Tag().allow().toGetTagKeys());
        prowlerBuild.addToRolePolicy(new statement.Lambda().allow().toGetFunction());
        prowlerBuild.addToRolePolicy(new statement.Glue().allow().toSearchTables().toGetConnections());
        prowlerBuild.addToRolePolicy(new statement.Apigateway().allow().toGET());
        prowlerBuild.addToRolePolicy(new iam.PolicyStatement({
            actions: ['support:Describe*'],
            resources: ['*'],
        }));
        reportBucket.grantPut(prowlerBuild);
        const myRole = new iam.Role(this, 'MyRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        const prowlerStartBuildLambda = new lambda.Function(this, 'prowlerStartBuildLambda', {
            runtime: lambda.Runtime.PYTHON_3_9,
            timeout: aws_cdk_lib_1.Duration.seconds(120),
            handler: 'index.lambda_handler',
            code: lambda.Code.fromInline(`import boto3
import cfnresponse
from botocore.exceptions import ClientError
def lambda_handler(event,context):
  props = event['ResourceProperties']
  codebuild_client = boto3.client('codebuild')
  if (event['RequestType'] == 'Create' or event['RequestType'] == 'Update'):
    try:
        response = codebuild_client.start_build(projectName=props['Build'])
        print(response)
        print("Respond: SUCCESS")
        cfnresponse.send(event, context, cfnresponse.SUCCESS, {})
    except Exception as ex:
        print(ex.response['Error']['Message'])
        cfnresponse.send(event, context, cfnresponse.FAILED, ex.response)
      `),
        });
        prowlerStartBuildLambda.addToRolePolicy(new statement.Codebuild().allow().toStartBuild()); // onResource project ...
        const myProvider = new cr.Provider(this, 'MyProvider', {
            onEventHandler: prowlerStartBuildLambda,
            logRetention: this.logsRetentionInDays,
            role: myRole,
        });
        new aws_cdk_lib_1.CustomResource(this, 'Resource1', {
            serviceToken: myProvider.serviceToken,
            properties: {
                Build: prowlerBuild.projectName,
                RERUN_PROWLER: Boolean(this.node.tryGetContext('reRunProwler'))
                    ? Date.now().toString()
                    : '',
            },
        });
        if (this.enableScheduler) {
            const prowlerSchedulerLambda = new lambda.Function(this, 'ScheduleLambda', {
                runtime: lambda.Runtime.PYTHON_3_9,
                timeout: aws_cdk_lib_1.Duration.seconds(120),
                handler: 'index.lambda_handler',
                environment: {
                    buildName: prowlerBuild.projectName,
                },
                code: lambda.Code.fromInline(`import boto3
        import os
        def lambda_handler(event,context):
          codebuild_client = boto3.client('codebuild')
          print("Running Prowler scheduled!: " + os.environ['buildName'])
          project_name = os.environ['buildName']
          response = codebuild_client.start_build(projectName=project_name)
          print(response)
          print("Respond: SUCCESS")
        `),
            });
            new events.Rule(this, 'Scheduler', {
                description: 'A schedule for the Lambda function that triggers Prowler in CodeBuild.',
                targets: [new targets.LambdaFunction(prowlerSchedulerLambda)],
                schedule: events.Schedule.expression(this.prowlerScheduler || ''),
            });
        }
    }
}
exports.ProwlerAudit = ProwlerAudit;
_a = JSII_RTTI_SYMBOL_1;
ProwlerAudit[_a] = { fqn: "cdk-prowler.ProwlerAudit", version: "2.30.1" };
//# sourceMappingURL=data:application/json;base64,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