//! Mapping betwwen country code and subregion code

/// Subregions regions as defined by UN M49 standards, the enum value corresponds to the region code.
/// See "Geographical regions" at <https://unstats.un.org/unsd/methodology/m49/.>.
#[allow(missing_docs)] // We don't need to document each variant, as it can all be explained by the description
pub enum Subregion {
    NorthernAmerica = 21,
    CentralAmerica = 13,
    Caribbean = 29,
    SouthAmerica = 5,
    NorthernEurope = 154,
    WesternEurope = 155,
    SouthernEurope = 39,
    EasternEurope = 151,
    EasternAsia = 30,
    SouthernAsia = 34,
    SouthEasternAsia = 35,
    WesternAsia = 145,
    CentralAsia = 143,
    NorthernAfrica = 15,
    WesternAfrica = 11,
    MiddleAfrica = 17,
    EasternAfrica = 14,
    SouthernAfrica = 18,
    Melanesia = 54,
    Micronesia = 57,
    Polynesia = 61,
    AustraliaAndNewZealand = 53,
}

impl Subregion {
    /// Maps 2 ISO letter country codes to geoscheme regions based on the UN M.49 standard.
    pub fn from_iso2(iso2_letter: &str) -> Option<Self> {
        Some(match iso2_letter {
            "AF" => Subregion::SouthernAsia,
            "AL" => Subregion::SouthernEurope,
            "DZ" => Subregion::NorthernAfrica,
            "AS" => Subregion::Polynesia,
            "AD" => Subregion::SouthernEurope,
            "AO" => Subregion::MiddleAfrica,
            "AI" => Subregion::Caribbean,
            // "AQ" => Subregion::Antarctica,
            "AG" => Subregion::Caribbean,
            "AR" => Subregion::SouthAmerica,
            "AM" => Subregion::WesternAsia,
            "AW" => Subregion::Caribbean,
            "AU" => Subregion::AustraliaAndNewZealand,
            "AT" => Subregion::WesternEurope,
            "AZ" => Subregion::WesternAsia,
            "BS" => Subregion::Caribbean,
            "BH" => Subregion::WesternAsia,
            "BD" => Subregion::SouthernAsia,
            "BB" => Subregion::Caribbean,
            "BY" => Subregion::EasternEurope,
            "BE" => Subregion::WesternEurope,
            "BZ" => Subregion::CentralAmerica,
            "BJ" => Subregion::WesternAfrica,
            "BM" => Subregion::NorthernAmerica,
            "BT" => Subregion::SouthernAsia,
            "BO" => Subregion::SouthAmerica,
            "BA" => Subregion::SouthernEurope,
            "BW" => Subregion::SouthernAfrica,
            // "BV" => Subregion::Antarctica,
            "BR" => Subregion::SouthAmerica,
            "IO" => Subregion::EasternAfrica,
            "BN" => Subregion::SouthEasternAsia,
            "BG" => Subregion::EasternEurope,
            "BF" => Subregion::WesternAfrica,
            "BI" => Subregion::EasternAfrica,
            "CV" => Subregion::WesternAfrica,
            "KH" => Subregion::SouthEasternAsia,
            "CM" => Subregion::MiddleAfrica,
            "CA" => Subregion::NorthernAmerica,
            "KY" => Subregion::Caribbean,
            "CF" => Subregion::MiddleAfrica,
            "TD" => Subregion::MiddleAfrica,
            "CL" => Subregion::SouthAmerica,
            "CN" => Subregion::EasternAsia,
            "CX" => Subregion::AustraliaAndNewZealand,
            "CC" => Subregion::AustraliaAndNewZealand,
            "CO" => Subregion::SouthAmerica,
            "KM" => Subregion::EasternAfrica,
            "CG" => Subregion::MiddleAfrica,
            "CD" => Subregion::MiddleAfrica,
            "CK" => Subregion::Polynesia,
            "CR" => Subregion::CentralAmerica,
            "CI" => Subregion::WesternAfrica,
            "HR" => Subregion::SouthernEurope,
            "CU" => Subregion::Caribbean,
            "CY" => Subregion::WesternAsia,
            "CZ" => Subregion::EasternEurope,
            "DK" => Subregion::NorthernEurope,
            "DJ" => Subregion::EasternAfrica,
            "DM" => Subregion::Caribbean,
            "DO" => Subregion::Caribbean,
            "EC" => Subregion::SouthAmerica,
            "EG" => Subregion::NorthernAfrica,
            "SV" => Subregion::CentralAmerica,
            "GQ" => Subregion::MiddleAfrica,
            "ER" => Subregion::EasternAfrica,
            "EE" => Subregion::NorthernEurope,
            "SZ" => Subregion::SouthernAfrica,
            "ET" => Subregion::EasternAfrica,
            "FK" => Subregion::SouthAmerica,
            "FO" => Subregion::NorthernEurope,
            "FJ" => Subregion::Melanesia,
            "FI" => Subregion::NorthernEurope,
            "FR" => Subregion::WesternEurope,
            "GF" => Subregion::SouthAmerica,
            "PF" => Subregion::Polynesia,
            // "TF" => Subregion::Antarctica,
            "GA" => Subregion::MiddleAfrica,
            "GM" => Subregion::WesternAfrica,
            "GE" => Subregion::WesternAsia,
            "DE" => Subregion::WesternEurope,
            "GH" => Subregion::WesternAfrica,
            "GI" => Subregion::SouthernEurope,
            "GR" => Subregion::SouthernEurope,
            "GL" => Subregion::NorthernAmerica,
            "GD" => Subregion::Caribbean,
            "GP" => Subregion::Caribbean,
            "GU" => Subregion::Micronesia,
            "GT" => Subregion::CentralAmerica,
            "GG" => Subregion::NorthernEurope,
            "GN" => Subregion::WesternAfrica,
            "GW" => Subregion::WesternAfrica,
            "GY" => Subregion::SouthAmerica,
            "HT" => Subregion::Caribbean,
            // "HM" => Subregion::Antarctica,
            "VA" => Subregion::SouthernEurope,
            "HN" => Subregion::CentralAmerica,
            "HK" => Subregion::EasternAsia,
            "HU" => Subregion::EasternEurope,
            "IS" => Subregion::NorthernEurope,
            "IN" => Subregion::SouthernAsia,
            "ID" => Subregion::SouthEasternAsia,
            "IR" => Subregion::SouthernAsia,
            "IQ" => Subregion::WesternAsia,
            "IE" => Subregion::NorthernEurope,
            "IL" => Subregion::WesternAsia,
            "IT" => Subregion::SouthernEurope,
            "JM" => Subregion::Caribbean,
            "JP" => Subregion::EasternAsia,
            "JE" => Subregion::NorthernEurope,
            "JO" => Subregion::WesternAsia,
            "KZ" => Subregion::CentralAsia,
            "KE" => Subregion::EasternAfrica,
            "KI" => Subregion::Micronesia,
            "KP" => Subregion::EasternAsia,
            "KR" => Subregion::EasternAsia,
            "KW" => Subregion::WesternAsia,
            "KG" => Subregion::CentralAsia,
            "LA" => Subregion::SouthEasternAsia,
            "LV" => Subregion::NorthernEurope,
            "LB" => Subregion::WesternAsia,
            "LS" => Subregion::SouthernAfrica,
            "LR" => Subregion::WesternAfrica,
            "LY" => Subregion::NorthernAfrica,
            "LI" => Subregion::WesternEurope,
            "LT" => Subregion::NorthernEurope,
            "LU" => Subregion::WesternEurope,
            "MO" => Subregion::EasternAsia,
            "MG" => Subregion::EasternAfrica,
            "MW" => Subregion::EasternAfrica,
            "MY" => Subregion::SouthEasternAsia,
            "MV" => Subregion::SouthernAsia,
            "ML" => Subregion::WesternAfrica,
            "MT" => Subregion::SouthernEurope,
            "MH" => Subregion::Micronesia,
            "MQ" => Subregion::Caribbean,
            "MR" => Subregion::WesternAfrica,
            "MU" => Subregion::EasternAfrica,
            "YT" => Subregion::EasternAfrica,
            "MX" => Subregion::CentralAmerica,
            "FM" => Subregion::Micronesia,
            "MD" => Subregion::EasternEurope,
            "MC" => Subregion::WesternEurope,
            "MN" => Subregion::EasternAsia,
            "ME" => Subregion::SouthernEurope,
            "MS" => Subregion::Caribbean,
            "MA" => Subregion::NorthernAfrica,
            "MZ" => Subregion::EasternAfrica,
            "MM" => Subregion::SouthEasternAsia,
            "NA" => Subregion::SouthernAfrica,
            "NR" => Subregion::Micronesia,
            "NP" => Subregion::SouthernAsia,
            "NL" => Subregion::WesternEurope,
            "NC" => Subregion::Melanesia,
            "NZ" => Subregion::AustraliaAndNewZealand,
            "NI" => Subregion::CentralAmerica,
            "NE" => Subregion::WesternAfrica,
            "NG" => Subregion::WesternAfrica,
            "NU" => Subregion::Polynesia,
            "NF" => Subregion::AustraliaAndNewZealand,
            "MK" => Subregion::SouthernEurope,
            "MP" => Subregion::Micronesia,
            "NO" => Subregion::NorthernEurope,
            "OM" => Subregion::WesternAsia,
            "PK" => Subregion::SouthernAsia,
            "PW" => Subregion::Micronesia,
            "PS" => Subregion::WesternAsia,
            "PA" => Subregion::CentralAmerica,
            "PG" => Subregion::Melanesia,
            "PY" => Subregion::SouthAmerica,
            "PE" => Subregion::SouthAmerica,
            "PH" => Subregion::SouthEasternAsia,
            "PN" => Subregion::Polynesia,
            "PL" => Subregion::EasternEurope,
            "PT" => Subregion::SouthernEurope,
            "PR" => Subregion::Caribbean,
            "QA" => Subregion::WesternAsia,
            "RE" => Subregion::EasternAfrica,
            "RO" => Subregion::EasternEurope,
            "RU" => Subregion::EasternEurope,
            "RW" => Subregion::EasternAfrica,
            "BL" => Subregion::Caribbean,
            "SH" => Subregion::WesternAfrica,
            "KN" => Subregion::Caribbean,
            "LC" => Subregion::Caribbean,
            "MF" => Subregion::Caribbean,
            "PM" => Subregion::NorthernAmerica,
            "VC" => Subregion::Caribbean,
            "WS" => Subregion::Polynesia,
            "SM" => Subregion::SouthernEurope,
            "ST" => Subregion::MiddleAfrica,
            "SA" => Subregion::WesternAsia,
            "SN" => Subregion::WesternAfrica,
            "RS" => Subregion::SouthernEurope,
            "SC" => Subregion::EasternAfrica,
            "SL" => Subregion::WesternAfrica,
            "SG" => Subregion::SouthEasternAsia,
            "SX" => Subregion::Caribbean,
            "SK" => Subregion::EasternEurope,
            "SI" => Subregion::SouthernEurope,
            "SB" => Subregion::Melanesia,
            "SO" => Subregion::EasternAfrica,
            "ZA" => Subregion::SouthernAfrica,
            "SS" => Subregion::EasternAfrica,
            "ES" => Subregion::SouthernEurope,
            "LK" => Subregion::SouthernAsia,
            "SD" => Subregion::NorthernAfrica,
            "SR" => Subregion::SouthAmerica,
            "SJ" => Subregion::NorthernEurope,
            "SE" => Subregion::NorthernEurope,
            "CH" => Subregion::WesternEurope,
            "SY" => Subregion::WesternAsia,
            "TW" => Subregion::EasternAsia,
            "TJ" => Subregion::CentralAsia,
            "TZ" => Subregion::EasternAfrica,
            "TH" => Subregion::SouthEasternAsia,
            "TL" => Subregion::SouthEasternAsia,
            "TG" => Subregion::WesternAfrica,
            "TK" => Subregion::Polynesia,
            "TO" => Subregion::Polynesia,
            "TT" => Subregion::Caribbean,
            "TN" => Subregion::NorthernAfrica,
            "TR" => Subregion::WesternAsia,
            "TM" => Subregion::CentralAsia,
            "TC" => Subregion::Caribbean,
            "TV" => Subregion::Polynesia,
            "UG" => Subregion::EasternAfrica,
            "UA" => Subregion::EasternEurope,
            "AE" => Subregion::WesternAsia,
            "GB" => Subregion::NorthernEurope,
            "US" => Subregion::NorthernAmerica,
            "UY" => Subregion::SouthAmerica,
            "UZ" => Subregion::CentralAsia,
            "VU" => Subregion::Melanesia,
            "VE" => Subregion::SouthAmerica,
            "VN" => Subregion::SouthEasternAsia,
            "VG" => Subregion::Caribbean,
            "VI" => Subregion::Caribbean,
            "WF" => Subregion::Polynesia,
            "EH" => Subregion::NorthernAfrica,
            "YE" => Subregion::WesternAsia,
            "ZM" => Subregion::EasternAfrica,
            "ZW" => Subregion::EasternAfrica,
            // Additional ISO codes for territories
            "AX" => Subregion::NorthernEurope,
            "BQ" => Subregion::Caribbean,
            "CW" => Subregion::Caribbean,
            // "TF" => Subregion::Antarctica,
            "IM" => Subregion::NorthernEurope,
            _ => return None,
        })
    }
}
