# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['coveo_testing', 'coveo_testing.temporary_resource']

package_data = \
{'': ['*']}

install_requires = \
['attrs', 'pytest']

setup_kwargs = {
    'name': 'coveo-testing',
    'version': '1.0.1',
    'description': 'Lightweight testing helpers',
    'long_description': '# coveo-testing\n\nA set of test/pytest helpers to facilitate common routines.\n\n\nContent in a nutshell:\n\n- Reusable pytest markers (UnitTest, IntegrationTest)\n- Unique ID generation for tests\n- Multiline logging assertions with includes, excludes, levels and comprehensive assertion output\n- Refactorable `unittest.mock.patch(\'this.module\')` module references\n\n- Human-readable (but still customizable) display for parametrized tests\n\n\nThis project is used as the test base for all other projects in this repository.\n\nTherefore, it cannot depend on any of them.\n\nMore complex use cases may be implemented in the `coveo-testing-extras` project. That\'s also where you can depend on projects that depend on `coveo-testing`. \n\n\n# pytest markers and auto-registration\n\nThis enables code completion on markers.\n\nThree markers are already provided: `[UnitTest, Integration, Interactive]`\n\nHere\'s how you can create additional markers:\n\n```python\n# /test_some_module/markers.py\nimport pytest\n\nDockerTest = pytest.mark.docker_test\nCloudTest = pytest.mark.cloud_test\n\nALL_MARKERS = [DockerTest, CloudTest]\n```\n\nYou can then import these markers and decorate your test functions accordingly:\n\n```python\n# /test_some_module/test_something.py\nfrom coveo_testing.markers import UnitTest, Integration, Interactive\nfrom test_some_module.markers import CloudTest, DockerTest\n\n@UnitTest\ndef test_unit() -> None:\n    ...  # designed to be fast and lightweight, most likely parametrized\n\n\n@Integration\ndef test_integration() -> None:\n    ...  # combines multiple features to achieve a test\n\n\n@CloudTest\ndef test_in_the_cloud() -> None:\n    ...  # this could be a post-deployment test, for instance.\n\n\n@DockerTest\n@Integration\ndef test_through_docker() -> None:\n    ... # will run whenever docker tests or integration tests are requested\n\n\n@Interactive\ndef test_interactive() -> None:\n    ...  # these tests rely on eye-validations, special developer setups, etc  \n\n```\n\nPytest will issue a warning when markers are not registered.\n\nTo register coveo-testing\'s markers along with your custom markers, use the provided `register_markers` method:\n\n```python\n# /test_some_module/conftest.py\nfrom _pytest.config import Config\nfrom coveo_testing.markers import register_markers\nfrom test_some_module.markers import ALL_MARKERS\n\ndef pytest_configure(config: Config) -> None:\n    """This pytest hook is ran once, before collecting tests."""\n    register_markers(config, *ALL_MARKERS)\n```\n\n\n# Human-readable unique ID generation\n\nThe generated ID has this format:\n\n`friendly-name.timestamp.pid.host.executor.sequence`\n\n- friendly-name:\n  - provided by you, for your own benefit\n    \n- timestamp: \n  - format "%m%d%H%M%S" (month, day, hour, minutes, seconds)\n  - computed once, when TestId is imported\n    \n- pid:\n  - the pid of the python process\n    \n- host:\n  - the network name of the machine\n\n- executor:\n  - the content of the `EXECUTOR_NUMBER` environment variable\n  - returns \'default\' when not defined  \n  - historically, this variable comes from jenkins\n  - conceptually, it can be used to help distribute (and identify) tests and executors\n\n- sequence:\n  - Thread-safe\n  - Each `friendly-name` has an isolated `sequence` that starts at 0\n  - Incremented on each new instance\n  - Enables support for parallel parametrized tests\n\n```python\nfrom coveo_testing.temporary_resource.unique_id import TestId, unique_test_id\n\n\n# the friendly name is the only thing you need to specify\ntest_id = TestId(\'friendly-name\')\nstr(test_id)\n\'friendly-name.0202152243.18836.WORKSTATION.default.0\'\n\n\n# you can pass the instance around to share the ID\nstr(test_id)\n\'friendly-name.0202152243.18836.WORKSTATION.default.0\'\n\n\n# create new instances to increment the sequence number\ntest_id = TestId(\'friendly-name\')\nstr(test_id)\n\'friendly-name.0202152243.18836.WORKSTATION.default.1\'\n\n\n# use it in parallel parameterized tests\nimport pytest\n\n@pytest.mark.parametrize(\'param\', (True, False))\ndef test_param(param: bool, unique_test_id: TestId) -> None:\n    # in this case, the friendly name is the function name and\n    # the sequence will increase on each parameter\n    # test_param.0202152243.18836.WORKSTATION.default.0\n    # test_param.0202152243.18836.WORKSTATION.default.1\n    ...\n```\n\n\n# multiline logging assertions\n\nMaybe pytest\'s `caplog` is enough for your needs, or maybe you need more options.\nThis tool uses `in` and `not in` to match strings in a case-sensitive way.\n\n```python\nimport logging\nfrom coveo_testing.logging import assert_logging\n\nwith assert_logging(\n        logging.getLogger(\'logger-name\'),\n        present=[\'evidence1\', \'evidence2\'], \n        absent=[...], \n        level=logging.WARN):\n    ...\n```\n\n\n# Human-readable (but still customizable) display for parametrized tests\n\nIf you\'re like me, you typed `@pytest.mark.parametrize` wrong a couple of times!\n\nEnable IDE completion by using this one instead:\n\n```python\nfrom coveo_testing.parametrize import parametrize\n\n@parametrize(\'var\', (True, False))\ndef test_var(var: bool) -> None:\n    ...\n```\n\nIt has one difference vs the pytest one, and it\'s the way it formats the "parameter name" for each iteration of the test.\n\nPytest will skip a lot of types and will simply name your test "var0", "var1" and so on.\nUsing this `@parametrize` instead, the variable\'s content will be inspected:\n\n```python\nfrom typing import Any\nfrom coveo_testing.parametrize import parametrize\nimport pytest\n\n\nclass StrMe:\n  def __init__(self, var: Any) -> None:\n    self.var = var\n      \n  def __str__(self) -> str:\n    return f"Value: {self.var}"\n\n\n@parametrize(\'var\', [[\'list\', \'display\'], [StrMe(\'hello\')]])\ndef test_param(var: bool) -> None:\n    ...\n\n@pytest.mark.parametrize(\'var\', [[\'list\', \'display\'], [StrMe(\'hello\')]])\ndef test_param_from_pytest(var: bool) -> None:\n    ...\n```\n\nIf you run `pytest --collect-only` you will obtain the following:\n```\n    <Function test_param[list-display]>\n    <Function test_param[Value: hello]>\n    <Function test_param_from_pytest[var0]>\n    <Function test_param_from_pytest[var1]>\n```\n',
    'author': 'Jonathan Piché',
    'author_email': 'tools@coveo.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/coveooss/coveo-python-oss',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8',
}


setup(**setup_kwargs)
