#!/usr/bin/env python
import os
import pickle
import traceback
import utils
from typing import List
from pkl_organizer import PklOrganizer
from configuration_facade import AutosubmitConfigurationFacade
from joblist_facade import JobListFacade
from job_factory import get_jobs_with_no_outliers

class PerformanceMetrics(object):
  """ Manages Performance Metrics """  
  
  def __init__(self, expid):
    # type (str) -> None    
    self.expid = expid
    self.error = False
    self.error_message = ""    
    self.total_sim_run_time = 0 # type : int
    self.total_sim_queue_time = 0 # type : int    
    self.SYPD = 0 # type : int
    self.ASYPD = 0 # type : int
    self.CHSY = 0 # type : int
    self.JPSY = 0 # type : int 
    self.RSYPD = 0 # type : int
    self._considered = [] # type : List
    self._sim_processors = 0 # type : int
    self.warnings = [] # type : List
    self.post_jobs_total_time_average = 0 # type : int 
    try:
      self._configuration = AutosubmitConfigurationFacade(self.expid) # type : AutosubmitConfigurationFacade
      self._pkl_content = PklOrganizer(self._configuration.pkl_path) # type : PklOrganizer      
      self._sim_processors = self._configuration.sim_processors      
      self._job_list_package_time = JobListFacade(self.expid, self._pkl_content.get_simple_jobs(self._configuration.tmp_path)) # type : JobListFacade
    except Exception as exp:
      self.error = True
      self.error_message = str(exp)
      print(traceback.format_exc())
      print(exp.message)
    if self.error == False:   
      self._configuration.update_sim_jobs(self._pkl_content.sim_jobs)     
      self._update_jobs_with_time_data()
      self._calculate_post_jobs_total_time_average()      
      self.sim_jobs_valid = get_jobs_with_no_outliers(self._pkl_content.get_completed_section_jobs(utils.JobSection.SIM)) # type : List[Job]    
      self._identify_outlied_jobs()
      self._update_valid_sim_jobs_with_post_data()
      self._add_valid_sim_jobs_to_considered()
      self._calculate_total_sim_queue_time()
      self._calculate_total_sim_run_time()
      self._calculate_global_metrics()
      self._unify_warnings()
  
  def _update_jobs_with_time_data(self):
      self._job_list_package_time.update_with_timedata(self._pkl_content.sim_jobs)
      self._job_list_package_time.update_with_timedata(self._pkl_content.post_jobs)
      self._job_list_package_time.update_with_timedata(self._pkl_content.clean_jobs)
      self._job_list_package_time.update_with_timedata(self._pkl_content.transfer_jobs)
      self._job_list_package_time.update_with_yps_per_run(self._pkl_content.sim_jobs)     

  def _calculate_global_metrics(self):
      self._calculate_SYPD()
      self._calculate_ASYPD()
      self._calculate_RSYPD()
      self._calculate_JPSY()
      self._calculate_CHSY()

  def _identify_outlied_jobs(self):
    """ Generates warnings """
    outlied = [job for job in self._pkl_content.get_completed_section_jobs(utils.JobSection.SIM) if job not in self.sim_jobs_valid]
    for job in outlied:
      self.warnings.append("Considered | Job {0} (Package {1}) has no energy information and is not going to be considered for energy calculations.".format(job.name, self._job_list_package_time.job_to_package.get(job.name, "No Package")))

  def _unify_warnings(self):
    self.warnings.extend(self._pkl_content.warnings)
    self.warnings.extend(self._configuration.warnings)
    self.warnings.extend(self._job_list_package_time.warning_messages)
  
  def _calculate_post_jobs_total_time_average(self):
    """ Average run+queue of all completed POST jobs """
    completed_post_jobs = self._pkl_content.get_completed_section_jobs(utils.JobSection.POST)
    if len(completed_post_jobs):
      average = sum(job.total_time for job in completed_post_jobs)/ len(completed_post_jobs)
      self.post_jobs_total_time_average = round(average, 4)


  def _get_sims_with_energy_count(self):
    return sum(1 for job in self.sim_jobs_valid if job.energy > 0)

  def _update_valid_sim_jobs_with_post_data(self):
    """ Updates required value in sim job """
    for simjob in self.sim_jobs_valid:
      if self.post_jobs_total_time_average > 0:
        simjob.set_post_jobs_total_average(self.post_jobs_total_time_average)
      # self._add_to_considered(simjob)
  
  def _add_valid_sim_jobs_to_considered(self):
    for simjob in self.sim_jobs_valid:
      self._add_to_considered(simjob)

  def _calculate_total_sim_run_time(self):
    self.total_sim_run_time =  sum(job.run_time for job in self.sim_jobs_valid)
  
  def _calculate_total_sim_queue_time(self):
    self.total_sim_queue_time = sum(job.queue_time for job in self.sim_jobs_valid)
  
  def _calculate_SYPD(self):
    if self.total_sim_run_time > 0:
      SYPD = ((self._configuration.current_years_per_sim * len(self._considered) * utils.SECONDS_IN_A_DAY) /
                  (self.total_sim_run_time))
      self.SYPD = round(SYPD, 4)
  
  def _calculate_ASYPD(self):
    if len(self.sim_jobs_valid) > 0:
      ASYPD = (self._configuration.current_years_per_sim * len(self.sim_jobs_valid) * utils.SECONDS_IN_A_DAY) / (self.total_sim_run_time + self.total_sim_queue_time + self.post_jobs_total_time_average) 
      self.ASYPD = round(ASYPD, 4)      
  
  def _calculate_RSYPD(self):  
    divisor = self._get_RSYPD_divisor()  
    if len(self.sim_jobs_valid) > 0 and divisor > 0:
      RSYPD = (self._configuration.current_years_per_sim * len(self.sim_jobs_valid) * utils.SECONDS_IN_A_DAY) / divisor
      self.RSYPD = round(RSYPD, 4)
  
  def _calculate_JPSY(self):
    """ Joules per Simulated Year """
    sims_with_energy_count = self._get_sims_with_energy_count()
    if len(self.sim_jobs_valid) > 0 and sims_with_energy_count > 0:
      JPSY = sum(job.JPSY for job in self.sim_jobs_valid)/sims_with_energy_count
      self.JPSY = round(JPSY, 4)
  
  def _calculate_CHSY(self):
    if len(self.sim_jobs_valid) > 0:
      CHSY = sum(job.CHSY for job in self.sim_jobs_valid)/len(self.sim_jobs_valid)
      self.CHSY = round(CHSY, 4)

  def _get_RSYPD_support_list(self):
    """ The support list for the divisor can have a different source """
    # type () -> List[Job]
    completed_transfer_jobs = self._pkl_content.get_completed_section_jobs(utils.JobSection.TRANSFER)
    completed_clean_jobs = self._pkl_content.get_completed_section_jobs(utils.JobSection.CLEAN)
    if len(completed_transfer_jobs) > 0:
      return completed_transfer_jobs
    elif len(completed_clean_jobs) > 0:
      return completed_clean_jobs
    else:
      return []

  def _get_RSYPD_divisor(self):
    support_list = self._get_RSYPD_support_list()
    divisor = 0
    if len(support_list) > 0 and len(self.sim_jobs_valid):            
      divisor = support_list[-1].finish - self.sim_jobs_valid[0].start
    return divisor
    
  
  def _add_to_considered(self, simjob):
    # type : (Job) -> None
    self._considered.append({
      "name": simjob.name,
      "queue": simjob.queue_time,
      "running": simjob.run_time,
      "CHSY": simjob.CHSY,
      "SYPD": simjob.SYPD,
      "ASYPD": simjob.ASYPD,
      "JPSY": simjob.JPSY,
      "energy": simjob.energy,
      "yps": simjob.years_per_sim,
      "ncpus": simjob.ncpus
    })
  
  def to_json(self):
    # type : () -> Dict
    return {"SYPD": self.SYPD,
            "ASYPD": self.ASYPD,
            "RSYPD": self.RSYPD,
            "CHSY": self.CHSY,
            "JPSY": self.JPSY,
            "Parallelization": self._sim_processors,
            "considered": self._considered,
            "error": self.error,
            "error_message": self.error_message,
            "warnings_job_data": self.warnings,
            }
  
  
  
    




    


