"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedServiceBase = void 0;
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
/**
 * The base class for ApplicationLoadBalancedEc2Service and ApplicationLoadBalancedFargateService services.
 */
class ApplicationLoadBalancedServiceBase extends cdk.Construct {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedServiceBase class.
     */
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        this.desiredCount = props.desiredCount || 1;
        const internetFacing = props.publicLoadBalancer !== undefined ? props.publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        const loadBalancer = props.loadBalancer !== undefined ? props.loadBalancer
            : new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', lbProps);
        if (props.certificate !== undefined && props.protocol !== undefined && props.protocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            throw new Error('The HTTPS protocol must be used when a certificate is given');
        }
        const protocol = props.protocol !== undefined ? props.protocol :
            (props.certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
        if (protocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS && props.redirectHTTP === true) {
            throw new Error('The HTTPS protocol must be used when redirecting HTTP traffic');
        }
        const targetProps = {
            port: 80,
        };
        this.listener = loadBalancer.addListener('PublicListener', {
            protocol,
            port: props.listenerPort,
            open: (_a = props.openListener) !== null && _a !== void 0 ? _a : true,
        });
        this.targetGroup = this.listener.addTargets('ECS', targetProps);
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            if (typeof props.domainName === 'undefined' || typeof props.domainZone === 'undefined') {
                throw new Error('A domain name and zone is required when using the HTTPS protocol');
            }
            if (props.certificate !== undefined) {
                this.certificate = props.certificate;
            }
            else {
                this.certificate = new aws_certificatemanager_1.Certificate(this, 'Certificate', {
                    domainName: props.domainName,
                    validation: aws_certificatemanager_1.CertificateValidation.fromDns(props.domainZone),
                });
            }
        }
        if (this.certificate !== undefined) {
            this.listener.addCertificates('Arns', [aws_elasticloadbalancingv2_1.ListenerCertificate.fromCertificateManager(this.certificate)]);
        }
        if (props.redirectHTTP) {
            this.redirectListener = loadBalancer.addListener('PublicRedirectListener', {
                protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                port: 80,
                open: true,
                defaultAction: aws_elasticloadbalancingv2_1.ListenerAction.redirect({
                    port: ((_b = props.listenerPort) === null || _b === void 0 ? void 0 : _b.toString()) || '443',
                    protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
                    permanent: true,
                }),
            });
        }
        let domainName = loadBalancer.loadBalancerDnsName;
        if (typeof props.domainName !== 'undefined') {
            if (typeof props.domainZone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            const record = new aws_route53_1.ARecord(this, 'DNS', {
                zone: props.domainZone,
                recordName: props.domainName,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
            domainName = record.domainName;
        }
        if (loadBalancer instanceof aws_elasticloadbalancingv2_1.ApplicationLoadBalancer) {
            this._applicationLoadBalancer = loadBalancer;
        }
        new cdk.CfnOutput(this, 'LoadBalancerDNS', { value: loadBalancer.loadBalancerDnsName });
        new cdk.CfnOutput(this, 'ServiceURL', { value: protocol.toLowerCase() + '://' + domainName });
    }
    /**
     * The Application Load Balancer for the service.
     */
    get loadBalancer() {
        if (!this._applicationLoadBalancer) {
            throw new Error('.loadBalancer can only be accessed if the class was constructed with an owned, not imported, load balancer');
        }
        return this._applicationLoadBalancer;
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = cdk.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Adds service as a target of the target group.
     */
    addServiceAsTarget(service) {
        this.targetGroup.addTarget(service);
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.ApplicationLoadBalancedServiceBase = ApplicationLoadBalancedServiceBase;
//# sourceMappingURL=data:application/json;base64,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