"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const cdk = require("@aws-cdk/core");
const ecsPatterns = require("../../lib");
module.exports = {
    'test ECS loadbalanced construct'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            memoryLimitMiB: 1024,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
                environment: {
                    TEST_ENVIRONMENT_VARIABLE1: 'test environment variable 1 value',
                    TEST_ENVIRONMENT_VARIABLE2: 'test environment variable 2 value',
                },
            },
            desiredCount: 2,
        });
        // THEN - stack contains a load balancer and a service
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
            DesiredCount: 2,
            LaunchType: 'EC2',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Environment: [
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE1',
                            Value: 'test environment variable 1 value',
                        },
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE2',
                            Value: 'test environment variable 2 value',
                        },
                    ],
                    Memory: 1024,
                },
            ],
        }));
        test.done();
    },
    'set vpc instead of cluster'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedEc2Service(stack, 'Service', {
            vpc,
            memoryLimitMiB: 1024,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
                environment: {
                    TEST_ENVIRONMENT_VARIABLE1: 'test environment variable 1 value',
                    TEST_ENVIRONMENT_VARIABLE2: 'test environment variable 2 value',
                },
            },
            desiredCount: 2,
        });
        // THEN - stack does not contain a LaunchConfiguration
        assert_1.expect(stack, true).notTo(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration'));
        test.throws(() => assert_1.expect(stack));
        test.done();
    },
    'setting vpc and cluster throws error'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        test.throws(() => new ecsPatterns.NetworkLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            vpc,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
            },
        }));
        test.done();
    },
    'test ECS loadbalanced construct with memoryReservationMiB'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            memoryReservationMiB: 1024,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
            },
        });
        // THEN - stack contains a load balancer and a service
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    MemoryReservation: 1024,
                },
            ],
        }));
        test.done();
    },
    'creates AWS Cloud Map service for Private DNS namespace with application load balanced ec2 service'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // WHEN
        cluster.addDefaultCloudMapNamespace({
            name: 'foo.com',
            type: cloudmap.NamespaceType.DNS_PRIVATE,
        });
        new ecsPatterns.ApplicationLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            taskImageOptions: {
                containerPort: 8000,
                image: ecs.ContainerImage.fromRegistry('hello'),
            },
            cloudMapOptions: {
                name: 'myApp',
            },
            memoryLimitMiB: 512,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
            ServiceRegistries: [
                {
                    ContainerName: 'web',
                    ContainerPort: 8000,
                    RegistryArn: {
                        'Fn::GetAtt': [
                            'ServiceCloudmapServiceDE76B29D',
                            'Arn',
                        ],
                    },
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
            DnsConfig: {
                DnsRecords: [
                    {
                        TTL: 60,
                        Type: 'SRV',
                    },
                ],
                NamespaceId: {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                        'Id',
                    ],
                },
                RoutingPolicy: 'MULTIVALUE',
            },
            HealthCheckCustomConfig: {
                FailureThreshold: 1,
            },
            Name: 'myApp',
            NamespaceId: {
                'Fn::GetAtt': [
                    'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                    'Id',
                ],
            },
        }));
        test.done();
    },
    'creates AWS Cloud Map service for Private DNS namespace with network load balanced fargate service'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // WHEN
        cluster.addDefaultCloudMapNamespace({
            name: 'foo.com',
            type: cloudmap.NamespaceType.DNS_PRIVATE,
        });
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                containerPort: 8000,
                image: ecs.ContainerImage.fromRegistry('hello'),
            },
            cloudMapOptions: {
                name: 'myApp',
            },
            memoryLimitMiB: 512,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
            ServiceRegistries: [
                {
                    RegistryArn: {
                        'Fn::GetAtt': [
                            'ServiceCloudmapServiceDE76B29D',
                            'Arn',
                        ],
                    },
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
            DnsConfig: {
                DnsRecords: [
                    {
                        TTL: 60,
                        Type: 'A',
                    },
                ],
                NamespaceId: {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                        'Id',
                    ],
                },
                RoutingPolicy: 'MULTIVALUE',
            },
            HealthCheckCustomConfig: {
                FailureThreshold: 1,
            },
            Name: 'myApp',
            NamespaceId: {
                'Fn::GetAtt': [
                    'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                    'Id',
                ],
            },
        }));
        test.done();
    },
    'test Fargate loadbalanced construct'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
                environment: {
                    TEST_ENVIRONMENT_VARIABLE1: 'test environment variable 1 value',
                    TEST_ENVIRONMENT_VARIABLE2: 'test environment variable 2 value',
                },
            },
            desiredCount: 2,
        });
        // THEN - stack contains a load balancer and a service
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Environment: [
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE1',
                            Value: 'test environment variable 1 value',
                        },
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE2',
                            Value: 'test environment variable 2 value',
                        },
                    ],
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': { Ref: 'ServiceTaskDefwebLogGroup2A898F61' },
                            'awslogs-stream-prefix': 'Service',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
            DesiredCount: 2,
            LaunchType: 'FARGATE',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 80,
            Protocol: 'HTTP',
        }));
        test.done();
    },
    'test Fargate loadbalanced construct opting out of log driver creation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
                enableLogging: false,
                environment: {
                    TEST_ENVIRONMENT_VARIABLE1: 'test environment variable 1 value',
                    TEST_ENVIRONMENT_VARIABLE2: 'test environment variable 2 value',
                },
            },
            desiredCount: 2,
        });
        // THEN - stack contains a load balancer and a service
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Environment: [
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE1',
                            Value: 'test environment variable 1 value',
                        },
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE2',
                            Value: 'test environment variable 2 value',
                        },
                    ],
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': { Ref: 'ServiceTaskDefwebLogGroup2A898F61' },
                            'awslogs-stream-prefix': 'Service',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                },
            ],
        }));
        test.done();
    },
    'test Fargate loadbalanced construct with TLS'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        const zone = new aws_route53_1.PublicHostedZone(stack, 'HostedZone', { zoneName: 'example.com' });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
            },
            domainName: 'api.example.com',
            domainZone: zone,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'helloworld'),
        });
        // THEN - stack contains a load balancer and a service
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 443,
            Protocol: 'HTTPS',
            Certificates: [{
                    CertificateArn: 'helloworld',
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            Port: 80,
            Protocol: 'HTTP',
            TargetType: 'ip',
            VpcId: {
                Ref: 'VPCB9E5F0B4',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
            DesiredCount: 1,
            LaunchType: 'FARGATE',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: 'api.example.com.',
            HostedZoneId: {
                Ref: 'HostedZoneDB99F866',
            },
            Type: 'A',
            AliasTarget: {
                HostedZoneId: { 'Fn::GetAtt': ['ServiceLBE9A1ADBC', 'CanonicalHostedZoneID'] },
                DNSName: { 'Fn::Join': ['', ['dualstack.', { 'Fn::GetAtt': ['ServiceLBE9A1ADBC', 'DNSName'] }]] },
            },
        }));
        test.done();
    },
    'test Fargateloadbalanced construct with TLS and default certificate'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        const zone = new aws_route53_1.PublicHostedZone(stack, 'HostedZone', { zoneName: 'example.com' });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
            },
            domainName: 'api.example.com',
            domainZone: zone,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
        });
        // THEN - stack contains a load balancer, a service, and a certificate
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CertificateManager::Certificate', {
            DomainName: 'api.example.com',
            DomainValidationOptions: [
                {
                    DomainName: 'api.example.com',
                    HostedZoneId: {
                        Ref: 'HostedZoneDB99F866',
                    },
                },
            ],
            ValidationMethod: 'DNS',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 443,
            Protocol: 'HTTPS',
            Certificates: [{
                    CertificateArn: {
                        Ref: 'ServiceCertificateA7C65FE6',
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
            DesiredCount: 1,
            LaunchType: 'FARGATE',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: 'api.example.com.',
            HostedZoneId: {
                Ref: 'HostedZoneDB99F866',
            },
            Type: 'A',
            AliasTarget: {
                HostedZoneId: { 'Fn::GetAtt': ['ServiceLBE9A1ADBC', 'CanonicalHostedZoneID'] },
                DNSName: { 'Fn::Join': ['', ['dualstack.', { 'Fn::GetAtt': ['ServiceLBE9A1ADBC', 'DNSName'] }]] },
            },
        }));
        test.done();
    },
    'errors when setting domainName but not domainZone'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // THEN
        test.throws(() => {
            new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
                cluster,
                taskImageOptions: {
                    image: ecs.ContainerImage.fromRegistry('test'),
                },
                domainName: 'api.example.com',
            });
        });
        test.done();
    },
    'errors when setting both HTTP protocol and certificate'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // THEN
        test.throws(() => {
            new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
                cluster,
                taskImageOptions: {
                    image: ecs.ContainerImage.fromRegistry('test'),
                },
                protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'helloworld'),
            });
        });
        test.done();
    },
    'errors when setting both HTTP protocol and redirectHTTP'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // THEN
        test.throws(() => {
            new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
                cluster,
                taskImageOptions: {
                    image: ecs.ContainerImage.fromRegistry('test'),
                },
                protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                redirectHTTP: true,
            });
        });
        test.done();
    },
    'does not throw errors when not setting HTTPS protocol but certificate for redirectHTTP'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        const zone = new aws_route53_1.PublicHostedZone(stack, 'HostedZone', { zoneName: 'example.com' });
        // THEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
            },
            domainName: 'api.example.com',
            domainZone: zone,
            redirectHTTP: true,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'helloworld'),
        });
        test.done();
    },
    'errors when setting HTTPS protocol but not domain name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // THEN
        test.throws(() => {
            new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
                cluster,
                taskImageOptions: {
                    image: ecs.ContainerImage.fromRegistry('test'),
                },
                protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
            });
        });
        test.done();
    },
    'test Fargate loadbalanced construct with optional log driver input'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
                enableLogging: false,
                environment: {
                    TEST_ENVIRONMENT_VARIABLE1: 'test environment variable 1 value',
                    TEST_ENVIRONMENT_VARIABLE2: 'test environment variable 2 value',
                },
                logDriver: new ecs.AwsLogDriver({
                    streamPrefix: 'TestStream',
                }),
            },
            desiredCount: 2,
        });
        // THEN - stack contains a load balancer and a service
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Environment: [
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE1',
                            Value: 'test environment variable 1 value',
                        },
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE2',
                            Value: 'test environment variable 2 value',
                        },
                    ],
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': { Ref: 'ServiceTaskDefwebLogGroup2A898F61' },
                            'awslogs-stream-prefix': 'TestStream',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                },
            ],
        }));
        test.done();
    },
    'test Fargate loadbalanced construct with logging enabled'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
                enableLogging: true,
                environment: {
                    TEST_ENVIRONMENT_VARIABLE1: 'test environment variable 1 value',
                    TEST_ENVIRONMENT_VARIABLE2: 'test environment variable 2 value',
                },
            },
            desiredCount: 2,
        });
        // THEN - stack contains a load balancer and a service
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Environment: [
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE1',
                            Value: 'test environment variable 1 value',
                        },
                        {
                            Name: 'TEST_ENVIRONMENT_VARIABLE2',
                            Value: 'test environment variable 2 value',
                        },
                    ],
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': { Ref: 'ServiceTaskDefwebLogGroup2A898F61' },
                            'awslogs-stream-prefix': 'Service',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                },
            ],
        }));
        test.done();
    },
    'test Fargate loadbalanced construct with both image and taskDefinition provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
        taskDefinition.addContainer('web', {
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            memoryLimitMiB: 512,
        });
        // WHEN
        test.throws(() => new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
                enableLogging: true,
                environment: {
                    TEST_ENVIRONMENT_VARIABLE1: 'test environment variable 1 value',
                    TEST_ENVIRONMENT_VARIABLE2: 'test environment variable 2 value',
                },
            },
            desiredCount: 2,
            taskDefinition,
        }));
        test.done();
    },
    'test Fargate application loadbalanced construct with taskDefinition provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('passedTaskDef', {
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            memoryLimitMiB: 512,
        });
        container.addPortMappings({
            containerPort: 80,
        });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'Service', {
            cluster,
            taskDefinition,
            desiredCount: 2,
            memoryLimitMiB: 1024,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Image: 'amazon/amazon-ecs-sample',
                    Memory: 512,
                    Name: 'passedTaskDef',
                    PortMappings: [
                        {
                            ContainerPort: 80,
                            Protocol: 'tcp',
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'ALB - throws if desiredTaskCount is 0'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // THEN
        test.throws(() => new ecsPatterns.ApplicationLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            memoryLimitMiB: 1024,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
            },
            desiredCount: 0,
        }), /You must specify a desiredCount greater than 0/);
        test.done();
    },
    'NLB - throws if desiredTaskCount is 0'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // THEN
        test.throws(() => new ecsPatterns.NetworkLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            memoryLimitMiB: 1024,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
            },
            desiredCount: 0,
        }), /You must specify a desiredCount greater than 0/);
        test.done();
    },
    'ALBFargate - having *HealthyPercent properties'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedFargateService(stack, 'ALB123Service', {
            cluster,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            },
            minHealthyPercent: 100,
            maxHealthyPercent: 200,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            DeploymentConfiguration: {
                MinimumHealthyPercent: 100,
                MaximumPercent: 200,
            },
        }));
        test.done();
    },
    'NLBFargate - having *HealthyPercent properties'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedFargateService(stack, 'Service', {
            cluster,
            memoryLimitMiB: 1024,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            },
            desiredCount: 1,
            minHealthyPercent: 100,
            maxHealthyPercent: 200,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            DeploymentConfiguration: {
                MinimumHealthyPercent: 100,
                MaximumPercent: 200,
            },
        }));
        test.done();
    },
    'ALB - having *HealthyPercent properties'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            memoryLimitMiB: 1024,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            },
            desiredCount: 1,
            minHealthyPercent: 100,
            maxHealthyPercent: 200,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            DeploymentConfiguration: {
                MinimumHealthyPercent: 100,
                MaximumPercent: 200,
            },
        }));
        test.done();
    },
    'NLB - having *HealthyPercent properties'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            memoryLimitMiB: 1024,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            },
            desiredCount: 1,
            minHealthyPercent: 100,
            maxHealthyPercent: 200,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            DeploymentConfiguration: {
                MinimumHealthyPercent: 100,
                MaximumPercent: 200,
            },
        }));
        test.done();
    },
    'NetworkLoadbalancedEC2Service accepts previously created load balancer'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc, clusterName: 'MyCluster' });
        cluster.addCapacity('Capacity', { instanceType: new ec2.InstanceType('t2.micro') });
        const nlb = new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(stack, 'NLB', { vpc });
        const taskDef = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const container = taskDef.addContainer('Container', {
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            memoryLimitMiB: 1024,
        });
        container.addPortMappings({ containerPort: 80 });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            loadBalancer: nlb,
            taskDefinition: taskDef,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            LaunchType: 'EC2',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Type: 'network',
        }));
        test.done();
    },
    'NetworkLoadBalancedEC2Service accepts imported load balancer'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const nlbArn = 'arn:aws:elasticloadbalancing::000000000000::dummyloadbalancer';
        const vpc = new ec2.Vpc(stack, 'Vpc');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc, clusterName: 'MyCluster' });
        cluster.addCapacity('Capacity', { instanceType: new ec2.InstanceType('t2.micro') });
        const nlb = aws_elasticloadbalancingv2_1.NetworkLoadBalancer.fromNetworkLoadBalancerAttributes(stack, 'NLB', {
            loadBalancerArn: nlbArn,
            vpc,
        });
        const taskDef = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const container = taskDef.addContainer('Container', {
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            memoryLimitMiB: 1024,
        });
        container.addPortMappings({
            containerPort: 80,
        });
        // WHEN
        new ecsPatterns.NetworkLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            loadBalancer: nlb,
            desiredCount: 1,
            taskDefinition: taskDef,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            LaunchType: 'EC2',
            LoadBalancers: [{ ContainerName: 'Container', ContainerPort: 80 }],
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            LoadBalancerArn: nlb.loadBalancerArn,
            Port: 80,
        }));
        test.done();
    },
    'ApplicationLoadBalancedEC2Service accepts previously created load balancer'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc, clusterName: 'MyCluster' });
        cluster.addCapacity('Capacity', { instanceType: new ec2.InstanceType('t2.micro') });
        const sg = new ec2.SecurityGroup(stack, 'SG', { vpc });
        const alb = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(stack, 'NLB', {
            vpc,
            securityGroup: sg,
        });
        const taskDef = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const container = taskDef.addContainer('Container', {
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            memoryLimitMiB: 1024,
        });
        container.addPortMappings({ containerPort: 80 });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            loadBalancer: alb,
            taskDefinition: taskDef,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            LaunchType: 'EC2',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Type: 'application',
        }));
        test.done();
    },
    'ApplicationLoadBalancedEC2Service accepts imported load balancer'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const albArn = 'arn:aws:elasticloadbalancing::000000000000::dummyloadbalancer';
        const vpc = new ec2.Vpc(stack, 'Vpc');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc, clusterName: 'MyCluster' });
        cluster.addCapacity('Capacity', { instanceType: new ec2.InstanceType('t2.micro') });
        const sg = new ec2.SecurityGroup(stack, 'SG', { vpc });
        const alb = aws_elasticloadbalancingv2_1.ApplicationLoadBalancer.fromApplicationLoadBalancerAttributes(stack, 'ALB', {
            loadBalancerArn: albArn,
            vpc,
            securityGroupId: sg.securityGroupId,
            loadBalancerDnsName: 'MyName',
        });
        const taskDef = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const container = taskDef.addContainer('Container', {
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            memoryLimitMiB: 1024,
        });
        container.addPortMappings({
            containerPort: 80,
        });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            loadBalancer: alb,
            taskDefinition: taskDef,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
            LaunchType: 'EC2',
            LoadBalancers: [{ ContainerName: 'Container', ContainerPort: 80 }],
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            LoadBalancerArn: alb.loadBalancerArn,
            Port: 80,
        }));
        test.done();
    },
    'test ECS loadbalanced construct default/open security group'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            memoryReservationMiB: 1024,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
            },
        });
        // THEN - Stack contains no ingress security group rules
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: [{
                    CidrIp: '0.0.0.0/0',
                    FromPort: 80,
                    IpProtocol: 'tcp',
                    ToPort: 80,
                }],
        }));
        test.done();
    },
    'test ECS loadbalanced construct closed security group'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        // WHEN
        new ecsPatterns.ApplicationLoadBalancedEc2Service(stack, 'Service', {
            cluster,
            memoryReservationMiB: 1024,
            taskImageOptions: {
                image: ecs.ContainerImage.fromRegistry('test'),
            },
            openListener: false,
        });
        // THEN - Stack contains no ingress security group rules
        assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: assert_1.arrayWith(assert_1.objectLike({})),
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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