"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DomainName = exports.SecurityPolicy = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const base_path_mapping_1 = require("./base-path-mapping");
const restapi_1 = require("./restapi");
/**
 * The minimum version of the SSL protocol that you want API Gateway to use for HTTPS connections.
 *
 * @stability stable
 */
var SecurityPolicy;
(function (SecurityPolicy) {
    SecurityPolicy["TLS_1_0"] = "TLS_1_0";
    SecurityPolicy["TLS_1_2"] = "TLS_1_2";
})(SecurityPolicy = exports.SecurityPolicy || (exports.SecurityPolicy = {}));
/**
 * @stability stable
 */
class DomainName extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const endpointType = props.endpointType || restapi_1.EndpointType.REGIONAL;
        const edge = endpointType === restapi_1.EndpointType.EDGE;
        if (!core_1.Token.isUnresolved(props.domainName) && /[A-Z]/.test(props.domainName)) {
            throw new Error('domainName does not support uppercase letters. ' +
                `got: '${props.domainName}'`);
        }
        const mtlsConfig = this.configureMTLS(props.mtls);
        const resource = new apigateway_generated_1.CfnDomainName(this, 'Resource', {
            domainName: props.domainName,
            certificateArn: edge ? props.certificate.certificateArn : undefined,
            regionalCertificateArn: edge ? undefined : props.certificate.certificateArn,
            endpointConfiguration: { types: [endpointType] },
            mutualTlsAuthentication: mtlsConfig,
            securityPolicy: props.securityPolicy,
        });
        this.domainName = resource.ref;
        this.domainNameAliasDomainName = edge
            ? resource.attrDistributionDomainName
            : resource.attrRegionalDomainName;
        this.domainNameAliasHostedZoneId = edge
            ? resource.attrDistributionHostedZoneId
            : resource.attrRegionalHostedZoneId;
        if (props.mapping) {
            this.addBasePathMapping(props.mapping);
        }
    }
    /**
     * Imports an existing domain name.
     *
     * @stability stable
     */
    static fromDomainNameAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = attrs.domainName;
                this.domainNameAliasDomainName = attrs.domainNameAliasTarget;
                this.domainNameAliasHostedZoneId = attrs.domainNameAliasHostedZoneId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Maps this domain to an API endpoint.
     *
     * @param targetApi That target API endpoint, requests will be mapped to the deployment stage.
     * @param options Options for mapping to base path with or without a stage.
     * @stability stable
     */
    addBasePathMapping(targetApi, options = {}) {
        const basePath = options.basePath || '/';
        const id = `Map:${basePath}=>${core_1.Names.nodeUniqueId(targetApi.node)}`;
        return new base_path_mapping_1.BasePathMapping(this, id, {
            domainName: this,
            restApi: targetApi,
            ...options,
        });
    }
    configureMTLS(mtlsConfig) {
        if (!mtlsConfig)
            return undefined;
        return {
            truststoreUri: mtlsConfig.bucket.s3UrlForObject(mtlsConfig.key),
            truststoreVersion: mtlsConfig.version,
        };
    }
}
exports.DomainName = DomainName;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZG9tYWluLW5hbWUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJkb21haW4tbmFtZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFFQSx3Q0FBa0U7QUFFbEUsaUVBQXVEO0FBQ3ZELDJEQUE4RTtBQUM5RSx1Q0FBbUQ7Ozs7OztBQUtuRCxJQUFZLGNBS1g7QUFMRCxXQUFZLGNBQWM7SUFFeEIscUNBQW1CLENBQUE7SUFFbkIscUNBQW1CLENBQUE7QUFDckIsQ0FBQyxFQUxXLGNBQWMsR0FBZCxzQkFBYyxLQUFkLHNCQUFjLFFBS3pCOzs7O0FBd0VELE1BQWEsVUFBVyxTQUFRLGVBQVE7Ozs7SUFtQnRDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0I7UUFDOUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLHNCQUFZLENBQUMsUUFBUSxDQUFDO1FBQ2pFLE1BQU0sSUFBSSxHQUFHLFlBQVksS0FBSyxzQkFBWSxDQUFDLElBQUksQ0FBQztRQUVoRCxJQUFJLENBQUMsWUFBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDM0UsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQ7Z0JBQy9ELFNBQVMsS0FBSyxDQUFDLFVBQVUsR0FBRyxDQUFDLENBQUM7U0FDakM7UUFDRCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNsRCxNQUFNLFFBQVEsR0FBRyxJQUFJLG9DQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNuRCxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIsY0FBYyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDbkUsc0JBQXNCLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsY0FBYztZQUMzRSxxQkFBcUIsRUFBRSxFQUFFLEtBQUssRUFBRSxDQUFDLFlBQVksQ0FBQyxFQUFFO1lBQ2hELHVCQUF1QixFQUFFLFVBQVU7WUFDbkMsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO1NBQ3JDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxVQUFVLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUUvQixJQUFJLENBQUMseUJBQXlCLEdBQUcsSUFBSTtZQUNuQyxDQUFDLENBQUMsUUFBUSxDQUFDLDBCQUEwQjtZQUNyQyxDQUFDLENBQUMsUUFBUSxDQUFDLHNCQUFzQixDQUFDO1FBRXBDLElBQUksQ0FBQywyQkFBMkIsR0FBRyxJQUFJO1lBQ3JDLENBQUMsQ0FBQyxRQUFRLENBQUMsNEJBQTRCO1lBQ3ZDLENBQUMsQ0FBQyxRQUFRLENBQUMsd0JBQXdCLENBQUM7UUFFdEMsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFO1lBQ2pCLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7U0FDeEM7SUFDSCxDQUFDOzs7Ozs7SUEvQ00sTUFBTSxDQUFDLHdCQUF3QixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTJCO1FBQzlGLE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQixlQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztnQkFDOUIsOEJBQXlCLEdBQUcsS0FBSyxDQUFDLHFCQUFxQixDQUFDO2dCQUN4RCxnQ0FBMkIsR0FBRyxLQUFLLENBQUMsMkJBQTJCLENBQUM7WUFDbEYsQ0FBQztTQUFBO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDL0IsQ0FBQzs7Ozs7Ozs7SUE4Q00sa0JBQWtCLENBQUMsU0FBbUIsRUFBRSxVQUFrQyxFQUFHO1FBQ2xGLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxRQUFRLElBQUksR0FBRyxDQUFDO1FBQ3pDLE1BQU0sRUFBRSxHQUFHLE9BQU8sUUFBUSxLQUFLLFlBQUssQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7UUFDcEUsT0FBTyxJQUFJLG1DQUFlLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUNuQyxVQUFVLEVBQUUsSUFBSTtZQUNoQixPQUFPLEVBQUUsU0FBUztZQUNsQixHQUFHLE9BQU87U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sYUFBYSxDQUFDLFVBQXVCO1FBQzNDLElBQUksQ0FBQyxVQUFVO1lBQUUsT0FBTyxTQUFTLENBQUM7UUFDbEMsT0FBTztZQUNMLGFBQWEsRUFBRSxVQUFVLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDO1lBQy9ELGlCQUFpQixFQUFFLFVBQVUsQ0FBQyxPQUFPO1NBQ3RDLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUE1RUQsZ0NBNEVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgYWNtIGZyb20gJ0Bhd3MtY2RrL2F3cy1jZXJ0aWZpY2F0ZW1hbmFnZXInO1xuaW1wb3J0IHsgSUJ1Y2tldCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgeyBJUmVzb3VyY2UsIE5hbWVzLCBSZXNvdXJjZSwgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuRG9tYWluTmFtZSB9IGZyb20gJy4vYXBpZ2F0ZXdheS5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgQmFzZVBhdGhNYXBwaW5nLCBCYXNlUGF0aE1hcHBpbmdPcHRpb25zIH0gZnJvbSAnLi9iYXNlLXBhdGgtbWFwcGluZyc7XG5pbXBvcnQgeyBFbmRwb2ludFR5cGUsIElSZXN0QXBpIH0gZnJvbSAnLi9yZXN0YXBpJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gU2VjdXJpdHlQb2xpY3kge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgVExTXzFfMCA9ICdUTFNfMV8wJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFRMU18xXzIgPSAnVExTXzFfMidcbn1cblxuZXhwb3J0IGludGVyZmFjZSBEb21haW5OYW1lT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNlcnRpZmljYXRlOiBhY20uSUNlcnRpZmljYXRlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5kcG9pbnRUeXBlPzogRW5kcG9pbnRUeXBlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VjdXJpdHlQb2xpY3k/OiBTZWN1cml0eVBvbGljeVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbXRscz86IE1UTFNDb25maWdcbn1cblxuZXhwb3J0IGludGVyZmFjZSBEb21haW5OYW1lUHJvcHMgZXh0ZW5kcyBEb21haW5OYW1lT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1hcHBpbmc/OiBJUmVzdEFwaTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBJRG9tYWluTmFtZSBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWVBbGlhc0RvbWFpbk5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWVBbGlhc0hvc3RlZFpvbmVJZDogc3RyaW5nO1xuXG59XG5cbmV4cG9ydCBjbGFzcyBEb21haW5OYW1lIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJRG9tYWluTmFtZSB7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tRG9tYWluTmFtZUF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IERvbWFpbk5hbWVBdHRyaWJ1dGVzKTogSURvbWFpbk5hbWUge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSURvbWFpbk5hbWUge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGRvbWFpbk5hbWUgPSBhdHRycy5kb21haW5OYW1lO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGRvbWFpbk5hbWVBbGlhc0RvbWFpbk5hbWUgPSBhdHRycy5kb21haW5OYW1lQWxpYXNUYXJnZXQ7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluTmFtZUFsaWFzSG9zdGVkWm9uZUlkID0gYXR0cnMuZG9tYWluTmFtZUFsaWFzSG9zdGVkWm9uZUlkO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluTmFtZUFsaWFzRG9tYWluTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluTmFtZUFsaWFzSG9zdGVkWm9uZUlkOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IERvbWFpbk5hbWVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBlbmRwb2ludFR5cGUgPSBwcm9wcy5lbmRwb2ludFR5cGUgfHwgRW5kcG9pbnRUeXBlLlJFR0lPTkFMO1xuICAgIGNvbnN0IGVkZ2UgPSBlbmRwb2ludFR5cGUgPT09IEVuZHBvaW50VHlwZS5FREdFO1xuXG4gICAgaWYgKCFUb2tlbi5pc1VucmVzb2x2ZWQocHJvcHMuZG9tYWluTmFtZSkgJiYgL1tBLVpdLy50ZXN0KHByb3BzLmRvbWFpbk5hbWUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2RvbWFpbk5hbWUgZG9lcyBub3Qgc3VwcG9ydCB1cHBlcmNhc2UgbGV0dGVycy4gJyArXG4gICAgICAgIGBnb3Q6ICcke3Byb3BzLmRvbWFpbk5hbWV9J2ApO1xuICAgIH1cbiAgICBjb25zdCBtdGxzQ29uZmlnID0gdGhpcy5jb25maWd1cmVNVExTKHByb3BzLm10bHMpO1xuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmbkRvbWFpbk5hbWUodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgZG9tYWluTmFtZTogcHJvcHMuZG9tYWluTmFtZSxcbiAgICAgIGNlcnRpZmljYXRlQXJuOiBlZGdlID8gcHJvcHMuY2VydGlmaWNhdGUuY2VydGlmaWNhdGVBcm4gOiB1bmRlZmluZWQsXG4gICAgICByZWdpb25hbENlcnRpZmljYXRlQXJuOiBlZGdlID8gdW5kZWZpbmVkIDogcHJvcHMuY2VydGlmaWNhdGUuY2VydGlmaWNhdGVBcm4sXG4gICAgICBlbmRwb2ludENvbmZpZ3VyYXRpb246IHsgdHlwZXM6IFtlbmRwb2ludFR5cGVdIH0sXG4gICAgICBtdXR1YWxUbHNBdXRoZW50aWNhdGlvbjogbXRsc0NvbmZpZyxcbiAgICAgIHNlY3VyaXR5UG9saWN5OiBwcm9wcy5zZWN1cml0eVBvbGljeSxcbiAgICB9KTtcblxuICAgIHRoaXMuZG9tYWluTmFtZSA9IHJlc291cmNlLnJlZjtcblxuICAgIHRoaXMuZG9tYWluTmFtZUFsaWFzRG9tYWluTmFtZSA9IGVkZ2VcbiAgICAgID8gcmVzb3VyY2UuYXR0ckRpc3RyaWJ1dGlvbkRvbWFpbk5hbWVcbiAgICAgIDogcmVzb3VyY2UuYXR0clJlZ2lvbmFsRG9tYWluTmFtZTtcblxuICAgIHRoaXMuZG9tYWluTmFtZUFsaWFzSG9zdGVkWm9uZUlkID0gZWRnZVxuICAgICAgPyByZXNvdXJjZS5hdHRyRGlzdHJpYnV0aW9uSG9zdGVkWm9uZUlkXG4gICAgICA6IHJlc291cmNlLmF0dHJSZWdpb25hbEhvc3RlZFpvbmVJZDtcblxuICAgIGlmIChwcm9wcy5tYXBwaW5nKSB7XG4gICAgICB0aGlzLmFkZEJhc2VQYXRoTWFwcGluZyhwcm9wcy5tYXBwaW5nKTtcbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZEJhc2VQYXRoTWFwcGluZyh0YXJnZXRBcGk6IElSZXN0QXBpLCBvcHRpb25zOiBCYXNlUGF0aE1hcHBpbmdPcHRpb25zID0geyB9KSB7XG4gICAgY29uc3QgYmFzZVBhdGggPSBvcHRpb25zLmJhc2VQYXRoIHx8ICcvJztcbiAgICBjb25zdCBpZCA9IGBNYXA6JHtiYXNlUGF0aH09PiR7TmFtZXMubm9kZVVuaXF1ZUlkKHRhcmdldEFwaS5ub2RlKX1gO1xuICAgIHJldHVybiBuZXcgQmFzZVBhdGhNYXBwaW5nKHRoaXMsIGlkLCB7XG4gICAgICBkb21haW5OYW1lOiB0aGlzLFxuICAgICAgcmVzdEFwaTogdGFyZ2V0QXBpLFxuICAgICAgLi4ub3B0aW9ucyxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgY29uZmlndXJlTVRMUyhtdGxzQ29uZmlnPzogTVRMU0NvbmZpZyk6IENmbkRvbWFpbk5hbWUuTXV0dWFsVGxzQXV0aGVudGljYXRpb25Qcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKCFtdGxzQ29uZmlnKSByZXR1cm4gdW5kZWZpbmVkO1xuICAgIHJldHVybiB7XG4gICAgICB0cnVzdHN0b3JlVXJpOiBtdGxzQ29uZmlnLmJ1Y2tldC5zM1VybEZvck9iamVjdChtdGxzQ29uZmlnLmtleSksXG4gICAgICB0cnVzdHN0b3JlVmVyc2lvbjogbXRsc0NvbmZpZy52ZXJzaW9uLFxuICAgIH07XG4gIH1cbn1cblxuZXhwb3J0IGludGVyZmFjZSBEb21haW5OYW1lQXR0cmlidXRlcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkb21haW5OYW1lQWxpYXNUYXJnZXQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWVBbGlhc0hvc3RlZFpvbmVJZDogc3RyaW5nO1xuXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIE1UTFNDb25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYnVja2V0OiBJQnVja2V0O1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBrZXk6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZlcnNpb24/OiBzdHJpbmc7XG59XG4iXX0=