"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcLink = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
/**
 * Define a new VPC Link Specifies an API Gateway VPC link for a RestApi to access resources in an Amazon Virtual Private Cloud (VPC).
 *
 * @stability stable
 */
class VpcLink extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.vpcLinkName ||
                core_1.Lazy.string({ produce: () => core_1.Names.nodeUniqueId(this.node) }),
        });
        this._targets = new Array();
        const cfnResource = new apigateway_generated_1.CfnVpcLink(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            targetArns: core_1.Lazy.list({ produce: () => this.renderTargets() }),
        });
        this.vpcLinkId = cfnResource.ref;
        if (props.targets) {
            this.addTargets(...props.targets);
        }
    }
    /**
     * Import a VPC Link by its Id.
     *
     * @stability stable
     */
    static fromVpcLinkId(scope, id, vpcLinkId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcLinkId = vpcLinkId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * @stability stable
     */
    addTargets(...targets) {
        this._targets.push(...targets);
    }
    /**
     * Return the list of DNS names from the target NLBs.
     * @internal
     * */
    get _targetDnsNames() {
        return this._targets.map(t => t.loadBalancerDnsName);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        if (this._targets.length === 0) {
            return ['No targets added to vpc link'];
        }
        return [];
    }
    renderTargets() {
        return this._targets.map(nlb => nlb.loadBalancerArn);
    }
}
exports.VpcLink = VpcLink;
//# sourceMappingURL=data:application/json;base64,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