"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Archiver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib/");
const aws_cdk_lib_1 = require("aws-cdk-lib/");
const codebuild = require("aws-cdk-lib/aws-codebuild");
const s3 = require("aws-cdk-lib/aws-s3");
const snsNotifications = require("aws-cdk-lib/aws-s3-notifications");
const sns = require("aws-cdk-lib/aws-sns");
const constructs_1 = require("constructs");
class Archiver extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        const topic = new sns.Topic(this, 'notifications', {
            displayName: 'archiver-notifications',
        });
        const bucket = this.createArchiveBucket();
        bucket.addEventNotification(s3.EventType.OBJECT_CREATED, new snsNotifications.SnsDestination(topic));
        this.createProjects(bucket);
    }
    /**
     *Create the S3 bucket that will later store the repositories.
     *
     * @private
     * @return {*}
     * @memberof Archiver
     */
    createArchiveBucket() {
        return new s3.Bucket(this, 'destination', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
            lifecycleRules: [
                {
                    expiration: cdk.Duration.days(360),
                    transitions: [
                        {
                            storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: cdk.Duration.days(30),
                        },
                        {
                            storageClass: s3.StorageClass.GLACIER,
                            transitionAfter: cdk.Duration.days(90),
                        },
                        {
                            storageClass: s3.StorageClass.DEEP_ARCHIVE,
                            transitionAfter: cdk.Duration.days(180),
                        },
                    ],
                },
            ],
            publicReadAccess: false,
            versioned: true,
        });
    }
    /**
     * Creates for each backup configuration a separate CodeBuild project
     *
     * @private
     * @param {s3.Bucket} bucket
     * @memberof Archiver
     */
    createProjects(bucket) {
        this.props.backupConfigurations.forEach((element) => {
            const project = this.createProject(element, bucket);
            project.enableBatchBuilds();
            bucket.grantReadWrite(project);
        });
    }
    /**
     * Create a CodeBuild project
     *
     * @private
     * @param {BackupConfiguration} element
     * @param {cdk.aws_s3.Bucket} bucket
     * @return {*}
     * @memberof Archiver
     */
    createProject(element, bucket) {
        return new codebuild.Project(this, 'archiver-' +
            element.organizationName +
            '-' +
            element.projectName, {
            timeout: cdk.Duration.hours(5),
            projectName: 'AzureDevOpsGitBackup' +
                '-' +
                element.organizationName +
                '-' +
                element.projectName,
            description: 'Backup Azure DevOps git repositories to an S3 bucket.',
            checkSecretsInPlainTextEnvVariables: true,
            concurrentBuildLimit: 90,
            environment: {
                environmentVariables: {
                    TOKEN: {
                        value: element.secretArn + ':pat',
                        type: codebuild.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    },
                    ORGANIZATION: { value: element.organizationName },
                    PROJECT: { value: element.projectName },
                },
                buildImage: codebuild.LinuxBuildImage.STANDARD_6_0,
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: 0.2,
                batch: {
                    'fail-fast': false,
                    'build-list': this.createBatchConfiguration(element.repositoryNames),
                },
                phases: {
                    build: {
                        commands: [
                            'git clone --mirror "https://${TOKEN}@dev.azure.com/${ORGANIZATION}/${PROJECT}/_git/${REPOSITORY}"',
                            'tar czf ${REPOSITORY}.tgz ./${REPOSITORY}.git',
                            'aws s3 cp ./${REPOSITORY}.tgz ' +
                                bucket.s3UrlForObject() +
                                '/${ORGANIZATION}/${PROJECT}/${REPOSITORY}.tgz',
                        ],
                    },
                },
            }),
        });
    }
    createBatchConfiguration(repositoryNames) {
        const output = [];
        repositoryNames.forEach((element) => {
            output.push({
                identifier: 'build_' + element.replace(/-/g, '_'),
                env: {
                    variables: {
                        REPOSITORY: element,
                    },
                },
            });
        });
        return output;
    }
}
exports.Archiver = Archiver;
_a = JSII_RTTI_SYMBOL_1;
Archiver[_a] = { fqn: "azure-devops-repository-archiver.Archiver", version: "0.0.4" };
//# sourceMappingURL=data:application/json;base64,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