# SPDX-License-Identifier: GPL-2.0-or-later OR AGPL-3.0-or-later OR CERN-OHL-S-2.0+
import os
from textwrap import dedent
from c4m.pdk import freepdk45


def gen_files_var(*, rootdir, dirname, suffix):
    udirname = dirname.upper()

    def gen_lib_files_var(*, lib):
        varname = f"{lib.name}_{udirname}_FILES"

        files = f"export {varname} := \\\n"
        files += "".join(
            f"  {rootdir}/{lib.name}/{dirname}/{cell.name}.{suffix} \\\n"
            for cell in lib.cells
        )
        files += f"# {varname} end\n"

        return files

    return (
        "".join(gen_lib_files_var(lib=lib) for lib in freepdk45.__libs__)
        + f"export {udirname}_FILES := "
        + " ".join(
            f"$({lib.name}_{udirname}_FILES)" for lib in freepdk45.__libs__
        )
    )

lib_cells = "".join(
    (
        f"export {lib.name}_CELLS := \\\n"
        + "".join(f"  {cell.name} \\\n" for cell in lib.cells)
        + f"# {lib.name}_CELLS end"
    ) for lib in freepdk45.__libs__
)

spice_files = gen_files_var(rootdir="$(VIEWS_DIR)", dirname="spice", suffix="spi")
gds_files = gen_files_var(rootdir="$(VIEWS_DIR)", dirname="gds", suffix="gds")
verilog_files = (
    gen_files_var(rootdir="$(VIEWS_DIR)", dirname="verilog", suffix="v") + "\n"
    + "\n".join(
        f"$({lib.name}_VERILOG_FILES): $({lib.name}_SPICE_FILES)"
        for lib in freepdk45.__libs__
    )
)
vhdl_files = (
    gen_files_var(rootdir="$(VIEWS_DIR)", dirname="vhdl", suffix="vhdl") + "\n"
    + "\n".join(
        f"$({lib.name}_VHDL_FILES): $({lib.name}_SPICE_FILES)"
        for lib in freepdk45.__libs__
    )
)

def gen_verif_var(dirname, suffix, spice_dep):
    def gen_verif_deps(lib):
        s_cells = []
        for cell in lib.cells:
            verif_file = f"$(VERIFICATION_DIR)/{lib.name}/{dirname}/{cell.name}.{suffix}"
            gds_file = f"$(VIEWS_DIR)/{lib.name}/gds/{cell.name}.gds"
            s_cells.append(f"\n{verif_file}: {gds_file}")
            if spice_dep:
                spice_file = f"$(VIEWS_DIR)/{lib.name}/spice/{cell.name}.spi"
                s_cells.append(f"\n{verif_file}: {spice_file}")
        return "".join(s_cells)

    return (
        gen_files_var(rootdir="$(VERIFICATION_DIR)", dirname=dirname, suffix=suffix)
        + "\n".join(gen_verif_deps(lib) for lib in freepdk45.__libs__)
    )

drc_files = gen_verif_var(dirname="drc", suffix="drc.out", spice_dep=False)
lvs_files = gen_verif_var(dirname="lvs", suffix="lvs.out", spice_dep=True)

liberty_files = (
    "".join(
        dedent(f"""
            export {lib.name}_LIBERTY_FILES := \\
                $(VIEWS_DIR)/{lib.name}/liberty/{lib.name}_nom.lib \\
                $(VIEWS_DIR)/{lib.name}/liberty/{lib.name}_slow.lib \\
                $(VIEWS_DIR)/{lib.name}/liberty/{lib.name}_fast.lib
            $(VIEWS_DIR)/{lib.name}/liberty/{lib.name}_nom.lib: CORNER:=nom
            $(VIEWS_DIR)/{lib.name}/liberty/{lib.name}_slow.lib: CORNER:=slow
            $(VIEWS_DIR)/{lib.name}/liberty/{lib.name}_fast.lib: CORNER:=fast
            $({lib.name}_LIBERTY_FILES): LIB:={lib.name}
            $({lib.name}_LIBERTY_FILES): $({lib.name}_SPICE_FILES)
            $({lib.name}_LIBERTY_FILES): $({lib.name}_VHDL_FILES)
        """[1:])
        for lib in freepdk45.__libs__
    )
    + f"export LIBERTY_FILES := "
    + " ".join(f"$({lib.name}_LIBERTY_FILES)" for lib in freepdk45.__libs__)
)

with open(os.environ["LIBS_MAKEFILE"], "w") as f:
    f.write(dedent("""
        # Autogenerated file
        # SPDX-License-Identifier: GPL-2.0-or-later OR AGPL-3.0-or-later OR CERN-OHL-S-2.0+
        LIB_NAMES := {lib_names}
        {lib_cells}
        {spice_files}
        SPICE_DIRS := $(sort $(dir $(SPICE_FILES)))
        {gds_files}
        GDS_DIRS := $(sort $(dir $(GDS_FILES)))
        {verilog_files}
        VERILOG_DIRS := $(sort $(dir $(VERILOG_FILES)))
        {vhdl_files}
        VHDL_DIRS := $(sort $(dir $(VHDL_FILES)))
        {liberty_files}
        LIBERTY_DIRS := $(sort $(dir $(LIBERTY_FILES)))
        {drc_files}
        DRC_DIRS := $(sort $(dir $(DRC_FILES)))
        {lvs_files}
        LVS_DIRS := $(sort $(dir $(LVS_FILES)))

        $(SPICE_FILES): | $(SPICE_DIRS)
        $(GDS_FILES): | $(GDS_DIRS)
        $(VERILOG_FILES): | $(VERILOG_DIRS)
        $(VHDL_FILES): | $(VHDL_DIRS)
        $(LIBERTY_FILES): | $(LIBERTY_DIRS)
        $(DRC_FILES): | $(DRC_DIRS)
        $(LVS_FILES): | $(LVS_DIRS)
        MKDIRS += $(SPICE_DIRS) $(GDS_DIRS) $(VERILOG_DIRS) $(VHDL_DIRS) $(LIBERTY_DIRS) $(DRC_DIRS) $(LVS_DIRS)
    """[1:]).format(
        lib_names="" "".join(lib.name for lib in freepdk45.__libs__),
        lib_cells=lib_cells, spice_files=spice_files, gds_files=gds_files,
        verilog_files=verilog_files, vhdl_files=vhdl_files,
        liberty_files=liberty_files, drc_files=drc_files, lvs_files=lvs_files,
    ))
