"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
const subscription_1 = require("./subscription");
/**
 * Either a new or imported Topic
 */
class TopicBase extends core_1.Resource {
    /**
     * Subscribe some endpoint to this topic
     */
    addSubscription(subscription) {
        const subscriptionConfig = subscription.bind(this);
        const scope = subscriptionConfig.subscriberScope || this;
        const id = subscriptionConfig.subscriberId;
        // We use the subscriber's id as the construct id. There's no meaning
        // to subscribing the same subscriber twice on the same topic.
        if (scope.node.tryFindChild(id)) {
            throw new Error(`A subscription with id "${id}" already exists under the scope ${scope.node.path}`);
        }
        new subscription_1.Subscription(scope, id, Object.assign({ topic: this }, subscriptionConfig));
    }
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is improted (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.TopicPolicy(this, 'Policy', { topics: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
        }
    }
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['sns:Publish'],
            resourceArns: [this.topicArn],
            resource: this,
        });
    }
}
exports.TopicBase = TopicBase;
//# sourceMappingURL=data:application/json;base64,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