"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool Client', () => {
    test('default setup', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
        });
    });
    test('client name', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        const client1 = new lib_1.UserPoolClient(stack, 'Client1', {
            userPool: pool,
            userPoolClientName: 'myclient',
        });
        const client2 = new lib_1.UserPoolClient(stack, 'Client2', {
            userPool: pool,
        });
        // THEN
        expect(client1.userPoolClientName).toEqual('myclient');
        expect(() => client2.userPoolClientName).toThrow(/available only if specified on the UserPoolClient during initialization/);
    });
    test('import', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const client = lib_1.UserPoolClient.fromUserPoolClientId(stack, 'Client', 'client-id-1');
        // THEN
        expect(client.userPoolClientId).toEqual('client-id-1');
    });
    test('ExplicitAuthFlows is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ExplicitAuthFlows: assert_1.ABSENT,
        });
    });
    test('ExplicitAuthFlows are correctly named', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client', {
            authFlows: {
                adminUserPassword: true,
                custom: true,
                refreshToken: true,
                userPassword: true,
                userSrp: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ExplicitAuthFlows: [
                'ALLOW_USER_PASSWORD_AUTH',
                'ALLOW_ADMIN_USER_PASSWORD_AUTH',
                'ALLOW_CUSTOM_AUTH',
                'ALLOW_USER_SRP_AUTH',
                'ALLOW_REFRESH_TOKEN_AUTH',
            ],
        });
    });
    test('AllowedOAuthFlows is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: assert_1.ABSENT,
        });
    });
    test('AllowedOAuthFlows are correctly named', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client1', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                    implicitCodeGrant: true,
                },
                callbackUrls: ['redirect-url'],
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        pool.addClient('Client2', {
            oAuth: {
                flows: {
                    clientCredentials: true,
                },
                callbackUrls: ['redirect-url'],
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['implicit', 'code'],
            AllowedOAuthFlowsUserPoolClient: true,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['client_credentials'],
            AllowedOAuthFlowsUserPoolClient: true,
        });
    });
    test('fails when callbackUrls are not specified for codeGrant or implicitGrant', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addClient('Client1', {
            oAuth: {
                flows: { authorizationCodeGrant: true },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).toThrow(/callbackUrl must be specified/);
        expect(() => pool.addClient('Client2', {
            oAuth: {
                flows: { implicitCodeGrant: true },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).toThrow(/callbackUrl must be specified/);
        expect(() => pool.addClient('Client3', {
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).not.toThrow();
    });
    test('fails when clientCredentials OAuth flow is selected along with codeGrant or implicitGrant', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addClient('Client1', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                    clientCredentials: true,
                },
                callbackUrls: ['redirect-url'],
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).toThrow(/clientCredentials OAuth flow cannot be selected/);
        expect(() => pool.addClient('Client2', {
            oAuth: {
                flows: {
                    implicitCodeGrant: true,
                    clientCredentials: true,
                },
                callbackUrls: ['redirect-url'],
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).toThrow(/clientCredentials OAuth flow cannot be selected/);
    });
    test('OAuth scopes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client', {
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [
                    lib_1.OAuthScope.PHONE,
                    lib_1.OAuthScope.EMAIL,
                    lib_1.OAuthScope.OPENID,
                    lib_1.OAuthScope.PROFILE,
                    lib_1.OAuthScope.COGNITO_ADMIN,
                    lib_1.OAuthScope.custom('my-resource-server/my-own-scope'),
                ],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthScopes: [
                'phone',
                'email',
                'openid',
                'profile',
                'aws.cognito.signin.user.admin',
                'my-resource-server/my-own-scope',
            ],
        });
    });
    test('OAuthScope - openid is included when email or phone is specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client1', {
            userPoolClientName: 'Client1',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        pool.addClient('Client2', {
            userPoolClientName: 'Client2',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.EMAIL],
            },
        });
        pool.addClient('Client3', {
            userPoolClientName: 'Client3',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.PROFILE],
            },
        });
        pool.addClient('Client4', {
            userPoolClientName: 'Client4',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.COGNITO_ADMIN],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client1',
            AllowedOAuthScopes: ['phone', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client2',
            AllowedOAuthScopes: ['email', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client3',
            AllowedOAuthScopes: ['profile', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client4',
            AllowedOAuthScopes: ['aws.cognito.signin.user.admin'],
        });
    });
    test('enable user existence errors prevention', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
            preventUserExistenceErrors: true,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: 'ENABLED',
        });
    });
    test('disable user existence errors prevention', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
            preventUserExistenceErrors: false,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: 'LEGACY',
        });
    });
    test('user existence errors prevention is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: assert_1.ABSENT,
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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