import asyncio
import os

import async_timeout
import nextcord
from nextcord.ext import commands
from .ext import *
from wavelink import Player

from ._classes import Loop
from .errors import InvalidLoopMode, NotEnoughSong, NothingIsPlaying


class NextPlayer(Player):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.queue = asyncio.Queue()
        self.loop = Loop.NONE  # CURRENT, PLAYLIST
        self.bound_channel = None
        self.track_provider = "yt"

    async def destroy(self) -> None:
        self.queue = None

        await super().stop()
        await super().disconnect()

    async def do_next(self) -> None:
        if self.is_playing():
            return

        timeout = int(os.getenv("nextmusic_TIMEOUT", 300))

        try:
            with async_timeout.timeout(timeout):
                track = await self.queue.get()
        except asyncio.TimeoutError:
            if not self.is_playing():
                await self.destroy()

            return

        self._source = track
        await self.play(track)
        self.client.dispatch("nextmusic_track_start", self, track)
        await self.invoke_player()

    async def set_loop(self, loop_type: str) -> None:
        if not self.is_playing():
            raise NothingIsPlaying("Player is not playing any track. Can't loop")

        if not loop_type:
            if Loop.TYPES.index(self.loop) >= 2:
                loop_type = "NONE"
            else:
                loop_type = Loop.TYPES[Loop.TYPES.index(self.loop) + 1]

            if loop_type == "PLAYLIST" and len(self.queue._queue) < 1:
                loop_type = "NONE"

        if loop_type.upper() == "PLAYLIST" and len(self.queue._queue) < 1:
            raise NotEnoughSong(
                "There must be 2 songs in the queue in order to use the PLAYLIST loop"
            )

        if loop_type.upper() not in Loop.TYPES:
            raise InvalidLoopMode("Loop type must be `NONE`, `CURRENT` or `PLAYLIST`.")

        self.loop = loop_type.upper()

        return self.loop

    async def invoke_player(self, ctx: commands.Context = None) -> None:
        track = self.source

        if not track:
            raise NothingIsPlaying("Player is not playing anything.")

        embed = nextcord.Embed(description=f'[{track.title}]({track.uri})', colour=Commands.embed_colour(self))
        embed.set_author(name='Now playing', url=track.uri)
        try:
          embed.set_thumbnail(url=track.thumbnail)
        except:
          embed.set_thumbnail(url=self.client.user.display_avatar)
        embed.set_footer(text=f"Duration {int(track.length // 60)}:{int(track.length % 60)}")


        # embed = nextcord.Embed(
        #     title=track.title, url=track.uri, color=Commands.embed_colour(self)
        # )
        # embed.set_author(
        #     name=track.author,
        #     url=track.uri,
        #     icon_url=self.client.user.display_avatar.url,
        # )
        # try:
        #     embed.set_thumbnail(url=track.thumb)
        # except AttributeError:
        #     embed.set_thumbnail(
        #         url="https://cdn.nextcordapp.com/attachments/776345413132877854/940540758442795028/unknown.png"
        #     )
        # embed.add_field(
        #     name="Length",
        #     value=f"{int(track.length // 60)}:{int(track.length % 60)}",
        # )
        # embed.add_field(name="Looping", value=self.loop)
        # embed.add_field(name="Volume", value=self.volume)
        # next_song = ""

        # if self.loop == "CURRENT":
        #     next_song = self.source.title
        # else:
        #     if len(self.queue._queue) > 0:
        #         next_song = self.queue._queue[0].title

        # if next_song:
        #     pass
        # #     embed.add_field(name="Next Song", value=next_song, inline=False)

        if not ctx:
            return await self.bound_channel.send(embed=embed, view=MusicController(), delete_after=track.length)

        await ctx.send(embed=embed)
