# Copyright (c) HashiCorp, Inc.
# SPDX-License-Identifier: MPL-2.0

"""Google GSuite Activities connector for Grove."""

import json
from datetime import datetime, timedelta

import google_auth_httplib2
import httplib2
from google.auth.exceptions import GoogleAuthError
from google.oauth2 import service_account
from googleapiclient.discovery import build
from googleapiclient.errors import Error

from grove.connectors import BaseConnector
from grove.constants import REVERSE_CHRONOLOGICAL
from grove.exceptions import (
    ConfigurationException,
    NotFoundException,
    RequestFailedException,
)

# This connector is only interested in the GSuite Reports API.
SCOPES = ["https://www.googleapis.com/auth/admin.reports.audit.readonly"]

# Use consistent flags when formatting via isoformat() on datetime objects.
ISO_FORMAT_FLAGS = {"sep": "T", "timespec": "milliseconds"}


class Connector(BaseConnector):
    NAME = "gsuite_activities"
    POINTER_PATH = "id.time"
    LOG_ORDER = REVERSE_CHRONOLOGICAL

    def collect(self):
        """Collects activities from the Google GSuite Reports API.

        As the Google APIs use OAuth 2.0 2LO ('two-legged OAuth') which contains a
        number of fields inside of a JSON 'service account file' the key and identity
        are treated a little differently in this connector.

        Rather than the key being a single authentication token, the key should contain
        the entire 'service account file' in JSON format - as generated by the Google
        API console.

        The identity must be the name of a service account which has been granted domain
        wide delegation. Please see the following guides for more information:

          https://developers.google.com/admin-sdk/directory/v1/guides/delegation
          https://developers.google.com/admin-sdk/reports/reference/rest/v1/activities

        :raises RequestFailedException: An HTTP request failed.
        """
        cursor = str()
        http = google_auth_httplib2.AuthorizedHttp(
            self.get_credentials(),
            http=self.get_http_transport(),
        )

        # If no pointer is stored then a previous run hasn't been performed, so set the
        # pointer to a week ago. In the case of the GSuite reports API the pointer is
        # the value of the id.time field from the latest record retrieved from the API.
        try:
            _ = self.pointer
        except NotFoundException:
            # Sorry about the + 'Z' here. It turns out, if you pass the API a startTime
            # with a timezone in '+00:00' format, which is permitted according to API
            # validation, filtering seems to break and just return you ALL data
            # available. Cool.
            self.pointer = (datetime.utcnow() - timedelta(days=7)).isoformat(
                **ISO_FORMAT_FLAGS
            ) + "Z"

        # Page over all activities since the last pointer.
        more_requests = True

        with build("admin", "reports_v1", http=http) as service:
            while more_requests:
                self.logger.debug(
                    "Requesting GSuite activity list.",
                    extra={"operation": self.operation},
                )
                if cursor:
                    self.logger.debug(
                        "Using pageToken as cursor to request next page of results",
                        extra={"cursor": cursor},
                    )
                    request = service.activities().list(
                        userKey="all",
                        applicationName=self.operation,
                        startTime=self.pointer,
                        pageToken=cursor,
                    )
                else:
                    self.logger.debug(
                        "Using startTime from pointer", extra={"pointer": self.pointer}
                    )
                    request = service.activities().list(
                        userKey="all",
                        startTime=self.pointer,
                        applicationName=self.operation,
                    )

                # As both the _entries and activities objects are Lists, we can extend
                # them in order to support appending more data on any subsequent
                # iterations (due to paging).
                try:
                    results = request.execute()
                    activities = results.get("items", [])

                    self.logger.debug(
                        "Got activities from the GSuite API",
                        extra={"count": len(activities)},
                    )
                    self.save(activities)
                except (GoogleAuthError, Error) as err:
                    raise RequestFailedException(f"Request to GSuite API failed: {err}")

                # Determine whether we're still paging.
                if "nextPageToken" not in results:
                    self.logger.debug("No nextPageToken, finishing collection.")
                    more_requests = False
                    break

                self.logger.debug("nextPageToken is present in response, paging.")
                cursor = results["nextPageToken"]
                more_requests = True

    def get_http_transport(self):
        """Creates an HTTP object for use by the Google API Client.

        :return: An httplib2.Http object for use with the Google API client.
        """
        return httplib2.Http()

    def get_credentials(self):
        """Generates and returns a credentials instance from the connector's configured
        service account info. This is used for required to perform operations using the
        Google API client.

        :return: A credentials instance built from configured service account info.

        :raises ConfigurationException: There is an issue with the configuration
            for this connector.
        """
        try:
            service_account_info = json.loads(self.key)
        except json.JSONDecodeError as err:
            raise ConfigurationException(
                f"Unable to load service account JSON for {self.identity}: {err}"
            )

        # Construct the credentials, including scopes and delegation.
        try:
            credentials = service_account.Credentials.from_service_account_info(
                service_account_info,
                scopes=SCOPES,
                subject=self.identity,
            )
        except ValueError as err:
            raise ConfigurationException(
                "Unable to generate credentials from service account info for "
                f" {self.identity}: {err}"
            )

        return credentials
