"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcEndpointServiceDomainName = void 0;
const crypto = require("crypto");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const lib_1 = require("../lib");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A Private DNS configuration for a VPC endpoint service.
 *
 * @stability stable
 */
class VpcEndpointServiceDomainName extends core_2.Construct {
    // The way this class works is by using three custom resources and a TxtRecord in conjunction
    // The first custom resource tells the VPC endpoint service to use the given DNS name
    // The VPC endpoint service will then say:
    // "ok, create a TXT record using these two values to prove you own the domain"
    // The second custom resource retrieves these two values from the service
    // The TxtRecord is created from these two values
    // The third custom resource tells the VPC Endpoint Service to verify the domain ownership
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        const serviceId = props.endpointService.vpcEndpointServiceId;
        const privateDnsName = props.domainName;
        // Make sure a user doesn't accidentally add multiple domains
        this.validateProps(props);
        VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId] = privateDnsName;
        VpcEndpointServiceDomainName.endpointServices.push(props.endpointService);
        // Enable Private DNS on the endpoint service and retrieve the AWS-generated configuration
        const privateDnsConfiguration = this.getPrivateDnsConfiguration(serviceUniqueId, serviceId, privateDnsName);
        // Tell AWS to verify that this account owns the domain attached to the service
        this.verifyPrivateDnsConfiguration(privateDnsConfiguration, props.publicHostedZone);
        // Finally, don't do any of the above before the endpoint service is created
        this.node.addDependency(props.endpointService);
    }
    validateProps(props) {
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        if (serviceUniqueId in VpcEndpointServiceDomainName.endpointServicesMap) {
            const endpoint = VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId];
            throw new Error(`Cannot create a VpcEndpointServiceDomainName for service ${serviceUniqueId}, another VpcEndpointServiceDomainName (${endpoint}) is already associated with it`);
        }
    }
    /**
     * Sets up Custom Resources to make AWS calls to set up Private DNS on an endpoint service,
     * returning the values to use in a TxtRecord, which AWS uses to verify domain ownership.
     */
    getPrivateDnsConfiguration(serviceUniqueId, serviceId, privateDnsName) {
        // The custom resource which tells AWS to enable Private DNS on the given service, using the given domain name
        // AWS will generate a name/value pair for use in a TxtRecord, which is used to verify domain ownership.
        const enablePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                PrivateDnsName: privateDnsName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(serviceUniqueId),
        };
        const removePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                RemovePrivateDnsName: true,
            },
        };
        const enable = new custom_resources_1.AwsCustomResource(this, 'EnableDns', {
            onCreate: enablePrivateDnsAction,
            onUpdate: enablePrivateDnsAction,
            onDelete: removePrivateDnsAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            serviceId,
                        ]),
                    ]),
                ],
            }),
        });
        // Look up the name/value pair if the domain changes, or the service changes,
        // which would cause the values to be different. If the unique ID changes,
        // the resource may be entirely recreated, so we will need to look it up again.
        const lookup = hashcode(core_1.Names.uniqueId(this) + serviceUniqueId + privateDnsName);
        // Create the custom resource to look up the name/value pair generated by AWS
        // after the previous API call
        const retriveNameValuePairAction = {
            service: 'EC2',
            action: 'describeVpcEndpointServiceConfigurations',
            parameters: {
                ServiceIds: [serviceId],
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(lookup),
        };
        const getNames = new custom_resources_1.AwsCustomResource(this, 'GetNames', {
            onCreate: retriveNameValuePairAction,
            onUpdate: retriveNameValuePairAction,
            // describeVpcEndpointServiceConfigurations can't take an ARN for granular permissions
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // We only want to call and get the name/value pair after we've told AWS to enable Private DNS
        // If we call before then, we'll get an empty pair of values.
        getNames.node.addDependency(enable);
        // Get the references to the name/value pair associated with the endpoint service
        const name = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Name');
        const value = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Value');
        return { name, value, serviceId };
    }
    /**
     * Creates a Route53 entry and a Custom Resource which explicitly tells AWS to verify ownership
     * of the domain name attached to an endpoint service.
     */
    verifyPrivateDnsConfiguration(config, publicHostedZone) {
        // Create the TXT record in the provided hosted zone
        const verificationRecord = new lib_1.TxtRecord(this, 'DnsVerificationRecord', {
            recordName: config.name,
            values: [config.value],
            zone: publicHostedZone,
        });
        // Tell the endpoint service to verify the domain ownership
        const startVerificationAction = {
            service: 'EC2',
            action: 'startVpcEndpointServicePrivateDnsVerification',
            parameters: {
                ServiceId: config.serviceId,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(core_1.Fn.join(':', [config.name, config.value])),
        };
        const startVerification = new custom_resources_1.AwsCustomResource(this, 'StartVerification', {
            onCreate: startVerificationAction,
            onUpdate: startVerificationAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            config.serviceId,
                        ]),
                    ]),
                ],
            }),
        });
        // Only verify after the record has been created
        startVerification.node.addDependency(verificationRecord);
    }
}
exports.VpcEndpointServiceDomainName = VpcEndpointServiceDomainName;
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServices = [];
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServicesMap = {};
/**
 * Hash a string
 */
function hashcode(s) {
    const hash = crypto.createHash('md5');
    hash.update(s);
    return hash.digest('hex');
}
;
//# sourceMappingURL=data:application/json;base64,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