import pytest
from pyproj import CRS
from to4326.exceptions import *
import to4326.transform as transform


def test_transform():
    assert transform._transform(
        [[-10, -10], [10, -10], [10, 10], [-10, 10], [-10, -10]], 4326, 4326
    ) == [[-10, -10], [10, -10], [10, 10], [-10, 10], [-10, -10]]

    assert transform._transform(
        [
            [223069, -268407],
            [-275846, -495187],
        ],
        3411,
        4326,
    ) == [
        [-5.270541933285495, 86.77915026771586],
        [-74.12017710269978, 84.77098843082935],
    ]


def test_transform_enclosing_pole_ring():
    assert transform._transform_enclosing_pole_ring(
        [
            [-1096140.177763625513762, 633075.779011003673077],
            [-1156379.742859589401633, -520081.609968872275203],
            [919322.002944564330392, -628513.79071983625181],
            [979561.568040528334677, 524643.59826003969647],
            [-1096140.177763625513762, 633075.779011003673077],
        ],
        3411,
        partition=0,
        north=True,
    ) == [
        [-165.0085828152573, 78.3537208449233],
        [-110.78417499347034, 78.33427559463203],
        [10.640716221111093, 79.7462878857949],
        [73.17306663110058, 79.7684835157352],
        [180, 82.08068862192995],
        [180, 90],
        [-180, 90],
        [-180, 82.08068862192995],
        [-165.0085828152573, 78.3537208449233],
    ]

    assert transform._transform_enclosing_pole_ring(
        [
            [-1096140.177763625513762, 633075.779011003673077],
            [-1156379.742859589401633, -520081.609968872275203],
            [919322.002944564330392, -628513.79071983625181],
            [979561.568040528334677, 524643.59826003969647],
            [-1096140.177763625513762, 633075.779011003673077],
        ],
        3412,
        partition=0,
        north=False,
    ) == [
        [-59.9914171847427, -78.3537208449233],
        [-114.21582500652967, -78.33427559463203],
        [-180, -84.6451943499158],
        [-180, -90],
        [180, -90],
        [180, -84.6451943499158],
        [124.3592837788889, -79.7462878857949],
        [61.82693336889944, -79.7684835157352],
        [-59.9914171847427, -78.3537208449233],
    ]

    assert transform._transform_enclosing_pole_ring(
        [
            [-1096140.177763625513762, 633075.779011003673077],
            [979561.568040528334677, 524643.59826003969647],
            [919322.002944564330392, -628513.79071983625181],
            [-1156379.742859589401633, -520081.609968872275203],
            [-1096140.177763625513762, 633075.779011003673077],
        ],
        3411,
        partition=0,
        north=True,
    ) == [
        [-165.0085828152573, 78.3537208449233],
        [-180, 82.08068862192995],
        [-180, 90],
        [180, 90],
        [180, 82.08068862192995],
        [73.17306663110058, 79.7684835157352],
        [10.640716221111093, 79.7462878857949],
        [-110.78417499347034, 78.33427559463203],
        [-165.0085828152573, 78.3537208449233],
    ]
    assert transform._transform_enclosing_pole_ring(
        [
            [-1096140.177763625513762, 633075.779011003673077],
            [979561.568040528334677, 524643.59826003969647],
            [919322.002944564330392, -628513.79071983625181],
            [-1156379.742859589401633, -520081.609968872275203],
            [-1096140.177763625513762, 633075.779011003673077],
        ],
        3412,
        partition=0,
        north=False,
    ) == [
        [-59.9914171847427, -78.3537208449233],
        [61.82693336889944, -79.7684835157352],
        [124.3592837788889, -79.7462878857949],
        [180, -84.6451943499158],
        [180, -90],
        [-180, -90],
        [-180, -84.6451943499158],
        [-114.21582500652967, -78.33427559463203],
        [-59.9914171847427, -78.3537208449233],
    ]
    assert transform._transform_enclosing_pole_ring(
        [
            [-1096140.177763625513762, 633075.779011003673077],
            [-1156379.742859589401633, -520081.609968872275203],
            [919322.002944564330392, -628513.79071983625181],
            [979561.568040528334677, 524643.59826003969647],
            [-1096140.177763625513762, 633075.779011003673077],
        ],
        3411,
        partition=5,
        north=True,
    ) == [
        [-165.0085828152573, 78.3537208449233],
        [-156.7304154704545, 79.03978596130258],
        [-147.56016241240812, 79.47196311680196],
        [-137.871746756679, 79.61767780469414],
        [-128.1900442711871, 79.46477183460594],
        [-119.03782920095914, 79.02598714163038],
        [-110.78417499347034, 78.33427559463203],
        [-101.41444194424896, 81.03727076699988],
        [-84.86374614619683, 83.31790529580597],
        [-56.661492985017695, 84.59073393509807],
        [-23.997303500485867, 84.14752952709192],
        [-1.7935794108101304, 82.28025812231863],
        [10.640716221111093, 79.7462878857949],
        [19.85067874527038, 80.54320737156078],
        [30.43243897767354, 81.05772172159837],
        [41.869003138215085, 81.23893808369635],
        [53.316637650632835, 81.06622091117508],
        [63.92693309051733, 80.55927219284123],
        [73.17306663110058, 79.7684835157352],
        [81.9815223904838, 81.75083902053998],
        [95.79408023672825, 83.42764891884764],
        [116.81087569236769, 84.50349273924772],
        [142.71035788751252, 84.60173769508268],
        [165.01742621551023, 83.67635700611854],
        [180, 82.08068862192995],
        [180, 90],
        [-180, 90],
        [-180, 82.08068862192995],
        [-176.604555128726, 81.50928142471932],
        [-173.6474814909233, 80.91238836229148],
        [-171.06209142427136, 80.29488780447296],
        [-168.79122851403028, 79.66065491509582],
        [-166.7867724748905, 79.01277578737971],
        [-165.0085828152573, 78.3537208449233],
    ]


def test_transform_ring():
    assert transform.transform_ring(
        [
            [382200.000, 2512500.000],
            [382200.000, 2279400.000],
            [610500.000, 2279400.000],
            [610500.000, 2512500.000],
            [382200.000, 2512500.000],
        ],
        32645,
    ) == [
        [85.85296718933647, 22.71566587084141],
        [85.8694977275825, 20.610041795245515],
        [88.06045501053292, 20.610485722030123],
        [88.07596179098907, 22.71615986368381],
        [85.85296718933647, 22.71566587084141],
    ]

    assert transform.transform_ring(
        [
            [508800.000, 7247400.000],
            [508800.000, 7001100.000],
            [753000.000, 7001100.000],
            [753000.000, 7247400.000],
            [508800.000, 7247400.000],
        ],
        32660,
    ) == [
        [177.18908505580515, 65.34932256544839],
        [177.17456390562774, 63.13910500179646],
        [-177.99275205341496, 63.05068519969726],
        [-177.57860702499983, 65.25180997065199],
        [177.18908505580515, 65.34932256544839],
    ]

    assert transform.transform_ring(
        [[170, -10], [-170, -10], [-170, 10], [170, 10], [170, -10]], 4326
    ) == [[170, -10], [-170, -10], [-170, 10], [170, 10], [170, -10]]

    assert transform.transform_ring(
        [
            [-1096140.177763625513762, 633075.779011003673077],
            [-1156379.742859589401633, -520081.609968872275203],
            [919322.002944564330392, -628513.79071983625181],
            [979561.568040528334677, 524643.59826003969647],
            [-1096140.177763625513762, 633075.779011003673077],
        ],
        3411,
    ) == [
        [-165.0085828152573, 78.3537208449233],
        [-110.78417499347034, 78.33427559463203],
        [10.640716221111093, 79.7462878857949],
        [73.17306663110058, 79.7684835157352],
        [180, 82.08068862192995],
        [180, 90],
        [-180, 90],
        [-180, 82.08068862192995],
        [-165.0085828152573, 78.3537208449233],
    ]

    assert transform.transform_ring(
        [
            [-1096140.177763625513762, 633075.779011003673077],
            [-1156379.742859589401633, -520081.609968872275203],
            [919322.002944564330392, -628513.79071983625181],
            [979561.568040528334677, 524643.59826003969647],
            [-1096140.177763625513762, 633075.779011003673077],
        ],
        3412,
    ) == [
        [-59.9914171847427, -78.3537208449233],
        [-114.21582500652967, -78.33427559463203],
        [-180, -84.6451943499158],
        [-180, -90],
        [180, -90],
        [180, -84.6451943499158],
        [124.3592837788889, -79.7462878857949],
        [61.82693336889944, -79.7684835157352],
        [-59.9914171847427, -78.3537208449233],
    ]

    assert transform.transform_ring(
        [
            [-1096140.177763625513762, 633075.779011003673077],
            [979561.568040528334677, 524643.59826003969647],
            [919322.002944564330392, -628513.79071983625181],
            [-1156379.742859589401633, -520081.609968872275203],
            [-1096140.177763625513762, 633075.779011003673077],
        ],
        3411,
    ) == [
        [-165.0085828152573, 78.3537208449233],
        [-180, 82.08068862192995],
        [-180, 90],
        [180, 90],
        [180, 82.08068862192995],
        [73.17306663110058, 79.7684835157352],
        [10.640716221111093, 79.7462878857949],
        [-110.78417499347034, 78.33427559463203],
        [-165.0085828152573, 78.3537208449233],
    ]
    assert transform.transform_ring(
        [
            [-1096140.177763625513762, 633075.779011003673077],
            [979561.568040528334677, 524643.59826003969647],
            [919322.002944564330392, -628513.79071983625181],
            [-1156379.742859589401633, -520081.609968872275203],
            [-1096140.177763625513762, 633075.779011003673077],
        ],
        3412,
    ) == [
        [-59.9914171847427, -78.3537208449233],
        [61.82693336889944, -79.7684835157352],
        [124.3592837788889, -79.7462878857949],
        [180, -84.6451943499158],
        [180, -90],
        [-180, -90],
        [-180, -84.6451943499158],
        [-114.21582500652967, -78.33427559463203],
        [-59.9914171847427, -78.3537208449233],
    ]
    assert transform.transform_ring(
        [
            [-1096140.177763625513762, 633075.779011003673077],
            [-1156379.742859589401633, -520081.609968872275203],
            [919322.002944564330392, -628513.79071983625181],
            [979561.568040528334677, 524643.59826003969647],
            [-1096140.177763625513762, 633075.779011003673077],
        ],
        3411,
        partition=5,
    ) == [
        [-165.0085828152573, 78.3537208449233],
        [-156.7304154704545, 79.03978596130258],
        [-147.56016241240812, 79.47196311680196],
        [-137.871746756679, 79.61767780469414],
        [-128.1900442711871, 79.46477183460594],
        [-119.03782920095914, 79.02598714163038],
        [-110.78417499347034, 78.33427559463203],
        [-101.41444194424896, 81.03727076699988],
        [-84.86374614619683, 83.31790529580597],
        [-56.661492985017695, 84.59073393509807],
        [-23.997303500485867, 84.14752952709192],
        [-1.7935794108101304, 82.28025812231863],
        [10.640716221111093, 79.7462878857949],
        [19.85067874527038, 80.54320737156078],
        [30.43243897767354, 81.05772172159837],
        [41.869003138215085, 81.23893808369635],
        [53.316637650632835, 81.06622091117508],
        [63.92693309051733, 80.55927219284123],
        [73.17306663110058, 79.7684835157352],
        [81.9815223904838, 81.75083902053998],
        [95.79408023672825, 83.42764891884764],
        [116.81087569236769, 84.50349273924772],
        [142.71035788751252, 84.60173769508268],
        [165.01742621551023, 83.67635700611854],
        [180, 82.08068862192995],
        [180, 90],
        [-180, 90],
        [-180, 82.08068862192995],
        [-176.604555128726, 81.50928142471932],
        [-173.6474814909233, 80.91238836229148],
        [-171.06209142427136, 80.29488780447296],
        [-168.79122851403028, 79.66065491509582],
        [-166.7867724748905, 79.01277578737971],
        [-165.0085828152573, 78.3537208449233],
    ]


def test_transform_bbox():
    assert transform.transform_bbox([382200, 2279400, 610500, 2512500], 32645) == [
        85.85296718933647,
        20.610041795245515,
        88.07596179098907,
        22.719775713801845,
    ]
    assert transform.transform_bbox([508800, 7001100, 753000, 7247400], 32660) == [
        177.17456390562774,
        63.05068519969726,
        -177.57860702499983,
        65.34932256544839,
    ]
    assert transform.transform_bbox(
        [508800, 7001100, 753000, 7247400], 32660, expand=True
    ) == [
        177.17456390562774,
        63.05068519969726,
        -177.57860702499983 + 360,
        65.34932256544839,
    ]
    assert transform.transform_bbox(
        [382200, 2279400, 10, 610500, 2512500, 50], 32645
    ) == [
        85.85296718933647,
        20.610041795245515,
        10,
        88.07596179098907,
        22.719775713801845,
        50,
    ]
    assert transform.transform_bbox(
        [508800, 7001100, 5, 753000, 7247400, 10], 32660
    ) == [
        177.17456390562774,
        63.05068519969726,
        5,
        -177.57860702499983,
        65.34932256544839,
        10,
    ]


def test_geojson_from_linear_ring():
    print(
        transform.geojson_from_linear_ring(
            [
                [382200.000, 2512500.000],
                [382200.000, 2279400.000],
                [610500.000, 2279400.000],
                [610500.000, 2512500.000],
                [382200.000, 2512500.000],
            ],
            32645,
        )
    )
    assert transform.geojson_from_linear_ring(
        [[-10, -10], [10, -10], [10, 10], [-10, 10], [-10, -10]], 4326, partition=0
    ) == {
        "type": "Feature",
        "bbox": [-10, -10, 10, 10],
        "properties": {},
        "geometry": {
            "type": "Polygon",
            "coordinates": [[[-10, -10], [10, -10], [10, 10], [-10, 10], [-10, -10]]],
        },
    }
    assert transform.geojson_from_linear_ring(
        [[-10, -10], [10, -10], [10, 10], [-10, 10], [-10, -10]], 4326, partition=1
    ) == {
        "type": "Feature",
        "bbox": [-10.0, -10, 10, 10],
        "properties": {},
        "geometry": {
            "type": "Polygon",
            "coordinates": [
                [
                    [-10.0, -10],
                    [0.0, -10],
                    [10, -10.0],
                    [10, 0.0],
                    [10.0, 10],
                    [0.0, 10],
                    [-10, 10.0],
                    [-10, 0.0],
                    [-10, -10],
                ]
            ],
        },
    }

    assert transform.geojson_from_linear_ring(
        [
            [223069.075613658875227, -268407.860357680357993],
            [-275846.995845806901343, -495187.89283925422933],
            [-163320.733313913689926, -742745.670409420854412],
            [335595.338145552086644, -515965.637927847041283],
            [223069.075613658875227, -268407.860357680357993],
        ],
        3411,
    ) == {
        "type": "Feature",
        "bbox": [
            -74.12022112082042,
            82.98826228650147,
            -5.270622665103448,
            86.88878705473334,
        ],
        "properties": {},
        "geometry": {
            "type": "Polygon",
            "coordinates": [
                [
                    [-5.270622665103448, 86.77914371855618],
                    [-14.250061369548213, 86.87412426312318],
                    [-23.548864966957215, 86.88878705473334],
                    [-32.693812151359616, 86.82201802001164],
                    [-41.25560995422688, 86.67873416307306],
                    [-48.9538596657912, 86.46825557920494],
                    [-55.68003779057252, 86.20176494269847],
                    [-61.45601993315672, 85.89017913245713],
                    [-66.37532859434766, 85.54298702755824],
                    [-70.55724119088521, 85.16790348854887],
                    [-74.12022112082042, 84.77097678044598],
                    [-71.97110864705208, 84.61841509935437],
                    [-69.94331397518876, 84.45873464719025],
                    [-68.0315123231705, 84.29253849885926],
                    [-66.22993488762883, 84.12038499165931],
                    [-64.5325844049517, 83.94278809292268],
                    [-62.93340319357807, 83.76021880654855],
                    [-61.42639995777989, 83.57310727610472],
                    [-60.00574206724918, 83.38184531482337],
                    [-58.665819773133734, 83.18678915779185],
                    [-57.40128818663171, 82.98826228650147],
                    [-53.951992624649165, 83.27843524136415],
                    [-50.20571088005186, 83.54224084412589],
                    [-46.15863801687481, 83.77628363534642],
                    [-41.818277399785174, 83.97705623949825],
                    [-37.20658593714978, 84.14110603045513],
                    [-32.36219263801153, 84.2652552572159],
                    [-27.340749255999818, 84.34685551746712],
                    [-22.212645835564576, 84.38403943161795],
                    [-17.05792785558349, 84.37592038404522],
                    [-11.959143304085023, 84.32269393312409],
                    [-11.56347384314647, 84.5702079431649],
                    [-11.129970887643898, 84.81748727101969],
                    [-10.652993182505446, 85.06448689529408],
                    [-10.12573285508205, 85.31115283779752],
                    [-9.53990164907278, 85.55741971247636],
                    [-8.885311834454305, 85.80320743194677],
                    [-8.149308950579606, 86.04841671801508],
                    [-7.315992915614546, 86.29292288494149],
                    [-6.365131918099119, 86.53656708154573],
                    [-5.270622665103448, 86.77914371855618],
                ]
            ],
        },
    }

    assert transform.geojson_from_linear_ring(
        [
            [508800.000, 7247400.000],
            [508800.000, 7001100.000],
            [753000.000, 7001100.000],
            [753000.000, 7247400.000],
            [508800.000, 7247400.000],
        ],
        32660,
        partition=1,
    ) == {
        "type": "Feature",
        "bbox": [
            177.17456390562774,
            63.05068519969726,
            -177.57860702499983,
            65.34932256544839,
        ],
        "properties": {},
        "geometry": {
            "type": "MultiPolygon",
            "coordinates": [
                [
                    [
                        [180, 65.31807448056004],
                        [179.81058927281353, 65.323257946911],
                        [177.18908505580515, 65.34932256544839],
                        [177.18150083071316, 64.24429821916637],
                        [177.17456390562774, 63.13910500179646],
                        [179.59505058865193, 63.11547654369982],
                        [180, 63.104599649017885],
                        [180, 65.31807448056004],
                    ]
                ],
                [
                    [
                        [-180, 63.104599649017885],
                        [-177.99275205341496, 63.05068519969726],
                        [-177.79484203871405, 64.15151244109893],
                        [-177.57860702499983, 65.25180997065199],
                        [-180, 65.31807448056004],
                        [-180, 63.104599649017885],
                    ]
                ],
            ],
        },
    }


def test_geojson_from_corner_coordinates():
    assert transform.geojson_from_corner_coordinates(
        [223069.075613658875227, -268407.860357680357993],
        [-275846.995845806901343, -495187.89283925422933],
        [335595.338145552086644, -515965.637927847041283],
        [-163320.733313913689926, -742745.670409420854412],
        3411,
    ) == {
        "type": "Feature",
        "bbox": [
            -74.12022112082042,
            82.98826228650147,
            -5.270622665103448,
            86.88878705473334,
        ],
        "properties": {},
        "geometry": {
            "type": "Polygon",
            "coordinates": [
                [
                    [-5.270622665103448, 86.77914371855618],
                    [-14.250061369548213, 86.87412426312318],
                    [-23.548864966957215, 86.88878705473334],
                    [-32.693812151359616, 86.82201802001164],
                    [-41.25560995422688, 86.67873416307306],
                    [-48.9538596657912, 86.46825557920494],
                    [-55.68003779057252, 86.20176494269847],
                    [-61.45601993315672, 85.89017913245713],
                    [-66.37532859434766, 85.54298702755824],
                    [-70.55724119088521, 85.16790348854887],
                    [-74.12022112082042, 84.77097678044598],
                    [-71.97110864705208, 84.61841509935437],
                    [-69.94331397518876, 84.45873464719025],
                    [-68.0315123231705, 84.29253849885926],
                    [-66.22993488762883, 84.12038499165931],
                    [-64.5325844049517, 83.94278809292268],
                    [-62.93340319357807, 83.76021880654855],
                    [-61.42639995777989, 83.57310727610472],
                    [-60.00574206724918, 83.38184531482337],
                    [-58.665819773133734, 83.18678915779185],
                    [-57.40128818663171, 82.98826228650147],
                    [-53.951992624649165, 83.27843524136415],
                    [-50.20571088005186, 83.54224084412589],
                    [-46.15863801687481, 83.77628363534642],
                    [-41.818277399785174, 83.97705623949825],
                    [-37.20658593714978, 84.14110603045513],
                    [-32.36219263801153, 84.2652552572159],
                    [-27.340749255999818, 84.34685551746712],
                    [-22.212645835564576, 84.38403943161795],
                    [-17.05792785558349, 84.37592038404522],
                    [-11.959143304085023, 84.32269393312409],
                    [-11.56347384314647, 84.5702079431649],
                    [-11.129970887643898, 84.81748727101969],
                    [-10.652993182505446, 85.06448689529408],
                    [-10.12573285508205, 85.31115283779752],
                    [-9.53990164907278, 85.55741971247636],
                    [-8.885311834454305, 85.80320743194677],
                    [-8.149308950579606, 86.04841671801508],
                    [-7.315992915614546, 86.29292288494149],
                    [-6.365131918099119, 86.53656708154573],
                    [-5.270622665103448, 86.77914371855618],
                ]
            ],
        },
    }

    assert transform.geojson_from_corner_coordinates(
        [508800.000, 7247400.000],
        [508800.000, 7001100.000],
        [753000.000, 7247400.000],
        [753000.000, 7001100.000],
        32660,
        partition=1,
    ) == {
        "type": "Feature",
        "bbox": [
            177.17456390562774,
            63.05068519969726,
            -177.57860702499983,
            65.34932256544839,
        ],
        "properties": {},
        "geometry": {
            "type": "MultiPolygon",
            "coordinates": [
                [
                    [
                        [180, 65.31807448056004],
                        [179.81058927281353, 65.323257946911],
                        [177.18908505580515, 65.34932256544839],
                        [177.18150083071316, 64.24429821916637],
                        [177.17456390562774, 63.13910500179646],
                        [179.59505058865193, 63.11547654369982],
                        [180, 63.104599649017885],
                        [180, 65.31807448056004],
                    ]
                ],
                [
                    [
                        [-180, 63.104599649017885],
                        [-177.99275205341496, 63.05068519969726],
                        [-177.79484203871405, 64.15151244109893],
                        [-177.57860702499983, 65.25180997065199],
                        [-180, 65.31807448056004],
                        [-180, 63.104599649017885],
                    ]
                ],
            ],
        },
    }


def test_invalid_error():
    with pytest.raises(InvalidLinearRing):
        transform.transform_ring([[0, 0], [10, 10]], 4326)
    with pytest.raises(EnclosingBothPoles):
        transform.transform_ring(
            [
                [-4525934.615208394825459, 8626327.724166244268417],
                [7103066.913057595491409, 6594075.02990616671741],
                [3037818.062780980952084, -16668182.280009945854545],
                [-8591183.465485010296106, -14635929.585749868303537],
                [-4525934.615208394825459, 8626327.724166244268417],
            ],
            CRS(
                "+proj=aeqd +lat_0=35.7 +lon_0=139.8 +x_0=0 +y_0=0 +ellps=WGS84 +units=m +no_defs"
            ),
        )
    with pytest.raises(InvalidBounds):
        transform.transform_bbox([508800, 7001100, 753000, 7247400, 1000], 32660)
    with pytest.raises(NotAllowedWarpBounds):
        transform.transform_bbox([508800, 7001100, -753000, 7247400], 32660)
    with pytest.raises(NotAllowedWarpBounds):
        transform.transform_bbox([508800, 7001100, 10, -753000, 7247400, 100], 32660)
    with pytest.raises(InvalidLinearRing):
        transform.geojson_from_linear_ring(
            [
                [508800.000, 7247400.000],
                [508800.000, 7001100.000],
                [753000.000, 7001100.000],
                [753000.000, 7247400.000],
            ],
            32660,
        )
    with pytest.raises(NotAllowedCwLinearRing):
        transform.geojson_from_corner_coordinates(
            [508800.000, 7247400.000],
            [508800.000, 7001100.000],
            [-753000.000, 7247400.000],
            [-753000.000, 7001100.000],
            32660,
        )
    with pytest.raises(InvalidLinearRingEnclosingPole):
        transform._transform_enclosing_pole_ring(
            [
                [223069.075613658875227, -268407.860357680357993],
                [-275846.995845806901343, -495187.89283925422933],
                [-163320.733313913689926, -742745.670409420854412],
                [335595.338145552086644, -515965.637927847041283],
                [223069.075613658875227, -268407.860357680357993],
            ],
            3411,
            partition=0,
            north=True,
        )