# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['uniserde']

package_data = \
{'': ['*']}

install_requires = \
['motor>=3.0.0,<4.0.0', 'python-dateutil>=2.8.2,<3.0.0']

setup_kwargs = {
    'name': 'uniserde',
    'version': '0.2.0',
    'description': 'Convention based, effortless serialization and deserialization',
    'long_description': '# Convention based, effortless serialization and deserialization\n\nUniserde can convert Python classes to/from JSON and BSON without any input\nfrom your side. Simply define the classes, and the library does the rest.\n\nSimply define your types as classes with type annotations, and call one of the\nserialization/deserialization functions:\n\n```py\nfrom uniserde import Serde\nfrom datetime import datetime, timezone\nfrom dataclasses import dataclass\nfrom bson import ObjectId\n\n\n@dataclass\nclass Person(Serde):\n    id: ObjectId\n    name: str\n    birth_date: datetime\n\n\nBetty = Person(\n    id=ObjectId(),\n    name="Betty",\n    birth_date=datetime(year=1988, month=12, day=1, tzinfo=timezone.utc),\n)\n\nprint(Betty.as_json())\n```\n\nThis will print a dictionary similar to this one\n\n```py\n{\n    \'id\': \'62bc6c77792fc617c52499d0\',\n    \'name\': \'Betty\',\n    \'birthDate\': \'1988-12-01T00:00:00+00:00\'\n}\n```\n\nYou can easily convert this to a string using Python\'s built-in `json` module if\nthat\'s what you need.\n\n# API\n\nThe API is intentionally simple. Functions/Classes you might be interested in\nare:\n\n- `as_json`, `as_bson`\n\n  Given a class with type annotations, these create a JSON/BSON like dictionary.\n  You can feed those into functions like `json.dump`, or use them as is.\n\n- `from_json`, `from_bson`\n\n  Given a JSON/BSON like dictionary, these will instantiate the corresponding\n  Python class. Raise `SerdeError` if the values are invalid.\n\n- `Serde` is a helper class you can optionally apply to your models. It adds the\n  convenience functions `as_json`, `as_bson`, `from_json`, and `from_bson`\n  directly to the models.\n\n- Sometimes a class simply acts as a type-safe base, but you really just want to\n  serialize the children of that class. In that case you can decorate the class\n  with `@as_child`. This will store an additional `type` field in the result, so\n  the correct child class can be parsed.\n\n- `as_mongodb_schema` automatically creates JSON schemas compatible with MongoDB\n  from models\n\n# Types & Conventions\n\nThe library tries to stick to the naming conventions used by the target formats:\n\n- names in JSON are written in lowerCamelCase, as is commonly done in\n  JavaScript\n- BSON uses the same conventions as JSON\n- Python class names are expected to be written in UpperCamelCase\n- Python enum values must be in ALL_UPPER_CASE\n\n## JSON\n\n| Python         | JSON            | Notes                                                                                                               |\n| -------------- | --------------- | ------------------------------------------------------------------------------------------------------------------- |\n| bool           | bool            |\n| int            | float           |\n| float          | float           |\n| str            | str             |\n| List           | list            |\n| Optional       | value or `None` |\n| Any            | as-is           |\n| Literal[str]   | str             |\n| enum.Enum      | str             | Enum values are mapped to their name (NOT value!)                                                                   |\n| custom class   | dict            | Each attribute is stored as key, in lowerCamelCase. If marked with `as_child`, an additional `type` field is added. |\n| bytes          | str             | base64 encoded                                                                                                      |\n| datetime       | str             | as ISO 8601 - With timezone. Naïve datetimes are not supported.                                                     |\n| Dict[str, ...] | dict            |\n| bson.ObjectID  | str             |\n\n## BSON\n\nBSON uses the same conventions as JSON, with just a few changes\n\n| Python        | BSON          | Notes                                                                                               |\n| ------------- | ------------- | -----------------------------------------------------------------------------                       |\n| custom class  | object        | Same as JSON, but any fields named `id` are renamed to `_id` to match MongoDB                       |\n| bytes         | bytes         |\n| datetime      | datetime      | Serialization requires a timezone be set. Deserialization imputes UTC, to match MongoDB convention. |\n| bson.ObjectID | bson.ObjectId |\n\n# Schema Generation\n\nIf you are working with MongoDB you will come to appreciate the automatic schema\ngeneration. Calling `uniserde.as_mongodb_schema` on any supported class will return\na MongoDB compatible JSON schema without hassle.\n\nFor example `uniserde.as_mongodb_schema(Person)` with the `Person` class from above:\n\n```py\n{\n    \'type\': \'object\',\n    \'properties\': {\n        \'_id\': {\n            \'bsonType\': \'objectId\'\n        },\n        \'name\': {\n            \'type\': \'string\'\n        },\n        \'birthDate\': {\n            \'bsonType\': \'date\'\n        }\n    },\n    \'additionalProperties\': False,\n    \'required\': [\n        \'name\',\n        \'birthDate\'\n    ]\n}\n```\n\n# TODO\n\n- Support for `Union` is currently very limited. Really only `Optional` is\n  supported (which maps to `Union`)\n- `Literal` currently only supports strings\n- Make support for BSON optional, so the library doesn\'t depend on MongoDB\n- Extend `as_child`, to allow marking some classes as abstract. i.e. their\n  parents/children can be serialized, but not those classes themselves\n- Datetime handling is weird. MongoDB does not store the timezone, so the BSON\n  code assumes there is none. But really, every datetime should have a timezone\n  set.\n- Being able to specify additional limitations to fields would be nice:\n  - must match regex\n  - minimum / maximum\n  - custom validation functions\n- more Unit tests\n- Add more examples to the README\n  - show custom serializers/deserializers\n  - recommended usage\n',
    'author': 'Jakob Pinterits',
    'author_email': 'jakob.pinterits@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/Vivern/uniserde',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
