########################################################################################################
#
# ddrclass.py
#
# This module defines the DDR class used to instantiate instances of the DDR expert system 
# machine reasoning framework
#
# Each instance of the DDR class implements an independent machine reasoning environment with
# indepedent instances of the CLIPs expert system.  Multiple instances can be run simultaneously.
# A common way to deploy DDR is to instantiate an DDR instance for each use case
#
# DDR can be run "on-box" in the guestshell on XE and NX-OS platforms and in a Docker container
# on XR eXR enabled platforms in the app-hosting environment
#
# DDR can be run "off-box" on PC, MAC or Linux platforms
#
# DDR execution is controlled by content contained in a collection of files loaded by DDR on startup:
#
#   facts - This file contains information used by DDR to collect "FACTS" from the device.  Also contains
#           "control" flags used to control DDR operating modes and execution
#   rules - This file contains the CLIPS rules and other CLIPS constructs used to implement the use case
#   flags - This file contains control flags for DDR execution
#   devices - This file defines the devices that are used by the workflow
#   tests - This optional file contains FACTS used to enable execution of CLIPS using pre-defined fact data
#           instead of collecting FACTS from the device.  When 'test-facts' is enabled DDR does not interact
#           with devices and instead reads and asserts lists of facts from the file.  This can be used
#           for testing and demonstrating DDR use cases
#   log - This optional file when 'debug-logging' is enabled can be used to persist messages generated by DDR in
#         a log file on the device, typically stored on bootflash
#
#
##### Running DDR #####
#
# DDR is run on-box or off-box using a command of the form below.  The relative paths to the control files are parameters. 
# If no parameters are specified DDR finds files in the directory DDR is started in with names 'ddr-facts', 'ddr-rules.',
# 'ddr-flags', 'ddr-tests', and 'ddr-log'
#
#   $ python ddrrun.py --facts=input/assurance/memory-leak.txt --rules=input/assurance/memory-leak.clp  
#                       --tests=input/assuracne/tests.txt --log=bootflash/ddr-log
#
# Execution is stopped by CTRL-/ in the window where the script is started
#
# For information contact: Peter Van Horne petervh@cisco.com
#
###################################################################################################################
import ncclient.manager
from ncclient.xml_ import to_ele
from lxml import etree
import xml.dom.minidom
import xml.etree.ElementTree as ET
import clips
import datetime
from datetime import datetime
import time
import os
import sys
import logging
import logging.handlers
from logging.handlers import RotatingFileHandler
import imp
import json
import pprint
import copy
import xmltodict
import re
from threading import Timer
import pexpect

try:
    import resource
except:
    pass
#
# If DDR is running on a device, import the python cli library for running cli commands in guestshell
#
try:
    import cli
except:
    pass

#
# Import library of parser Classes to translate show commands into python dictionaries that are
# translated into CLIPs FACTS using FACT definitions
#
try:
    from genie_parsers import *
except:
    pass

    #############################################################################
    #############################################################################
    #############################################################################
class DDR:
    #
    # DDR entry point - Call this function to load and start CLIPs execution
    #
    #############################################################################
    #############################################################################
    #############################################################################
    def ddr(self,flist, alist, model_control, single_run):
        self.ddr_init(flist, model_control, single_run)
    # 
    # Argument definitions:
    # ~~~ flist - list of file names containing FACT definitions, RULEs, test FACTs and log file name
    # ~~~ model_control - If set to "1" the script reads configuration from a YANG model
    # ~~~ single_run - If set to "1" run the main processing loop once and exit
    #
    # Maintain a counter for the notification logs generated during the session
    # The "control" dictionary stores configuration and state information
    #  
        self.control["session-log-count"] = 1
   
    ######################################################################################    
    ######################################################################################
    # while loop runs until interrupted by CTRL-/ or an error occurs that causes the
    # DDR function to return.
    # Loop continuosly looking for triggering events which include a syslog message
    # containing a string that matches syslog_triggers, an RFC5277 notification including
    # a string that matches the content of a notification or timed execution
    ######################################################################################
    ######################################################################################
        while True:
            self.engine_process(alist)

    def engine_process(self, alist):
        ######################################################################################
        ######################################################################################
        # MAIN RULES ENGINE PROCESSING LOOP
        ######################################################################################
        ######################################################################################

    # 
    # Triggering conditions cause execution of the FACT collection and 
    # RULEs engine execution
    #
        syslog_trigger = False
        timed_trigger = False
        notification_trigger = False
        control_trigger = False

#######################################################################################
#
# control["run-mode"] = 0 Run after time delay
#
#######################################################################################
        if self.control["run-mode"] == 0:
            self.print_log("\n**** DDR Notice: Wait for Trigger Event")
            time.sleep(self.control["run-wait"]/1000)
            timed_trigger = True
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
            self.print_log("\n**** DDR Notice: Trigger Event: Timed Execution at: " + str(timestamp))

#######################################################################################
#
#   control["run-mode"] = 1 waits for a Syslog message to trigger execution
#   The DDR  script runs a Syslog listener on port 9514 (default Syslog port is 514)
#   Devices sending "logging" Syslog messages should use the DDR  script IP address and
#   port 9514.  Typicall port forwarding must be configured using NAT to map
#   Packets addressed to the hosting device on Port 9514 to forward to the receiver in DDR 
#   Example XR configuration: logging 10.24.121.9 vrf default severity notifications port 9514
#   Example XE configuration: logging host 10.24.72.167 vrf Mgmt-vrf transport udp port 9514
#
#######################################################################################

        if self.control["run-mode"] == 1:
#
# mesq.get() checks to see if there is a message available on the Queue that
# contains device Syslog messages received and processed by "queue_syslog" function.
# If there are no syslog messages in the queue, do not wait
#
            self.print_log("**** DDR Notice: Wait for Syslog Trigger Event")
#
# script waits to receive a syslog message sent the IP and port defined for the syslog listener
#
            syslog = self.mesq.get()
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
            self.print_log("\n**** DDR Notice: Trigger Event: Syslog Message at: " + str(timestamp))
            if self.control["debug-syslog"] == 1:
                self.print_log("**** DDR Debug: queued Syslog message: ")
                self.print_log(syslog)
                self.print_log("**** DDR Debug: syslog_triggers: " + str(self.control["syslog-triggers"]))
#
# Test to see if the syslog message receive matches one of the syslog triggering list entries
#
            for trigger in self.control["syslog-triggers"]:
                if self.control["debug-syslog"] == 1:
                    self.print_log("**** DDR Debug: syslog_trigger: " + str(trigger))
#
# Loop through all match string for the syslog trigger entry.  All of the strings must be found
# in the syslog or the trigger condition is not satisfied
#
                syslog_trigger = False
                for match_string in trigger[0]:
                    if syslog.find(match_string) != -1:
                        syslog_trigger = True
                    elif syslog_trigger == True:
                        syslog_trigger = False
                        break # one of the match_strings was not present in the syslog message

                if syslog_trigger == True:
                    try:
                        if trigger[2] == 'True':
                            self.print_log("**** DDR Notice: Assert Syslog FACT: " + str(trigger))
                            self.assert_syslog_fact(syslog)
#
#  Check to see if the matching this syslog requires asserting additional facts
#
                        if trigger[1] != []:
                            try:
                                for syslog_fact in trigger[1]:
                                    self.env.assert_string(str(syslog_fact))
                            except Exception as e:
                                self.print_log("%%%%% DDR Error: Asserting syslog fact: " + str(trigger[1]) + str(e))
                        break
                    except:
                        self.print_log("%%%%% DDR Error: Asserting syslog fact for: " + str(syslog))

#######################################################################################
#
# control["run-mode"] = 2 Trigger on NETCONF notification
#                         The notification is generated with the DMI infrastructure generates
#                         an RFC5277 notification using the content of Syslog message or SNMP trap
#
#######################################################################################
        if self.control["run-mode"] == 2:

#
# Check for NETCONF notifications from the device
# If netconf notification is received, check if the content of the notification
# Matches one of the "triggers".
# Check to see if the notification contains content that should be used to directly assert a fact
#
#######################################################################################
            try:
#
# block and wait until an RFC5277 NETCONF notification is received
#
                if self.control["single-notify"] == 1:
                    input("\n\nHit Enter to accept one notification event\n\n")  # useful when manually debugging
                if self.control["run-mode"] == 2:
                    notification_found = False
                    notif = self.notify_conn.take_notification(block=True) # block on IO waiting for the NETCONF notification
                    notify_xml = notif.notification_xml
                    if self.control["debug-notify"] == 1:
                        self.print_log("**** DDR Debug: RFC5277 notification received: " + notify_xml)

######################################################################################
# Test to see if the snmpevents notification included any of the strings that
# are included in the notification_triggers definitions in the FACTs file.  If so, run the rules engine
######################################################################################

                    if notify_xml != '':
                        for trigger in self.control["notification-triggers"]:
                            notification_trigger = False
#
# Loop through all match strings for the notification.  All of the strings must be found
# in the notification or the trigger condition is not satisfied
#
#   element 0 - List containing strings that must all be matched
#   element 1 - Optional statically defined FACT with no paramters that is asserted if element 2 is 'true'
#   element 2 - 'true' if the FACT defined in element 1 should be asserted
#   element 3[0] - The regular expression in this entry extracts all of the text in the indicated xml tag for processing
#   element 3[1] - Regular expression used to extract fields from the message and insert into variables.
#   element 3[2] - List variable names that will be used to populate the FACT
#   element 3[3] - Prototype FACT template.  The variables are inserted in order into the template and the FACT is asserted
#
# notification_triggers = [
#      [['ADJCHANGE', 'Down'], 
#       ['(assert (sample-fact (sample-slot sample-value)))'], 
#       'false', 
#       ['<clogHistMsgText>(.*)</clogHistMsgText>', 
#        'neighbor.{1}(?P<neighbor>(\S+)).{1}(?P<state>(\S+)).{1}(?P<message>(.+))', 
#        ['neighbor', 'state', 'message'],
#        '(bgp-event (device {0}) (event-time {1}) (event-type NEIGHBOR-DOWN) (neighbor {2}) (state {3}) (message {4})  (syslog-time #{5}))']
#     ]
# ]
#
                            if all([val in str(notify_xml) for val in trigger[0]]): 
                                notification_trigger = True                        
                                try:
                                    timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
                                    self.print_log("\n**** DDR Notice: Triggering Notification Event Found: " + str(trigger[0]) + " at: " + str(timestamp))
                                    if trigger[2] == 'True':
                                        self.print_log("**** DDR Notice: Assert RFC5277 FACT: " + str(trigger))
                                        self.assert_rfc5277_fact(notify_xml)
#
# Get the event time - Extract syslog mesasge time from notification and translate into seconds since midnight
# Insert the seconds into the generated syslog FACT
# Sample time: <eventTime>2020-09-29T09:47:38+00:00</eventTime>
#
                                    try:
                                        event_seconds = 0
                                        event_content = re.search('<eventTime>(.*)</eventTime>', str(notify_xml)) #extract event_time field from message
                                        p1 = re.compile('.{11}(?P<mtime>(.{8}))') #regex to extract time in seconds
                                        results = p1.match(str(event_content.group(1))) #extract seconds field from message
                                        syslog_time = str(event_content.group(1))
                                        group = results.groupdict() #dictionary contains objects
                                        times = str(group["mtime"]).split(":")
                                        event_seconds = sum(int(x) * 60 ** i for i, x in enumerate(reversed(times)))
                                    except Exception as e:
                                        self.print_log("%%%%% Error generating notification time: " + str(event_content.group(1)) + " " + str(e))
#
# Extract FACTs from the notification message and assert a FACT in CLIPs
# Process in a try clause for backward compatibility with older FACT files that do not have this option
#
                                    if trigger[3] != []:
                                        mdata = trigger[3]
                                        Slog = re.search(str(mdata[0]), str(notify_xml))
                                        SlogClean = Slog.group(1).replace(',',' ')
                                        if self.control["debug-fact"] == 1:
                                            self.print_log("**** DDR Debug: Notification Syslog: " + str(SlogClean))
                                        p1 = re.compile(str(mdata[1])) #generate regex object
#
# If there is no match generate a logging message
#
                                        if (str(results) == 'none') or (str(results) == 'None'):
                                            self.print_log("**** DDR Debug: Notification Syslog no data found: " + str(mdata[1]))
                                            break
#
# Build the notification FACT using data parsed from the notification text
# This code supports selecting up to 5 values from the Syslog message to assert in the FACT
# 
                                        try:
                                            try:
                                                results = p1.match(SlogClean) #extract fields from message
                                            except Exception as e:
                                                self.print_log("%%%% Error notification parsing: \n" + str(trigger[3][2]) + " " + str(SlogClean) + "\n" +str(e))
                                                break
                                            
                                            group = results.groupdict() #dictionary contains objects
                                            keys = mdata[2] #get the field names
                                            protofact = str(mdata[3]) #string to update with values
                                            if self.control["debug-fact"] == 1:
                                                self.print_log("\n**** DDR Debug: Notification Protofact: " + str(protofact))
                                            if len(keys) == 1:
                                                value1 = str(group[str(keys[0])])
                                                notify_fact = protofact.format(str(self.control["mgmt-device"][4]), str(event_seconds), str(value1).replace(' ', '_'), syslog_time)
                                            elif len(keys) == 2:
                                                value1 = str(group[str(keys[0])])
                                                value2 = str(group[str(keys[1])])
                                                notify_fact = protofact.format(str(self.control["mgmt-device"][4]), str(event_seconds), str(value1).replace(' ', '_'), str(value2).replace(' ', '_'), syslog_time)
                                            elif len(keys) == 3:
                                                value1 = str(group[str(keys[0])])
                                                value2 = str(group[str(keys[1])])
                                                value3 = str(group[str(keys[2])])
                                                notify_fact = protofact.format(str(self.control["mgmt-device"][4]), str(event_seconds), str(value1).replace(' ', '_'), str(value2).replace(' ', '_'), str(value3).replace(' ', '_'), syslog_time)
                                            elif len(keys) == 4:
                                                value1 = str(group[str(keys[0])])
                                                value2 = str(group[str(keys[1])])
                                                value3 = str(group[str(keys[2])])
                                                value4 = str(group[str(keys[3])])
                                                notify_fact = protofact.format(str(self.control["mgmt-device"][4]), str(event_seconds), str(value1).replace(' ', '_'), str(value2).replace(' ', '_'), str(value3).replace(' ', '_'), str(value4).replace(' ', '_'), syslog_time)
                                            elif len(keys) == 5:
                                                value1 = str(group[str(keys[0])])
                                                value2 = str(group[str(keys[1])])
                                                value3 = str(group[str(keys[2])])
                                                value4 = str(group[str(keys[3])])
                                                value5 = str(group[str(keys[4])])
                                                notify_fact = protofact.format(str(self.control["mgmt-device"][4]), str(event_seconds), str(value1).replace(' ', '_'), str(value2).replace(' ', '_'), str(value3).replace(' ', '_'), str(value4).replace(' ', '_'), str(value5).replace(' ', '_'), syslog_time)
                                            elif len(keys) == 6:
                                                value1 = str(group[str(keys[0])])
                                                value2 = str(group[str(keys[1])])
                                                value3 = str(group[str(keys[2])])
                                                value4 = str(group[str(keys[3])])
                                                value5 = str(group[str(keys[4])])
                                                value6 = str(group[str(keys[5])])
                                                notify_fact = protofact.format(str(self.control["mgmt-device"][4]), str(event_seconds), str(value1).replace(' ', '_'), str(value2).replace(' ', '_'), str(value3).replace(' ', '_'), str(value4).replace(' ', '_'), str(value5).replace(' ', '_'), str(value6).replace(' ', '_'),syslog_time)
                                            else:
                                                self.print_log("%%%% Error notification fact invalid: " + str(trigger[1]) + " " + str(e))
                                                notification = False
                                        except Exception as e:
                                            self.print_log("%%%% Error notification flag keys: " + str(keys) + " " + str(SlogClean) + " " + str(e))
#
# If trigger[3] is empty there are no values to extract from the notification, assert the static FACT defined in the second list element in the trigger FACT
# for the example above: (sample-fact (sample-slot sample-value))
#
                                    else:
                                        notify_fact = str(trigger[1][0])
                                        if self.control["debug-fact"] == 1:
                                            self.print_log("**** DDR Debug: Static notification FACT: " + str(notify_fact))
#
# Assert the notification FACT
#
                                    try:
                                        if self.control["debug-fact"] == 1:
                                            self.print_log("**** DDR Debug: Notification FACT: " + str(notify_fact))
                                        self.env.assert_string(str(notify_fact))
                                    except: 
                                        if self.control["debug-fact"] == 1:
                                            self.print_log("**** DDR Debug: Notification FACT Exists: " + str(notify_fact))

                                except Exception as e:
                                    self.print_log("%%%%% Error asserting notification fact: " + str(notify_fact) + " " + str(e))
                                break
                            else:
                                if self.control["debug-notify"] == 1:
                                    self.print_log("**** DDR Debug: RFC5277 Notification Event Does Not Match Trigger - Ignored")

            except Exception as e:
                self.print_log("%%%%% DDR Error: In notification event processing: " + str(e))


#######################################################################################
#
# control["run-mode"] = 3 Run CLIPs immediately - Subsequent triggers caused by
#                         control files written by external systems to /bootflash/guest-share/ddr
#
#######################################################################################
        if self.control["run-mode"] == 3:
            self.print_log("\n**** DDR Notice: Trigger from external application control-file write to device")
            run_delay = self.control["run-wait"]/1000
            self.run_read_control_file(self.control["control-file"], run_delay)
            control_trigger = True
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
            self.print_log("\n**** DDR Notice: Trigger Event: Enable external application trigger at: " + str(timestamp))
            
###################################################################################
#
# Determine if DDR is in the "active" mode with standby-active set to 1
# If standby-active is set to 0, skip processing for this execution cycle
# The standby-active state is updated at the end of each execution loop
#
###################################################################################

        if self.control["standby-active"] == 1:
         
###################################################################################
#
# Test to see if any triggering event was found
#
###################################################################################
            if syslog_trigger or timed_trigger or notification_trigger or control_trigger:

######################################################################################
#
# For each fact in the fact_list get the operational or configuration data required
# to populate the facts required by the expert system
# Assert the facts in the expert system
# Measure the amount of time required to collect the facts from the device
#
######################################################################################

#####################################################################
#####################################################################
#
# Assert all FACTS here
#
####################################################################
####################################################################
                starttime = time.time()
#
# If test facts are being used, get the facts from the test_data list 
# loaded from the 'ddr-tests' file
# The test FACTs are loaded into memory so the FACTs can be "played"
# to DDR  for testing when the actual use case environment is not available
#PPP
                if self.control["test-facts"] == 1:
                    starttime = time.time()
                    max_facts = len(self.test_data)
                    if self.test_index < max_facts:
                        if self.control["debug-fact"] == 1:
                            self.print_log(self.test_data[self.test_index])

                        self.assert_test_facts(self.test_data[self.test_index])
                        self.test_index = self.test_index + 1
                    if self.test_index == max_facts:
                        self.control["standby-active"] = 2       # If all test facts have been used exit the main DDR loop
#
# Log time to load test facts
#
                    endtime = time.time()
                    fact_runtime = endtime - starttime
                    self.ddr_timing(' **** DDR  Time: Load and assert test facts(ms): %8.3f', fact_runtime, "load-test-facts")
#
# If use case requires asserting a FACT to advance the "period" in a state machine in the rules, assert the
# template fact "update-period-fact" at the end of each run of CLIPS
# Asserting this FACT will cause rules in the RULEs file that include this FACT as a Condition to fire
#
                    if self.control["update-period"] == 1:  
                        try:
                            self.env.assert_string("(update-period-fact (update TRUE))")
                        except: pass #ignore case where FACT already exists    
#
# If control["test-facts"] not set, read the FACTs from the devices
#
                else:
                    starttime = time.time()
                    try:
                        for fact in self.control["fact-list"]:
                            if self.control["debug-fact"] == 1:
                                self.print_log("**** DDR Debug: Main fact loop: " + str(fact))
                            if fact["fact_type"] == "show_and_assert":
                                if "log_message_while_running" in fact:
                                    self.print_log(fact["log_message_while_running"])
                                status = self.show_and_assert_fact(fact)
                            elif fact["fact_type"] == "multitemplate":
                                status = self.get_template_multifacts(fact["data"], 'none', 'none')
                            elif fact["fact_type"] == "multitemplate_protofact":
                                status = self.get_template_multifacts_protofact(fact, 'none', 'none')
                            else:
                                self.print_log("%%%%% DDR Error: Invalid fact type: " + str(fact))
                            if status != "success":
                                self.print_log("%%%%% DDR Error: Fact Read Error: " + str(status))
                    except Exception as e:
                    
                        self.print_log("%%%%% DDR Error: Fact type selection error: " + str(e))
#
# If required assert fact to advance the "period" state in the knowledge base
#
# If use case requires asserting a FACT to advance the "period" in a state machine in the rules, assert the
# template fact "update-period-fact" at the end of each run of CLIPS
#
                    if self.control["update-period"] == 1:  
                        try:
                            self.env.assert_string("(update-period-fact (update TRUE))")
                        except: pass #ignore case where FACT already exists    

                    endtime = time.time()
                    fact_runtime = endtime - starttime
                    self.ddr_timing(' **** DDR  Time: Get and process FACTS from devices(ms): %8.3f', fact_runtime, "get-device-facts")

#
# If action facts are being used, get the facts from the action_data list 
# loaded from the 'action.txt' file
# The action FACTs are loaded into memory so the FACTs can be "played"
# to DDR for testing the external applications facts
#
                if self.control["action-facts"] == 1:
                    starttime = time.time()
                    self.read_action_facts(alist)
                    
#
# Log time to load test facts
#
                    endtime = time.time()
                    fact_runtime = endtime - starttime
                    self.ddr_timing(' **** DDR  Time: Load and assert action facts(ms): %8.3f', fact_runtime, "load-action-facts")
#
# If use case requires asserting a FACT to advance the "period" in a state machine in the rules, assert the
# template fact "update-period-fact" at the end of each run of CLIPS
# Asserting this FACT will cause rules in the RULEs file that include this FACT as a Condition to fire
#
                    if self.control["update-period"] == 1:  
                        try:
                            self.env.assert_string("(update-period-fact (update TRUE))")
                        except: pass #ignore case where FACT already exists    
#
# log the FACTS asserted and triggered RULES on the agenda before CLIPs runs 
#
                self.print_clips_info()
#
# If required return clips facts and a dictionary representation of facts in the Service Impact Notification
# FACTs are sent before running clips to capture all facts that were present before the rules were run
#
                self.clips_facts = ''
                self.dict_facts = ''
                if self.control["send-before"] == 1:
                    self.send_clips_info()
                    self.save_dict_facts()

                for item in self.env.activations():
                    root_cause_rule = str(item)
                    break

##################################################################################
#
# env.run() causes execution of any action functions on the RHS of rules
# These control["actions"] can include: asserting additional FACTS, calling a python function
# adding a message to the service-impact notification
# Compute the run time for the CLIPs rules
#
##################################################################################
                starttime = time.time()
                try:
                    self.memory_use(" **** DDR  Memory: Before Running Inference Engine in KBytes: ", "before-clips-run")
                    self.env.run()
                except Exception as e:
                    self.print_log("\n%%%% DDR Error: Exception when running inference engine " + str(e))
                    self.close_sessions()
                    sys.exit(0)
                self.memory_use(" **** DDR  Memory: After Running Inference Engine in KBytes: ", "after-clips-run")

                endtime = time.time()
                clips_runtime = endtime - starttime
                self.ddr_timing(' **** DDR  Time: Run Inference Engine(ms): %8.3f\n', clips_runtime, "clips-runtime")
#
# Add facts to service impact notification after CLIPs is run if required
#
                if self.control["send-after"] == 1:
                    self.clips_facts = ''
                    self.dict_facts = ''
                    self.send_clips_info()
                    self.save_dict_facts()
#
# Generate service-impact-notification message 
# Include all messages asserted by rules.  Optionally include the CLIPS facts in
# CLIPS fact format and/or in Python dictionary format
#
                try:
                    event_time = datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
                    impactMessage = ''.join(self.impactList)
                    self.impactList = []
#
# Generate RFC5277 formated Service Impact notification message
#
                    try:
                        si_notification = self.control["service-impact"].format("DDR Service Impact Notification: " + str(self.control["use-case"]), impactMessage, self.clips_facts, self.dict_facts, str(self.control["session-time"]), datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f"))
                        if self.control["json-notify"] == 1:
                            data_dict = xmltodict.parse(si_notification)
                            si_notification = json.dumps(data_dict, indent=2, sort_keys=True)
                    except Exception as e:
                        self.print_log("%%%%% DDR Error: Content error: " + str(e))
      
                    if self.control["show-notification"] == 1:
                        self.print_log("\n################################ DDR Service Impact Notification ############################\n")
                        self.print_log(si_notification)
#
# Send RFC 5277 notification if platform with "send-notification" model implementation
# The RFC 5277 notification is not currently included in the Polaris implementation
#
                    if self.control["send-notification"] == 1 or self.control["send-notification"] == 3:
                        if self.control["debug-notify"] == 1:
                            self.print_log("\n**** DDR  Debug: Notification Message: \n" + str(si_notification))
                        try:
                            self.netconf_connection.dispatch(to_ele(si_notification))
                        except Exception as e:
                            self.print_log("%%%%% DDR Error: Notification request send error: " + str(e))
#
# Send Syslog message using IOX infrastructure if required
# This Syslog message will have the following form:
#
#    *Nov 3 17:22:13.514: %IM-5-IOX_INST_NOTICE: Switch 1 R0/0: ioxman:  
#                         IOX SERVICE guestshell LOG: DDR_MESSAGE BGPIC_Event: Log: /bootflash/guest-share/BGPIC_Event_11-03-2020_17:20:36.896019.1
#
# The Syslog indicates the message came from the IOX infrastructure which hosts the guestshell.  The first section is generated by the device Syslog feature.  The second 
# section (shown on a separate line) is generated by DDR  from content provided by RULEs and FACTs.  The "Log:" contains the full text of the service impact message.
# There can be multiple logs generated for an DDR  execution.  The timestamp on the log is the time the session started.  The ".1" increments for each additional log
#
                    if self.control["send-notification"] == 2 or self.control["send-notification"] == 3:
                        try:
                            filenum = self.control["session-log-count"]
                            self.control["session-log-count"] = self.control["session-log-count"] + 1
                            filename = str(self.control["log-path"]) + str(self.control["use-case"]) + "_" + str(self.control["session-time"]) + "." + str(filenum)
                            syslog_notification = "DDR_MESSAGE " + str(self.control["use-case"]) + ": Log: " + str(filename)
                            self.run_write_to_syslog(syslog_notification, 0)
                        except Exception as e:
                            self.print_log("%%%% DDR: Error sending Syslog notification: " + str(e))
                        if self.control["debug-notify"] == 1:
                            self.print_log("\n**** DDR  Debug: Syslog Notification Message: \n" + str(syslog_notification))
#
# Write notification log file to guest-share if syslog message notification mode selected
#
                        try:
                            with open(filename, "a+") as fd:
                                fd.write(str(si_notification))
                                fd.write("\n")
                            if self.control["debug-notify"] == 1:
                                self.print_log("\n**** DDR  Debug: Syslog Notification Log: \n" + str(filename))
                        except Exception as e:
                            self.print_log("%%%% DDR: Error writing Notification log: " + str(e) + "\n" + str(filename))
#
# Save notification if required
#
                    if self.control["save-notifications"] == 1:
                        self.save_si_notification(si_notification)

                except Exception as e:
                    self.print_log("%%%%% DDR Error: Notification send error: " + str(e))

#
# log the FACTS asserted and RULES on the agenda after CLIPs runs
#
                self.print_clips_info()
#
# Show memory statistics
#
            if self.control["show-memory"] == 1:
                self.print_log("\n**** DDR  Memory: Memory use statistics (kb)")
                self.print_log(self.memory)
                self.assert_statistics_fact("ddr-memory", self.memory)

#
# Show timing statistics
#
            if self.control["show-timing"] == 1:
                self.print_log("\n**** DDR  Timing: Execution time statistics (ms)")
                self.print_log(self.timing)
                self.assert_statistics_fact("ddr-timing", self.timing)
                self.timing = {
                "load-initial-facts" : 0,
                "load-test-facts" : 0,
                "load-action-facts" : 0,
                "get-device-facts" : 0,
                "clips-runtime" : 0}    
#
# If required clear selected template facts before running CLIPS again
# A list of FACTs to clear is optionally included in the FACT definitions
#
            self.clear_selected_facts()
#
# If using test-facts and all facts have been used exit the main loop
# test-facts loop above sets standby-active to force exit
#
            if self.control["standby-active"] == 2:
                sys.exit(0)
#
# If "run-one" is set, exit DDR  after running only one iteration
#
            if self.control["run-one"] == 1:
                self.print_log("\n!!!!!!!!!!!!!!!!!!!!!!!!!!!!! DDR Execution Cycle Completed !!!!!!!!!!!!!!!!!!!!!!!!!!!\n")
                sys.exit(0)
#
# If manual execution control wait for operator input
#
            if self.control["user-control"] == 1:
                self.print_log("\n!!!!!!!!!!!!!!!!!!!!!!!!!!!!! DDR Execution Cycle Completed !!!!!!!!!!!!!!!!!!!!!!!!!!!\n")
                continueRunning = input("**** DDR Notice: Enter 1 continue, 0 to exit: ")

                if continueRunning == str(1):
                    pass
                else:
                    sys.exit(0)
#
# call set_ddr_runmode to apply any changes to DDR execution applied
# to the facts file after DDR is started
# if standby-active has been set to 2, exit DDR
#
        self.set_ddr_runmode()
        if self.control["standby-active"] == 2:
            sys.exit(0)

################################################################
################################################################
################################################################
#
# End of DDR  While loop
#
################################################################
################################################################
################################################################


########################################################################
########################################################################
#
# Class Methods
#
########################################################################
########################################################################

    ###########################################################################################
    #
    # save_si_notification - Save service impact notification in ddr-control model list
    # list if required.  Save up to "max-notifications" service impact notifications in the 
    # saved-notifications list
    #
    ###########################################################################################

    def save_si_notification(self, si_notification):
        if self.control["save-notifications"] == 1:
            if self.control["notify-index"] > self.control["max-notifications"]:
                self.control["notify-index"] = 1
            try:
                store_notification = self.control["save-notification"].format(self.control["ddr-id"], self.control["notify-index"], datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f"), 'new', si_notification.replace("<", "&lt;").replace(">", "&gt;"), self.nc_facts)
                store_notification_text = store_notification
                if self.control["debug-notify"] == 1:
                    self.print_log("\n**** DDR  Debug: Saved Notificaion: \n" + str(store_notification_text))
                self.netconf_connection.edit_config(store_notification_text, target='running')
                self.control["notify-index"] = self.control["notify-index"] + 1

            except Exception as e:
                self.print_log("%%%%% DDR Error: Notification save error: " + str(e))

    ###########################################################################################
    #
    # clear_selected_facts - Clear facts in the knowledge base for templates in the clear-filter list
    #
    # NOTE: All FACTS in the knowledge-base that have a template name in the clear-filter list
    #       all cleared at the end of the DDR  execution cycle
    #
    ###########################################################################################
    def clear_selected_facts(self):
        if self.control["clear-selected"] == 1:
            for cfilter in self.control["clear-filter"]:
                notify_found = True
                while notify_found:
                    for fact in self.env.facts():
                        notify_found = False
                        try:
                            if str(cfilter) in str(fact):
                                notify_found = True
                                fact.retract()
                                break
                        except Exception as e:
                            self.print_log("%%%% DDR Error: Error clearing fact: " + str(e))

########################################################
#
#  ddr_init - initialize ddr runtime
#
########################################################
    def ddr_init(self, flist, model_control, single_run):
    #
    # Flags that control ddr  execution are read from the cisco-ios-xe-ddr-control.yang model
    # The values of the control flags are read when ddr  is started and after the
    # execution of each DDR pass (while loop execution)
    # The default values for the flags are shown below
    # These values will be updated by reading the ddr-control model
    # If the 'control["sync"]' flag in the ddr-control model is set, the values in the model
    # are updated in the control dictionary and selected values applied to ddr  control variables
    #
        self.control = dict()
    #
    # If model_control which is passed in when ddr  is started is == 1, control DDR using
    # the ddr-control.yang model
    #
        try:
            self.control.update({"model-control" : model_control})
            self.control.update({"send-messages" : 0})
            self.control.update({"debug-logging" : 0})
            self.control.update({"session-time" : ''})
            self.control["session-time"] = str(datetime.now().strftime('%m-%d-%Y_%H:%M:%S.%f'))
            self.control.update({"session-log-count" : 0})
    #
    # variable to collect facts that are converted from dictionary format and saved
    #
            self.nc_facts = ''
    #
    # Dictionary for collecting timing information
    #
            self.timing = {
            "load-initial-facts" : 0,
            "load-test-facts" : 0,
            "load-sim-facts" : 0,
            "load-action-facts" : 0,
            "get-device-facts" : 0,
            "clips-runtime" : 0
            }
    #
    # Dictionary for collecting memory use information
    #
            self.memory = {
            "entry" : 0,
            "before-test-facts" : 0,
            "after-test-facts" : 0,
            "before-sim-facts" : 0,
            "after-sim-facts" : 0,
            "before-action-facts" : 0,
            "after-action-facts" : 0,
            "before-clips-env" : 0,
            "after-clips-env" : 0,
            "before-rules-load" : 0,
            "after-rules-load" : 0,
            "before-load-init-facts" : 0,
            "after-load-init-facts" : 0,
            "after-assert-init-facts" : 0,
            "before-clips-run": 0,
            "after-clips-run": 0
            }
    #
    # impactList is used to collect messages to include in a service-impact notification
    #
            self.impactList = []
    #
    # Template for saving notifications
    #
            self.control["save-notification"] = '''
 <config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
  <ddr-control xmlns="http://cisco.com/ns/yang/cisco-ios-xe-ddr-control">
    <instances>
      <instance>
        <ddr-id>{0}</ddr-id>
        <saved-notifications>
          <notifications>
            <notification>
              <id>{1}</id>
              <datetime>{2}</datetime>
              <status>{3}</status>
              <content>{4}</content>
              <dictionary-facts>{5}</dictionary-facts>
            </notification>
          </notifications>
        </saved-notifications>
      </instance>
    </instances>
  </ddr-control>
 </config>'''
        except Exception as e:
            self.print_log('%%%%% DDR Error: ddr_init Intialization error: ' + str(e))
            self.print_log('%%%%% DDR Error: flist: ' + str(flist))
            sys.exit(0)

    ###################################################################################################
    #
    # Check to see if the use case FACTS file is available in /bootflash/guest-share/ddr-facts
    # If the ddr-facts file is present, read all DDR control files from guestshare:
    #
    #    ddr-facts - FACT collection and use case control definitions
    #    ddr-rules - RULE definitions for CLIPs execution
    #    ddr-flags - Control flags for DDR execution
    #    ddr-devices - Devices used for use case
    #    ddr-tests - FACT definitions to assert when ddr  is running if 'test-facts' is set
    #    ddr-init - File containing collection of FACTs to assert on ddr startup if 'initial-facts' is set
    #    ddr-control - File containing facts defined by external application used to control ddr execution
    #    ddr-action - File containing collection of FACTs to assert on
    #    ddrstartup if 'initial-facts' is not  set
    #    
    # If the files are not present in /guest-share use the the files provided as command line arguments
    #
    ###################################################################################################

        try:
            if os.path.exists('/bootflash/guest-share/ddr' + 'ddr-facts'):
                self.control.update({"facts-file" : '/bootflash/guest-share/ddr' + 'ddr-facts'})
                self.control.update({"rules-file" : '/bootflash/guest-share/ddr' + 'ddr-rules'})
                self.control.update({"flags-file" : '/bootflash/guest-share/ddr' + 'ddr-flags'})
                self.control.update({"devices-file" : '/bootflash/guest-share/ddr' + 'ddr-devices'})
                self.control.update({"test-file" : '/bootflash/guest-share/ddr' + 'ddr-tests'})
                #self.control.update({"action-file" : '/bootflash/guest-share/ddr' + 'ddr-action'})
                self.control.update({"init-facts-file" : '/bootflash/guest-share/ddr' + 'ddr-init'})
                self.control.update({"control-file" : '/bootflash/guest-share/ddr' + 'ddr-control'})
                self.control.update({"sim-file" : '/bootflash/guest-share/ddr' + 'ddr-sim'})
            else:
                if not os.path.exists(flist[0]):
                    self.print_log("%%%%% DDR Error: FACTS file not found")
                if not os.path.exists(flist[1]):
                    self.print_log("%%%%% DDR Error: RULES file not found")
                self.control.update({"facts-file" : flist[0]})
                self.control.update({"rules-file" : flist[1]})
                self.control.update({"flags-file" : flist[2]})
                self.control.update({"devices-file" : flist[3]})
                self.control.update({"test-file" : flist[4]})
                #self.control.update({"action-file" : flist[5]})
                #self.control.update({"init-facts-file" : flist[6]})
                self.control.update({"init-facts-file" : flist[5]})
                self.control.update({"control-file" : flist[7]})
                self.control.update({"sim-file" : flist[8]})
   #
    # Read the ddr control flags from the ddr-flags
    #
            fd = open(self.control["flags-file"])
            inputdata = imp.load_source('inputdata', self.control["flags-file"], fd)
            fd.close()
        except Exception as e:
            self.print_log('%%%%% DDR Error: initialization files read error: ' + str(e))
            sys.exit(0)
    #
    # DDR instance control parameters will be loaded
    # from the ddr-flags file to set the 'control' dictionary parameters
    #
        try:
            self.control.update({"initial-facts" : inputdata.initialFacts})
            self.control.update({"nc-timeout" : inputdata.ncTimeout})
            self.control.update({"log-file" : inputdata.logFile})
            self.control.update({"ddr-id" :  1})
            self.control.update({"run-mode" : inputdata.runMode})
            self.control.update({"run-one" : inputdata.runOne})
            self.control.update({"user-control" : inputdata.userControl})
            self.control.update({"run-wait" : inputdata.runWait})
            self.control.update({"update-period" : inputdata.updatePeriod})
            self.control.update({"actions" : inputdata.actions})
            self.control.update({"clear-facts" : inputdata.clearFacts})
            self.control.update({"clear-selected" : inputdata.clearSelected})
            self.control.update({"test-facts" : inputdata.testFacts})
            self.control.update({"sim-facts" : inputdata.simFacts})
            self.control.update({"show-facts" : inputdata.showFacts})
            self.control.update({"show-dict" : inputdata.showDict})
            self.control.update({"show-rules" : inputdata.showRules})
            self.control.update({"show-messages" : inputdata.showMessages})
            self.control.update({"send-messages" : inputdata.sendMessages})
            self.control.update({"show-notification" : inputdata.showNotification})
            self.control.update({"show-memory" : inputdata.showMemory})
            self.control.update({"show-timing" : inputdata.showTiming})
            self.control.update({"debug-action" : inputdata.debugAction})
            self.control.update({"debug-CLI" : inputdata.debugCLI})
            self.control.update({"debug-notify" : inputdata.debugNotify})
            self.control.update({"debug-syslog" : inputdata.debugSyslog})
            self.control.update({"debug-fact" : inputdata.debugFact})
            self.control.update({"debug-config" : inputdata.debugConfig})
            self.control.update({"debug-parser" : inputdata.debugParser})
            self.control.update({"debug-file" : inputdata.debugFile})
            self.control.update({"debug-logging" : inputdata.debugLogging})
            self.control.update({"logging-null" : inputdata.loggingNull})
            self.control.update({"send-notification" : inputdata.sendNotification})
            self.control.update({"save-notifications" : inputdata.saveNotifications})
            self.control.update({"save-dict-facts" : inputdata.saveDictFacts})
            self.control.update({"max-notifications" : inputdata.maxNotifications})
            self.control.update({"send-before" : inputdata.sendBefore})
            self.control.update({"send-after" : inputdata.sendAfter})
            self.control.update({"send-clips" : inputdata.sendClips})
            self.control.update({"send-dict" : inputdata.sendDict})
            self.control.update({"syslog-address" : inputdata.syslogAddress})
            self.control.update({"syslog-port" : inputdata.syslogPort})
            self.control.update({"notify-index" : 1})
            self.control.update({"service-impact" : inputdata.service_impact})
            self.control.update({"use-case" : inputdata.useCase})
            self.control.update({"log-path" : inputdata.logPath})
            self.control.update({"single-notify" : inputdata.singleNotify})
            self.control.update({"fact-timer" : inputdata.factTimer})
            self.control.update({"timer-fact-name" : inputdata.factName})
            self.control.update({"notify-path" : inputdata.notifyPath})
            self.control.update({"local-path" : inputdata.localPath})
            self.control.update({"json-notify" : inputdata.jsonNotify})
            self.control.update({"retry-count" : inputdata.retryCount})
            self.control.update({"retry-time" : inputdata.retryTime})
            self.control.update({"startup-delay" : inputdata.startupDelay})
            try:
                self.control.update({"action-facts" : inputdata.actionFacts})
            except: pass

    #
    # Add standby-active flag to control whether DDR should execute during the next
    # execution cycle or should be in "standby" mode, not perform FACT collection
    # and wait to standby-active set to cause full execution
            try:
                self.control.update({"standby-active" : inputdata.standbyActive})
            except:
                self.control.update({"standby-active" : 1})
    #
    # Set logging to null device if required
    #
            self.stdout_save = sys.stdout
            if self.control["logging-null"] == 1:
                f = open(os.devnull, 'w')
                sys.stdout = f
        except Exception as e:
            self.print_log('%%%%% DDR Error: ddr-flags file missing content: ' + str(e))


    ###################################################################################################
    #
    # Check to see if device configuration information is available in /bootflash/guest-share/ddr-devices
    # If the file is present, update the "device-list" and "mgmt-device" with configurations from ddr-devices
    #
    ###################################################################################################

        try:
            if os.path.exists('/bootflash/guest-share/ddr/ddr-devices'):
                fd = open('/bootflash/guest-share/ddr/ddr-devices')
                devicedata = imp.load_source('devicedata', '/bootflash/guest-share/ddr/ddr-devices', fd)
                self.control.update({"device-list" : devicedata.device_list})
                self.control.update({"mgmt-device" : devicedata.mgmt_device})
                self.print_log("**** DDR Notice: /bootflash/guest-share/ddr/ddr-devices used to set device information")
            else:
                self.print_log("**** DDR Notice: ddr-devices in local directory used to set device information")
                fd = open(self.control["devices-file"])
                devicedata = imp.load_source('devicedata', self.control["devices-file"], fd)
                self.control.update({"device-list" : devicedata.device_list})
                self.control.update({"mgmt-device" : devicedata.mgmt_device})

        except Exception as e:
            self.print_log('%%%%% DDR Error: DDR ddr-devices file read error: ' + str(e))
            sys.exit(0)
            
    ###########################################################################
    #
    # Read the FACTs definitions from ddr-facts
    #
    ###########################################################################
        try:
            fd = open(self.control["facts-file"])
            inputdata = imp.load_source('inputdata', self.control["facts-file"], fd)
            fd.close()
    #
    # Read the FACT collection instructions from the ddr-fact file
    #
            self.control.update({"initial-facts-list" : inputdata.initial_facts})
            self.control.update({"fact-list" : inputdata.fact_list})        
            self.control.update({"nc-fact-list" : inputdata.nc_fact_list})
            self.control.update({"show-fact-list" : inputdata.show_fact_list})
            self.control.update({"show-run-facts" : inputdata.showRunFacts})
            self.control.update({"show-parameter-fact-list" : inputdata.show_parameter_fact_list})
            self.control.update({"file-fact-list" : inputdata.file_fact_list})
            self.control.update({"decode-btrace-fact-list" : inputdata.decode_btrace_fact_list})
            self.control.update({"logging-trigger-list" : inputdata.logging_trigger_list})
            self.control.update({"fact-filter" : inputdata.fact_filter})
            self.control.update({"dict-filter" : inputdata.dict_filter})
            self.control.update({"clear-filter" : inputdata.clear_filter})
            self.control.update({"notification-triggers" : inputdata.notification_triggers})
            self.control.update({"syslog-triggers" : inputdata.syslog_triggers})
            self.control.update({"edit-configs" : inputdata.edit_configs})
    #
    # optional configuration content
    #
            try:
                self.control.update({"action-fact-list" : inputdata.action_fact_list})

                self.telem_list = inputdata.telem_list
                self.telemetry_config = inputdata.telemetry_config
                self.cli_cmd = inputdata.cli_cmd
                self.translation_dict = inputdata.translation_dict
                self.append_slot_dict = inputdata.append_slot_dict
            except: pass
        except Exception as e:
            self.print_log('%%%%% DDR Error: ddr-facts file read error: ' + str(e))
            sys.exit(0)

        self.memory_use("**** DDR Memory: On Entry(kb): ", "entry")
    #
    # If this DDR instance is run on a platform with the 'cisco-ios-xe-ddr-control.yang" model
    # control configurations can be read from the device NETCONF datastore
    # DDR uses the device credentials contained in the FACTS file to connect to the device NETCONF instance
    #
        if self.control["model-control"] == 1:            
            self.get_ddr_control()

    ######################################################################################
    ######################################################################################
    #
    # If test-facts=0, fact data is collected from devices and device connections must
    # be established.  If test-facts=1, simulated facts are read from the ddr-tests file
    #
    ######################################################################################
    ######################################################################################
        if self.control["test-facts"] == 0:
    #
    # If required delay before attempting the first device connection
    #
            if self.control["startup-delay"] != 0:            
                self.print_log("**** DDR Notice: Delay Seconds before Connecting to Device: " + str(self.control["startup-delay"]))
                time.sleep(self.control["startup-delay"])
                self.print_log("**** DDR Notice: Startup Connect Delay Complete")
    #
    # Connect to device identified as the management device
    #
            retry_count = 1
            connect_success = False
            while retry_count <= self.control["retry-count"]:
    #
    #    Try multiple times to connect (may be delayed for DHCP lease)
    #
                try:
                    self.netconf_connection = (ncclient.manager.connect_ssh(host=self.control["mgmt-device"][0], port=self.control["mgmt-device"][1],
                        username=self.control["mgmt-device"][2],
                        password=self.control["mgmt-device"][3],
                        hostkey_verify=False,
                        look_for_keys=False,
                        allow_agent=False,
                        timeout=self.control["nc-timeout"]))
    #
    # If connections successful break out of the retry loop
    #
                    connect_success = True
                    break
    #
    # If connection failed, retry after waiting "retry-time" seconds
    #
                except Exception as e:
                    timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
                
                    self.print_log("%%%%% DDR Error: Unable to connect to management device: " + str(self.control["mgmt-device"][0]) + " retry: " + str(retry_count) + " at: " + str(timestamp) + " " + str(e))
                    retry_count = retry_count + 1
                    time.sleep(self.control["retry-time"])
    #
    # If device connections failed exit
    #
            if connect_success == False:
                timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
                self.print_log("%%%%% DDR Error: Unable to connect management device: " + str(self.control["mgmt-device"][0]) + " at: " + str(timestamp))
                self.close_sessions()
                sys.exit(0)
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
        
            self.print_log("**** DDR Notice: Connected to management device: " + str(self.control["mgmt-device"][0]) + " at: " + str(timestamp))

    #
    # If run-mode requires waiting for notifications create NETCONF listener for snmpevents stream
    # Create a notification listener for run-mode 1, timed execution, to support if the runmode is changed by a rule to 2
    #
            if (self.control["run-mode"] == 2) or (self.control["run-mode"] == 1):
                retry_count = 1
                connect_success = False
                while retry_count <= self.control["retry-count"]:
    #
    #    Try multiple times to connect for notifications (may be delayed for DHCP lease)
    #
                    try:
                        self.notify_conn = (ncclient.manager.connect_ssh(host=self.control["mgmt-device"][0], port=self.control["mgmt-device"][1],
                            username=self.control["mgmt-device"][2],
                            password=self.control["mgmt-device"][3],
                            hostkey_verify=False,
                            look_for_keys=False,
                            allow_agent=False,
                            timeout=self.control["nc-timeout"]))
                        self.notify_conn.async_mode = False
                        self.notify_conn.create_subscription(stream_name='snmpevents')
    #
    # If connections successful break out of the retry loop
    #
                        connect_success = True
                        break
    #
    # If connection failed, retry after waiting "retry-time" seconds
    #
                    except Exception as e:
                        timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
                    
                        self.print_log("%%%%% DDR Error: Unable to connect to notification stream from: " + str(self.control["mgmt-device"][0]) + " retry: " + str(retry_count) + " at: " + str(timestamp) + " " + str(e))
                        retry_count = retry_count + 1
                        time.sleep(self.control["retry-time"])
    #
    # If device connection failed exit
    #
                if connect_success == False:
                    timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
                
                    self.print_log("%%%%% DDR Error: Unable to connect to notification stream from: " + str(self.control["mgmt-device"][0]) + " at: " + str(timestamp))
                    self.close_sessions()
                timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
                self.print_log("**** DDR Notice: Connected to notification stream from: " + str(self.control["mgmt-device"][0]) + " at: " + str(timestamp))

    ###########################################################################
    #
    # Connect to devices used to implement the use case
    # Create a list with entries for each device containing the
    # ncclient connection object.  The indexes into the 'device' list
    # are one of the parameters in configurations in the 'facts.txt' instructions
    # for collecting FACT information from devices
    # If control["test-facts"] is not set, connect to the devices
    #
    ###########################################################################
            retry_count = 1
            connect_success = False
            while retry_count <= self.control["retry-count"]:
                try:
                    self.device = []
                    for device_dat in self.control["device-list"]:
                        self.device.append(ncclient.manager.connect_ssh(host=device_dat[0], port=device_dat[1],
                                    username=device_dat[2],
                                    password=device_dat[3],
                                    hostkey_verify=False,
                                    look_for_keys=False,
                                    allow_agent=False,
                                    timeout=self.control["nc-timeout"]))
    #
    # If connections successful break out of the retry loop
    #
                    connect_success = True
                    break
    #
    # If connection failed, retry after waiting "retry-time" seconds
    #
                except Exception as e:
                    timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
                
                    self.print_log("%%%%% DDR Error: Unable to connect to device-list entry: " + str(device_dat[0]) + " retry: " + str(retry_count) + " at: " + str(timestamp) + " " + str(e))
                    retry_count = retry_count + 1
                    time.sleep(self.control["retry-time"])
    #
    # If device connections failed exit
    #
            if connect_success == False:
                timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
                self.print_log("%%%%% DDR Error: Unable to connect to device-list entries at: " + str(timestamp))
                self.close_sessions()
                sys.exit(0)
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
        
            self.print_log("**** DDR Notice: Connected to all device-list entry devices at: " + str(timestamp))

    ################################################################################################
    ################################################################################################
    #
    # Read test facts to assert if 'test-facts' is configured
    # The list of list of facts will be available in self.test_data
    #
    ################################################################################################
    ################################################################################################

        self.test_index = 0
        if self.control["test-facts"] == 1:
            self.memory_use("**** DDR Memory: Before Loading Test Facts(kb): ", "before-test-facts")

            try:
                if not os.path.exists(self.control["test-file"]):
                    self.print_log('%%%%% DDR Error: Test fact file does not exist')

                else:
                    with open(self.control["test-file"]) as file:
                        self.test_data = file.readlines()
                        self.test_data = [line.rstrip() for line in self.test_data]
            except Exception as e:
                self.print_log('%%%%% DDR Error: Test fact file read error: ' + str(e))
            self.memory_use("**** DDR Memory: After Loading Test Facts(kb): ", "after-test-facts")

    ################################################################################################
    ################################################################################################
    #
    # Read sim-facts to assert in rules if 'sim-facts = 1' is configured
    # The list of list of sim-facts will be available in self.sim_data
    #
    ################################################################################################
    ################################################################################################

        if self.control["sim-facts"] == 1:
            self.memory_use("**** DDR Memory: Before Loading Sim Facts(kb): ", "before-sim-facts")
            try:
                if not os.path.exists(self.control["sim-file"]):
                    self.print_log('%%%%% DDR Error: sim fact file does not exist')

                else:
                    with open(self.control["sim-file"]) as file:
                        self.sim_data = file.readlines()
                        self.sim_data= [line.rstrip() for line in self.sim_data]
                        if self.control["debug-fact"] == 1:
                            self.print_log("**** DDR Debug: sim_data file content: \n" + str(self.sim_data))

                        fd.close()
            except Exception as e:
                self.print_log('%%%%% DDR Error: Sim fact file read error: ' + str(e))
            self.memory_use("**** DDR Memory: After Loading Sim Facts(kb): ", "after-sim-facts")
    
    ################################################################################################
    #
    # Read action facts to assert if 'action-facts' is configured
    # The list of list of facts will be available in self.action_data
    #
    ################################################################################################
    ################################################################################################

        #self.action_index = 0
        #if self.control["action-facts"] == 1:
        #    print(f"karthik inside action facts \n")
        #    self.memory_use("**** DDR Memory: Before Loading Action Facts(kb): ", "before-action-facts")
        #    try:
        #        fd = open(self.control["action-file"])
        #        actiondata = imp.load_source('actiondata', self.control["action-file"], fd)
        #        fd.close()
        #        self.action_data = actiondata.action_data
        #        print(f"karthik inside action facts \n {self.action_data}")
        #    except Exception as e:
        #        self.print_log('%%%%% DDR Error: Action fact file read error: ' + str(e))
        #    self.memory_use("**** DDR Memory: After Loading Action Facts(kb): ", "after-action-facts")

    ###########################################################################################
    #
    # Initialize CLIPS environment
    #
    ###########################################################################################
        try:
            self.memory_use("**** DDR Memory: Before Creating CLIPs env(kb): ", "before-clips-env")
            self.env = clips.Environment()
            if self.control["debug-action"] == 1:
                self.print_log("**** DDR Debug: CLIPs Environment: " + str(self.env))

        except:
            self.print_log("%%%%% DDR Error: Failed to clear inference engine")
            sys.exit(0)
        self.memory_use("**** DDR Memory: After Creating CLIPs env(kb): ", "after-clips-env")

    ###########################################################################################
    #
    # action_functions - Names of any python functions used as control["actions"] in CLIPS rules
    # The functions in the list must be defined before being referenced below
    # The functions could be defined in a library that is imported by the basic script.
    # As new action functions are created they are added to the library
    # dynamically load action functions
    ###########################################################################################
        try:
            action_functions = self.get_action_functions()
            for function in action_functions:
                if self.control["debug-action"] == 1:
                    self.print_log("**** DDR Debug: intialize action functions: " + str(function))
                self.env.define_function(function)
        except Exception as e:
            self.print_log("%%%%% DDR Error: registering action_functions: " + str(e))
            self.close_sessions()
            sys.exit(0)
    #
    # Load the clips constructs file including rules deftemplates and deffacts
    #
        try:
            self.memory_use("**** DDR Memory: Before load CLIPS rules in KBytes: ", "before-rules-load")
            self.env.load(self.control["rules-file"])
            self.memory_use("**** DDR Memory: After load CLIPS rules in KBytes: ", "after-rules-load")

            self.print_log("**** DDE Notice: Inference Engine rules file loaded: " + self.control["rules-file"])
        except Exception as e:
            self.print_log("%%%%% DDR Error: failed to load inference engine rules" + str(e))
            self.close_sessions()
            sys.exit(0)

        try:
            self.env.reset() # Initialize any "deffacts" defined in the ddr-rules CLIPS file
            self.print_log("**** DDR Notice: Inference Engine Reset")
        except:
            self.print_log("%%%%% DDR Error: failed resetting inference engine")
            self.close_sessions()
            sys.exit(0)

    #
    # Assert initial facts defined in the FACTS configuration file
    #
        starttime = time.time()
        if self.control["initial-facts-list"] != []:
            self.get_initial_facts()
    #
    # Read and assert facts in the init-facts-file
    #
        if self.control["initial-facts"] == 1:
            self.memory_use("**** DDR Memory: After Loading Initial Facts(kb): ", "before-load-init-facts")

            try:
                fd = open(self.control["init-facts-file"], 'r')
                facts = fd.readlines()
                fd.close()
                self.memory_use("**** DDR Memory: After Loading Initial Facts(kb): ", "after-load-init-facts")

    #
    # Assert each initial fact
    #
                try:
                    for fact in facts:
                        if self.control["debug-fact"] == 1:
                            self.print_log("**** DDR Debug: Assert init-file fact: " + str(fact))
                        try:
                            self.env.assert_string(str(fact))
                        except: pass #ignore case where FACT already exists
                except Exception as e:
                    self.print_log('%%%%% DDR Error: init-file-fact assert error' + str(e))
            except Exception as e:
                self.print_log('%%%%% DDR Error: init-file-fact read error: ' + str(e))
    #
    # Log time to load any inital facts
    #
        endtime = time.time()
        fact_runtime = endtime - starttime
        self.ddr_timing('**** DDR Time: to load and assert initial facts(ms): %8.3f', fact_runtime, "load-initial-facts")
        self.memory_use("**** DDR Memory: After Asserting Initial Facts(kb): ", "after-assert-init-facts")


    ##############################################################################################
    #
    # set_ddr_runmode - 
    #  This function checks the facts file at the end of each execution loop to determine if the runninng
    #  mode of DDR should be changed and to change debug logging settings
    #
    #  To change these options, update the FACTs file on the device with new settings for the parameters
    #  At the end of the main execution loop, the FACT file will be read and these parameters updated
    # 
    #        standbyActive - 0/do not perform action, 1/perform normal DDR actions, 2/terminate DDR
    #        runMode - 0/run continously using the updatePeriod delay between cycles
    #        updatePeriod - Set the continuous running mode delay before the next cycle starts in ms
    #        debugCLI - 1/log CLI command processing
    #        debugSyslog - 1/log Syslog processing actions
    #        debugFact - 1/log FACT generation actions
    #        debugConfig - 1/generate debug logging for configuration actions               
    #
    ##############################################################################################
    def set_ddr_runmode(self):
    #
    # Read the facts file and update parameters that control DDR execution behavior
    #
        try:
    #
    # Read the FACTs definitions from the facts-file
    #
            fd = open(self.control["facts-file"])
            inputdata = imp.load_source('inputdata', self.control["facts-file"], fd)
            fd.close()
        except Exception as e:
            self.print_log('%%%%% DDR Error: DDR  Intialization files read error: ' + str(e))
            sys.exit(0)
    #
    # If this DDR instance control parameters will be loaded
    # from files set the 'control' dictionary parameters
    #
        try:
            self.control.update({"standby-active" : inputdata.standbyActive})
        except:
            self.control.update({"standby-active" : 1})
    #
    # Update any changes in other running control flags
    #            
        self.control.update({"debug-CLI" : inputdata.debugCLI})
        self.control.update({"debug-notify" : inputdata.debugNotify})
        self.control.update({"debug-syslog" : inputdata.debugSyslog})
        self.control.update({"debug-fact" : inputdata.debugFact})
        self.control.update({"debug-config" : inputdata.debugConfig})

    ##############################################################################################
    #
    # run_write_to_syslog - This function sends the input syslog_message to the host through
    # serial device /dev/ttyS2 as described here: 
    # https://developer.cisco.com/docs/iox/#!app-hosting-loggingtracing-services/iox-logging-tracing-facility
    #
    ##############################################################################################
    def run_write_to_syslog(self, syslog_message, delay_ms):
        try:
            if int(delay_ms) != 0: time.sleep(delay_ms/1000)
            if str(self.control["local-path"]) == 'TRUE':
                timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
                
                outfile = str(self.control["notify-path"]) + str(self.control["use-case"]) + "_" + timestamp
                tty_fd = os.open(outfile, os.O_WRONLY | os.O_CREAT)
            else:
                tty_fd = os.open(str(self.control["notify-path"]), os.O_WRONLY)
            numbytes = os.write(tty_fd, str.encode(syslog_message + "\n"))
            os.close(tty_fd)
        except Exception as e:
            self.print_log('%%%%% DDR Error: Writing Syslog notification ' + str(e))

    ##############################################################################################
    #
    # memory_use - Measure and display memory used by the DDR Python script
    #
    ##############################################################################################
    def memory_use(self, message, key):
        if self.control["show-memory"] == 1:
            try:
                usage = resource.getrusage(resource.RUSAGE_SELF).ru_maxrss
                self.print_log(message + str(usage))
                self.memory[key] = usage
            except:
                pass

    ##############################################################################################
    #
    # timing - Report execution time used by DDR Python script
    #
    ##############################################################################################
    def ddr_timing(self, message, runtime, key):
        if self.control["show-timing"] == 1:
            try:
                self.print_log(message%(runtime*1000))
                self.timing[key] = runtime*1000
            except:
                pass

    ##############################################################################################
    #
    # run_assert_message - This function displays messages sent from triggered rules to the console
    # and also adds the messages as service-tags in the service-impact notification
    #
    ##############################################################################################
    def run_assert_message(self, string):
        if self.control["show-messages"] == 1:
            self.print_log("#### DDR Rule Message: " + string)
        if self.control["send-messages"] == 1:
    #
    # Add the message to a list that will be inserted into the service impact notification
    #
            try:
                self.impactList.append("      <message>")
                self.impactList.append(string)
                self.impactList.append("</message>")
                self.impactList.append("\n")
            except Exception as e:
                self.print_log("%%%% DDR Error: run_assert_message error: " + str(e))
        return

    ##############################################################################################
    #
    # run_assert_message_syslog - This function displays messages sent from triggered rules to the console,
    # adds the messages as service-tags in the service-impact notification, and generates a Syslog message
    #
    ##############################################################################################
    def run_assert_message_syslog(self, string):
        if self.control["show-messages"] == 1:
            self.print_log("#### DDR Rule Message: " + string)
        if self.control["send-messages"] == 1:
    #
    # Add the message to a list that will be inserted into the service impact notification
    #
            try:
                self.impactList.append("      <message>")
                self.impactList.append(string)
                self.impactList.append("</message>")
                self.impactList.append("\n")
            except Exception as e:
                self.print_log("%%%% DDR Error: run_assert_message error: " + str(e))
    #
    # Send a Syslog message
    #
            try:
                syslog_notification = "DDR_MESSAGE " + str(self.control["use-case"]) + ": " + string
                self.run_write_to_syslog(syslog_notification, 0)
            except Exception as e:
                self.print_log("%%%% DDR Error: run_assert_message_syslog error: " + str(e))
        return

    #########################################################################################
    #
    # Create a logging file
    #
    #########################################################################################
    def make_logger(self, path):

        logger = logging.getLogger('ddr Logger')
        logger.setLevel(logging.DEBUG)

        handler = logging.handlers.SysLogHandler('/var/log')
        handler_rot = RotatingFileHandler(path, maxBytes=10*1024*1024, backupCount=5)

        l_format = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
        handler.setFormatter(l_format)

        logger.addHandler(handler_rot)

        return logger


    ########################################################################################
    #
    # get_template_multifacts_protofact - For each element in the element_list substitute the element
    # value, for example the name of an interface, as the key in a template used to read
    # multiple facts from a device
    #
    ########################################################################################
    def get_template_multifacts_protofact(self, fact, slot, slot_value):
      try:
        get_result = self.device[fact['device_index']].get(filter=('subtree', str(fact['path'])))
        if self.control["debug-fact"] == 1:
            self.print_log("**** DDR Debug: get_template_multifacts_protofact get result: \n" + str(get_result))
        instances = xml.dom.minidom.parseString(get_result.xml).getElementsByTagName(fact["assert_fact_for_each"])

        # put all instances with desired fields in "filtered_instances"
        if "element_list" in fact:
            filtered_instances = []
            for each in instances:
                found = False
                slots = fact['protofact']["slots"]
                for slot in slots:
                    if found: break
                    if ("hardcoded_list" in fact) and (slot in fact["hardcoded_list"]): continue
                    node_list = each.getElementsByTagName(slots[slot])
                    if node_list:
                        node = node_list[0]
                    elif "/" in slots[slot]:
                        node = self.get_upper_value(each, slots[slot])
                    else: continue
                    for category in fact["element_list"]:
                        if slot == category:
                            if (str(node.firstChild.nodeValue).replace(" ", "") in fact["element_list"][category]):
                                filtered_instances.append(each)
                                found = True
                                break
        else:
            filtered_instances = instances

        # collect info for each instance and then assert it as a template fact
        for each in filtered_instances:
            protofact = copy.deepcopy(fact['protofact'])
            slots = fact['protofact']["slots"]
            for slot in slots:
                if ("hardcoded_list" in fact) and (slot in fact["hardcoded_list"]): continue
                node_list = each.getElementsByTagName(slots[slot])
                if node_list:
                    node = node_list[0]
                elif "/" in slots[slot]:
                    # for upper tags, node_list would have been empty above
                    node = self.get_upper_value(each, slots[slot])
                else: continue

                value_str = node.firstChild.nodeValue
                if value_str.isdecimal():
                    protofact["slots"][slot] = int(value_str)
                else:
                    protofact["slots"][slot] = value_str  
            self.assert_template_fact(protofact, slot, slot_value)
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: get_protofact: " + str(protofact))
        return "success"
      except Exception as e:
          self.print_log("%%%%% DDR Error: get_template_protofact: " + str(e))
          
    ##############################################################################
    #
    # get_upper_value - Get the value for model nodes above the target nodes
    #                   These nodes are normally keys in a nested list above the
    #                   target nodes
    #
    ##############################################################################
    def get_upper_value(self, node, upper_tag_key_combo):
        tag_key_list = upper_tag_key_combo.split("/")
        upper_tag = tag_key_list[0]
        key = tag_key_list[1]
        cur_node = node
        while cur_node.tagName != upper_tag:
            try:
                cur_node = cur_node.parentNode
            except:
                self.print_log("%%%%% DDR Error: Protofact could not find parent: " + str(upper_tag_key_combo))
                return
        return cur_node.getElementsByTagName(key)[0]

    def get_template_multifacts(self, fact, slot, slot_value):
        '''
            Use a NETCONF get operation to read operational or configuration model content.
            If the data is in a list, create a fact for each list entry optionally filtered using a list of key names.
            Extract identified leaf values for the response and insert into the facts

            {"fact_type": "multitemplate",
            "data": ["multitemplate", 0, "CAT9K-24", # fact type, index into device_list, device name
            """<interfaces xmlns='http://cisco.com/ns/yang/Cisco-IOS-XE-interfaces-oper'> # cut and paste from a YANG view of the model (suggest use YangSuite)
                    <interface>
                      <name/>
                      <statistics>
                        <in-errors/>
                        <in-crc-errors/>
                        <in-errors-64/>
                      </statistics>
                    </interface>
                  </interfaces>""",
            "interface-stats", "interface", #first element is fact template name, 2nd element is list name given by user
            ["name", "in-errors", "in-crc-errors", "in-errors-64"], # names of leafs that end in /> generated if /> at end
            ["name", "in-errors", "in-crc-errors", "in-errors-64"], # names of the slots in the interface-stats fact generate deftemplate slot names with same values as leaf names
            [['fact-name1', 'value1'], ['fact-name2', 'value2']], # list of facts to assert when this template is executed
            ['GigabitEthernet1', 'GigabitEthernet2']] # list of key filters, only return data if key (name) matches a list entry
            }

            :param fact: List of parameters used to control fact collection
               [0] - Fact type
               [1] - device_list index for device to perform operation
               [2] - device name to include in generated fact
               [3] - RPC content for get operation to get the required leafs
               [4] - name of deftemplate for the fact that will be generated
               [5] - name of the 'key' leaf.  One fact will be generated for each instance of the key
               [6] - parameter names in the RPC which will be extracted and asserted as slots in the fact
               [7] - slot names in the deftemplate to receive the parameters extracted from the get result
               [8] - optional list of additional facts to assert when this template is executed
               [9] - list of key values to use as filters to generate facts only for keys that match the list, if empty [] generate facts for all keys
        
        Usage::

              get_template_multifacts(self, test_fact, device, 'CAT9K-24')

            :raises none:

        '''
        try:
            device_id = self.device[fact[1]]
            device_name = fact[2]
            path =  fact[3]
            template = fact[4]
            key = fact[5]
            leafs = fact[6]  
            facts = fact[7]
            assert_list = fact[8] # list of facts to assert when this template is executed
            element_list = fact[9]
            totaltimestart = time.time()
        except Exception as e:
            self.print_log("%%%%% DDR Error: Get template multifact test_fact error: \n" + str(fact) + "\n" + str(e))
       
        try:
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: get_fact parameters: " + str(path) + " " + str(key) + " " + str(leafs) + " " + str(facts) + " " + str(assert_list) + " " + str(element_list))
    #
    # If there are facts that must be asserted for this call assert all facts in the required templates
    # The list contains entries of this form [[template_name, slot_name, slot_value]]
    #
            for assert_fact in assert_list:
                try:
                    self.env.assert_string("(" + str(assert_fact[0]) + " (" + str(assert_fact[1]) + " " + str(assert_fact[2]) + "))")
                except: pass #ignore case where FACT already exists
    #
    # get data for multiple instances
    # instances will be a list of the objects which match the "key"
    #
            get_result = device_id.get(filter=('subtree', str(path)))
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: get_template result: " + str(get_result))
            instances = xml.dom.minidom.parseString(get_result.xml).getElementsByTagName(key)
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: instances: " + str(instances))
    #
    # Filter entries in multitemplate list if a filter is specified to limit fact collection
    #
            if element_list != []:
                filtered_elements = []
                filtered_instances = []
                for each in instances:
                    instance = []
                    for leaf in leafs:
                        for node in each.getElementsByTagName(leaf):
                            if str(node.firstChild.nodeValue).replace(" ", "") in element_list:
                                if self.control["debug-fact"] == 1:
                                    self.print_log("**** DDR Debug: Multitemplate instance in list: " + str(node.firstChild.nodeValue))
                                filtered_elements.append(str(node.firstChild.nodeValue).replace(" ", ""))
                                filtered_instances.append(each)
                element_list = filtered_elements
                instances = filtered_instances

            instance_list = []
            for each in instances:
                if self.control["debug-fact"] == 1:
                    self.print_log("**** DDR Debug nodes: " + str(each.childNodes))
                instance = []
                for leaf in leafs:
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: leaf: " + str(leaf))
                    for node in each.getElementsByTagName(leaf):
                        if self.control["debug-fact"] == 1:
                            self.print_log("**** DDR Debug get_template_fact node: " + str(node.firstChild.nodeValue))
                        instance.append(node.firstChild.nodeValue)
                if instance != []:
                    instance_list.append(instance)
                instance = []
    #
    # For each instance in the list assert the information read as facts
    #
            for each in instance_list:
                flist = ["(", str(template)]
                j = 0
                try:
                    flist.append(" (")
                    flist.append("device")
                    flist.append(" ")
                    flist.append(str(device_name))
                    flist.append(")")
                    for fact in facts:
                        factlist_save = flist
                        try:
                            flist.append(" (")
                            flist.append(str(fact))
                            flist.append(" ")
                            flist.append(str(each[j]).replace(" ", ""))
                            flist.append(")")
                            j = j + 1
    #
    # Some instances of a fact object may not implement all keys causing a mismatch in length
    # catch exceptions and set missing fact values to 0
    #
                        except:
                            self.print_log("%%%%% DDR Error: Fact template: " + str(template) + " " + str(each[0]) + " requires a value not returned: " + str(fact))
                            flist.append("0")
                            flist.append(")")
                            j = j + 1
    # 
    # Append extra "slot" if there is a slot and slot_value passed in to the function
    #   
                    if slot != 'none':
                      slot_fact = " (" + str(slot) + " " + str(slot_value) + ")"
                      flist.append(slot_fact)
                    flist.append(")") #add closing paren for the FACT
    #
    # Join all strings in list into one string and assert as a FACT
    #
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: Assert template fact: ")
                        self.print_log(''.join(flist))
                    try:
                        self.env.assert_string(''.join(flist))
                    except: pass #ignore case where FACT already exists
                except Exception as e:
                    self.print_log(str(e))
                    self.print_log("%%%%% DDR Error: Error asserting template fact: " + fact + "  " + flist)
    #
    # If there are no instances found, create an empty FACT to indicate no data found
    #
            if len(instance_list) == 0:
                try:                
                    flist = ["(", str(template)]
                    flist.append(" (")
                    flist.append("device")
                    flist.append(" ")
                    flist.append(str(device_name))
                    flist.append(")")
                    flist.append(")")
    #
    # Join all strings in list into one string and assert as a FACT
    #
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: Assert template fact: ")
                        self.print_log(''.join(flist))
                    self.env.assert_string(''.join(flist))
                except: pass #ignore case where FACT already exists

        except Exception as e:
            self.print_log("%%%%% DDR Error: Get template multifact error: " + str(e))
            self.print_log("%%%%% DDR Error: No value found in get_template_multifacts")
        return "success"

##############################################################################
#
#
# show_and_assert_fact -
#            Execute a show command, parse the show command output,
#            and assert FACTS based on the output
#            Support using ssh for device access (for offbox use cases) or use the
#            Python cli package.
#            Function is backward compatible with older FACT files that do not have the
#            the "access-method" dictionary entry in the FACT definition
#            Return "success" on success else return error string
#
#############################################################################
    def show_and_assert_fact(self, fact):
    #
    # Determine if ssh should be used to access device
    # Execute in try clause for backward compatibility with older FACT files
    #
        try:
            if str(fact["access-method"]) == 'ssh':
                device_info = self.control["device-list"][int(fact["device-index"])]
                device_address = str(device_info[0])
                device_user = str(device_info[2])
                device_pass = str(device_info[3])
                options = '-q -oStrictHostKeyChecking=no -oUserKnownHostsFile=/dev/null -oPubkeyAuthentication=no'
    #
    # Select the device to use
    #
                ssh_cmd = 'ssh %s@%s %s "%s"' % (device_user, device_address, options, str(fact["command"]))
                if self.control["debug-fact"] == 1:
                    self.print_log("**** DDR Debug: run_show_and_assert SSH command: " + str(ssh_cmd) + "\n")
                try:
                    child = pexpect.spawn(ssh_cmd, timeout=15, encoding='utf-8')
                    child.delaybeforesend = None
                    if self.control["debug-CLI"] == 1:
                        child.logfile = sys.stdout
                    child.expect(['\r\nPassword: ', '\r\npassword: ', 'Password: ', 'password: '])
                    child.sendline(device_pass)
                    child.expect(pexpect.EOF)
                    response = child.before
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: run_show_and_assert SSH execution response: " + str(response))
                except Exception as e:
                    child.close()
                    self.print_log("%%%% DDR  show_and_assert ssh or timeout Error: " + str(ssh_cmd) + "\n")
                    return "show_and_assert ssh or timeout Error"

                child.close()
    #
    # process response and generate FACTs
    #
                try:
                    if "error" in response:
                        self.print_log("%%%% DDR Error: error in ssh show_and_assert show command response")
                        return "Error: show_and_assert show command response"
                    parser = genie_str_to_class(fact["genie_parser"])
                    if type(parser) == str:
                        return parser
                    parsed_genie_output = parser.cli(output=response)
                    if self.control["debug-fact"] == 1:
                        self.print_log("\n*** DDR Debug: parsed genie output: " + str(parsed_genie_output))
                    if parsed_genie_output == {}:
                        return "success"
                    sub_dictionary_list = self.find(fact["assert_fact_for_each_item_in"], parsed_genie_output)
                    import copy
                    for sub_dictionary in sub_dictionary_list:
                        for item in sub_dictionary:
                            protofact = copy.deepcopy(fact["protofact"])
                            for slot in protofact["slots"]:
                                value = protofact["slots"][slot]
	#
	# insert the device name into the fact
	#
                                if value == "device":
                                    protofact["slots"][slot] = value.replace("device", fact["device"])
                                elif type(value) == str and "$" in value:
                                    protofact["slots"][slot] = value.replace("$", item)
                            self.assert_template_fact(protofact, 'device', fact["device"], sub_dictionary)
                    return "success"
                except Exception as e:
                    self.print_log("%%%% DDR Error: Exception in ssh show_and_assert_fact response processing: " + str(e))
                    return "Exception in ssh show_and_assert_fact response processing"


            elif str(fact["access-method"]) == 'cli':
                try:
                    response = cli.cli(str(fact["command"]))
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: show_and_assert cli command result \n" + str(response))
                    if "error" in response:
                        self.print_log("%%%% DDR Error: show_and_assert show command response")
                        return "Error: show_and_assert show command response"

                    parser = genie_str_to_class(fact["genie_parser"])
                    if type(parser) == str:
                        return parser
    #
    # convert show command response to parsed dictionary
    #
                    parsed_genie_output = parser.cli(output=response)
                    if self.control["debug-fact"] == 1:
                        self.print_log("\n*** DDR Debug: parsed genie output: " + str(parsed_genie_output))
                    if parsed_genie_output == {}:
                        return "success"

                    sub_dictionary_list = self.find(fact["assert_fact_for_each_item_in"], parsed_genie_output)
                    import copy
                    for sub_dictionary in sub_dictionary_list:
                        for item in sub_dictionary:
                            protofact = copy.deepcopy(fact["protofact"])
                            for slot in protofact["slots"]:
                                value = protofact["slots"][slot]
	#
	# insert the device name into the fact
	#
                                if value == "device":
                                    protofact["slots"][slot] = value.replace("device", fact["device"])
                                elif type(value) == str and "$" in value:
                                    protofact["slots"][slot] = value.replace("$", item)
                            self.assert_template_fact(protofact, 'device', fact["device"], sub_dictionary)
                    return "success"
                except Exception as e:
                    self.print_log("%%%% DDR Error: Exception in cli show_and_assert_fact: " + str(e))
                    return "Error: Exception in cli show_and_assert_fact"
    #
    # If "access-method" not supported, return error
    #
            else:
                self.print_log("%%%% DDR Error: Exception in show_and_assert_fact: Invalid access-method")
                return "Exception in show_and_assert_fact: Invalid access-method"


    #
    # catch exception if using old FACT file without the "access-method" dictionary element
    # Process show_and_assert using the python cli package
    #
        except:
            try:
                response = cli.cli(str(fact["command"]))
                if self.control["debug-fact"] == 1:
                    self.print_log("\n*** DDR Debug: show_and_assert_fact cli response: \n" + str(response))
                if "error" in response:
                    self.print_log("%%%% DDR Error: show_and_assert show command response")
                    return "Error: show_and_assert show command response"

                parser = genie_str_to_class(fact["genie_parser"])
                if type(parser) == str:
                    return parser
                parsed_genie_output = parser.cli(output=response)
                if self.control["debug-fact"] == 1:
                    self.print_log("\n*** DDR Debug: show_and_assert_fact parsed genie output: " + str(parsed_genie_output))
                if parsed_genie_output == {}:
                    return "success"

                sub_dictionary_list = self.find(fact["assert_fact_for_each_item_in"], parsed_genie_output)
                import copy
                for sub_dictionary in sub_dictionary_list:
                    for item in sub_dictionary:
                        protofact = copy.deepcopy(fact["protofact"])
                        for slot in protofact["slots"]:
                            value = protofact["slots"][slot]
	#
	# insert the device name into the fact
	#
                            if value == "device":
                                protofact["slots"][slot] = value.replace("device", fact["device"])
                            elif type(value) == str and "$" in value:
                                protofact["slots"][slot] = value.replace("$", item)
                        self.assert_template_fact(protofact, 'device', fact["device"], sub_dictionary)
                return "success"
            except Exception as e:
                self.print_log("%%%% DDR Error: Exception in show_and_assert_fact: " + str(e))
                return "Error: Exception in show_and_assert_fact"

##############################################################################
#
# find - return nested dictionary value given a dictionary (j) and a string
#		 element (element) in the form of "Garden.Flowers.White"
#
#############################################################################
    def find(self, element, j):
        if element == "":
            return j
        keys = element.split('+')
        rv = j
        for i, key in enumerate(keys):
            if key == '*':
                new_list = []
                new_keys = copy.deepcopy(keys[i+1:]) # all the keys past the * entry
                for entry in rv: # for each entry in the * dictionary
                    new_rv = copy.deepcopy(rv[entry])
                    for new_key in new_keys:
                        new_rv = new_rv[new_key]
                    for e in new_rv:
                        new_rv[e]["upper_value"] = entry
                    new_list.append(new_rv)
                return new_list
            else:
    # normal stepping through dictionary
                rv = rv[key]
        return [rv]

##############################################################################
#
# assert_template_fact - given a protofact, assert the fact into the clips system
# 						 protofact examples:
# protofact = {"template": "person-template", "slots": {"name": "Leeroy", "surname": "Jenkins", "age": 23.1}}
# protofact2 = {"template": "person-template", "slots": {"name": "Leeroy", "surname": "Jenkins", "age": "Leeroy.age"}}
#						 in protofact2, the age is unknown, but can be looked up in a sub_dictionary
#						 sub_dictionary example:
# sub_dictionary = {"Leeroy: {"age": 23.1, "height": 182, "gender": "M"},
#				    "Maria": {"age": 32.5, "height": 160, "gender": "F"}}
#
#############################################################################
    def assert_template_fact(self, protofact, add_slot, slot_value, sub_dictionary=None):
        try:
            template = self.env.find_template(protofact["template"])
            fact1 = template.new_fact()
            for slot, value in protofact["slots"].items():
    #
    # If the "value" is in a subdirectory, look up the final value in the sub_dictionary
    #
                if type(value) is str and "+" in value:
                    value = self.find(value, sub_dictionary)[0]
    #
    # if the value is a string, assert using CLIPs Symbol
    # if value is a number assert as a number
    #
                if type(value) is str:
                    fact1[slot] = clips.Symbol(value)
                else:
                    fact1[slot] = value
    # 
    # Append extra "slot" if there is a slot and slot_value passed in to the function
    #   
            if add_slot != 'none':
                fact1[add_slot] = clips.Symbol(slot_value)
    #
    # Assert the FACT
    #
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: Assert Template FACT: " + str(fact1))
            try:
                fact1.assertit()
            except Exception as e:
                pass
        except Exception as e:
            self.print_log("%%%% DDR Error: Exception assert_template_fact: " + str(e))

    ##############################################################################
    #
    # assert_syslog_fact - Convert the syslog message passed in to a
    #            to a FACT and assert in CLIPS
    #
    # The device syslog message has this form:
    # XR: RP/0/RSP0/CPU0:Apr 26 20:30:07.567 UTC: ifmgr[257]:
    #                    %PKT_INFRA-LINK-3-UPDOWN : Interface Loopback5, changed state to Down
    #
    #    The syslog message fact has this form:
    #       (deftemplate syslog-message
    #         (slot device)
    #         (slot source)
    #         (slot date)
    #         (slot time)
    #         (slot component)
    #         (slot syslog)
    #         (slot content)
    #       )
    #
    #############################################################################
    def assert_syslog_fact(self, fact):
        try:
            if self.control["mgmt-device"][5] == 'XR':
                parser = genie_str_to_class("ParseXRSyslogMessage")
                parsed_genie_output = parser.syslog(message=fact)
            if self.control["mgmt-device"][5] == 'XE':
                parser = genie_str_to_class("ParseXESyslogMessage")
                parsed_genie_output = parser.syslog(message=fact)

            if self.control["debug-syslog"] == 1:
                self.print_log("*** DDR Debug: parsed Syslog output: " + str(parsed_genie_output))
            sub_dictionary = parsed_genie_output["syslog-message"]
            assert_fact = "(syslog-message (device " + str(self.control["mgmt-device"][4]) + ")"

            for key, value in sub_dictionary.items():
                if self.control["debug-syslog"] == 1:
                    self.print_log("**** DDR Debug: key,value: " + str(key) + " " + str(value))
                if self.control["mgmt-device"][5] == 'XE':
                    if key == 'source': value = self.control["mgmt-device"][4]
                assert_fact = assert_fact + " (" + str(key) + " " + str(value) + ")"

            assert_fact = assert_fact + "))"
            if self.control["debug-syslog"] == 1:
                self.print_log("*** DDR Debug: assert_syslog_fact: " + str(assert_fact))
    #
    # Assert the syslog FACT
    #
            try:
                self.env.assert_string(assert_fact)
            except: pass #ignore case where FACT already exists
            return
        except Exception as e:
            self.print_log("%%%% DDR Error: Exception assert_syslog_fact: " + str(e))
            return

    ##############################################################################
    #
    # assert_rfc5277_fact - Convert the RFC5277 notification message generated for a Syslog message 
    #            to a FACT and assert in CLIPS
    #
    # The device syslog message has this form:
    # XR: RP/0/RSP0/CPU0:Apr 26 20:30:07.567 UTC: ifmgr[257]:
    #                    %PKT_INFRA-LINK-3-UPDOWN : Interface Loopback5, changed state to Down
    #
    #    The syslog message fact has this form:
    #       (deftemplate rfc5277-message
    #         (slot device)
    #         (slot source)
    #         (slot date)
    #         (slot time)
    #         (slot component)
    #         (slot syslog)
    #         (slot content)
    #       )
    #
    #############################################################################
    def assert_rfc5277_fact(self, fact):
        try:
            if self.control["mgmt-device"][5] == 'XR':
                parser = genie_str_to_class("ParseRFC5277Message")
                parsed_genie_output = parser.rfc5277(message=fact)
            if self.control["mgmt-device"][5] == 'XE':
                parser = genie_str_to_class("ParseRFC5277Message")
                parsed_genie_output = parser.rfc5277(message=fact)

            if self.control["debug-notify"] == 1:
                self.print_log("*** DDR Debug: parsed RFC5277 output: " + str(parsed_genie_output))
            sub_dictionary = parsed_genie_output["rfc5277-message"]
            assert_fact = "(rfc5277-message (device " + str(self.control["mgmt-device"][4]) + ")"

            for key, value in sub_dictionary.items():
                if self.control["debug-notify"] == 1:
                    self.print_log("**** DDR Debug: key,value: " + str(key) + " " + str(value))
                if self.control["mgmt-device"][5] == 'XE':
                    if key == 'source': value = self.control["mgmt-device"][4]
                if self.control["mgmt-device"][5] == 'XR':
                    if key == 'source': value = self.control["mgmt-device"][4]
                assert_fact = assert_fact + " (" + str(key) + " " + str(value) + ")"

            assert_fact = assert_fact + "))"
            if self.control["debug-notify"] == 1:
                self.print_log("*** DDR Debug: assert_rfc5277_fact: " + str(assert_fact))
    #
    # Assert the syslog FACT
    #
            try:
                self.env.assert_string(assert_fact)
            except: pass #ignore case where FACT already exists
            return
        except Exception as e:
            self.print_log("%%%% DDR Error: Exception assert_rfc5277_fact: " + str(e))
            return

    ##############################################################################
    #
    # assert_statistics_fact
    #
    #############################################################################
    def assert_statistics_fact(self, template_name, statistics):
        try:
            template = self.env.find_template(template_name)
            fact = template.new_fact()
            for key, value in statistics.items():
                fact[key] = value
            fact.assertit()
        except Exception as e:
            self.print_log("%%%% DDR Error: Exception assert_statistics_fact: " + str(e))

    ##############################################################################
    #
    #  print_clips_info - Print facts and rules from main loop
    #
    #############################################################################
    def print_clips_info(self):
        try:
            if self.control["show-facts"] == 1:
                if self.control["fact-filter"] == ['all']:
                    self.print_log("\n####################### FACTs(all) Main - CLIPS Format ########################\n")
                    for fact in self.env.facts(): self.print_log(fact)

                elif self.control["fact-filter"] != ['none']:
                    self.print_log("\n####################### FACTs(filtered)  Main - CLIPS Format ########################\n")
                    for fact in self.env.facts():
                        for ffilter in self.control["fact-filter"]:
                            if str(ffilter) in str(fact): 
                                self.print_log(fact)

            if self.control["show-rules"] == 1:
                self.print_log("\n####################### DDR Activated RULES Main ########################\n")
                for item in self.env.activations(): self.print_log(item)

#
# Show all facts in dictionary encoded format without filtering if dict_filter specifies all FACTS
#
            if self.control["show-dict"] == 1 and self.control["dict-filter"] == ['all']:
                self.print_log("\n####################### FACTs(all) Main - Python Dictionary Format ########################\n")
                for fact in self.env.facts():
                    try:
                        cleanfact = str(fact).split('  ')[-1]
                        templatename = cleanfact.split('(')
                        template = templatename[1].lstrip('(')
                        tempdict = "{'template': '" + str(template) + "', "
                        strfact = str(dict(fact))
                        stripquotes = strfact.split('  ')[-1].strip('"').lstrip("{")
                        finalfact = tempdict + stripquotes
                        self.print_log(finalfact.strip('"'))
                    except Exception as e:
                        self.print_log("%%%% DDR Error: Print Dictionary exception: " + str(e))
                        pass
#
# Show dictionary encoded facts filtered by the dict_filter list unless 'none' is specified in the filter
#
            elif self.control["show-dict"] == 1 and self.control["dict-filter"] != ['none']:
                self.print_log("\n####################### FACTs(filtered) Main - Python Dictionary Format ########################\n")
                for fact in self.env.facts():
                    try:
                        cleanfact = str(fact).split('  ')[-1]
                        templatename = cleanfact.split('(')
                        template = templatename[1].lstrip('(')
                        for dfilter in self.control["dict-filter"]:
                            if str(dfilter) in template: 
                                tempdict = "{'template': '" + str(template) + "', "
                                strfact = str(dict(fact))
                                stripquotes = strfact.split('  ')[-1].strip('"').lstrip("{")
                                finalfact = tempdict + stripquotes
                                self.print_log(finalfact.strip('"'))
                    except Exception as e:
                        self.print_log("%%%% DDR Error: Print Dictionary exception: " + str(e))
                        pass

        except Exception as e:
            self.print_log("%%%% DDR Error: Exception print_clips_info: " + str(e))

    ##############################################################################
    #
    #  print_clips_info_run - Print facts and rules from run_ functions
    #
    #############################################################################
    def print_clips_info_run(self, when_run):
        try:
            if self.control["show-run-facts"] == 1:
                if self.control["fact-filter"] == ['all']:
                    self.print_log("\n####################### FACTs(all) Run Action " + str(when_run) +"- CLIPS Format ########################\n")
                    for fact in self.env.facts(): self.print_log(fact)

                elif self.control["fact-filter"] != ['none']:
                    self.print_log("\n####################### FACTs(filtered) Run Action " + str(when_run) +"- CLIPS Format ########################\n")
                    for fact in self.env.facts():
                        for ffilter in self.control["fact-filter"]:
                            if str(ffilter) in str(fact): 
                                self.print_log(fact)

            if self.control["show-rules"] == 1:
                self.print_log("\n####################### DDR Activated RULES Run Action ########################\n")
                for item in self.env.activations(): self.print_log(item)

#
# Show all facts in dictionary encoded format without filtering if dict_filter specifies all FACTS
#
            if self.control["show-dict"] == 1 and self.control["dict-filter"] == ['all']:
                self.print_log("\n####################### FACTs(all) Run Action - Python Dictionary Format ########################\n")
                for fact in self.env.facts():
                    try:
                        cleanfact = str(fact).split('  ')[-1]
                        templatename = cleanfact.split('(')
                        template = templatename[1].lstrip('(')
                        tempdict = "{'template': '" + str(template) + "', "
                        strfact = str(dict(fact))
                        stripquotes = strfact.split('  ')[-1].strip('"').lstrip("{")
                        finalfact = tempdict + stripquotes
                        self.print_log(finalfact.strip('"'))
                    except Exception as e:
                        self.print_log("%%%% DDR Error: Print Dictionary exception: " + str(e))
                        pass
#
# Send dictionary encoded facts filtered by the dict_filter list unless 'none' is specified in the filter
#
            elif self.control["show-dict"] == 1 and self.control["dict-filter"] != ['none']:
                self.print_log("\n####################### FACTs(filtered) Run Action - Python Dictionary Format ########################\n")
                for fact in self.env.facts():
                    try:
                        cleanfact = str(fact).split('  ')[-1]
                        templatename = cleanfact.split('(')
                        template = templatename[1].lstrip('(')
                        for dfilter in self.control["dict-filter"]:
                            if str(dfilter) in template: 
                                tempdict = "{'template': '" + str(template) + "', "
                                strfact = str(dict(fact))
                                stripquotes = strfact.split('  ')[-1].strip('"').lstrip("{")
                                finalfact = tempdict + stripquotes
                                self.print_log(finalfact.strip('"'))
                    except Exception as e:
                        self.print_log("%%%% DDR Error: Print Dictionary exception: " + str(e))
                        pass

        except Exception as e:
            self.print_log("%%%% DDR Error: Exception print_clips_info: " + str(e))

    ##############################################################################
    #
    #  send_clips_info - Send filtered facts and rules in the service impact notification
    #
    #############################################################################
    def send_clips_info(self):
        try:
            if self.control["send-clips"] == 1:
                if (self.control["fact-filter"] == ['all'] or self.control["fact-filter"] == []):
                    for fact in self.env.facts(): self.clips_facts = self.clips_facts + "      <cfact>" + str(fact) + "</cfact>\n"

                elif self.control["fact-filter"] != ['none']:
                    for fact in self.env.facts():
                        for ffilter in self.control["fact-filter"]:
                            if str(ffilter) in str(fact): 
                                self.clips_facts = self.clips_facts + "      <cfact>" + str(fact) + "</cfact>\n"
#
# Send all facts in dictionary encoded format without filtering if dict_filter specifies 'all'
#
            if self.control["send-dict"] == 1 and (self.control["dict-filter"] == ['all'] or self.control["dict-filter"] == []):
                for fact in self.env.facts():
                    try:
                        cleanfact = str(fact).split('  ')[-1]
                        templatename = cleanfact.split('(')
                        template = templatename[1].lstrip('(')
                        tempdict = "{'template': '" + str(template) + "', "
                        strfact = str(dict(fact))
                        stripquotes = strfact.split('  ')[-1].strip('"').lstrip("{")
                        finalfact = tempdict + stripquotes
                        self.dict_facts = self.dict_facts + "      <dfact>" + json.dumps(finalfact).strip('"') + "</dfact>\n"
                    except Exception as e:
                        if str(fact).find("retract") == -1:
                            self.print_log("%%%% DDR Error: Send Dictionary exception: " + str(fact))
                        pass
#
# Send dictionary encoded facts filtered by the dict_filter list
#
            elif self.control["send-dict"] == 1 and self.control["dict-filter"] != ['none']:
                for fact in self.env.facts():
                    try:
                        cleanfact = str(fact).split('  ')[-1]
                        templatename = cleanfact.split('(')
                        template = templatename[1].lstrip('(')
                        for dfilter in self.control["dict-filter"]:
                            if str(dfilter) in template: 
                                tempdict = "{'template': '" + str(template) + "', "
                                strfact = str(dict(fact))
                                stripquotes = strfact.split('  ')[-1].strip('"').lstrip("{")
                                finalfact = tempdict + stripquotes
                                self.dict_facts = self.dict_facts + "      <dfact>" + json.dumps(finalfact).strip('"') + "</dfact>\n"
                    except Exception as e:
                        if str(fact).find("retract") == -1:
                            self.print_log("%%%% DDR Error: Send Dictionary exception: " + str(fact))
                        pass

        except Exception as e:
            self.print_log("%%%% DDR Error: Exception send_clips_info: " + str(e))


    ##############################################################################
    #
    #  save_dict_facts - Save dictionary facts in the ddr-control dictionary-facts list
    #  (interfact-error-state (slot value) (slot value) (slot value))
    #
    #############################################################################
    def save_dict_facts(self):
        try:
            count = 1
            if self.control["save-dict-facts"] == 1 and self.control["dict-filter"] == ['all']:
                self.nc_facts = '<facts>'
                count = 1
                for fact in self.env.facts():
                    try:
                        cleanfact = str(fact).split('  ')[-1]
                        template = cleanfact.split('(')
                        name = template[1].lstrip('(').rstrip(' ')
                        templeaf = "<template>" + str(name) + "</template><slot-list>"
                        self.nc_facts = self.nc_facts + "<fact><id>" + str(count) + "</id>"
                        self.nc_facts = self.nc_facts + str(templeaf)
                        fact_len = len(template)
    #
    # add each fact to the slot-list
    #
                        index = 2
                        while index <= fact_len - 1:
                            strfact = str(template[index]).rstrip(' ').rstrip(')')
                            stripfact = strfact.split(' ')
                            self.nc_facts = self.nc_facts + "<slots><slot>" + str(stripfact[0]) + "</slot><value>" + str(stripfact[1]) + "</value></slots>"
                            index = index + 1
                        count = count + 1
                        self.nc_facts = self.nc_facts + "</slot-list></fact>"

                    except Exception as e:
                        self.print_log("%%%% DDR Error: save_dict_facts unfiltered: " + str(e))
                        pass
                self.nc_facts = self.nc_facts + '</facts>'
#
# Send dictionary encoded facts filtered by the dict_filter list
#
            elif self.control["save-dict-facts"] == 1 and self.control["dict-filter"] != ['none']:
                self.nc_facts = '<facts>'
                count = 1
                for fact in self.env.facts():
                    try:
                        cleanfact = str(fact).split('  ')[-1]
                        template = cleanfact.split('(')
                        name = template[1].lstrip('(').rstrip(' ')
                        for dfilter in self.control["dict-filter"]:
                            if str(dfilter) == name:
                                templeaf = "<template>" + str(name) + "</template><slot-list>"
                                self.nc_facts = self.nc_facts + "<fact><id>" + str(count) + "</id>"
                                self.nc_facts = self.nc_facts + str(templeaf)
                                fact_len = len(template)
    #
    # add each fact to the slot-list
    #
                                index = 2
                                while index <= fact_len - 1:
                                    strfact = str(template[index]).rstrip(' ').rstrip(')')
                                    stripfact = strfact.split(' ')
                                    self.nc_facts = self.nc_facts + "<slots><slot>" + str(stripfact[0]) + "</slot><value>" + str(stripfact[1]) + "</value></slots>"
                                    index = index + 1
                                count = count + 1
                                self.nc_facts = self.nc_facts + "</slot-list></fact>"

                    except Exception as e:
                        self.print_log("%%%% DDR Error: save_dict_facts filtered: " + str(e))
                        pass
                self.nc_facts = self.nc_facts + '</facts>'

        except Exception as e:
            self.print_log("%%%% DDR Error: Exception save_dict_facts: " + str(e))

    def get_initial_facts(self):
        '''
        Initial facts are defined in ddr use cases to provide basic configuration 
        information required before ddr starts execution
        
        Initial facts may be defined in a "python dictionary" format or may be defined
        as a string of strings to maintain backward compability with older use cases.
        The content of the initial_facts is used to assert facts defined in deftemplates in 
        the ddr use case rules file.  The 'key" for each dictionary entry in the name of the 
        the fact template and the items in the dictionary key/value pairs are used to initialize
        slots in the fact.  For example:
            
           initial_facts = [
           {'devices': {'device': 'CAT9K-24'},
            'thresholds': {'max-threshold': '0', 'used-threshold': '1', 'percent-used-threshold': '1'},
            'reporting': {'healthy': '0', 'degraded': '1'}}]
            
        results in the assertion of these facts in ddr:
        
            (devices (device CAT9K-24))
            (thresholds (max-threshold 0) (used-threshold 1) (percent-used-threshold 1))
            (reporting (healthy 0) (degraded 1))

        The list of lists format for initial facts directly defines the strings that will be asserted as facts
        
            initial_facts_list = [
            '(devices (device CAT9K-24))',
            '(thresholds (max-threshold 0) (used-threshold 1) (percent-used-threshold 1))',
            '(reporting (healthy 0) (degraded 1))'
        '''

    #
    # If the initial facts are in dictionary form generate facts from dictionary content
    #    
        initial_dict = self.control["initial-facts-list"][0]
        if type(initial_dict) is dict:
            for key in initial_dict:
                flist = ["(", str(key)]
                slot_dict = (initial_dict[key])
                try:
                    for slot, value in slot_dict.items():
                        flist.append(" (")
                        flist.append(str(slot))
                        flist.append(" ")
                        flist.append(str(value))
                        flist.append(")")
                    flist.append(")")
                except Exception as e:
                    self.print_log("%%%% DDR Error creating initial fact from dictionary: " + str(key) + " " + str(e))
    #
    # Assert the fact
    #
                fact = ''.join(flist)
                if self.control["debug-fact"] == 1:
                    self.print_log("**** DDR Debug: Assert initial dictionary fact: ")
                    self.print_log(fact)
                try:
                    self.env.assert_string(fact)
                except: pass #ignore case where FACT already exists
    #
    # If the initial-fact-list contains a list of strings assert each string as a fact
    #
        else:
            try:
    #
    # assert a fact for each instance in the fact list
    #
                for fact in self.control["initial-facts-list"]:
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: Assert initial string fact: ")
                        self.print_log(fact)
                    try:
                        self.env.assert_string(fact)
                    except: pass #ignore case where FACT already exists

            except Exception as e:
                self.print_log("%%%%% DDR Error: Error asserting initial string fact: " + str(fact))

##############################################################################################
#
# assert_test_facts
#
##############################################################################################
    def assert_test_facts(self, test_fact):
#
# assert a fact for each instance in the fact list
#
        try:
            fact_count = int(test_fact[0])
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: assert_test_facts - Assert test fact: ")
            for i in range(int(fact_count)):
                  try:
                      self.env.assert_string(str(test_fact[i + 1]))
                  except Exception as e: 
                      self.print_log("**** DDR Debug: assert_test_facts - assert_string: " + str(e))
                      pass #ignore case where FACT already exists  
        except Exception as e:
            self.print_log("%%%% DDR Error: assert_test_facts - Asserting test facts: " + str(e))  
        return

##############################################################################################
#
# assert_action_facts
#
##############################################################################################
    def assert_action_facts(self, action_fact):
#
# assert a fact for each instance in the fact list
#
        try:
            fact_count = int(action_fact[0])
            for i in range(int(fact_count)):
                  if self.control["debug-fact"] == 1:
                      self.print_log("**** DDR Debug: assert_action_facts -Assert Action fact: ")
                      self.print_log(action_fact[i + 1])
                  try:
                      self.env.assert_string(action_fact[i + 1])
                  except: pass #ignore case where FACT already exists  
        except Exception as e:
            self.print_log("%%%% DDR Error: assert_action_facts - Asserting action facts: " + str(e))  
        return

    #############################################################################
    #
    # Close open netconf device sessions and other open connections
    #
    #############################################################################
    def close_sessions(self):
        try:
            self.netconf_connection.close_session()
            self.notify_conn.close_session()
            for dev in self.device:
                dev.close_session()
        except Exception as e:
            self.print_log("%%%% DDR Error: Closing NETCONF sessions: " + str(e))
            sys.exit(0)

    ##############################################################################
    #
    #  print_log - log message to debug control["log-file"] if enabled otherwise only display
    #              logging messages on command line
    #     debug-logging controls logging behavior
    #        0 - Display log messages on stdout
    #        1 - Write log messages to the path defined in the FACT file with a timestamp added
    #        2 - Write log messages to stdout and log file path
    #
    #############################################################################
    def print_log(self, logMessage):
        try: 
            if self.control["debug-logging"] == 0 : print(logMessage)
            if self.control["debug-logging"] >= 1: 
                with open(self.control["log-path"] + self.control["log-file"] + "_" + str(self.control["session-time"]), 'a+') as debuglog_file:
                    debuglog_file.write(str(logMessage) + "\n")
                if self.control["debug-logging"] == 2: print(logMessage)
        except Exception as e:
            print("%%%% DDR Error: Error writing to log file: " + str(e))

######################################################################################################
######################################################################################################
#
# RULE action functions
#
# The following modules are called from the execution section in a CLIPs rule
# When a RULE is triggered 0 or more of these action functions can be called
# Action functions can perform any operation desired by the developer
# Typical action functions collect additional data and assert additional FACTS
#
#    run_cli_command - Runs a CLI command on the host using the Python "cli" package implemented for the host
#    run_show_parameter - Runs a show command on the host using either the Python "cli" library or an SSH connection
#                         Accepts up to 3 parameters from the CLIPs knowledge base to insert into the show command
#    run_cli_parameter - Runs any cli command on the host.  The CLI command can include up to 3 parameters
#    run_show_fact - Run show command defined in the 'show_facts_list' in the facts file using the rule action function run_show_fact
#    run_decode_btrace_log - RULE function processes btrace log, selects and asserts FACTs
#    run_copy_file - RULEs can copy a file from the device and save the file.  This gives the RULE control over persistence (e.g. prevent wrapping/deletion)
#    run_logging_trigger - Rule function that parses the content of the Syslog logging buffer to
#            and asserts facts when buffered log messages contain specific content.
#    run_process_file - Rule function that parses the content of "filename" stored in /bootflash/guest-share and asserts FACTs
#    run_command - Runs a CLI command on the host using the Python "cli" package.  No result is data is collected
#    run_clear_selected_facts - Clear facts in the knowledge base for the template passed in the argument.  Can be used by RULEs to remove types of FACTs
#    run_nc_fact - This function called from a RULE collects data using the NETCONF interface and asserts a FACT
#    run_apply_config - This function called from a RULE applies an edit-config operation to the device
#    run_ddr- This action function runs the CLIPs engine when called from a rule. Normally called when a RULE asserts new FACTs to trigger any RULEs satisfied by new FACTs
#    run_action - This action function invoked from a RULE causes collection of FACTs defined in the "action_facts_list" in the FACTs file
#    run_rule_timer - This action function starts a thread to generate a timer to assert a FACT on a regular time interval
#    run_ping_action - This action function called from CLIPs rules runs a ping command, collects results and asserts a FACT
#    run_trace - This action function called from CLIPs rules runs a traceroute command
#    run_trace_hops - This action function called from CLIPs rules asserts a FACT for the last good hop in a traceroute
#    run_delay - Rule function that delays execution of rules for specified number of seconds
#    run_set_runmode - Rule function used to change the DDR runmode, 0 for timed trigger, 1 for Syslog trigger (not available on XE), 2 for RFC5277 Notification trigger

    def run_delay(self, seconds):
        """
            Rule file function call: (run_delay 5)
            
            Invoked in the "RHS" of a triggered rule.
            This function called from a rule delays the specified number of seconds before continuing the execution of rule actions.
            This function can be called from a rule to insert a fixed delay in a worklow, for example, "delay 60 seconds after starting packet capture"
            could be implemented by:
               Start packet capture
               (run_delay 60)
            
            :param mode - cli/ssh run using python cli package or using ssh 
            :param vrf - vrf-name or none - do not include a vrf in the ping command
            :param neighbor - ip address of neighbor to ping
            :param count - number of pings to run
            :param min-success - minimum % of pings that must succeed
            :param src-interface - interface to run ping command through
            :param pkt_size - ping payload size in bytes
            :param fact_result - name of template to store results
        
        Usage::

              (run_delay 5)

            :raises none:

        """
        if self.control["actions"] == 1:
            if self.control["debug-action"] == 1:
                timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
                self.print_log("\n**** DDR  Debug: run_delay seconds: " + str(seconds) + " starting at: " + str(timestamp))
            time.sleep(float(seconds))
        
    def run_show_fact(self, fact_index):

        if self.control["debug-fact"] == 1:
            self.print_log("**** EMRE Debug: run_show_facts entry: " + str(fact_index))
        fact = self.control["show-fact-list"][int(fact_index)]
        if self.control["debug-fact"] == 1:
                self.print_log("**** EMRE Debug: run_show_facts fact: " + str(fact))
        status = self.show_and_assert_fact(fact)
    
    def run_cli_command(self, cli_index, filename, parameter1, parameter2, addtime):
        """
            Rule file function call: (run_cli_command 0 file_name)

            Invoked in the "RHS" of a triggered rule.
            Runs a CLI command in the guestshell on the management device using the
            Python "cli" package installed by default in the guestshell.  
            
            The response from the command can optionally be redirected,
            for example to bootflashthe path for the redirected output is provided.
            
            The CLI command executed is defined in the 'cli_cmd' list in the ddr-facts file.
            
            cli_cmd = [['flash:guest-share/', 'show device-tracking events | redirect RUN_FILE_NAME ']]
               first element - Prefix for the filename used to store CLI command response if saved to file
               second element - CLI command to execute including optional redirect to a file RUN_FILE_NAME
               RUN_FILE_NAME - string included in rule statement with name for file to subsitute in redirect command
               third element - 

            :param cli_index: 0 origin index into the cli_cmd list for command to execute
            :param filename: Name to include in file name which is prepended with the path argument in the cli_cmd entry and with a timestamp appended
        
        Usage::

            (run_cli_command 0 file_name)

            :raises none:

        """
        if self.control["actions"] == 1:
            command_idx = int(cli_index)
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
        #######################################################################
        #
        # Process each CLI command in the list identified in the RULE callback
        # If filename is not equal to 'none' include the filename argument in the data file name
        #
        #######################################################################
            try:
                cmdline = self.cli_cmd[command_idx]
                if self.control["debug-CLI"] == 1:
                    self.print_log("**** DDR Debug: run_cli_cmd: cmdline: " + str(cmdline))
                if addtime == 'time':
                    timestring = "_" + timestamp
                else:
                    timestring = ""
                if filename == 'none':
                    outfile = cmdline[0] + timestring
                else:
                    outfile = cmdline[0] + "_" + str(filename) + timestring
        #
        # substitute parameters in command string if included in call from rule
        #
                if cmdline[3] == 0:
                    command = str(cmdline[1])
                else:
                    translated_value = self.translation_dict.get(str(parameter1))
                    if not translated_value:
                        return
                    if cmdline[3] == 1:
                        first = translated_value
                        command = str(cmdline[1]).format(first)
                    elif cmdline[3] == 2:
                        (first, second) = translated_value
                        command = str(cmdline[1]).format(first, second)
                    elif cmdline[3] == 3:
                        (first, second) = translated_value
                        third = parameter2
                        command = str(cmdline[1]).format(first, second, third)
        #
        # if command result is written to a file in nvram insert the file name into the cli command
        #
                if cmdline[2] == 'none':
                    out_command = command
                else:
                    out_command = command.replace('RUN_FILE_NAME', str(outfile))
        #######################################################################
        #
        # If the location to save the data is 'nvram' only write the command results to the device flash/bootflash
        #
        #######################################################################
                try:
                    if self.control["debug-CLI"] == 1:
                        self.print_log("**** DDR Debug: run_cli_cmd: out_command: " + str(out_command))
                    cli.cli(out_command)
                    if self.control["debug-CLI"] == 1:
                        self.print_log("**** DDR Notice: CLI Command Executed: " + str(out_command) + " at: " + str(timestamp))
                except Exception as e:
                    self.print_log("%%%% DDR  run_cli_cmd Error: Exception writing to nvram: " + str(e) + "\n")
            except Exception as e:
                self.print_log("%%%% DDR  run_cli_cmd Error: Exception command processing: " + str(e) + "\n")

  
    def run_show_parameter(self, fact_index, access_type, show_template, pcount, par1, par2, par3, no_fact, yes_fact):
        """
            Rule file function call: (run_show_parameter 3 ssh "show platform software fed active vp key {0} {1}" 2 ?key ?vlan none no-fact no-fact)

            Invoked in the "RHS" of a triggered rule.
            Runs a show command in the guestshell on the management device using the
            Python "cli" package installed by default in the guestshell or ssh connection to a device  
            
            The actions performed by the method are defined in the 'show_parameter_fact_list' list in the ddr-facts file.
            Each entry in the list is indexed by the fact_index parameter as a Python dictionary
              fact_type - The type of action function
              device - optionally directly specify the device name otherwise management device used by default
              genie_parser - Name of parser class in genie_parsers.py for processing this show command
              assert_fact_for_each_item_in - Assert a fact for each dictionary entry key at this level in the dictionary
              protofact - prototype describing how to translate parser python dictionary output to a CLIPs fact
              template - name of the deftemplate in ddr-rules defining the fact
              slots - names of the slots/fields in the deftemplate for the fact
              $ - the value of the key in the dictionary.  In the example below local-host is the key and the slot 'local-host' is set to the key value
              $+neighbor - set the neighbor slot in the bgp-keepalive-messages fact to the value of 'neighbor' in the current dictionary entry
            
                {"fact_type": "run_show_parameter",
                 "device": "513E.C.24-9300-2",
                 "genie_parser": "ShowMonitorCapture",
                 "assert_fact_for_each_item_in": "bgp_keepalive",
                 "protofact": {"template": "bgp-keepalive-messages",
                                   "slots": {"local-host": "$",
                                   "neighbor": "$+neighbor",
                                   "message": "$+message"
                                            }
                              }
            }

            :param fact_index: 0 origin index into the show_parameter_fact_list with execution instructions
            :param access_type: 'cli' to use the Python cli package in the guestshell or 'ssh' for ssh device access
            :param show_template: show command template with parameters identified by {0}, {1}, {2}
            :param pcount: Number of parameters to substitute in the show_template 0 to 3
            :param parx: Values for parmeters 1, 2 and 3 which can be variables from the rule or defined values
            :param no_fact: Optional fact to assert if this function did not assert a fact that met rule conditions
            :param yes_fact: Optional fact to assert if this function did assert a fact that met rule conditions
        
        Usage::

              (run_show_parameter 3 ssh "show monitor capture CAP buffer brief" 0 none none none "(fact-found (value false))" "(fact-found (value true))")
              (run_show_parameter 3 ssh "show platform software fed active vp key {0} {1}" 2 ?key ?vlan none no-fact no-fact)

            :raises none:

        """
        if self.control["actions"] == 1:
            index = int(fact_index)
            fact = self.control["show-parameter-fact-list"][int(index)]
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
        #######################################################################
        #
        # Generate the show command with parameters
        #
        #######################################################################
            try:
                cmdline = str(show_template)
                if self.control["debug-CLI"] == 1:
                    self.print_log("**** DDR Debug: run_show_parameter: cmdline: " + str(cmdline))
        #
        # substitute parameters in command string if included in call from rule
        #
                if int(pcount) == 0:
                    command = str(cmdline)
                else:
                    if int(pcount) == 1:
                        command = str(show_template).format(str(par1))
                    elif int(pcount) == 2:
                        command = str(show_template).format(str(par1), str(par2))
                    elif int(pcount) == 3:
                        command = str(show_template).format(str(par1), str(par2), str(par3))
                try:
                    if self.control["debug-CLI"] == 1:
                        self.print_log("**** DDR Debug: run_show_parameter: command: " + str(command))
        #
        # If access-type is cli use the Python CLI package to run command
        #
                    if str(access_type) == 'cli':
                        response = cli.cli(command)
                        if self.control["debug-CLI"] == 1:
                            self.print_log("**** DDR Notice: CLI Command Executed: " + str(command) + " at: " + str(timestamp))
                    else:
        #
        # Use SSH to run the show command
        #
                        try:
                            options = '-q -oStrictHostKeyChecking=no -oUserKnownHostsFile=/dev/null -oPubkeyAuthentication=no'
                            ssh_cmd = 'ssh %s@%s %s "%s"' % (self.control["mgmt-device"][2], self.control["mgmt-device"][0], options, command)
                            if self.control["debug-CLI"] == 1:
                                self.print_log("**** DDR Debug: run_show_parameter SSH command: " + str(ssh_cmd) + "\n")
                            child = pexpect.spawn(ssh_cmd, timeout= 20, encoding='utf-8')
                            child.delaybeforesend = None
                            if self.control["debug-CLI"] == 1:
                                child.logfile = sys.stdout
                            child.expect(['\r\nPassword: ', '\r\npassword: ', 'Password: ', 'password: '])
                            child.sendline(self.control["mgmt-device"][3])
                            child.expect(pexpect.EOF) #result contains the ping result
                            response = child.before

                        except Exception as e:
                            child.close()
                            self.print_log("%%%% DDR  run_show_parameter SSH or timeout Error: " + str(ssh_cmd) + "\n")
                        child.close()
         #
         # Assert the FACTs for using the response
         #
                    fact_found = False
                    try:
                        if "error" in response:
                            self.print_log("%%%% DDR Error: error in run_show_parameter show command response")
                            return "Error: show_and_assert show command response"
                        parser = genie_str_to_class(fact["genie_parser"])
                        if type(parser) == str:
                            return parser
                        parsed_genie_output = parser.parse(output=response)
                        if self.control["debug-CLI"] == 1:
                            self.print_log("**** DDR Debug: run_show_parameter genie: " + str(parsed_genie_output) + "\n")
                        if parsed_genie_output == {}:
                            return "success"
        #
        # Get the starting point in the Python dictionary containing the parsed content
        #
                        sub_dictionary_list = self.find(fact["assert_fact_for_each_item_in"], parsed_genie_output)
                        import copy
                        for sub_dictionary in sub_dictionary_list:
                            for item in sub_dictionary:
                                protofact = copy.deepcopy(fact["protofact"])
                                for slot in protofact["slots"]:
                                    value = protofact["slots"][slot]
	#
	# insert the device name into the fact
	#
                                    if value == "device":
                                        protofact["slots"][slot] = value.replace("device", fact["device"])
                                    elif type(value) == str and "$" in value:
                                        protofact["slots"][slot] = value.replace("$", item)
                                fact_found = True
                                self.assert_template_fact(protofact, 'device', fact["device"], sub_dictionary)
        #
        # Test to see if a FACT should be asserted if no FACT was asserted as a result of executing the command
        #
                            try:
                                if (str(yes_fact) != 'no-fact') and (fact_found == True):
                                    self.env.assert_string(str(yes_fact))
                                if (str(no_fact) != 'no-fact') and (fact_found == False):
                                    self.env.assert_string(str(no_fact))
                            except: pass

                            return "success"

                    except Exception as e:
                        self.print_log("%%%% DDR Error: Exception in run_show_parameter response processing: " + str(e))
                        return "Exception in ssh run_show_parameter response processing"

                except Exception as e:
                    self.print_log("%%%% DDR  run_show_parameter Error: Exception sending show command: " + str(e) + "\n")
            except Exception as e:
                self.print_log("%%%% DDR  run_show_parameter Error: Exception generating show command: " + str(e) + "\n")

    def run_cli_parameter(self, access_type, show_template, pcount, par1, par2, par3):
        """
            Rule file function call: (run_cli_parameter ssh "monitor capture CAP interface {0} out" 1 ?port none none)
            
            Invoked in the "RHS" of a triggered rule.
            Runs an exec command in the guestshell on the management device using the
            Python "cli" package installed by default in the guestshell or ssh connection to a device  
            
            This method is used only to execute commands.  No facts are generated as a result of execution

            :param access_type: 'cli' to use the Python cli package in the guestshell or 'ssh' for ssh device access
            :param show_template: show command template with parameters identified by {0}, {1}, {2}
            :param pcount: Number of parameters to substitute in the show_template 0 to 3
            :param parx: Values for parmeters 1, 2 and 3 which can be variables from the rule or defined values
        
        Usage::

              (run_cli_parameter ssh "monitor capture CAP interface {0} out" 1 ?port none none)

            :raises none:

        """
        if self.control["actions"] == 1:
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
        #######################################################################
        #
        # Generate the show command with parameters
        #
        #######################################################################
            try:
                cmdline = str(show_template)
                if self.control["debug-CLI"] == 1:
                    self.print_log("**** DDR Debug: run_show_parameter: cmdline: " + str(cmdline))
        #
        # substitute parameters in command string if included in call from rule
        #
                if int(pcount) == 0:
                    command = str(cmdline)
                else:
                    if int(pcount) == 1:
                        command = str(show_template).format(str(par1))
                    elif int(pcount) == 2:
                        command = str(show_template).format(str(par1), str(par2))
                    elif int(pcount) == 3:
                        command = str(show_template).format(str(par1), str(par2), str(par3))
                try:
                    if self.control["debug-CLI"] == 1:
                        self.print_log("**** DDR Debug: run_cli_parameter: command: " + str(command))
        #
        # If access-type is cli use the Python CLI package to run command
        #
                    if str(access_type) == 'cli':
                        cli.cli(command)
                        if self.control["debug-CLI"] == 1:
                            self.print_log("**** DDR Notice: CLI Command Executed: " + str(command) + " at: " + str(timestamp))
                    else:
        #
        # Use SSH to run the show command
        #
                        try:
                            options = '-q -oStrictHostKeyChecking=no -oUserKnownHostsFile=/dev/null -oPubkeyAuthentication=no'
                            ssh_cmd = 'ssh %s@%s %s "%s"' % (self.control["mgmt-device"][2], self.control["mgmt-device"][0], options, command)
                            if self.control["debug-CLI"] == 1:
                                self.print_log("**** DDR Debug: run_cli_parameter SSH command: " + str(ssh_cmd) + "\n")
                            child = pexpect.spawn(ssh_cmd, timeout= 20, encoding='utf-8')
                            child.delaybeforesend = None
                            if self.control["debug-CLI"] == 1:
                                child.logfile = sys.stdout
                            child.expect(['\r\nPassword: ', '\r\npassword: ', 'Password: ', 'password: '])
                            child.sendline(self.control["mgmt-device"][3])
                            child.expect(pexpect.EOF) #result contains the ping result
                            response = child.before

                        except Exception as e:
                            child.close()
                            self.print_log("%%%% DDR  run_cli_parameter SSH or timeout Error: " + str(ssh_cmd) + "\n")
                        child.close()

                except Exception as e:
                    self.print_log("%%%% DDR  run_cli_parameter Error: Exception sending show command: " + str(e) + "\n")
            except Exception as e:
                self.print_log("%%%% DDR  run_cli_parameter Error: Exception generating show command: " + str(e) + "\n")

    
    def run_decode_btrace_log(self, fact_index, access_type, show_template, pcount, par1, par2, par3):

        """
            Rule file function call: (run_decode_btrace_log 0 ssh "show platform software trace message {0} {1}" 2 dmiauthd "switch active R0" none)
            
            Invoked in the "RHS" of a triggered rule.
            Runs a show command in the guestshell on the management device using the
            Python "cli" package installed by default in the guestshell or ssh connection to a device
            
            The show command uses 'show platform software trace..." to decode the current btrace log
            file for a process and extracts information from the file to assert facts.  The 'genie_parser' searches
            the lines in the decoded btrace log and asserts facts for each line that match the expressions defined in the parser 
            
            The actions performed by the method are defined in the 'decode_btrace_fact_list' list in the ddr-facts file.
            Each entry in the list is indexed by the fact_index parameter as a Python dictionary
              fact_type - The type of action function
              device - optionally directly specify the device name otherwise management device used by default
              genie_parser - Name of parser class in genie_parsers.py for processing this show command
              assert_fact_for_each_item_in - Assert a fact for each dictionary entry key at this level in the dictionary
              protofact - prototype describing how to translate parser python dictionary output to a CLIPs fact
              template - name of the deftemplate in ddr-rules defining the fact
              slots - names of the slots/fields in the deftemplate for the fact
              $+transaction_id - the value of the key in the dictionary.  In the example below the NETCONF transaction_id is the key and the slot 'transaction-id' is set to the key value
              $+date - set the date slot to the parser dictionary entry for 'date'
              etc.

                  {"fact_type": "show_and_assert",
                   "device": "cat9k-24",
                   "genie_parser": "BtraceDmiauthdConfigI",
                   "assert_fact_for_each_item_in": "config_transaction",
                   "protofact": {"template": "bt-dmi-config",
                                    "slots": {"device": "device",
                                          "transaction-id": "$+transaction_id",
                                          "date": "$+date",
                                          "time": "$+time",
                                          "method": "$+method",
                                          "config-by": "$+config_by"
                                             }
                               }
                 }

            :param fact_index: 0 origin index into the decode_btrace_fact_list with execution instructions
            :param access_type: 'cli' to use the Python cli package in the guestshell or 'ssh' for ssh device access
            :param show_template: show command template with parameters identified by {0}, {1}, {2}
            :param pcount: Number of parameters to substitute in the show_template 0 to 3
            :param parx: Values for parmeters 1, 2 and 3 which can be variables from the rule or defined values
        
        Usage::

              (run_decode_btrace_log 0 ssh "show platform software trace message {0} {1}" 2 dmiauthd "switch active R0" none)

            :raises none:

        """

        if self.control["actions"] == 1:
            index = int(fact_index)
            fact = self.control["decode-btrace-fact-list"][int(index)]

            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
          
        #######################################################################
        #
        # Generate the show command with parameters
        #
        #######################################################################
            try:
                cmdline = str(show_template)
                if self.control["debug-CLI"] == 1:
                    self.print_log("**** DDRDebug: run_decode_btrace_log: cmdline: " + str(cmdline))
        #
        # substitute parameters in command string if included in call from rule
        #
                if int(pcount) == 0:
                    command = str(cmdline)
                else:
                    if int(pcount) == 1:
                        command = str(show_template).format(str(par1))
                    elif int(pcount) == 2:
                        command = str(show_template).format(str(par1), str(par2))
                    elif int(pcount) == 3:
                        command = str(show_template).format(str(par1), str(par2), str(par3))
                try:
                    if self.control["debug-CLI"] == 1:
                        self.print_log("**** DDRDebug: run_decode_btrace_log: command: " + str(command))
        #
        # If access-type is cli use the Python CLI package to run command
        #
                    if str(access_type) == 'cli':
                        response = cli.cli(command)
                        if self.control["debug-CLI"] == 1:
                            self.print_log("**** DDR Notice: Decode Btrace log Command Executed: " + str(command) + " at: " + str(timestamp))
                            
                    else:
        #
        # Use SSH to run the show command
        #
                        try:
                            options = '-q -oStrictHostKeyChecking=no -oUserKnownHostsFile=/dev/null -oPubkeyAuthentication=no'
                            ssh_cmd = 'ssh %s@%s %s "%s"' % (self.control["mgmt-device"][2], self.control["mgmt-device"][0], options, command)
                            if self.control["debug-CLI"] == 1:
                                self.print_log("**** DDR Debug: run_decode_btrace_log SSH command: " + str(ssh_cmd) + "\n")
                            child = pexpect.spawn(ssh_cmd, timeout= 60, encoding='utf-8')
                            child.delaybeforesend = None
                            if self.control["debug-CLI"] == 1:
                                child.logfile = sys.stdout
                            child.expect(['\r\nPassword: ', '\r\npassword: ', 'Password: ', 'password: '])
                            child.sendline(self.control["mgmt-device"][3])
                            child.expect(pexpect.EOF)
                            response = child.before

                        except Exception as e:
                            child.close()
                            self.print_log("%%%% DDR run_decode_btrace_log SSH or timeout Error: " + str(ssh_cmd) + "\n")
                        child.close()

    ##########################################################################################
    #
    # Generate facts using log content
    #
    ##########################################################################################
                    fact_found = False
                    try:
                        parser = genie_str_to_class(fact["genie_parser"])
                        if type(parser) == str:
                            return parser
                        parsed_genie_output = parser.parse(output=response)
                        if self.control["debug-parser"] == 1:
                            self.print_log("**** DDR Debug: run_decode_btrace_log parser result: \n\n" + str(parsed_genie_output) + "\n")
                        if parsed_genie_output == {}:
                            return "success"
        #
        # Get the starting point in the Python dictionary containing the parsed content
        #
                        sub_dictionary_list = self.find(fact["assert_fact_for_each_item_in"], parsed_genie_output)
                        import copy
                        for sub_dictionary in sub_dictionary_list:
                            for item in sub_dictionary:
                                protofact = copy.deepcopy(fact["protofact"])
                                for slot in protofact["slots"]:
                                    value = protofact["slots"][slot]
	#
	# insert the device name into the fact
	#
                                    if value == "device":
                                        protofact["slots"][slot] = value.replace("device", fact["device"])
                                    elif type(value) == str and "$" in value:
                                        protofact["slots"][slot] = value.replace("$", item)
                                fact_found = True
                                self.assert_template_fact(protofact, 'device', fact["device"], sub_dictionary)

                            return "success"

                    except Exception as e:
                        self.print_log("%%%% DDR Error: Exception in run_decode_btrace_log response processing: " + str(e))
                        return "Exception in ssh run_show_parameter response processing"

                except Exception as e:
                    self.print_log("%%%% DDR run_decode_btrace_log Error: Exception sending show command: " + str(e) + "\n")
            except Exception as e:
                self.print_log("%%%% DDR run_decode_btrace_log Error: Exception generating show command: " + str(e) + "\n")
    
    def run_copy_file(self, infile, outfile):
        """
            Rule file function call: (run_copy_file "bootflash:/core/coredump" "bootflash:/guest_share/saved-coredump")
            
            Invoked in the "RHS" of a triggered rule.
            Copies a file from infile location to outfile location.
            Typical use would be to save a log, debug or capture file for further processing or to archive.  
            
            :param infile: full pathname for file to copy
            :param outfile: full pathname for file destination.  A time stamp is appended to the outfile name
        
        Usage::

              (run_copy_file "bootflash:/core/coredump" "bootflash:/guest_share/saved-coredump")
              (run_copy_file ?filename-variable "bootflash:/guest_share/saved-coredump")

            :raises none:

        """
        if self.control["actions"] == 1:
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
            try:
                with open(str(infile), 'r') as rfd:
                    indata = rfd.read()
                    rfd.close()
                out_path = str(outfile) + "_" + str(timestamp)
                with open(out_path, 'w') as wfd:
                    wfd.write(indata)
                    wfd.close()
                if self.control["debug-CLI"] == 1:
                    self.print_log("**** DDR Debug: run_copy_file infile: " + str(infile) + " outfile: " + out_path)
            except Exception as e:
                self.print_log("%%%% DDR  run_copy_file Error: Exception command processing: " + str(e))

    
    def run_logging_trigger(self, access_type, logging_trigger_index, number_lines, facility):

        """
            Rule file function call: (run_logging_trigger ssh 0 4 DMI)
            
            Invoked in the "RHS" of a triggered rule.
            Rule function that parses the content of the Syslog logging buffer to
            and asserts facts when buffered log messages contain specific content.
            This function can be used by a rule to trigger when specific Syslog messages
            with specific content are generated.  This may be used if the device or the deployment
            can't support the RFC5277 notification trigger method for DDR execution triggering.
            This function can also be used by a workflow that decides it needs to search the logging history
            to determine if an event occurred.   
            
            The actions performed by the method are defined in the 'logging_trigger_list' list in the ddr-facts file.
            The logging_trigger_list is a list of dictionary entries that provide instructions to the main DDR

              *Mar 29 17:01:20.953: %SEC_LOGIN-5-LOGIN_SUCCESS: Login Success [user: admin] [Source: 172.20.86.186] [localport: 22] at 17:01:20 UTC Mon Mar 29 2021

            Each entry in the list is indexed by the fact_index parameter as a Python dictionary
              fact_type - The type of action function
              device - optionally directly specify the device name otherwise management device used by default
              genie_parser - Name of parser class in genie_parsers.py for processing this show command
              assert_fact_for_each_item_in - Assert a fact for each dictionary entry key at this level in the dictionary
              protofact - prototype describing how to translate parser python dictionary output to a CLIPs fact
              template - name of the deftemplate in ddr-rules defining the fact
              slots - names of the slots/fields in the deftemplate for the fact
              $+datetime - Date and time extracted from the syslog message
              $+facility - type of syslog message, e.g. "SEC_LOGIN"
              $+level - Syslog level 0 - 7. e.g 5
              $+message - text to right of the level, e.g. "LOGIN_SUCCESS"
              $+note - free from text with remainder of syslog message 

                  {"fact_type": "show_and_assert",
                   "device": "cat9k-24",
                   "genie_parser": "ShowLoggingLast",
                   "assert_fact_for_each_item_in": "log_instance",
                   "protofact": {"template": "log-instance",
                                "slots": {"device": "device",
                                          "datetime": "$+datetime",
                                          "facility": "$+facility",
                                          "level": "$+level",
                                          "message": "$+message",
                                          "note": "$+note"
                                         }
                               }
                 }
             
            :param access_type: "cli" runs command using guestshell Python cli package, "ssh" uses SSH connectin
            :param logging_trigger_index: 0 origin index into the logging_trigger_list
            :param number_lines: Number of lines in the syslog buffer to process (starting with most recent)
            :param facility: String containing the Syslog "facility" name, e.g. "SEC_LOGIN" for the example above
        
        Usage::

              (run_logging_trigger ssh 0 4 DMI) ;use ssh, first entry in fact list, process 4 lines of the syslog buffer looking for "DMI" facility messages

            :raises none:

        """
        if self.control["actions"] == 1:
            try:
                command_idx = int(logging_trigger_index)
                cmdline = "show logging last {0}"
                command = cmdline.format(int(number_lines))

                if self.control["debug-CLI"] == 1:
                    self.print_log("\n**** DDR Debug: run_logging_trigger command: " + str(command))
        #
        # If access-type is cli use the Python CLI package to run command
        #
                if access_type == 'cli':
                    response = cli.cli(str(command))
                    if self.control["debug-CLI"] == 1:
                        self.print_log("**** DDR Debug: run_logging_trigger cli response: \n" + str(response))
        #
        # Use SSH to run the show command
        #
                else:
                    try:
                        options = '-q -oStrictHostKeyChecking=no -oUserKnownHostsFile=/dev/null -oPubkeyAuthentication=no'
                        ssh_cmd = 'ssh %s@%s %s "%s"' % (self.control["mgmt-device"][2], self.control["mgmt-device"][0], options, command)
                        if self.control["debug-CLI"] == 1:
                            self.print_log("**** DDR Debug: run_logging_trigger SSH command: " + str(ssh_cmd) + "\n")
                        child = pexpect.spawn(ssh_cmd, timeout= 20, encoding='utf-8')
                        child.delaybeforesend = None
                        if self.control["debug-CLI"] == 1:
                            child.logfile = sys.stdout
                        child.expect(['\r\nPassword: ', '\r\npassword: ', 'Password: ', 'password: '])
                        child.sendline(self.control["mgmt-device"][3])
                        child.expect(pexpect.EOF) #result contains the ping result
                        response = child.before
                        if self.control["debug-CLI"] == 1:
                            self.print_log("**** DDR Debug: run_logging_trigger ssh response: \n" + str(response))

                    except Exception as e:
                        child.close()
                        self.print_log("%%%% DDR run_logging_trigger SSH or timeout Error: " + str(ssh_cmd) + "\n")
                    child.close()
         #
         # Assert the syslog logging facts using the response
         #
                fact_found = False
        #
        # Read specified number of lines in Syslog logging buffer and assert facts
        #
                try:
                    fact = self.control["logging-trigger-list"][command_idx]
                    parser = genie_str_to_class(fact["genie_parser"])
                    parsed_genie_output = parser.parse(num_lines=int(number_lines), output=response, facility_filter=str(facility))
                    if self.control["debug-parser"] == 1:
                        self.print_log("**** DDR Debug: run_logging_trigger parser result: \n\n" + str(parsed_genie_output) + "\n")
                except Exception as e:
                    self.print_log("%%%% DDR Error: Exception in run_logging_trigger parser processing: \n" + str(parsed_genie_output) + "\n" + str(e))
                    return
               
                sub_dictionary_list = self.find(fact["assert_fact_for_each_item_in"], parsed_genie_output)
                import copy
                for sub_dictionary in sub_dictionary_list:
                    for item in sub_dictionary:
                        protofact = copy.deepcopy(fact["protofact"])
                        for slot in protofact["slots"]:
                            value = protofact["slots"][slot]
	#
	# insert the device name into the fact
	#
                            if value == "device":
                                protofact["slots"][slot] = value.replace("device", fact["device"])
                            elif type(value) == str and "$" in value:
                                protofact["slots"][slot] = value.replace("$", item)
                        self.assert_template_fact(protofact, 'device', fact["device"], sub_dictionary)
            except Exception as e:
                self.print_log("%%%% DDR Error: Exception in run_logging_trigger assert_fact response processing: " + str(fact) + "\n" + str(e))
   
    def run_process_file(self, file_fact_index, filename):
        """
            Rule file function call: (run_process_file 0 /bootflash/guest-share/emre_show-tech-acl)
            
            Invoked in the "RHS" of a triggered rule.
            Rule function that parses the content of "filename" stored in /bootflash/guest-share
            using the file_fact_list entry identified by file_fact_index to assert
            FACTs from the file content. A parser selects data from the file
            
            Each entry in the list is indexed by the file_fact_index parameter as a Python dictionary
             fact_type - The type of action function
             device - optionally directly specify the device name otherwise management device used by default
             genie_parser - Name of parser class in genie_parsers.py for processing this show command
             assert_fact_for_each_item_in - Assert a fact for each dictionary entry key at this level in the dictionary
             protofact - prototype describing how to translate parser python dictionary output to a CLIPs fact
             template - name of the deftemplate in ddr-rules defining the fact
             slots - names of the slots/fields in the deftemplate for the fact
             $+datetime - Date and time extracted from the syslog message
             $+facility - type of syslog message, e.g. "SEC_LOGIN"
             $+level - Syslog level 0 - 7. e.g 5
             $+message - text to right of the level, e.g. "LOGIN_SUCCESS"
             $+note - free from text with remainder of syslog message 
           
              {"fact_type": "run_process_file",
               "device": "CAT9K-24",
               "genie_parser": "ShowTechAclPlatform",
               "assert_fact_for_each_item_in": "acl_platform",
               "protofact": {"template": "acl-tech-platform-info",
                               "slots": {"switch": "$",
                                         "model": "$+model",
                                         "serial": "$+serial",
                                         "mac": "$+mac",
                                         "hwver": "$+hwver",
                                         "swver": "$+swver"}
                            }    
              },
            
            :param file_fact_index: 0 origin index into the file_fact_list in the ddr-facts file
            :param filename: full pathname for processed to generate facts
        
        Usage::

              (run_process_file 0 /bootflash/guest-share/emre_show-tech-acl) ; Get device image
              (run_process_file 1 /bootflash/guest-share/emre_show-tech-acl) ; Get platform information
              (run_process_file 2 /bootflash/guest-share/emre_show-tech-acl) ; Get ACL names
              (run_process_file 3 /bootflash/guest-share/emre_show-tech-acl) ; Get ACL counters

            :raises none:

        """
        if self.control["actions"] == 1:
            command_idx = int(file_fact_index)
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
        # Read the contents fo the file
        #
            try:
                with open(filename, "r") as fd:
                    file_data = fd.read()
                    fd.close()
                    if self.control["debug-file"] == 1:
                        self.print_log("\n**** DDR  Debug: run_process_file content: \n" + str(file_data))
            except Exception as e:
                self.print_log("%%%% DDR: Error reading run_process_file: " + str(e) + "\n" + str(filename))

    #
    # process file content and generate FACTs
    #
            try:
                fact = self.control["file-fact-list"][command_idx]
                parser = genie_str_to_class(fact["genie_parser"])
                parsed_genie_output = parser.parse(output=file_data)
                if self.control["debug-action"] == 1:
                    self.print_log("\n*** DDR Debug: run_process_file parsed output:\n" + str(parsed_genie_output))

                sub_dictionary_list = self.find(fact["assert_fact_for_each_item_in"], parsed_genie_output)
                import copy
                for sub_dictionary in sub_dictionary_list:
                    for item in sub_dictionary:
                        protofact = copy.deepcopy(fact["protofact"])
                        for slot in protofact["slots"]:
                            value = protofact["slots"][slot]
	#
	# insert the device name into the fact
	#
                            if value == "device":
                                protofact["slots"][slot] = value.replace("device", fact["device"])
                            elif type(value) == str and "$" in value:
                                protofact["slots"][slot] = value.replace("$", item)
                        self.assert_template_fact(protofact, 'device', fact["device"], sub_dictionary)
            except Exception as e:
                self.print_log("%%%% DDR Error: Exception in run_process_file assert_fact response processing: " + str(fact) + " " + str(e))

    def run_command(self, command, delay):
        """
            Rule file function call: (run_command "clear counters" 1000)
            
            Invoked in the "RHS" of a triggered rule.
            Runs the CLI command based in as an argument and delays for the number of ms in delay
            before continuing 
            
            :param command: cli command to execute
            :param delay: number of milliseconds to delay after running the command
        
        Usage::

              (run_command "clear counters") 1000

            :raises none:

        """
        if self.control["actions"] == 1:
            try:
                if self.control["debug-CLI"] == 1:
                    self.print_log("**** DDR Debug: run_command: " + str(command))
                result = cli.cli(command)
                if self.control["debug-CLI"] == 1:
                    self.print_log("**** DDR Debug: run_command result: " + str(result))
                if int(delay) > 0:
                    if self.control["debug-CLI"] == 1:
                        self.print_log("**** DDR Debug: run_command delay ms: " + str(delay))
                    time.sleep(int(delay)/1000)
                    if self.control["debug-CLI"] == 1:
                        self.print_log("**** DDR Debug: run_command delay complete")
            except Exception as e:
                self.print_log("%%%% DDR  run_command Error: Exception: " + str(e) + "\n")

    
    def run_clear_selected_facts(self, template):
        """
            Rule file function call: (run_clear_selected_facts interface-status-facts)
            
            Invoked in the "RHS" of a triggered rule.
            Deletes (retracts) all facts of the type passed in template from the knowledge base  
            
            :param template: deftemplate name in rule file which will have all existing facts deleted
        
        Usage::

              (run_clear_selected_facts interface-status-facts)

            :raises none:

        """
        for fact in self.env.facts():
            try:
                if str(template) in str(fact):
                    fact.retract()
                    break
            except Exception as e:
                self.print_log("%%%% DDR Error: run_clear_selected_facts: " + str(e))

    def run_nc_fact(self, *args):

        """
            Rule file function call: (run_nc_fact 0 1 neighbor ?neighbor none ?neighbor) ; use netconf to get hold-time for the bgp-neighbor
            
            Invoked in the "RHS" of a triggered rule.
            This function executes NETCONF operations defined in ddr-facts nc_fact_list entries
            For example: (run_nc_fact 0 1 neighbor ?neighbor none ?neighbor)
              This example runs a NETCONF operation using index "0" in the nc_fact_list.  The 1 indicates there is one
              parameter that will provided by the rule and will substituted into the NETCONF message.
              "none" indicates that splitting the contents of variable is not required 

              nc_fact_list = [
               {"fact_type": "multitemplate",
                "data": ["multitemplate", 0, "leaf2",
                      '''<native xmlns="http://cisco.com/ns/yang/Cisco-IOS-XE-native">
                           <router>
                             <bgp xmlns="http://cisco.com/ns/yang/Cisco-IOS-XE-bgp">
                               <neighbor>
                                 <id>{0}</id>
                                 <timers>
                                   <holdtime/>
                                 </timers>
                               </neighbor>
                             </bgp>
                           </router>
                         </native>
                ''',
                "nbr-hold-time", "timers",
                 ["holdtime"], 
                 ["hold-time"], [],
                 []]}]
            
            :param *args: Pointer to structure with passed arguments defined in function content below
                    args[0]: index = int(argval)
                    args[1]: num_params = int(argval)
                    args[2]: slot = argval is the slot name
                    args[3]: slot_value = argval is the slot value
                    args[4]: split = argval #This argument = 'split' if the next argument is an interface name
       
        Usage::

              (run_nc_fact 0 1 neighbor ?neighbor none ?neighbor) ;This use netconf to get hold-time for the bgp-neighbor
              (run_nc_fact 2 1 neighbor ?neighbor none ?neighbor) ;This use netconf to get session-state for the bgp-neighbor
              (run_nc_fact 1 1 neighbor ?neighbor none ?neighbor) ;This use netconf to routing state the bgp-neighbor

            :raises none:

        """

        if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: run_nc_facts facts: " + str(args))
        try:
    #
    # Extract the arguments from the call from the RULE
    # 1st argument is index into "nc-fact-list"
    # 2nd argument is the number of parameters to substitute into the NETCONF request
    # Remaining arguments are inserted into the NETCONF request using string.format method
    #            
            sub_params = []
            j = 0
            split = 'none'
            try:
                for argval in args:
                    if j == 0: index = int(argval)
                    if j == 1: num_params = int(argval)
                    if j == 2: slot = argval
                    if j == 3: slot_value = argval
                    if j == 4: split = argval #This argument = 'split' if the next argument is an interface name
                    if j > 4:
                        if (j == 5) and (split == 'split'):
                            interface = argval.partition('Ethernet')
                            sub_params.append(interface[0] + interface[1])
                            sub_params.append(interface[2])
                            split = 'none'
                            num_params = num_params + 1
                        else: sub_params.append(str(argval))
                    j = j + 1
            except Exception as e:
                self.print_log("%%%%% DDR Error: run_nc_fact argument error: " +str(sub_params) + " " + str(e))
                return

            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: run_nc_facts nc-fact-list:\n" + str(self.control["nc-fact-list"]))

            raw_fact = copy.deepcopy(self.control["nc-fact-list"][index])
            fact_data = raw_fact["data"]
            fact_msg = fact_data[3]
#
# Substitute arguments from the rule function call into the Netconf operation
#
            try:
                if num_params == 0:
                   formatted = str(fact_msg)
                elif num_params == 1:
                   formatted = str(fact_msg).format(str(sub_params[0]))
                elif num_params == 2:
                   formatted = str(fact_msg).format(str(sub_params[0]), str(sub_params[1]))
                elif num_params == 3:
                   formatted = str(fact_msg).format(str(sub_params[0]), str(sub_params[1]), str(sub_params[2]))
                elif num_params == 4:
                   formatted = str(fact_msg).format(str(sub_params[0]), str(sub_params[1]), str(sub_params[2]), str(sub_params[3]))
                else:
                    self.print_log("%%%%% DDR Error: run_nc_fact invalid # arguments: " + str(num_params))
                    return

            except Exception as e:
                self.print_log("%%%%% DDR Error: run_nc_fact format message: " + str(e))
               
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: run_nc_facts facts: " + str(fact_msg))

            fact_data[3] = formatted
            raw_fact["data"] = fact_data
            fact = raw_fact

            if fact["fact_type"] == "multitemplate":
                status = self.get_template_multifacts(fact["data"], slot, slot_value)
            elif fact["fact_type"] == "multitemplate_protofact":
                status = self.get_template_multifacts_protofact(fact, slot, slot_value)
            else:
                self.print_log("%%%%% DDR Error: run_nc_facts Invalid fact type: " + str(fact))
            if status != "success":
                self.print_log("%%%%% DDR Error: run_nc_facts Fact Read Error: " + str(status))

        except Exception as e:
            self.print_log("\n%%%% DDR Error: Exception in run_nc_fact: " + str(e))

    def run_apply_config(self, config_id):
        """
            Rule file function call: (run_apply_config config_id)
            
            Invoked in the "RHS" of a triggered rule.
            This function is called from the CLIPS RULES
            if an edit-config action is required.  'config_id' is the 0 origin index for the 
            edit-configs list in the ddr-facts file.

            If the device uses a candidate data store and a commit is required, the commit/RPC is sent.
            Configuration operations are stored as list entries.  This is an example:

              edit_configs = [[0, 'none',
                             '<config><native xmlns="http://cisco.com/ns/yang/Cisco-IOS-XE-native"><interface><Loopback>
                              <name>111</name><description>LB111</description></Loopback></interface></native></config>']]

             The first list element is the index of the device in the "devices" list used by ncclient
             The second argument is 'commit' if the device uses the candidate datastore and requires 'commit/'
             the argument is 'none' if the device uses the 'running-configuration'
             The third argument is the 'config' section of the edit-config operations, for example copied from YangSuite
            
            :param config_id: 0 origin index into the edit_configs list in the ddr-facts file
        
        Usage::

              run_apply_config(config_id)

            :raises none:

        """
        try:
            config_key = int(config_id)
            edit_msg = self.control["edit-configs"][config_key]
            device = self.device[edit_msg[1]]
            if self.control["actions"] == 1:
#
# Use candidate datastore and "commit" for XR configurations
#
                if edit_msg[2] == 'commit':
                    datastore = 'candidate'
                    try:
                        result = device.edit_config(str(edit_msg[3]), target=datastore)
                        if self.control["debug-config"] == 1:
                            self.print_log("**** DDR Debug: apply configuration:\n" + str(edit_msg[3]))
                            self.print_log("**** DDR Debug: edit_config result\n" + str(result.xml))
                        result = dedvice.commit()
                    except Exception as e:
                       self.print_log("%%%% Error applying configuration: ", str(config_key) + "\n" + str(e))
#
# Use running configuration for XE and NX-OS if commit_required = 0
#
                if edit_msg[2] == 'none':
                    datastore = 'running'
                    try:
                        result = device.edit_config(str(edit_msg[3]), target=datastore)
                        if self.control["debug-config"] == 1:
                            self.print_log("**** DDR Debug: apply configuration:\n" + str(edit_msg[3]))
                            self.print_log("**** DDR Debug: edit_config result\n" + str(result.xml))
                    except Exception as e:
                        self.print_log("%%%% Error applying configuration: " + str(edit_msg[3]) + "\n" + str(e))

        except Exception as e:
            self.print_log("%%%% Error in run_apply_config: " + str(e))


    
    def run_ddr(self):
        """
            Rule file function call: (run_ddr)
            
            Invoked in the "RHS" of a triggered rule.
            This function runs the CLIPs runtime to enable execution of actions for rules that are "triggered" by added facts.
            If a rule asserts or modifies a fact, the CLIPs system evaluates the state of all rules that are dependent on the added or changed fact.
            If the added or changed fact results in all of the conditions on the lefr-hand-side of rule being satisifed, the rule is "triggered".
            When a rule is triggered it is marked by CLIPs as ready to execute the right-hand-side actions.
            In order to execute the rule's actions, the CLIPs runtime must be invoked to run again.
            
            This function is typically used in a workflow where rule 1 asserts a fact and "knows" that another rule depends on the fact and needs to be
            executed immediately to implement the worklfow

            :param none: 
        
        Usage::

              (run_ddr)

            :raises none:

        """
        try:
            self.print_clips_info_run(' BEFORE ')
            self.env.run()
            self.print_clips_info_run(' AFTER' )
        except Exception as e:
            self.print_log("\n%%%% DDR Error: Exception when running inference engine in run_action" + str(e))
            self.close_sessions()
            sys.exit(0)

    
    def run_action(self, action_fact_index):
        """
            Rule file function call: (run_action 1)
            
            Invoked in the "RHS" of a triggered rule.
            This function is called from the CLIPS RULES to cause the execution of fact collection operations.
            This function can be used to run fact collection using show commands or NETCONF

            The action_fact_list in the ddr-facts file can have one or more entries defining fact collection operations.
            The run_action function selects the required collection list dictionary from action_fact_list and executes the collection.

            The instructions are the same as those used in the nc_fact_list and show command fact collection.  For example: 
            
               {"fact_type": "multitemplate",
                "data": ["multitemplate", 0, "leaf2",
                      '''<native xmlns="http://cisco.com/ns/yang/Cisco-IOS-XE-native">
                           <router>
                             <bgp xmlns="http://cisco.com/ns/yang/Cisco-IOS-XE-bgp">
                               <neighbor>
                                 <id>{0}</id>
                                 <timers>
                                   <holdtime/>
                                 </timers>
                               </neighbor>
                             </bgp>
                           </router>
                         </native>
                ''',
                "nbr-hold-time", "timers",
                 ["holdtime"], 
                 ["hold-time"], [],
                 []]}
                 
               {"fact_type": "show_and_assert",
                   "device": "cat9k-24",
                   "genie_parser": "ShowLoggingLast",
                   "assert_fact_for_each_item_in": "log_instance",
                   "protofact": {"template": "log-instance",
                                "slots": {"device": "device",
                                          "datetime": "$+datetime",
                                          "facility": "$+facility",
                                          "level": "$+level",
                                          "message": "$+message",
                                          "note": "$+note"
                                         }
                               }
                 }
                        
            :param action_fact_index: 0 origin index into the action_fact_list list in the ddr-facts file
        
        Usage::

              (run_action 1)

            :raises none:

        """
        try:
            index = int(action_fact_index)
            for fact in self.control["action-fact-list"][index]:
                if self.control["debug-fact"] == 1:
                    self.print_log("**** DDR Debug: run_action facts: " + str(fact))
                if fact["fact_type"] == "show_and_assert":
                    if "log_message_while_running" in fact:
                        self.print_log(fact["log_message_while_running"])
                    status = self.show_and_assert_fact(fact)
                elif fact["fact_type"] == "multitemplate":
                    status = self.get_template_multifacts(fact["data"], 'none', 'none')
                elif fact["fact_type"] == "multitemplate_protofact":
                    status = self.get_template_multifacts_protofact(fact, 'none', 'none')
                else:
                    self.print_log("%%%%% DDR Error: Invalid fact type: " + str(fact))
                if status != "success":
                    self.print_log("%%%%% DDR Error: Fact Read Error: " + str(status))

        except Exception as e:
            self.print_log("\n%%%% DDR Error: Exception in run_action: " + str(e))

    
    def run_rule_timer(self):
        """
            Rule file function call: (run_rule_timer)
            
            Invoked in the "RHS" of a triggered rule.
            This action function starts a thread to generate a timer to assert a FACT on
            a regular time interval. When the timer is running FACTs are generated of the form:
            
              (timer-fact-name (timer TRUE) (time 4596) (system-time 2020-10-01_03:09:55) 
              
              Where timer-fact-name is the name of a deftemplate defining a fact that should be asserted automatically 
              based on the timer execution.  "timer-fact-name" is defined in ddr-flags in the "timerFactName" parameter                       

            :param none:
        
        Usage::

              (run_rule_timer)

            :raises none:

        """

        try:
            timestamp =  datetime.now().strftime("%m-%d-%Y_%H:%M:%S.%f")
            
            hms = datetime.now().strftime("%H:%M:%S")
            h, m, s = [int(i) for i in hms.split(':')]
            timer_time = int(3600*h + 60*m + s)
        #
        # Build timer fact
        #
            message = '(' + str(self.control["timer-fact-name"]) + ' (timer TRUE) (time ' + str(timer_time) + ') (system-time ' + str(timestamp) + '))'
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: run_time fact: " + str(message))
            try:
                self.env.assert_string(message)
                self.env.run() #Run CLIPs to execute RULEs activated as a result of asserting the timer FACT
            except Exception as e:
                self.print_log("%%%% DDR: Exception in run_time: " + str(e))
    #
    # Run the timer
    #
            Timer(float(self.control["fact-timer"]), self.rule_timer).start()
        except Exception as e:
            self.print_log("%%%% DDR: run_timer exception: " + str(e))

    
    def run_ping_action(self, mode , vrf, neighbor, ping_count, min_success, src_interface, pkt_size, fact_result):
        """
            Rule file function call: (run_ping_action cli vrf ?neighbor count ?min-success ?src-interface 100 action-ping)
            
            Invoked in the "RHS" of a triggered rule.
            This function executes a ping to a remote address and asserts the results of the ping in a fact.
            The "ping result fact" can be used by additional rules in the workflow to make decisions.
            
            :param mode - cli/ssh run using python cli package or using ssh 
            :param vrf - vrf-name or none - do not include a vrf in the ping command
            :param neighbor - ip address of neighbor to ping
            :param count - number of pings to run
            :param min-success - minimum % of pings that must succeed
            :param src-interface - interface to run ping command through
            :param pkt_size - ping payload size in bytes
            :param fact_result - name of template to store results
        
        Usage::

              (run_ping_action none ?neighbor count ?min-success ?src-interface 100 action-ping-2)

            :raises none:

        """
        if self.control["actions"] == 1:
            try:
                if vrf == 'none':         
                    cmdline = 'ping ' + str(neighbor) + " source " + str(src_interface) + " size " + str(pkt_size) + " repeat " + str(ping_count)
                else:
                    cmdline = 'ping vrf ' + str(vrf) + ' ' + str(neighbor) + " source " + str(src_interface) + " size " + str(pkt_size) + " repeat " + str(ping_count)
                if self.control["debug-CLI"] == 1:
                    self.print_log("**** DDR Debug: ssh run_ping_action: cmdline: " + str(cmdline))
    #
    # Use ssh to the device to perform the ping
    #
                if mode == 'ssh':
                    options = '-q -oStrictHostKeyChecking=no -oUserKnownHostsFile=/dev/null -oPubkeyAuthentication=no'
                    ssh_cmd = 'ssh %s@%s %s "%s"' % (self.control["mgmt-device"][2], self.control["mgmt-device"][0], options, cmdline)
                    if self.control["debug-CLI"] == 1:
                        self.print_log("**** DDR Debug: ssh run_ping_action SSH command: " + str(ssh_cmd) + "\n")
                    try:
                        child = pexpect.spawn(ssh_cmd, timeout= 20, encoding='utf-8')
                        child.delaybeforesend = None
                        if self.control["debug-CLI"] == 1:
                            child.logfile = sys.stdout
                        child.expect(['\r\nPassword: ', '\r\npassword: ', 'Password: ', 'password: '])
                        child.sendline(self.control["mgmt-device"][3])
                        child.expect(pexpect.EOF) #result contains the ping result
                        result = child.before
                    except Exception as e:
                        child.close()
                        self.print_log("%%%% DDR  ssh run_ping_action SSH or timeout Error: " + str(ssh_cmd) + "\n")
                        message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent 0) (result FALSE))'
                        if self.control["debug-CLI"] == 1:
                            self.print_log("**** DDR Debug: Error ssh Assert run_ping_action SSH FACT: " + str(message))
                        try:
                            self.env.assert_string(message)
                        except: pass
                        return

                    child.close()
                    if self.control["debug-CLI"] == 1:
                        self.print_log("**** DDR Debug: ssh command result \n" + str(result)) #successful ping
                    try:
                        regex = 'Success rate is.{1}(?P<percent>(\d+))'
                        p1 = re.compile(regex)

                        for line in result.splitlines():
                            line = line.strip()
                            try:
                                results = p1.match(line)
                                group = results.groupdict()
                                percent = group['percent']
                                if self.control["debug-CLI"] == 1:
                                    self.print_log("**** DDR Debug: ssh run_ping_action: percent: " + str(percent))
                            except:  #skip lines that don't match the expected ping result
                                if self.control["debug-CLI"] == 1:
                                    self.print_log("**** DDR Debug: ssh run_ping_action: skipline: " + str(line))

                                if result.find("% Invalid source interface") != -1:
                                    message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent 0) (result FALSE))'
                                    if self.control["debug-fact"] == 1:
                                        self.print_log("**** DDR Debug: ssh Assert run_ping_action SSH FACT: " + str(message))
                                    try:
                                        self.env.assert_string(message)
                                    except: pass

                        if int(percent) >= int(min_success):
                            message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent ' + str(percent) + ') (result TRUE))'
                        else:
                            message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent ' + str(percent) + ') (result FALSE))'

                        if self.control["debug-fact"] == 1:
                            self.print_log("**** DDR Debug: Assert ssh run_ping_action FACT: " + str(message))
                        try:
                            self.env.assert_string(message)
                        except: pass
           
                    except Exception as e:
                        self.print_log("\n%%%% DDR  ssh run_ping_action ssh command Error: " + str(e) + "\n")
                        message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent 0) (result FALSE))'
                        if self.control["debug-fact"] == 1:
                            self.print_log("**** DDR Debug: ssh Error Assert run_ping_action SSH FACT: " + str(message))
                        try:
                            self.env.assert_string(message)
                        except: pass
    #
    # Use python cli package to perform ping
    #
                elif mode == 'cli':
                    try:
                        if vrf == 'none':         
                            cmdline = 'ping ' + str(neighbor) + " source " + str(src_interface) + " size " + str(pkt_size) + " repeat " + str(ping_count)
                        else:
                             cmdline = 'ping vrf ' + str(vrf) + ' ' + str(neighbor) + " source " + str(src_interface) + " size " + str(pkt_size) + " repeat " + str(ping_count)
                        if self.control["debug-CLI"] == 1:
                            self.print_log("**** DDR Debug: cli run_ping_action: cmdline: " + str(cmdline))
                        try:
                            regex = 'Success rate is.{1}(?P<percent>(\d+))'
                            p1 = re.compile(regex)

                            result = cli.cli(cmdline) # Run ping command to get ping results
                            for line in result.splitlines():
                                line = line.strip()
                                try:
                                    results = p1.match(line)
                                    group = results.groupdict()
                                    percent = group['percent']
                                    if self.control["debug-CLI"] == 1:
                                        self.print_log("**** DDR Debug: cli run_ping_action: percent: " + str(percent))
                                except:  #skip lines that don't match the expected ping result
                                    if self.control["debug-CLI"] == 1:
                                        self.print_log("**** DDR Debug: cli run_ping_action: skipline: " + str(line))

                                    if result.find("% Invalid source interface") != -1:
                                        message = message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent ' + str(0) + ') (result DOWN))'
                                        if self.control["debug-fact"] == 1:
                                            self.print_log("**** DDR Debug: Assert run_ping_action FACT: " + str(message))
                                        self.env.assert_string(message)
                                        return
                                    else: continue

                            if int(percent) >= int(min_success):
                                message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent ' + str(percent) + ') (result TRUE))'
                            else:
                                message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent ' + str(percent) + ') (result FALSE))'

                            if self.control["debug-fact"] == 1:
                                self.print_log("**** DDR Debug: cli Assert run_ping_action FACT: " + str(message))
                            self.env.assert_string(message) # Assert the ping action FACT
           
                        except Exception as e:
                            self.print_log("%%%% DDR  run_ping_action CLI command Error: " + str(e) + "\n")
                            message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent 0) (result FALSE))'
                            if self.control["debug-fact"] == 1:
                                self.print_log("**** DDR Debug: cli Error Assert run_ping_action SSH FACT: " + str(message))
                            try:
                                self.env.assert_string(message)
                            except: pass
                            return

                    except Exception as e:
                        self.print_log("%%%% DDR  cli run_ping_action ssh command Error: " + str(e) + "\n")
                        message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent 0) (result FALSE))'
                        if self.control["debug-fact"] == 1:
                            self.print_log("**** DDR Debug: cli Error Assert run_ping_action SSH FACT: " + str(message))
                        try:
                            self.env.assert_string(message)
                        except: pass
                        return
   #
   # else invalid mode 
   #
                else:
                    self.print_log("%%%% DDR  invalid run_ping_action ssh command Error: " + str(e) + "\n")
                    message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent 0) (result FALSE))'
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: invalid Error Assert run_ping_action SSH FACT: " + str(message))
                    try:
                        self.env.assert_string(message)
                    except: pass

            except Exception as e:
                self.print_log("%%%% DDR  global run_ping_action ssh command Error: " + str(e) + "\n")
                message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (percent 0) (result FALSE))'
                if self.control["debug-fact"] == 1:
                    self.print_log("**** DDR Debug: global Error Assert run_ping_action SSH FACT: " + str(message))
                try:
                    self.env.assert_string(message)
                except: pass
                return

    
    def run_trace(self, mode, vrf, trace_type, neighbor, src_interface, probe, timeout, ttl, trace_timeout, all_addresses, system_time_raw, fact_result):
        """
            Rule file function call: (run_trace ?mode ?vrf ?trace-type ?neighbor ?src-interface ?probe ?timeout ?ttl ?trace-timeout ?all-addresses ?syslog-time action-trace)
            
            Invoked in the "RHS" of a triggered rule.
            This function executes a trace route to a remote address and asserts the results of the trace route in a fact.
            Sample command to send to device: traceroute [vrf Mgmt-vrf] ip 172.24.115.25 source GigabitEthernet0/0 probe 1 timeout 1
            The "action_trace fact" can be used by additional rules in the workflow to make decisions.
            Sample trace FACT: 
              (action-trace (neighbor 10.1.7.2) (hop 8) (last-ip 10.1.7.2) (result TRUE))
            
            :param mode: cli/ssh run using python cli package or using ssh 
            :param vrf: vrf-name or none - do not include a vrf in the ping command
            :param trace_type: default is 'ip'
            :param neighbor: ip address of trace neighbor endpoint
            :param src-interface: interface to run trace command through
            :param probe: integer typically 1 for number of times to try each path before failing
            :param timeout: timeout in seconds for complete trace
            :param trace-timeout: timeout in seconds for each trace hop
            :param ttl: maximum trace hop length
            :param all_addresses: TRUE to return all addresses in path, FALSE to return last address only
            :param system_time_raw: device system time
            :param fact_result: name of deftemplate that will be used to generate the trace route result fact
        
        Usage::

              (run_trace ?mode ?vrf ?trace-type ?neighbor ?src-interface ?probe ?timeout ?ttl ?trace-timeout ?all-addresses ?syslog-time action-trace)

            :raises none:

        """
        if self.control["actions"] == 1:
            try:
                system_time = str(system_time_raw).replace(" ", "_")
                if vrf == 'none':         
                    cmdline = 'traceroute ' + str(trace_type) + " " + str(neighbor) + " source " + str(src_interface) + " probe " + str(probe) + " timeout " + str(timeout) + " ttl 1 " + str(ttl)
                else:
                    cmdline = 'traceroute vrf ' + str(vrf) + " " + str(trace_type) + " " + str(neighbor) + " source " + str(src_interface) + " probe " + str(probe) + " timeout " + str(timeout) + " ttl 1 " + str(ttl)
                if self.control["debug-CLI"] == 1:
                    self.print_log("**** DDR Debug: run_trace: cmdline: " + str(cmdline))
    #
    # Use ssh to the device to perform the traceroute
    #
                if mode == 'ssh':
                    p1 = re.compile('(?P<hop>(\d+)).{1}(?P<address>(\S+)).*')
                    p2 = re.compile('(?P<nohop>(\d+)).{1}(?P<failed>(\S+)).*')
                    options = '-q -oStrictHostKeyChecking=no -oUserKnownHostsFile=/dev/null -oPubkeyAuthentication=no'
                    ssh_cmd = 'ssh %s@%s %s "%s"' % (self.control["mgmt-device"][2], self.control["mgmt-device"][0], options, cmdline)
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: run_trace SSH command: " + str(ssh_cmd) + "\n")
                    try:
                        child = pexpect.spawn(ssh_cmd, timeout= trace_timeout, encoding='utf-8')
                        child.delaybeforesend = None
                        if self.control["debug-CLI"] == 1:
                            child.logfile = sys.stdout
                        child.expect(['\r\nPassword: ', '\r\npassword: ', 'Password: ', 'password: '])
                        child.sendline(self.control["mgmt-device"][3])
                        child.expect(pexpect.EOF) #result contains the traceroute result
                        result = child.before
                    except Exception as e:
                        child.close()
                        self.print_log("%%%% DDR  run_trace ssh or timeout Error: " + str(ssh_cmd) + "\n")
                        message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (system-time ' + str(system_time) + ') (result FALSE))'
                        if self.control["debug-fact"] == 1:
                            self.print_log("**** DDR Debug: Assert run_trace FACT: " + str(message))
                        try:
                            self.env.assert_string(message)
                        except: pass
                        return

                    child.close()
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: ssh command result \n", result) #successful traceroute execution
    #
    # Use python cli package to perform the traceroute
    #
                elif mode == 'cli':
                    try:
                        p1 = re.compile('(?P<hop>(\d+)).{1}(?P<address>(\S+)).*')
                        p2 = re.compile('(?P<nohop>(\d+)).{1}(?P<failed>(\S+)).*')
                        result = cli.cli(cmdline) # Run trace command
                    except Exception as e:
                        self.print_log("%%%% DDR  run_trace cli Error: " + str(cmdline) + "\n" + str(e))

                    message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (system-time ' + str(system_time) + ') (result FALSE))'
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: Assert run_trace FACT: " + str(message))
                    try:
                        self.env.assert_string(message)
                    except: pass
                    return
    #
    # If invalid traceroute mode is selected
    #
                else:
                    self.print_log("%%%% DDR  run_trace Error: Invalid mode" + str(mode))
                    message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (system-time ' + str(system_time) + ') (result DOWN))'
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: run_trace Error: Invalid mode: " + str(message))
                    try:
                        self.env.assert_string(message)
                    except: return

    #############################################################
    #
    #  The traceroute reply is now in the "result" string variable
    #  Select information and build the result FACT
    #
    #############################################################
                ipaddress = 'none'
                hop = 'none'
                domain = 'none'
                last_address = 'none'

                if self.control["debug-CLI"] == 1:
                    self.print_log("**** DDR Debug: run_trace: result: " + str(result))
                for line in result.splitlines():
                    line = line.strip()
                    try:
                        results = p1.match(line)
                        group = results.groupdict()
                        hop = group['hop']
                        ipaddress = group['address'].strip().lstrip('(').rstrip(')')
                        if str(ipaddress) != '*':
                            last_address = ipaddress        
                        if self.control["debug-CLI"] == 1:
                            self.print_log("**** DDR Debug: run_trace: hop address: " + str(hop) + " "+ str(ipaddress))

                    except Exception as ex: #skip lines that don't match the expected trace result
                        if self.control["debug-CLI"] == 1:
                            self.print_log("**** DDR Debug: run_trace: skipline: " + str(line))
                        continue
    #
    # After parsing the response for the tracerout results, check to see if the interface was down
    #
                if result.find("% Invalid source interface") != -1:
                    message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (system-time ' + str(system_time) + ') (result DOWN))'
                    if self.control["debug-fact"] == 1:
                        self.print_log("**** DDR Debug: Assert run_trace Interface Down: " + str(message))
                    try:
                        self.env.assert_string(message)
                    except: return

                else: 

    ###########################################################################
    #
    # Test to see if neighbor is found as the last element in the trace list
    #
    ###########################################################################
                    failed = 'none'
                    nohop = 'none'
                    try:
                        last_result = p2.match(line)
                        group = results.groupdict()
                        failed = group['failed']
                        nohop = group['nohop']
                    except: pass
    #
    # test for timeout as the last hop reported which indicates trace failed to complete
    #
                    if str(failed) == '*':
                        message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (hop ' + str(nohop) + ') (last-ip ' + str(last_address) + ') (system-time ' + str(system_time) +') (result FALSE))'
                    else:
                        message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (hop ' + str(hop) + ') (last-ip ' + str(last_address) + ') (system-time ' + str(system_time) +') (result TRUE))'
                    try:
                        if self.control["debug-fact"] == 1:
                            self.print_log("**** DDR Debug: Assert run_trace Error: " + str(e))
                        self.env.assert_string(message)
                    except: return

            except Exception as e:
                self.print_log("%%%% DDR  run_trace command Error: " + str(e) + "\n")
                message = '(' + str(fact_result) + ' (neighbor ' + str(neighbor) + ') (result FALSE))'
                if self.control["debug-fact"] == 1:
                    self.print_log("**** DDR Debug: Assert run_trace Error: " + str(e))
                try:
                    self.env.assert_string(message)
                except: return

        if self.control["debug-fact"] == 1:
            self.print_log("**** DDR Debug: Assert run_trace FACT: " + str(message))
        try:
            self.env.assert_string(message) # Assert the action FACT
        except: return

    
    def run_trace_hops(self, mode, vrf, trace_type, neighbor, src_interface, probe, timeout, ttl, trace_timeout, all_addresses, system_time_raw, fact_result):
        """
            Rule file function call: (run_trace_hops ?mode ?vrf ?trace-type ?neighbor ?src-interface ?probe ?timeout ?ttl ?trace-timeout ?all-addresses ?syslog-time action-trace)
            
            Invoked in the "RHS" of a triggered rule.
            This function executes a trace route to a remote address and asserts a fact with the last good address in the trace in a fact.
            Sample command to send to device: traceroute [vrf Mgmt-vrf] ip 172.24.115.25 source GigabitEthernet0/0 probe 1 timeout 1
            The "action_trace fact" can be used by additional rules in the workflow to make decisions.
            Sample trace FACT: 
              (action-trace (neighbor 10.1.7.2) (hop 8) (last-ip 10.1.7.2) (result TRUE))
            
            :param mode: cli/ssh run using python cli package or using ssh 
            :param vrf: vrf-name or none - do not include a vrf in the ping command
            :param trace_type: default is 'ip'
            :param neighbor: ip address of trace neighbor endpoint
            :param src-interface: interface to run trace command through
            :param probe: integer typically 1 for number of times to try each path before failing
            :param timeout: timeout in seconds for complete trace
            :param trace-timeout: timeout in seconds for each trace hop
            :param ttl: maximum trace hop length
            :param all_addresses: TRUE to return all addresses in path, FALSE to return last address only
            :param system_time_raw: device system time
            :param fact_result: name of deftemplate that will be used to generate the trace route result fact
        
        Usage::

              (run_trace_hops ?mode ?vrf ?trace-type ?neighbor ?src-interface ?probe ?timeout ?ttl ?trace-timeout ?all-addresses ?syslog-time action-trace)

            :raises none:

        """
        try:
            message = '(trace-hops (neighbor ' + str(neighbor) + ') (hop ' + str(1) + ') (last-ip 192.168.1.24) (system-time ' + str(system_time_raw) +') (result TRUE))'
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: Assert run_trace_hops last good hop: " + str(message))
            try:
                self.env.assert_string(message)
            except Exception as e:
                self.print_log("**** Assert run_trace_hops error: " + str(e) + str(message))

        except Exception as e:
            self.print_log("**** Assert run_trace_hops error: " + str(e))

    def run_set_runmode(self, runmode):
        """
            Rule file function call: (run_set_runmode RUNMODE)
            
            Invoked in the "RHS" of a triggered rule.
            This function sets the ddr run-mode control variable
            The function allows a rule to change the execution mode, for example from "0" for timed execution
            to "2" for trigger off NETCONF notification
            
            :param runmode: 0/timed execution, 1/trigger on syslog, 2/trigger on NETCONF notification 
        
        Usage::

              (run_set_runmode 2)

            :raises none:

        """
        try:
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: run_set_runmode change mode to: " + str(runmode))
            self.control["run-mode"] = int(runmode)
        except Exception as e:
            self.print_log("%%%% ERROR run_set_runmode change mode value: " +str(runmode) + " " + str(e))


    def run_map_network_feature(self, *args):
        """
            Rule function call: (run_map_network_feature "action_function_name" param1 param2 ...)
            
            Invoked in the "RHS" of a triggered rule.
            This function provides a single entry point for action function calls.  The first argument arg[0] is the name of
            the action function that will be called.  The remaining elements in the args array are additional function arguments.
            
            This function uses the passed in function name to select which action function method to invoke.  The function extracts
            arguments from the args variable and performs transformations on the arguments if required before calling the 
            action function by executing a "return" to the method
            
            :param args: pointer to array of function arguments.  arg[0] is the name of the function to call.  args[1] to args[n]
                         are additional arguments for the function
        
        Usage::

              (run_map_network_feature "run_assert_message" "string")

            :raises none:

        """
        try:
            if self.control["debug-fact"] == 1:
                self.print_log("**** DDR Debug: run_map_network_feature: " + str(args))

            method = getattr(self, args[0], lambda: "invalid_network_feature call")

    # (run_assert_message string)
            if args[0] == "run_assert_message":
                return method(args[1])
                
    # (run_nc_fact 0 0 none none none none)
    #   args[1]: index = int(argval)
    #   args[2]: num_params = int(argval)
    #   args[3]: slot = argval is the slot name
    #   args[4]: slot_value = argval is the slot value
    #   args[5]: split = argval #This argument = 'split' if the next argument is an interface name
            elif args[0] == "run_nc_fact":
                return method(args[1], args[2], args[3], args[4], args[5])

    # (run_show_fact index)
            elif args[0] == "run_show_fact":
                return method(args[1])

    #  (run_show_parameter 0 cli "show vlan id {0}" 1 1 none none no-fact yes-fact)
    #   args[1]: index = int(argval)
    #   args[2]: access_type = argval string
    #   args[3]: show_template = argval string
    #   args[4]: num_params = int(argval)
    #   args[5]: first param = argval
    #   args[6]: second param = argval
    #   args[7]: third param = argval
    #   args[8]: no_fact string = argval is the "no fact"
    #   args[9]: yes_fact string = argval is "yes fact"
            elif args[0] == "run_show_parameter":
                return method(args[1], args[2], args[3], args[4], args[5], args[6], args[7], args[8], args[9])

    #    (run_map_network_feature "run_decode_btrace_log" 0 cli "show platform software trace message {0} {1}" 2 dmiauthd "switch active R0" none)
    #   args[1]: index = int(argval)
    #   args[2]: access_type = argval string
    #   args[3]: show_template = argval string
    #   args[4]: num_params = int(argval)
    #   args[5]: first param = argval
    #   args[6]: second param = argval
    #   args[7]: third param = argval
            elif args[0] == "run_decode_btrace_log":
                return method(args[1], args[2], args[3], args[4], args[5], args[6], args[7])

    #    (run_map_network_feature "run_ping_action" cli Mgmt-vrf 172.27.255.23 5 90 GigabitEthernet0/0 100 action-ping)
    #   args[1]: mode = int(argval)
    #   args[2]: vrf = argval string
    #   args[3]: destination_ip = argval string
    #   args[4]: number_ping = int(argval)
    #   args[5]: success_threshold = int(argval)
    #   args[6]: source_interface = argval string
    #   args[7]: packet_size = int(argval)
    #   args[8]: fact_name = argval string
            elif args[0] == "run_ping_action":
                return method(args[1], args[2], args[3], args[4], args[5], args[6], args[7], args[8])

    #    (run_map_network_feature "run_trace" cli Mgmt-vrf ip 172.27.255.23 GigabitEthernet0/0 3 5 3 10 TRUE 12:00 action-trace)
    #   args[1]: mode = int(argval)
    #   args[2]: vrf = argval string
    #   args[3]: packet_type = argval string
    #   args[4]: destination_ip = argval string
    #   args[5]: source_interface = argval string
    #   args[6]: probe = int(argval)
    #   args[7]: total_timeout = int(argval)
    #   args[8]: ttl = int(argval)
    #   args[9]: timeout_per_hop = int(argval)
    #   args[10]: all_addresses = argval string
    #   args[11]: syslog_time = argval string
    #   args[12]: fact_name = argval string
            elif args[0] == "run_trace":
                return method(args[1], args[2], args[3], args[4], args[5], args[6], args[7], args[8], args[9], args[10], args[11], args[12])


    #    (run_map_network_feature "run_trace_hops" cli Mgmt-vrf ip 172.27.255.23 GigabitEthernet0/0 3 5 3 10 TRUE 12:00 action-trace)
    #   args[1]: mode = int(argval)
    #   args[2]: vrf = argval string
    #   args[3]: packet_type = argval string
    #   args[4]: destination_ip = argval string
    #   args[5]: source_interface = argval string
    #   args[6]: probe = int(argval)
    #   args[7]: total_timeout = int(argval)
    #   args[8]: ttl = int(argval)
    #   args[9]: timeout_per_hop = int(argval)
    #   args[10]: all_addresses = argval string
    #   args[11]: syslog_time = argval string
    #   args[12]: fact_name = argval string
            elif args[0] == "run_trace_hops":
                return method(args[1], args[2], args[3], args[4], args[5], args[6], args[7], args[8], args[9], args[10], args[11], args[12])

    # (run_cli_command index)
            elif args[0] == "run_cli_command":
                return method(args[1], args[2], args[3], args[4], args[5])

    # (run_copy_file source_full_path dest_full_path)
            elif args[0] == "run_copy_file":
                return method(args[1], args[2])

    # (run_process_file index full_file_pathname)
            elif args[0] == "run_process_file":
                return method(args[1], args[2])

    # (run_delay time_seconds)
            elif args[0] == "run_delay":
                return method(args[1])


    # (run_logging_trigger mode index num_lines facility) e.g. "IM-5-IOX"
            elif args[0] == "run_logging_trigger":
                return method(args[1], args[2], args[3], args[4])

            else:
                self.print_log("%%%% DDR Error: Invalid run_map_network_feature call: " + str(args))

        except Exception as e:
            self.print_log("%%%% ERROR run_map_network_feature: " + str(args) + " " + str(e))
        
        
    def invalid_network_feature_call(self, *args):
        self.print_log("%%%% DDR Error: Invalid run_map_network_feature call: " + str(args))

    def run_read_control_file(self, control_file, delay):
        """
            Rule function call: (run_read_control_file "filename" delay)
            
            Invoked in the "RHS" of a triggered rule.
            This function looks in the /bootflash/guest-share/ddr with the name 'filename'
            This file is written to the guest-share by an external system and results in assertion
            of facts defined in the file.
            These facts are asserted into the CLIPs knowledge base and processed by the usecase CLIPs rules.
            This function would typically be used to look for information/instructions used to control
            the execution of the facts.
            If no file is present, the function returns with no error.  Typically the rule implementation would
            run_wait for a number of seconds then look for external facts again.
                        
            :param action_file: string with name of file in the guest-share/ddr directory
            :param delay: delay in seconds to wait before trying to find the action-facts file again
        
        Usage::

              (run_read_control_file "ddr-control" 10)

            :raises none:

        """

        while True: # Look for control file until the file is found or usecase terminates
            if control_file:
                self.control.update({"action-file" : control_file})
                self.action_index = 0
                     
                try:
                    with open(self.control["action-file"]) as file:
                        self.action_data = file.readlines()
                        self.action_data = [line.rstrip() for line in self.action_data]
                        if self.control["debug-fact"] == 1:
                            self.print_log("**** DDR Debug: run_read_control_file file content: \n" + str(self.action_data))
        #
        # assert each fact in the action_data file content
        #
                    for line in self.action_data:
                        try:                                          
                            self.env.assert_string(str(line))
                        except: pass # ignore if fact already exists in knowledge base

        #
        # delete the control_file after it is successfully processed
        # the application that wrote the control_file to the deivce checks to see if the file has been removed
        #
                    if os.path.exists(control_file):
                        os.remove(control_file)
                    return # return if control file is found

                except Exception as e:
                    if delay == 0:
                        return
                    else:
                        time.sleep(delay)
                        if self.control["debug-fact"] == 1:
                            self.print_log('**** DDR Notice: run_read_control_file wait for file: ' + str(e))
                    
            else:
                if self.control["debug-fact"] == 1:
                    self.print_log("**** DDR Debug: run_read_control_file wait for control file write to guest-share: " + str(control_file))

                if delay == 0:
                    return
                else:
                    time.sleep(delay)

    def run_assert_sim_fact(self, fact_index):
        """
            Rule function call: (run_assert_sim_fact fact_index)
            
            Invoked in the "RHS" of a triggered rule.
            This function asserts a fact in the "sim_data" list that is used
            to simulate the collection of fact data when facts can't be read from a real device.
            This allows inserted simulated facts at any step in the rule workflow execution
                        
            :param fact_index: 0 origin index into the self.sim_data list optionally loaded 
                               when the usecase locaes
        
        Usage::

              (run_assert_sim_fact 5)

            :raises none:

        """

        try:
            int_index = int(fact_index)
            if self.control["debug-fact"] == 1:
                self.print_log(self.sim_data[int_index])
            try:
                if self.control["debug-fact"] == 1:
                    self.print_log(str(self.sim_data[int_index]))
                self.env.assert_string(str(self.sim_data[int_index]))
            except: pass #ignore case where FACT already exists  
        except Exception as e:
            self.print_log('%%%%% DDR Error: run_assert_sim_fact fact assert error: ' + str(self.sim_data[int_index]))
 
    #################################################################################################
    #
    # get_action_functions - return a list of all action functions defined for DDR 
    #
    #################################################################################################
    def get_action_functions(self):
        action_functions = [self.run_show_fact, self.run_assert_message, self.run_assert_message_syslog, self.run_cli_command, self.run_apply_config, self.run_write_to_syslog, self.run_action,  self.run_ping_action, self.run_ddr, self.run_rule_timer, self.run_command, self.run_clear_selected_facts, self.run_nc_fact, self.run_trace, self.run_trace_hops, self.run_process_file, self.run_copy_file, self.run_delay, self.run_show_parameter, self.run_cli_parameter, self.run_decode_btrace_log, self.run_logging_trigger, self.run_set_runmode, self.run_map_network_feature, self.run_read_control_file, self.run_assert_sim_fact]
        return action_functions


    ########################################################################################
    #
    # read_action_facts
    #
    ########################################################################################

    def read_action_facts(self, alist):
        try:
            if os.path.exists('/bootflash/guest-share/ddr' + 'ddr-facts'):
                entries = os.listdir('/bootflash/guest-share/ddr')
                alist = []
                p = re.compile(r'^ddr-action\.(?P<no>\d)')
                for entry in entries:
                    m = p.match(entry)
                    if m:
                        alist.append(line)
                if alist:
                    for a_file in alist:
                        self.control.update({"action-file" : a_file})

                        try:
                            with open(self.control["action-file"]) as file:
                                self.action_data = file.readlines()
                                self.action_data = [line.rstrip() for line in self.action_data]
                                if self.control["debug-fact"] == 1:
                                    self.print_log("**** DDR Debug: read_action_facts file content: \n" + str(self.action_data))
        #
        # assert each fact in the action_data file content
        #
                                for line in self.action_data:
                                    try:                                          
                                        self.env.assert_string(str(line))
                                    except: pass # ignore if fact already exists in knowledge base

                        except Exception as e:
                                self.print_log('%%%%% DDR Error: read_action_facts: ' + str(e))
                else:    
                    self.print_log('%%%%% DDR Error: read_action_facts no action file present: ' + str(e))
        except Exception as e:
            self.print_log('%%%%% DDR Error: read_action_facts error: ' + str(e))
 

    ########################################################################################
    #
    # assert__action_facts
    #
    ########################################################################################

    def assert_each_action_facts(self):
        try:
            max_facts = len(self.action_data)
            if self.action_index < max_facts:
                if self.control["debug-fact"] == 1:
                    self.print_log(self.action_data[self.action_index])
                self.assert_action_facts(self.action_data[self.action_index])
                self.action_index = self.action_index + 1
        except Exception as e:
            self.print_log('%%%%% DDR Error: Action fact file assert error: ' + str(e))

