import csv
import os
import time
from typing import Iterable, Any

import requests
from bs4 import BeautifulSoup

from .Err import TransformError
from .transformers.Android import Android
from .transformers.Jsi18n import Jsi18n

statement = 'End : {}, IO File {}'


class Reader:
    Transformers = {
        "i18n": Jsi18n,
        "Android": Android
    }

    def __init__(self):
        self.key = "KEY"
        self.column = "CN"
        self.row_index = 1
        self.transformerEngine = None
        self._output_file_format = None
        self._defaultEncoding = "utf8"
        self._output_key_lowercase = True
        self.__debug = False
        self.lines = list()
        self.target = ""
        self.useEngine("")

    def setDebug(self, debug: bool) -> "Reader":
        self.__debug = debug
        return self

    def setKey(self, _key: str) -> "Reader":
        """
        specification for the KEY column
        :param _key: KEY column in string
        :return:
        """
        self.key = _key
        return self

    def overrideFileFormat(self, format_file: str, forceLowercase: bool) -> "Reader":
        """
        overrider for the file format
        :param format_file:
        :param forceLowercase:
        :return:
        """
        self._output_file_format = format_file
        self._output_key_lowercase = forceLowercase
        return self

    def setLang(self, tag: str) -> "Reader":
        """
        specification for the Language column
        :param tag: it can be the short form of the language column key
        :return:
        """
        self.column = tag
        return self

    def setTarget(self, f: str) -> "Reader":
        """
        to save the translation at the specific file folder target
        :param f:
        :return:
        """
        self.target = f
        return self

    def useEngine(self, tag: str) -> "Reader":
        """
        using the engine for the reader to get up
        :param tag:
        :return:
        """
        if tag in self.Transformers:
            self.transformerEngine = self.Transformers[tag]()
        else:
            self.transformerEngine = self.Transformers["i18n"]()

        return self

    def AppendCacheLine(self, info: str):
        """
        for debug purpose the rendering result on each line
        :param info: the line string
        :return:
        """
        if self.__debug:
            print(info)
        self.lines.append(info)

    def newSheet(self):
        """
        to remove the cache and start the newline
        :return:
        """
        self.lines = list()
        self._output_file_format = False

    def LoopMatrix(self, reader: Iterable[Iterable[Any]]):
        """
        enable the reader to start working
        :param reader:
        :return:
        """
        if not self.transformerEngine:
            raise TransformError
        if not self._output_file_format:
            self._output_file_format = self.transformerEngine.autoFileName()
        self.row_index = 0
        count = 0
        column_key = 0
        column_val = 0
        raw_key = ""
        raw_value = ""
        # print("{} - now".format(len(reader)))
        for row in reader:
            col = 0

            for rawcc in row:

                if count == self.row_index:
                    name = rawcc.strip()

                    if name == self.key:
                        column_key = col

                    if name == self.column:
                        print("checked - {}".format(name))
                        column_val = col

                if count > self.row_index:
                    name = rawcc.strip()
                    if col is column_key:
                        raw_key = name

                    if col is column_val:
                        raw_value = name

                col = col + 1

            if count == 0:
                self.AppendCacheLine(self.transformerEngine.autoGeneratedTag())

            if count > 0:
                self.AppendCacheLine(
                    self.transformerEngine.transformKeyValue(raw_key, raw_value, count == len(reader) - 1)
                )

            count = count + 1

        self.lines = self.transformerEngine.wrap_file("", self.lines)

        file = self.getFileNameInternal()
        self.writeFile("", file)
        file_io = open(file, 'a')
        for line in self.lines:
            file_io.write("{}\n".format(line))
        file_io.close()

        return self

    def LoopTable(self, reader: Iterable[Iterable[Any]]):
        """
        enable the reader to start working
        :param reader:
        :return:
        """
        if not self.transformerEngine:
            raise TransformError
        if not self._output_file_format:
            self._output_file_format = self.transformerEngine.autoFileName()
        self.row_index = 1
        count = 0
        column_key = 0
        column_val = 0
        raw_key = ""
        raw_value = ""
        # print("{} - now".format(len(reader)))
        for row in reader:
            col = 0

            for rawcc in row:

                if count == self.row_index:
                    name = rawcc.strip()

                    if name == self.key:
                        column_key = col

                    if name == self.column:
                        print("checked - {}".format(name))
                        column_val = col

                if count > self.row_index:
                    name = rawcc.strip()
                    if col is column_key:
                        raw_key = name

                    if col is column_val:
                        raw_value = name

                col = col + 1

            if count == 0:
                self.AppendCacheLine(self.transformerEngine.autoGeneratedTag())

            if count > 0:
                self.AppendCacheLine(
                    self.transformerEngine.transformKeyValue(raw_key, raw_value, count == len(reader) - 1)
                )

            count = count + 1

        self.lines = self.transformerEngine.wrap_file("", self.lines)

        file = self.getFileNameInternal()
        self.writeFile("", file)
        file_io = open(file, 'a')
        for line in self.lines:
            file_io.write("{}\n".format(line))
        file_io.close()

        return self

    def getFileNameInternal(self) -> str:
        if self._output_key_lowercase:
            keyname = self.column.lower()
        else:
            keyname = self.column
        return os.path.join(self.target, self._output_file_format.format(keyname))

    def writeFile(self, content, filename):
        fo = open(filename, "w")
        fo.write(content)
        fo.close()
        print(statement.format(time.ctime(), filename))

    def SortAskOut(self):
        """
        self.writeFile("", self.exchange_data_output)
        with opens(self.downline_batch) as lines:
            for line in lines:
                num = line.translate(str.maketrans('', '', ' \n\t\r'))
                self.NewSort().TradeSortByNumber(num)
                file_object = open(self.exchange_data_output, 'a')
                if len(self.list_downline_people) > 0:
                    file_object.write("[{}]:\n".format(num))
                    file_object.write('\n'.join(self.list_downline_people))
                else:
                    file_object.write("[{}]: NO MEMBERS \n".format(num))
                file_object.close()"""


class GoogleTranslationSheet:
    """
    featured translation service does not need to use creditentials
    """

    headers = {
        'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.12; rv:79.0) Gecko/20100101 Firefox/79.0',
        'Accept-Language': 'en-US,en;q=0.5',
        'X-Requested-With': 'XMLHttpRequest'
    }

    proxies = {
        "http": "socks5://127.0.0.1:1086",
        "https": "socks5://127.0.0.1:1086",
    }

    def __init__(self):
        self.exportedSheetUrl = ""
        self.html = ""
        self.tab = ""
        self.engine_name = ""
        self.target_folder = ""
        self.input_filepath = ""
        self.tab_content = dict()
        self.saveToCvs = True
        self.reader_cvs = 0
        self.reader_url = 0
        self.reader_csv = 0
        self.reader_debug = False
        self.connection_url_with_proxies = False
        self._readerEngine = Reader()

    def EnabledProxy(self) -> "GoogleTranslationSheet":
        self.connection_url_with_proxies = True
        return self

    def builderCvs(self, enabled: bool) -> "GoogleTranslationSheet":
        """
        enable CVS file save to the local path
        :param enabled:
        :return:
        """
        self.saveToCvs = enabled
        return self

    def builderOutputTarget(self, tar: str) -> "GoogleTranslationSheet":
        """
        save files at the specific target folder
        :param tar:
        :return:
        """
        self.target_folder = tar
        return self

    def builderTransformers(self, engine_name: str) -> "GoogleTranslationSheet":
        """
        using the transform engine from the specification tool
        :param engine_name:
        :return:
        """
        self.engine_name = engine_name
        return self

    def builderReaderDebug(self, de: bool) -> "GoogleTranslationSheet":
        """
        given the extra log detail to the reader
        :param de: yes or no
        :return:
        """
        self.reader_debug = de
        return self

    def builderReader(self, module_reader: Reader) -> "GoogleTranslationSheet":
        """
        building reader module by external instance
        :param module_reader:
        :return:
        """
        self._readerEngine = module_reader
        return self

    def builderFromCVS(self, filename: str = "") -> "GoogleTranslationSheet":
        """
        CVS file started
        :param filename:
        :return:
        """
        self.input_filepath = filename
        self.reader_cvs = 1
        return self

    def builderFromCSV(self, filename: str = "") -> "GoogleTranslationSheet":
        """
        CVS file started
        :param filename:
        :return:
        """
        self.input_filepath = filename
        self.reader_csv = 1
        return self

    def builderMeta(self, url: str, tabname: str = "") -> "GoogleTranslationSheet":
        """
        this is the required for the basic meta information for loading google sheet
        :param url:
        :param tabname:
        :return:
        """
        self.exportedSheetUrl = url
        self.tab = tabname
        self.reader_url = 1
        return self

    def builderGoogleSheet(self, sheetID: str = "", gridID: str = "", tabg: str = "") -> "GoogleTranslationSheet":
        """
        building by the sheet IDs
        :param sheetID:
        :param gridID:
        :param tabg:
        :return:
        """
        self.exportedSheetUrl = "https://docs.google.com/spreadsheets/d/e/{}/pubhtml?gid={}&single=true".format(sheetID,
                                                                                                                gridID)
        self.tab = tabg
        self.reader_url = 1
        return self

    def GetReader(self) -> Reader:
        """
        make modification setting for the internal Reader instance
        :return:
        """
        return self._readerEngine

    def getFileNameInternal(self, index: int) -> str:
        """
        get the naming for the cvs file
        :param index:
        :return:
        """
        return os.path.join(self.target_folder, "data{}.cvs".format(str(index)))

    def _run_csv(self, Lang="CN"):
        """
        read the cvs now
        :return:
        """
        if self.reader_csv == 0:
            raise TypeError("You have not set proper csv file path for the translation source.")

        self._readerEngine.newSheet()
        self._readerEngine.setLang(Lang)

        with open(self.input_filepath, newline='') as csvfile:
            line_ex = csv.reader(csvfile, delimiter=',', quotechar='|')
            self.tab_content = [[y for y in row] for row in line_ex]
            self._readerEngine.setDebug(self.reader_debug).setTarget(self.target_folder).useEngine(
                self.engine_name).LoopMatrix(self.tab_content)

    def _run_cvs(self, Lang="CN"):
        """
        read the cvs now
        :return:
        """
        if self.reader_cvs == 0:
            raise TypeError("You have not set proper cvs file path for the translation source.")

        self._readerEngine.newSheet()
        self._readerEngine.setLang(Lang)

        with open(self.input_filepath, newline='') as csvfile:
            line_ex = csv.reader(csvfile, delimiter=' ', quotechar='|')
            self.tab_content = [[y for y in row[0].split(",")] for row in line_ex]
            self._readerEngine.setDebug(self.reader_debug).setTarget(self.target_folder).useEngine(
                self.engine_name).LoopMatrix(self.tab_content)

    def _run_google_url(self, Lang="CN"):
        """
                run up the engine for given parameters
                :param proxies: whether the connection is using VPN or no
                :param Lang: the language column that matched to the sheet
                :return:
                """

        if self.reader_url == 0:
            raise TypeError("You have not set proper google sheet url for the translation source.")
        self._readerEngine.newSheet()
        self._readerEngine.setLang(Lang)

        if self.connection_url_with_proxies == True:
            self.html = requests.get(self.exportedSheetUrl, headers=self.headers, proxies=self.proxies).text
        else:
            self.html = requests.get(self.exportedSheetUrl, headers=self.headers).text

        soup = BeautifulSoup(self.html, "lxml")
        tables = soup.find_all("table")
        index = 0

        for table in tables:
            output_f = self.getFileNameInternal(index)
            self.tab_content = [[td.text for td in row.find_all("td")] for row in table.find_all("tr")]
            if self.saveToCvs:
                with open(output_f, "w") as f:
                    wr = csv.writer(f, quoting=csv.QUOTE_NONNUMERIC)
                    wr.writerows(self.tab_content)

            self._readerEngine.setDebug(self.reader_debug).setTarget(self.target_folder).useEngine(
                self.engine_name).LoopTable(self.tab_content)

            index = index + 1

    def run(self, Lang="CN"):
        if self.reader_cvs == 1:
            self._run_cvs(Lang)
        elif self.reader_csv == 1:
            self._run_csv(Lang)
        elif self.reader_url == 1:
            self._run_google_url(Lang)
        else:
            raise TypeError("Incomplete source set.")
