# coding: utf-8

# (C) Copyright IBM Corp. 2021.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# IBM OpenAPI SDK Code Generator Version: 3.29.1-b338fb38-20210313-010605
 
"""
Management interface of Vulnerability Advisor for IBM Cloud Container Registry
"""

from typing import Dict, List
import json

from ibm_cloud_sdk_core import BaseService, DetailedResponse
from ibm_cloud_sdk_core.authenticators.authenticator import Authenticator
from ibm_cloud_sdk_core.get_authenticator import get_authenticator_from_environment

from .common import get_sdk_headers

##############################################################################
# Service
##############################################################################

class VulnerabilityAdvisorV3(BaseService):
    """The Vulnerability Advisor V3 service."""

    DEFAULT_SERVICE_URL = 'https://us.icr.io'
    DEFAULT_SERVICE_NAME = 'vulnerability_advisor'

    @classmethod
    def new_instance(cls,
                     account: str,
                     service_name: str = DEFAULT_SERVICE_NAME,
                     accept_language: str = None,
                    ) -> 'VulnerabilityAdvisorV3':
        """
        Return a new client for the Vulnerability Advisor service using the
               specified parameters and external configuration.

        :param str account: The unique ID for your IBM Cloud account. Run 'ibmcloud
               cr info' to get the ID of the target account.

        :param str accept_language: (optional) The preferred language code for this
               request.
        """
        if account is None:
            raise ValueError('account must be provided')

        authenticator = get_authenticator_from_environment(service_name)
        service = cls(
            account,
            authenticator,
            accept_language,
            )
        service.configure_service(service_name)
        return service

    def __init__(self,
                 account: str,
                 authenticator: Authenticator = None,
                 accept_language: str = None,
                ) -> None:
        """
        Construct a new client for the Vulnerability Advisor service.

        :param str account: The unique ID for your IBM Cloud account. Run 'ibmcloud
               cr info' to get the ID of the target account.

        :param Authenticator authenticator: The authenticator specifies the authentication mechanism.
               Get up to date information from https://github.com/IBM/python-sdk-core/blob/master/README.md
               about initializing the authenticator of your choice.

        :param str accept_language: (optional) The preferred language code for this
               request.
        """
        if account is None:
            raise ValueError('account must be provided')

        BaseService.__init__(self,
                             service_url=self.DEFAULT_SERVICE_URL,
                             authenticator=authenticator)
        self.account = account
        self.accept_language = accept_language


    #########################
    # Report
    #########################


    def account_report_query_path(self,
        *,
        repository: str = None,
        include_ibm: str = None,
        include_private: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Get the vulnerability assessment for all images.

        Get the vulnerability assessment for the list of registry images that belong to a
        specific account.

        :param str repository: (optional) The name of the repository that you want
               to see image vulnerability assessments for. For example,
               us.icr.io/namespace/image.
        :param str include_ibm: (optional) When set to true, the returned list
               contains IBM public images and the account images. If not set, or set to
               false, the list contains only the account images.
        :param str include_private: (optional) When set to false, the returned list
               does not contain the private account images. If not set, or set to true,
               the list contains the private account images.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ScanReportList` object
        """

        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='account_report_query_path')
        headers.update(sdk_headers)

        params = {
            'repository': repository,
            'includeIBM': include_ibm,
            'includePrivate': include_private
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        url = '/va/api/v3/report/account'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response


    def account_status_query_path(self,
        *,
        repository: str = None,
        include_ibm: str = None,
        include_private: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Get vulnerability assessment status for all images.

        Get the vulnerability assessment status for the list of registry images that
        belong to a specific account.

        :param str repository: (optional) The name of the repository that you want
               to see image vulnerability assessments for. For example,
               us.icr.io/namespace/image.
        :param str include_ibm: (optional) When set to true, the returned list
               contains IBM public images and the account images. If not set, or set to
               false, the list contains only the account images.
        :param str include_private: (optional) When set to false, the returned list
               does not contain the private account images. If not set, or set to true,
               the list contains the private account images.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ScanreportImageSummaryList` object
        """

        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='account_status_query_path')
        headers.update(sdk_headers)

        params = {
            'repository': repository,
            'includeIBM': include_ibm,
            'includePrivate': include_private
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        url = '/va/api/v3/report/account/status'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response


    def image_report_query_path(self,
        name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Get vulnerability assessment.

        Get the vulnerability assessment for a registry image.

        :param str name: The name of the image. For example,
               us.icr.io/namespace/repository:tag.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ScanReport` object
        """

        if name is None:
            raise ValueError('name must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='image_report_query_path')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['name']
        path_param_values = self.encode_path_vars(name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/va/api/v3/report/image/{name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def image_status_query_path(self,
        name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Get vulnerability status.

        Get the overall vulnerability status for a registry image.

        :param str name: The name of the image. For example,
               us.icr.io/namespace/repository:tag.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ScanreportSummary` object
        """

        if name is None:
            raise ValueError('name must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='image_status_query_path')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['name']
        path_param_values = self.encode_path_vars(name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/va/api/v3/report/image/status/{name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response

    #########################
    # Exemption
    #########################


    def list_exemption_account(self,
        **kwargs
    ) -> DetailedResponse:
        """
        List account level exemptions.

        List the exemptions that are specified with account level scope.

        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `List[Exemption]` result
        """

        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='list_exemption_account')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        url = '/va/api/v3/exempt/image'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def get_exemption_account(self,
        issue_type: str,
        issue_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Get an account level exemption.

        Get details of an exemption that is specified with account level scope.

        :param str issue_type: Exemption type, e.g. 'cve' or 'sn' or
               'configuration'. See /va/api/v3/exempt/types for more details.
        :param str issue_id: Exemption ID, e.g. 'CVE-2018-9999'. See
               /va/api/v3/exempt/types for more details.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Exemption` object
        """

        if issue_type is None:
            raise ValueError('issue_type must be provided')
        if issue_id is None:
            raise ValueError('issue_id must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='get_exemption_account')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['issueType', 'issueID']
        path_param_values = self.encode_path_vars(issue_type, issue_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/va/api/v3/exempt/image/issue/{issueType}/{issueID}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def create_exemption_account(self,
        issue_type: str,
        issue_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Create or update an account level exemption.

        Create or update an exemption that is specified with account level scope.

        :param str issue_type: Exemption type, e.g. 'cve' or 'sn' or
               'configuration'. See /va/api/v3/exempt/types for more details.
        :param str issue_id: Exemption ID, e.g. 'CVE-2018-9999'. See
               /va/api/v3/exempt/types for more details.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Exemption` object
        """

        if issue_type is None:
            raise ValueError('issue_type must be provided')
        if issue_id is None:
            raise ValueError('issue_id must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='create_exemption_account')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['issueType', 'issueID']
        path_param_values = self.encode_path_vars(issue_type, issue_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/va/api/v3/exempt/image/issue/{issueType}/{issueID}'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def delete_exemption_account(self,
        issue_type: str,
        issue_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an account level exemption.

        Delete an exemption that is specified with account level scope.

        :param str issue_type: Exemption type, e.g. 'cve' or 'sn' or
               'configuration'. See /va/api/v3/exempt/types for more details.
        :param str issue_id: Exemption ID, e.g. 'CVE-2018-9999'. See
               /va/api/v3/exempt/types for more details.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if issue_type is None:
            raise ValueError('issue_type must be provided')
        if issue_id is None:
            raise ValueError('issue_id must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='delete_exemption_account')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        path_param_keys = ['issueType', 'issueID']
        path_param_values = self.encode_path_vars(issue_type, issue_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/va/api/v3/exempt/image/issue/{issueType}/{issueID}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def list_exemption_resource(self,
        resource: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List resource exemptions.

        List the exemptions that are specified for a resource (account, registry
        namespace, repository, or image).

        :param str resource: IBM Cloud Registry resource (namespace,
               namespace/repository, namespace/repository:tag, or
               namespace/repository@sha256:hash).
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `List[Exemption]` result
        """

        if resource is None:
            raise ValueError('resource must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='list_exemption_resource')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['resource']
        path_param_values = self.encode_path_vars(resource)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/va/api/v3/exempt/image/{resource}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def get_exemption_resource(self,
        resource: str,
        issue_type: str,
        issue_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Get details of a resource exemption.

        Get an exemption that is specified for a resource (account, registry namespace,
        repository, or image).

        :param str resource: IBM Cloud Registry resource (namespace,
               namespace/repository, namespace/repository:tag, or
               namespace/repository@sha256:hash).
        :param str issue_type: Exemption type, e.g. 'cve' or 'sn' or
               'configuration'. See /va/api/v3/exempt/types for more details.
        :param str issue_id: Exemption ID, e.g. 'CVE-2018-9999'. See
               /va/api/v3/exempt/types for more details.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Exemption` object
        """

        if resource is None:
            raise ValueError('resource must be provided')
        if issue_type is None:
            raise ValueError('issue_type must be provided')
        if issue_id is None:
            raise ValueError('issue_id must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='get_exemption_resource')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['resource', 'issueType', 'issueID']
        path_param_values = self.encode_path_vars(resource, issue_type, issue_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/va/api/v3/exempt/image/{resource}/issue/{issueType}/{issueID}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def create_exemption_resource(self,
        resource: str,
        issue_type: str,
        issue_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Create or update a resource exemption.

        Create or update an exemption that is specified for a resource (account, registry
        namespace, repository, or image).

        :param str resource: IBM Cloud Registry resource (namespace,
               namespace/repository, namespace/repository:tag, or
               namespace/repository@sha256:hash).
        :param str issue_type: Exemption type, e.g. 'cve' or 'sn' or
               'configuration'. See /va/api/v3/exempt/types for more details.
        :param str issue_id: Exemption ID, e.g. 'CVE-2018-9999'. See
               /va/api/v3/exempt/types for more details.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Exemption` object
        """

        if resource is None:
            raise ValueError('resource must be provided')
        if issue_type is None:
            raise ValueError('issue_type must be provided')
        if issue_id is None:
            raise ValueError('issue_id must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='create_exemption_resource')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['resource', 'issueType', 'issueID']
        path_param_values = self.encode_path_vars(resource, issue_type, issue_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/va/api/v3/exempt/image/{resource}/issue/{issueType}/{issueID}'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def delete_exemption_resource(self,
        resource: str,
        issue_type: str,
        issue_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a resource exemption.

        Delete an exemption that is specified for a resource (account, registry namespace,
        repository, or image).

        :param str resource: IBM Cloud Registry resource (namespace,
               namespace/repository, namespace/repository:tag, or
               namespace/repository@sha256:hash).
        :param str issue_type: Exemption type, e.g. 'cve' or 'sn' or
               'configuration'. See /va/api/v3/exempt/types for more details.
        :param str issue_id: Exemption ID, e.g. 'CVE-2018-9999'. See
               /va/api/v3/exempt/types for more details.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if resource is None:
            raise ValueError('resource must be provided')
        if issue_type is None:
            raise ValueError('issue_type must be provided')
        if issue_id is None:
            raise ValueError('issue_id must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='delete_exemption_resource')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))

        path_param_keys = ['resource', 'issueType', 'issueID']
        path_param_values = self.encode_path_vars(resource, issue_type, issue_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/va/api/v3/exempt/image/{resource}/issue/{issueType}/{issueID}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def exempt_handler(self,
        **kwargs
    ) -> DetailedResponse:
        """
        List the types of exemption.

        List the types of exemption.

        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `List[ExemptionTypeInfo]` result
        """

        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='exempt_handler')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        url = '/va/api/v3/exempt/types'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def list_account_exemptions(self,
        **kwargs
    ) -> DetailedResponse:
        """
        List all exemptions.

        List all of the exemptions in the given account.

        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `List[Exemption]` result
        """

        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='list_account_exemptions')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        url = '/va/api/v3/exemptions/account'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def exemptions_account_delete_handler(self,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete all exemptions.

        Delete all of the exemptions in the given account.

        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ExemptionDeletionInfo` object
        """

        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='exemptions_account_delete_handler')
        headers.update(sdk_headers)

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        url = '/va/api/v3/exemptions/deleteAll'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers)

        response = self.send(request)
        return response


    def list_image_exemptions(self,
        resource: str,
        *,
        include_scope: bool = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List image exemptions.

        List all of the exemptions for an image.

        :param str resource: IBM Cloud Registry resource (namespace,
               namespace/repository, namespace/repository:tag, or
               namespace/repository@sha256:hash).
        :param bool include_scope: (optional) Include scope on returned exemptions.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `List[Exemption]` result
        """

        if resource is None:
            raise ValueError('resource must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='list_image_exemptions')
        headers.update(sdk_headers)

        params = {
            'includeScope': include_scope
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['resource']
        path_param_values = self.encode_path_vars(resource)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/va/api/v3/exemptions/image/{resource}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request)
        return response


    def list_bulk_image_exemptions(self,
        body: List[str],
        **kwargs
    ) -> DetailedResponse:
        """
        List exemptions for images.

        List the exemptions for the given list of images.

        :param List[str] body: List of images.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result
        """

        if body is None:
            raise ValueError('body must be provided')
        headers = {
            'Account': self.account,
            'Accept-Language': self.accept_language
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V3',
                                      operation_id='list_bulk_image_exemptions')
        headers.update(sdk_headers)

        data = json.dumps(body)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        url = '/va/api/v3/exemptions/images'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       data=data)

        response = self.send(request)
        return response


##############################################################################
# Models
##############################################################################


class Exemption():
    """
    Exemption.

    :attr str account_id: The unique ID for the IBM Cloud account for this policy.
    :attr str issue_id: ID of issue being exempted.
    :attr str issue_type: Type of issue being exempted.
    :attr ExemptionScope scope: (optional)
    """

    def __init__(self,
                 account_id: str,
                 issue_id: str,
                 issue_type: str,
                 *,
                 scope: 'ExemptionScope' = None) -> None:
        """
        Initialize a Exemption object.

        :param str account_id: The unique ID for the IBM Cloud account for this
               policy.
        :param str issue_id: ID of issue being exempted.
        :param str issue_type: Type of issue being exempted.
        :param ExemptionScope scope: (optional)
        """
        self.account_id = account_id
        self.issue_id = issue_id
        self.issue_type = issue_type
        self.scope = scope

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'Exemption':
        """Initialize a Exemption object from a json dictionary."""
        args = {}
        if 'account_id' in _dict:
            args['account_id'] = _dict.get('account_id')
        else:
            raise ValueError('Required property \'account_id\' not present in Exemption JSON')
        if 'issue_id' in _dict:
            args['issue_id'] = _dict.get('issue_id')
        else:
            raise ValueError('Required property \'issue_id\' not present in Exemption JSON')
        if 'issue_type' in _dict:
            args['issue_type'] = _dict.get('issue_type')
        else:
            raise ValueError('Required property \'issue_type\' not present in Exemption JSON')
        if 'scope' in _dict:
            args['scope'] = ExemptionScope.from_dict(_dict.get('scope'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a Exemption object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'account_id') and self.account_id is not None:
            _dict['account_id'] = self.account_id
        if hasattr(self, 'issue_id') and self.issue_id is not None:
            _dict['issue_id'] = self.issue_id
        if hasattr(self, 'issue_type') and self.issue_type is not None:
            _dict['issue_type'] = self.issue_type
        if hasattr(self, 'scope') and self.scope is not None:
            _dict['scope'] = self.scope.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this Exemption object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'Exemption') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'Exemption') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ExemptionScope():
    """
    ExemptionScope.

    :attr str scope_type: The type of scope the exemption applies to: account,
          namespace, repository, image.
    :attr str namespace: (optional) The IBM Cloud Registry namespace included in
          scope.
    :attr str repository: (optional) The IBM Cloud Registry repository included in
          scope.
    :attr str tag: (optional) The IBM Cloud Registry tag included in scope.
    """

    def __init__(self,
                 scope_type: str,
                 *,
                 namespace: str = None,
                 repository: str = None,
                 tag: str = None) -> None:
        """
        Initialize a ExemptionScope object.

        :param str scope_type: The type of scope the exemption applies to: account,
               namespace, repository, image.
        :param str namespace: (optional) The IBM Cloud Registry namespace included
               in scope.
        :param str repository: (optional) The IBM Cloud Registry repository
               included in scope.
        :param str tag: (optional) The IBM Cloud Registry tag included in scope.
        """
        self.scope_type = scope_type
        self.namespace = namespace
        self.repository = repository
        self.tag = tag

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ExemptionScope':
        """Initialize a ExemptionScope object from a json dictionary."""
        args = {}
        if 'scope_type' in _dict:
            args['scope_type'] = _dict.get('scope_type')
        else:
            raise ValueError('Required property \'scope_type\' not present in ExemptionScope JSON')
        if 'namespace' in _dict:
            args['namespace'] = _dict.get('namespace')
        if 'repository' in _dict:
            args['repository'] = _dict.get('repository')
        if 'tag' in _dict:
            args['tag'] = _dict.get('tag')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ExemptionScope object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'scope_type') and self.scope_type is not None:
            _dict['scope_type'] = self.scope_type
        if hasattr(self, 'namespace') and self.namespace is not None:
            _dict['namespace'] = self.namespace
        if hasattr(self, 'repository') and self.repository is not None:
            _dict['repository'] = self.repository
        if hasattr(self, 'tag') and self.tag is not None:
            _dict['tag'] = self.tag
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ExemptionScope object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ExemptionScope') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ExemptionScope') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ScanReport():
    """
    ScanReport.

    :attr List[ScanresultConfigurationIssue] configuration_issues: Possible
          configuration issues found in the docker object.
    :attr str id: The unique ID of the report.
    :attr int scan_time: The scan time of the report as a UNIX timestamp.
    :attr str status: Overall vulnerability assessment status: OK, WARN, FAIL,
          UNSUPPORTED, INCOMPLETE, UNSCANNED. For more information about these status
          codes, see
          https://{DomainName}/apidocs/container-registry/va#getting-started-vulnerability-report-status-codes.
    :attr List[ScanresultCVE] vulnerabilities: Vulnerabilities found in the docker
          object at the scan time.
    """

    def __init__(self,
                 configuration_issues: List['ScanresultConfigurationIssue'],
                 id: str,
                 scan_time: int,
                 status: str,
                 vulnerabilities: List['ScanresultCVE']) -> None:
        """
        Initialize a ScanReport object.

        :param List[ScanresultConfigurationIssue] configuration_issues: Possible
               configuration issues found in the docker object.
        :param str id: The unique ID of the report.
        :param int scan_time: The scan time of the report as a UNIX timestamp.
        :param str status: Overall vulnerability assessment status: OK, WARN, FAIL,
               UNSUPPORTED, INCOMPLETE, UNSCANNED. For more information about these status
               codes, see
               https://{DomainName}/apidocs/container-registry/va#getting-started-vulnerability-report-status-codes.
        :param List[ScanresultCVE] vulnerabilities: Vulnerabilities found in the
               docker object at the scan time.
        """
        self.configuration_issues = configuration_issues
        self.id = id
        self.scan_time = scan_time
        self.status = status
        self.vulnerabilities = vulnerabilities

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ScanReport':
        """Initialize a ScanReport object from a json dictionary."""
        args = {}
        if 'configuration_issues' in _dict:
            args['configuration_issues'] = [ScanresultConfigurationIssue.from_dict(x) for x in _dict.get('configuration_issues')]
        else:
            raise ValueError('Required property \'configuration_issues\' not present in ScanReport JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError('Required property \'id\' not present in ScanReport JSON')
        if 'scan_time' in _dict:
            args['scan_time'] = _dict.get('scan_time')
        else:
            raise ValueError('Required property \'scan_time\' not present in ScanReport JSON')
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError('Required property \'status\' not present in ScanReport JSON')
        if 'vulnerabilities' in _dict:
            args['vulnerabilities'] = [ScanresultCVE.from_dict(x) for x in _dict.get('vulnerabilities')]
        else:
            raise ValueError('Required property \'vulnerabilities\' not present in ScanReport JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ScanReport object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'configuration_issues') and self.configuration_issues is not None:
            _dict['configuration_issues'] = [x.to_dict() for x in self.configuration_issues]
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'scan_time') and self.scan_time is not None:
            _dict['scan_time'] = self.scan_time
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'vulnerabilities') and self.vulnerabilities is not None:
            _dict['vulnerabilities'] = [x.to_dict() for x in self.vulnerabilities]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ScanReport object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ScanReport') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ScanReport') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ScanReportList():
    """
    ScanReportList.

    :attr dict assessments: A dictionary of image names as key and report.Report
          object as value.
    """

    def __init__(self,
                 assessments: dict) -> None:
        """
        Initialize a ScanReportList object.

        :param dict assessments: A dictionary of image names as key and
               report.Report object as value.
        """
        self.assessments = assessments

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ScanReportList':
        """Initialize a ScanReportList object from a json dictionary."""
        args = {}
        if 'assessments' in _dict:
            args['assessments'] = {k : ScanReport.from_dict(v) for k, v in _dict.get('assessments').items()}
        else:
            raise ValueError('Required property \'assessments\' not present in ScanReportList JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ScanReportList object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'assessments') and self.assessments is not None:
            _dict['assessments'] = {k : v.to_dict() for k, v in self.assessments.items()}
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ScanReportList object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ScanReportList') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ScanReportList') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ExemptionDeletionInfo():
    """
    ExemptionDeletionInfo.

    :attr int number_of_exemptions_deleted:
    """

    def __init__(self,
                 number_of_exemptions_deleted: int) -> None:
        """
        Initialize a ExemptionDeletionInfo object.

        :param int number_of_exemptions_deleted:
        """
        self.number_of_exemptions_deleted = number_of_exemptions_deleted

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ExemptionDeletionInfo':
        """Initialize a ExemptionDeletionInfo object from a json dictionary."""
        args = {}
        if 'number_of_exemptions_deleted' in _dict:
            args['number_of_exemptions_deleted'] = _dict.get('number_of_exemptions_deleted')
        else:
            raise ValueError('Required property \'number_of_exemptions_deleted\' not present in ExemptionDeletionInfo JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ExemptionDeletionInfo object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'number_of_exemptions_deleted') and self.number_of_exemptions_deleted is not None:
            _dict['number_of_exemptions_deleted'] = self.number_of_exemptions_deleted
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ExemptionDeletionInfo object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ExemptionDeletionInfo') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ExemptionDeletionInfo') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ExemptionTypeInfo():
    """
    ExemptionTypeInfo.

    :attr str display_name:
    :attr str identifier:
    :attr str translation_key: (optional)
    :attr str validation_regex:
    """

    def __init__(self,
                 display_name: str,
                 identifier: str,
                 validation_regex: str,
                 *,
                 translation_key: str = None) -> None:
        """
        Initialize a ExemptionTypeInfo object.

        :param str display_name:
        :param str identifier:
        :param str validation_regex:
        :param str translation_key: (optional)
        """
        self.display_name = display_name
        self.identifier = identifier
        self.translation_key = translation_key
        self.validation_regex = validation_regex

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ExemptionTypeInfo':
        """Initialize a ExemptionTypeInfo object from a json dictionary."""
        args = {}
        if 'display_name' in _dict:
            args['display_name'] = _dict.get('display_name')
        else:
            raise ValueError('Required property \'display_name\' not present in ExemptionTypeInfo JSON')
        if 'identifier' in _dict:
            args['identifier'] = _dict.get('identifier')
        else:
            raise ValueError('Required property \'identifier\' not present in ExemptionTypeInfo JSON')
        if 'translationKey' in _dict:
            args['translation_key'] = _dict.get('translationKey')
        if 'validation_regex' in _dict:
            args['validation_regex'] = _dict.get('validation_regex')
        else:
            raise ValueError('Required property \'validation_regex\' not present in ExemptionTypeInfo JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ExemptionTypeInfo object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'display_name') and self.display_name is not None:
            _dict['display_name'] = self.display_name
        if hasattr(self, 'identifier') and self.identifier is not None:
            _dict['identifier'] = self.identifier
        if hasattr(self, 'translation_key') and self.translation_key is not None:
            _dict['translationKey'] = self.translation_key
        if hasattr(self, 'validation_regex') and self.validation_regex is not None:
            _dict['validation_regex'] = self.validation_regex
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ExemptionTypeInfo object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ExemptionTypeInfo') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ExemptionTypeInfo') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ScanreportImageSummary():
    """
    ScanreportImageSummary.

    :attr int configuration_issue_count: The number of configuration issues found.
    :attr int created_time: The image creation time as a UNIX timestamp.
    :attr int exempt_configuration_issue_count: The number of exempt configuration
          issues found.
    :attr int exempt_issue_count: The number of exempt issues found.
    :attr int exempt_vulnerability_count: The number of exempt vulnerability issues
          found.
    :attr int issue_count: The number of issues found.
    :attr str name: Full docker image name including tag e.g.
          us.icr.io/namespace/repository:tag.
    :attr int scan_time: The scan time of the report as a UNIX timestamp.
    :attr str status: Overall vulnerability assessment status: OK, WARN, FAIL,
          UNSUPPORTED, INCOMPLETE, UNSCANNED. For more information about these status
          codes, see
          https://{DomainName}/apidocs/container-registry/va#getting-started-vulnerability-report-status-codes.
    :attr int vulnerability_count: The number of vulnerability issues found.
    """

    def __init__(self,
                 configuration_issue_count: int,
                 created_time: int,
                 exempt_configuration_issue_count: int,
                 exempt_issue_count: int,
                 exempt_vulnerability_count: int,
                 issue_count: int,
                 name: str,
                 scan_time: int,
                 status: str,
                 vulnerability_count: int) -> None:
        """
        Initialize a ScanreportImageSummary object.

        :param int configuration_issue_count: The number of configuration issues
               found.
        :param int created_time: The image creation time as a UNIX timestamp.
        :param int exempt_configuration_issue_count: The number of exempt
               configuration issues found.
        :param int exempt_issue_count: The number of exempt issues found.
        :param int exempt_vulnerability_count: The number of exempt vulnerability
               issues found.
        :param int issue_count: The number of issues found.
        :param str name: Full docker image name including tag e.g.
               us.icr.io/namespace/repository:tag.
        :param int scan_time: The scan time of the report as a UNIX timestamp.
        :param str status: Overall vulnerability assessment status: OK, WARN, FAIL,
               UNSUPPORTED, INCOMPLETE, UNSCANNED. For more information about these status
               codes, see
               https://{DomainName}/apidocs/container-registry/va#getting-started-vulnerability-report-status-codes.
        :param int vulnerability_count: The number of vulnerability issues found.
        """
        self.configuration_issue_count = configuration_issue_count
        self.created_time = created_time
        self.exempt_configuration_issue_count = exempt_configuration_issue_count
        self.exempt_issue_count = exempt_issue_count
        self.exempt_vulnerability_count = exempt_vulnerability_count
        self.issue_count = issue_count
        self.name = name
        self.scan_time = scan_time
        self.status = status
        self.vulnerability_count = vulnerability_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ScanreportImageSummary':
        """Initialize a ScanreportImageSummary object from a json dictionary."""
        args = {}
        if 'configuration_issue_count' in _dict:
            args['configuration_issue_count'] = _dict.get('configuration_issue_count')
        else:
            raise ValueError('Required property \'configuration_issue_count\' not present in ScanreportImageSummary JSON')
        if 'created_time' in _dict:
            args['created_time'] = _dict.get('created_time')
        else:
            raise ValueError('Required property \'created_time\' not present in ScanreportImageSummary JSON')
        if 'exempt_configuration_issue_count' in _dict:
            args['exempt_configuration_issue_count'] = _dict.get('exempt_configuration_issue_count')
        else:
            raise ValueError('Required property \'exempt_configuration_issue_count\' not present in ScanreportImageSummary JSON')
        if 'exempt_issue_count' in _dict:
            args['exempt_issue_count'] = _dict.get('exempt_issue_count')
        else:
            raise ValueError('Required property \'exempt_issue_count\' not present in ScanreportImageSummary JSON')
        if 'exempt_vulnerability_count' in _dict:
            args['exempt_vulnerability_count'] = _dict.get('exempt_vulnerability_count')
        else:
            raise ValueError('Required property \'exempt_vulnerability_count\' not present in ScanreportImageSummary JSON')
        if 'issue_count' in _dict:
            args['issue_count'] = _dict.get('issue_count')
        else:
            raise ValueError('Required property \'issue_count\' not present in ScanreportImageSummary JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError('Required property \'name\' not present in ScanreportImageSummary JSON')
        if 'scan_time' in _dict:
            args['scan_time'] = _dict.get('scan_time')
        else:
            raise ValueError('Required property \'scan_time\' not present in ScanreportImageSummary JSON')
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError('Required property \'status\' not present in ScanreportImageSummary JSON')
        if 'vulnerability_count' in _dict:
            args['vulnerability_count'] = _dict.get('vulnerability_count')
        else:
            raise ValueError('Required property \'vulnerability_count\' not present in ScanreportImageSummary JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ScanreportImageSummary object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'configuration_issue_count') and self.configuration_issue_count is not None:
            _dict['configuration_issue_count'] = self.configuration_issue_count
        if hasattr(self, 'created_time') and self.created_time is not None:
            _dict['created_time'] = self.created_time
        if hasattr(self, 'exempt_configuration_issue_count') and self.exempt_configuration_issue_count is not None:
            _dict['exempt_configuration_issue_count'] = self.exempt_configuration_issue_count
        if hasattr(self, 'exempt_issue_count') and self.exempt_issue_count is not None:
            _dict['exempt_issue_count'] = self.exempt_issue_count
        if hasattr(self, 'exempt_vulnerability_count') and self.exempt_vulnerability_count is not None:
            _dict['exempt_vulnerability_count'] = self.exempt_vulnerability_count
        if hasattr(self, 'issue_count') and self.issue_count is not None:
            _dict['issue_count'] = self.issue_count
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'scan_time') and self.scan_time is not None:
            _dict['scan_time'] = self.scan_time
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'vulnerability_count') and self.vulnerability_count is not None:
            _dict['vulnerability_count'] = self.vulnerability_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ScanreportImageSummary object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ScanreportImageSummary') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ScanreportImageSummary') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ScanreportImageSummaryList():
    """
    ScanreportImageSummaryList.

    :attr List[ScanreportImageSummary] images: List of image summaries.
    """

    def __init__(self,
                 images: List['ScanreportImageSummary']) -> None:
        """
        Initialize a ScanreportImageSummaryList object.

        :param List[ScanreportImageSummary] images: List of image summaries.
        """
        self.images = images

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ScanreportImageSummaryList':
        """Initialize a ScanreportImageSummaryList object from a json dictionary."""
        args = {}
        if 'images' in _dict:
            args['images'] = [ScanreportImageSummary.from_dict(x) for x in _dict.get('images')]
        else:
            raise ValueError('Required property \'images\' not present in ScanreportImageSummaryList JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ScanreportImageSummaryList object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'images') and self.images is not None:
            _dict['images'] = [x.to_dict() for x in self.images]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ScanreportImageSummaryList object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ScanreportImageSummaryList') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ScanreportImageSummaryList') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ScanreportSummary():
    """
    ScanreportSummary.

    :attr int configuration_issue_count: The number of configuration issues found.
    :attr int exempt_configuration_issue_count: The number of exempt configuration
          issues found.
    :attr int exempt_issue_count: The number of exempt issues found.
    :attr int exempt_vulnerability_count: The number of exempt vulnerability issues
          found.
    :attr int issue_count: The number of issues found.
    :attr int scan_time: The scan time of the report as a UNIX timestamp.
    :attr str status: Overall vulnerability assessment status: OK, WARN, FAIL,
          UNSUPPORTED, INCOMPLETE, UNSCANNED. For more information about these status
          codes, see
          https://{DomainName}/apidocs/container-registry/va#getting-started-vulnerability-report-status-codes.
    :attr int vulnerability_count: The number of vulnerability issues found.
    """

    def __init__(self,
                 configuration_issue_count: int,
                 exempt_configuration_issue_count: int,
                 exempt_issue_count: int,
                 exempt_vulnerability_count: int,
                 issue_count: int,
                 scan_time: int,
                 status: str,
                 vulnerability_count: int) -> None:
        """
        Initialize a ScanreportSummary object.

        :param int configuration_issue_count: The number of configuration issues
               found.
        :param int exempt_configuration_issue_count: The number of exempt
               configuration issues found.
        :param int exempt_issue_count: The number of exempt issues found.
        :param int exempt_vulnerability_count: The number of exempt vulnerability
               issues found.
        :param int issue_count: The number of issues found.
        :param int scan_time: The scan time of the report as a UNIX timestamp.
        :param str status: Overall vulnerability assessment status: OK, WARN, FAIL,
               UNSUPPORTED, INCOMPLETE, UNSCANNED. For more information about these status
               codes, see
               https://{DomainName}/apidocs/container-registry/va#getting-started-vulnerability-report-status-codes.
        :param int vulnerability_count: The number of vulnerability issues found.
        """
        self.configuration_issue_count = configuration_issue_count
        self.exempt_configuration_issue_count = exempt_configuration_issue_count
        self.exempt_issue_count = exempt_issue_count
        self.exempt_vulnerability_count = exempt_vulnerability_count
        self.issue_count = issue_count
        self.scan_time = scan_time
        self.status = status
        self.vulnerability_count = vulnerability_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ScanreportSummary':
        """Initialize a ScanreportSummary object from a json dictionary."""
        args = {}
        if 'configuration_issue_count' in _dict:
            args['configuration_issue_count'] = _dict.get('configuration_issue_count')
        else:
            raise ValueError('Required property \'configuration_issue_count\' not present in ScanreportSummary JSON')
        if 'exempt_configuration_issue_count' in _dict:
            args['exempt_configuration_issue_count'] = _dict.get('exempt_configuration_issue_count')
        else:
            raise ValueError('Required property \'exempt_configuration_issue_count\' not present in ScanreportSummary JSON')
        if 'exempt_issue_count' in _dict:
            args['exempt_issue_count'] = _dict.get('exempt_issue_count')
        else:
            raise ValueError('Required property \'exempt_issue_count\' not present in ScanreportSummary JSON')
        if 'exempt_vulnerability_count' in _dict:
            args['exempt_vulnerability_count'] = _dict.get('exempt_vulnerability_count')
        else:
            raise ValueError('Required property \'exempt_vulnerability_count\' not present in ScanreportSummary JSON')
        if 'issue_count' in _dict:
            args['issue_count'] = _dict.get('issue_count')
        else:
            raise ValueError('Required property \'issue_count\' not present in ScanreportSummary JSON')
        if 'scan_time' in _dict:
            args['scan_time'] = _dict.get('scan_time')
        else:
            raise ValueError('Required property \'scan_time\' not present in ScanreportSummary JSON')
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError('Required property \'status\' not present in ScanreportSummary JSON')
        if 'vulnerability_count' in _dict:
            args['vulnerability_count'] = _dict.get('vulnerability_count')
        else:
            raise ValueError('Required property \'vulnerability_count\' not present in ScanreportSummary JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ScanreportSummary object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'configuration_issue_count') and self.configuration_issue_count is not None:
            _dict['configuration_issue_count'] = self.configuration_issue_count
        if hasattr(self, 'exempt_configuration_issue_count') and self.exempt_configuration_issue_count is not None:
            _dict['exempt_configuration_issue_count'] = self.exempt_configuration_issue_count
        if hasattr(self, 'exempt_issue_count') and self.exempt_issue_count is not None:
            _dict['exempt_issue_count'] = self.exempt_issue_count
        if hasattr(self, 'exempt_vulnerability_count') and self.exempt_vulnerability_count is not None:
            _dict['exempt_vulnerability_count'] = self.exempt_vulnerability_count
        if hasattr(self, 'issue_count') and self.issue_count is not None:
            _dict['issue_count'] = self.issue_count
        if hasattr(self, 'scan_time') and self.scan_time is not None:
            _dict['scan_time'] = self.scan_time
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'vulnerability_count') and self.vulnerability_count is not None:
            _dict['vulnerability_count'] = self.vulnerability_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ScanreportSummary object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ScanreportSummary') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ScanreportSummary') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ScanresultCVE():
    """
    ScanresultCVE.

    :attr bool cve_exempt: True if this CVE has been exempted by user policy, and
          false otherwise.
    :attr str cve_id: The ID for this CVE.
    :attr int exempt_security_notice_count: Number of security notices that contain
          fixes for this CVE and are exempted by user policy.
    :attr str exempt_status: 'exempt' if this CVE is exempt or all security notices
          for this CVE are exempt. 'partial' if this CVE is not exempt but a subset of
          security notices for this CVE are exempt. 'active' if this CVE is not exempt and
          no security notices for this CVE are exempt.
    :attr int security_notice_count: Number of security notices that contain fixes
          for this CVE and are not exempted by user policy.
    :attr List[ScanresultSecurityNotice] security_notices: Security notices that
          contain fixes for this CVE.
    :attr str summary: The summary of the security vulnerability for this CVE.
    :attr int total_security_notice_count: Total number of security notices that
          contain fixes for this CVE.
    """

    def __init__(self,
                 cve_exempt: bool,
                 cve_id: str,
                 exempt_security_notice_count: int,
                 exempt_status: str,
                 security_notice_count: int,
                 security_notices: List['ScanresultSecurityNotice'],
                 summary: str,
                 total_security_notice_count: int) -> None:
        """
        Initialize a ScanresultCVE object.

        :param bool cve_exempt: True if this CVE has been exempted by user policy,
               and false otherwise.
        :param str cve_id: The ID for this CVE.
        :param int exempt_security_notice_count: Number of security notices that
               contain fixes for this CVE and are exempted by user policy.
        :param str exempt_status: 'exempt' if this CVE is exempt or all security
               notices for this CVE are exempt. 'partial' if this CVE is not exempt but a
               subset of security notices for this CVE are exempt. 'active' if this CVE is
               not exempt and no security notices for this CVE are exempt.
        :param int security_notice_count: Number of security notices that contain
               fixes for this CVE and are not exempted by user policy.
        :param List[ScanresultSecurityNotice] security_notices: Security notices
               that contain fixes for this CVE.
        :param str summary: The summary of the security vulnerability for this CVE.
        :param int total_security_notice_count: Total number of security notices
               that contain fixes for this CVE.
        """
        self.cve_exempt = cve_exempt
        self.cve_id = cve_id
        self.exempt_security_notice_count = exempt_security_notice_count
        self.exempt_status = exempt_status
        self.security_notice_count = security_notice_count
        self.security_notices = security_notices
        self.summary = summary
        self.total_security_notice_count = total_security_notice_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ScanresultCVE':
        """Initialize a ScanresultCVE object from a json dictionary."""
        args = {}
        if 'cve_exempt' in _dict:
            args['cve_exempt'] = _dict.get('cve_exempt')
        else:
            raise ValueError('Required property \'cve_exempt\' not present in ScanresultCVE JSON')
        if 'cve_id' in _dict:
            args['cve_id'] = _dict.get('cve_id')
        else:
            raise ValueError('Required property \'cve_id\' not present in ScanresultCVE JSON')
        if 'exempt_security_notice_count' in _dict:
            args['exempt_security_notice_count'] = _dict.get('exempt_security_notice_count')
        else:
            raise ValueError('Required property \'exempt_security_notice_count\' not present in ScanresultCVE JSON')
        if 'exempt_status' in _dict:
            args['exempt_status'] = _dict.get('exempt_status')
        else:
            raise ValueError('Required property \'exempt_status\' not present in ScanresultCVE JSON')
        if 'security_notice_count' in _dict:
            args['security_notice_count'] = _dict.get('security_notice_count')
        else:
            raise ValueError('Required property \'security_notice_count\' not present in ScanresultCVE JSON')
        if 'security_notices' in _dict:
            args['security_notices'] = [ScanresultSecurityNotice.from_dict(x) for x in _dict.get('security_notices')]
        else:
            raise ValueError('Required property \'security_notices\' not present in ScanresultCVE JSON')
        if 'summary' in _dict:
            args['summary'] = _dict.get('summary')
        else:
            raise ValueError('Required property \'summary\' not present in ScanresultCVE JSON')
        if 'total_security_notice_count' in _dict:
            args['total_security_notice_count'] = _dict.get('total_security_notice_count')
        else:
            raise ValueError('Required property \'total_security_notice_count\' not present in ScanresultCVE JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ScanresultCVE object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'cve_exempt') and self.cve_exempt is not None:
            _dict['cve_exempt'] = self.cve_exempt
        if hasattr(self, 'cve_id') and self.cve_id is not None:
            _dict['cve_id'] = self.cve_id
        if hasattr(self, 'exempt_security_notice_count') and self.exempt_security_notice_count is not None:
            _dict['exempt_security_notice_count'] = self.exempt_security_notice_count
        if hasattr(self, 'exempt_status') and self.exempt_status is not None:
            _dict['exempt_status'] = self.exempt_status
        if hasattr(self, 'security_notice_count') and self.security_notice_count is not None:
            _dict['security_notice_count'] = self.security_notice_count
        if hasattr(self, 'security_notices') and self.security_notices is not None:
            _dict['security_notices'] = [x.to_dict() for x in self.security_notices]
        if hasattr(self, 'summary') and self.summary is not None:
            _dict['summary'] = self.summary
        if hasattr(self, 'total_security_notice_count') and self.total_security_notice_count is not None:
            _dict['total_security_notice_count'] = self.total_security_notice_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ScanresultCVE object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ScanresultCVE') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ScanresultCVE') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ScanresultConfigurationIssue():
    """
    ScanresultConfigurationIssue.

    :attr str corrective_action: Advice on how to solve this ConfigurationIssue.
    :attr str description: Description of this ConfigurationIssue.
    :attr bool exempt: True if this configuration issue is exempted by user policy,
          and false otherwise.
    :attr dict meta: Additional keys and string values about this
          ConfigurationIssue.
    :attr str type: The ID of the check which found this ConfigurationIssue.
    """

    def __init__(self,
                 corrective_action: str,
                 description: str,
                 exempt: bool,
                 meta: dict,
                 type: str) -> None:
        """
        Initialize a ScanresultConfigurationIssue object.

        :param str corrective_action: Advice on how to solve this
               ConfigurationIssue.
        :param str description: Description of this ConfigurationIssue.
        :param bool exempt: True if this configuration issue is exempted by user
               policy, and false otherwise.
        :param dict meta: Additional keys and string values about this
               ConfigurationIssue.
        :param str type: The ID of the check which found this ConfigurationIssue.
        """
        self.corrective_action = corrective_action
        self.description = description
        self.exempt = exempt
        self.meta = meta
        self.type = type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ScanresultConfigurationIssue':
        """Initialize a ScanresultConfigurationIssue object from a json dictionary."""
        args = {}
        if 'corrective_action' in _dict:
            args['corrective_action'] = _dict.get('corrective_action')
        else:
            raise ValueError('Required property \'corrective_action\' not present in ScanresultConfigurationIssue JSON')
        if 'description' in _dict:
            args['description'] = _dict.get('description')
        else:
            raise ValueError('Required property \'description\' not present in ScanresultConfigurationIssue JSON')
        if 'exempt' in _dict:
            args['exempt'] = _dict.get('exempt')
        else:
            raise ValueError('Required property \'exempt\' not present in ScanresultConfigurationIssue JSON')
        if 'meta' in _dict:
            args['meta'] = _dict.get('meta')
        else:
            raise ValueError('Required property \'meta\' not present in ScanresultConfigurationIssue JSON')
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError('Required property \'type\' not present in ScanresultConfigurationIssue JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ScanresultConfigurationIssue object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'corrective_action') and self.corrective_action is not None:
            _dict['corrective_action'] = self.corrective_action
        if hasattr(self, 'description') and self.description is not None:
            _dict['description'] = self.description
        if hasattr(self, 'exempt') and self.exempt is not None:
            _dict['exempt'] = self.exempt
        if hasattr(self, 'meta') and self.meta is not None:
            _dict['meta'] = self.meta
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ScanresultConfigurationIssue object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ScanresultConfigurationIssue') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ScanresultConfigurationIssue') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ScanresultPackageFixes():
    """
    ScanresultPackageFixes.

    :attr str corrective_action: Advice on how to solve this vulnerability.
    :attr str description: Description of the vulnerability.
    :attr str fix_version: The version of this package that contains the fix for
          this vulnerability.
    :attr str installed_version: The version of this package that was found
          installed at scan time.
    :attr str package_name: The name of the package.
    """

    def __init__(self,
                 corrective_action: str,
                 description: str,
                 fix_version: str,
                 installed_version: str,
                 package_name: str) -> None:
        """
        Initialize a ScanresultPackageFixes object.

        :param str corrective_action: Advice on how to solve this vulnerability.
        :param str description: Description of the vulnerability.
        :param str fix_version: The version of this package that contains the fix
               for this vulnerability.
        :param str installed_version: The version of this package that was found
               installed at scan time.
        :param str package_name: The name of the package.
        """
        self.corrective_action = corrective_action
        self.description = description
        self.fix_version = fix_version
        self.installed_version = installed_version
        self.package_name = package_name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ScanresultPackageFixes':
        """Initialize a ScanresultPackageFixes object from a json dictionary."""
        args = {}
        if 'corrective_action' in _dict:
            args['corrective_action'] = _dict.get('corrective_action')
        else:
            raise ValueError('Required property \'corrective_action\' not present in ScanresultPackageFixes JSON')
        if 'description' in _dict:
            args['description'] = _dict.get('description')
        else:
            raise ValueError('Required property \'description\' not present in ScanresultPackageFixes JSON')
        if 'fix_version' in _dict:
            args['fix_version'] = _dict.get('fix_version')
        else:
            raise ValueError('Required property \'fix_version\' not present in ScanresultPackageFixes JSON')
        if 'installed_version' in _dict:
            args['installed_version'] = _dict.get('installed_version')
        else:
            raise ValueError('Required property \'installed_version\' not present in ScanresultPackageFixes JSON')
        if 'package_name' in _dict:
            args['package_name'] = _dict.get('package_name')
        else:
            raise ValueError('Required property \'package_name\' not present in ScanresultPackageFixes JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ScanresultPackageFixes object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'corrective_action') and self.corrective_action is not None:
            _dict['corrective_action'] = self.corrective_action
        if hasattr(self, 'description') and self.description is not None:
            _dict['description'] = self.description
        if hasattr(self, 'fix_version') and self.fix_version is not None:
            _dict['fix_version'] = self.fix_version
        if hasattr(self, 'installed_version') and self.installed_version is not None:
            _dict['installed_version'] = self.installed_version
        if hasattr(self, 'package_name') and self.package_name is not None:
            _dict['package_name'] = self.package_name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ScanresultPackageFixes object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ScanresultPackageFixes') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ScanresultPackageFixes') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class ScanresultSecurityNotice():
    """
    ScanresultSecurityNotice.

    :attr str notice: Further information about this security notice.
    :attr bool notice_exempt: True if this security notice has been exempted by user
          policy, and false otherwise.
    :attr str notice_id: The ID for this security notice.
    :attr str summary: Summary of information about the security vulnerabilities
          fixed by this security notice.
    :attr List[ScanresultPackageFixes] vulnerable_packages: Package updates that
          contain fixes for this vulnerability.
    """

    def __init__(self,
                 notice: str,
                 notice_exempt: bool,
                 notice_id: str,
                 summary: str,
                 vulnerable_packages: List['ScanresultPackageFixes']) -> None:
        """
        Initialize a ScanresultSecurityNotice object.

        :param str notice: Further information about this security notice.
        :param bool notice_exempt: True if this security notice has been exempted
               by user policy, and false otherwise.
        :param str notice_id: The ID for this security notice.
        :param str summary: Summary of information about the security
               vulnerabilities fixed by this security notice.
        :param List[ScanresultPackageFixes] vulnerable_packages: Package updates
               that contain fixes for this vulnerability.
        """
        self.notice = notice
        self.notice_exempt = notice_exempt
        self.notice_id = notice_id
        self.summary = summary
        self.vulnerable_packages = vulnerable_packages

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ScanresultSecurityNotice':
        """Initialize a ScanresultSecurityNotice object from a json dictionary."""
        args = {}
        if 'notice' in _dict:
            args['notice'] = _dict.get('notice')
        else:
            raise ValueError('Required property \'notice\' not present in ScanresultSecurityNotice JSON')
        if 'notice_exempt' in _dict:
            args['notice_exempt'] = _dict.get('notice_exempt')
        else:
            raise ValueError('Required property \'notice_exempt\' not present in ScanresultSecurityNotice JSON')
        if 'notice_id' in _dict:
            args['notice_id'] = _dict.get('notice_id')
        else:
            raise ValueError('Required property \'notice_id\' not present in ScanresultSecurityNotice JSON')
        if 'summary' in _dict:
            args['summary'] = _dict.get('summary')
        else:
            raise ValueError('Required property \'summary\' not present in ScanresultSecurityNotice JSON')
        if 'vulnerable_packages' in _dict:
            args['vulnerable_packages'] = [ScanresultPackageFixes.from_dict(x) for x in _dict.get('vulnerable_packages')]
        else:
            raise ValueError('Required property \'vulnerable_packages\' not present in ScanresultSecurityNotice JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ScanresultSecurityNotice object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'notice') and self.notice is not None:
            _dict['notice'] = self.notice
        if hasattr(self, 'notice_exempt') and self.notice_exempt is not None:
            _dict['notice_exempt'] = self.notice_exempt
        if hasattr(self, 'notice_id') and self.notice_id is not None:
            _dict['notice_id'] = self.notice_id
        if hasattr(self, 'summary') and self.summary is not None:
            _dict['summary'] = self.summary
        if hasattr(self, 'vulnerable_packages') and self.vulnerable_packages is not None:
            _dict['vulnerable_packages'] = [x.to_dict() for x in self.vulnerable_packages]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ScanresultSecurityNotice object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ScanresultSecurityNotice') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ScanresultSecurityNotice') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other
