"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventBus = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const events_generated_1 = require("./events.generated");
/**
 * Define an EventBridge EventBus.
 *
 * @resource AWS::Events::EventBus
 */
class EventBus extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        const { eventBusName, eventSourceName } = EventBus.eventBusProps(core_1.Lazy.stringValue({ produce: () => this.node.uniqueId }), props);
        super(scope, id, { physicalName: eventBusName });
        const eventBus = new events_generated_1.CfnEventBus(this, 'Resource', {
            name: eventBusName,
            eventSourceName,
        });
        this.eventBusArn = this.getResourceArnAttribute(eventBus.attrArn, {
            service: 'events',
            resource: 'event-bus',
            resourceName: eventBus.name,
        });
        this.eventBusName = this.getResourceNameAttribute(eventBus.ref);
        this.eventBusPolicy = eventBus.attrPolicy;
        this.eventSourceName = eventBus.eventSourceName;
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param eventBusArn ARN of imported event bus.
     */
    static fromEventBusArn(scope, id, eventBusArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventBusArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.eventBusArn = eventBusArn;
                this.eventBusName = parts.resourceName || '';
                this.eventBusPolicy = '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param attrs Imported event bus properties.
     */
    static fromEventBusAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.eventBusArn = attrs.eventBusArn;
                this.eventBusName = attrs.eventBusName;
                this.eventBusPolicy = attrs.eventBusPolicy;
                this.eventSourceName = attrs.eventSourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Permits an IAM Principal to send custom events to EventBridge so that they can be matched to rules.
     *
     * @param grantee The principal (no-op if undefined).
     */
    static grantPutEvents(grantee) {
        // It's currently not possible to restrict PutEvents to specific resources.
        // See https://docs.aws.amazon.com/eventbridge/latest/userguide/permissions-reference-eventbridge.html
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: ['*'],
        });
    }
    static eventBusProps(defaultEventBusName, props) {
        if (props) {
            const { eventBusName, eventSourceName } = props;
            const eventBusNameRegex = /^[\/\.\-_A-Za-z0-9]{1,256}$/;
            if (eventBusName !== undefined && eventSourceName !== undefined) {
                throw new Error('\'eventBusName\' and \'eventSourceName\' cannot both be provided');
            }
            else if (eventBusName !== undefined) {
                if (eventBusName === 'default') {
                    throw new Error('\'eventBusName\' must not be \'default\'');
                }
                else if (eventBusName.indexOf('/') > -1) {
                    throw new Error('\'eventBusName\' must not contain \'/\'');
                }
                else if (!eventBusNameRegex.test(eventBusName)) {
                    throw new Error(`'eventBusName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName };
            }
            else if (eventSourceName !== undefined) {
                // Ex: aws.partner/PartnerName/acct1/repo1
                const eventSourceNameRegex = /^aws\.partner(\/[\.\-_A-Za-z0-9]+){2,}$/;
                if (!eventSourceNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventSourceNameRegex}`);
                }
                else if (!eventBusNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName: eventSourceName, eventSourceName };
            }
        }
        return { eventBusName: defaultEventBusName };
    }
}
exports.EventBus = EventBus;
//# sourceMappingURL=data:application/json;base64,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