# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['craterpy', 'craterpy.tests']

package_data = \
{'': ['*'], 'craterpy': ['data/*', 'data/_images/*']}

install_requires = \
['matplotlib>=3.4.2,<4.0.0',
 'numpy>=1.20.2,<2.0.0',
 'pandas>=1.2.4,<2.0.0',
 'rasterio>=1.2.3,<2.0.0']

extras_require = \
{':python_version < "3.8"': ['importlib-metadata>=1.0,<2.0']}

setup_kwargs = {
    'name': 'craterpy',
    'version': '0.5.0',
    'description': 'Impact crater data science in Python.',
    'long_description': '<div align="center">\n  <strong>Craterpy:</strong><em> Impact crater data science in Python.</em>\n</div>\n\n<div align="center">\n  <!-- PYPI version -->\n  <a href="https://badge.fury.io/py/craterpy">\n    <img src="https://badge.fury.io/py/craterpy.svg"\n      alt="PYPI version" />\n  </a>\n  <!-- Code quality and testing (CI) -->\n  <a href="https://github.com/cjtu/craterpy/actions">\n    <img src="https://github.com/cjtu/craterpy/workflows/Code%20Quality%20Checks/badge.svg"\n      alt="Code Quality Checks" />\n  </a>\n  <!-- Test Coverage (codecov) -->\n  <a href="https://codecov.io/gh/cjtu/craterpy">\n    <img src="https://codecov.io/gh/cjtu/craterpy/branch/trunk/graph/badge.svg?token=9K567x0YUJ"\n      alt="Code Coverage" />\n  </a>\n</div>\n<div align="center">\n  <!-- Zenodo citation -->\n  <a href="https://zenodo.org/badge/latestdoi/88457986">\n    <img src="https://zenodo.org/badge/88457986.svg"\n      alt="Cite on Zenodo" />\n  </a>\n  <!-- ReadTheDocs -->\n  <a href="http://craterpy.readthedocs.io/en/latest/?badge=latest">\n    <img src="http://readthedocs.org/projects/craterpy/badge/?version=latest"\n      alt="Cite on Zenodo" />\n  </a>\n  <!-- Code Style -->\n  <a href="https://github.com/psf/black">\n    <img src="https://img.shields.io/badge/code%20style-black-000000.svg"\n      alt="Code Style: Black" />\n      </a>\n</div>\n\n# Overview\n\nCraterpy simplifies the extraction and statistical analysis of impact craters in planetary datasets. It can:\n\n- work with tables of crater data in Python (using pandas)\n- load and manipulate planetary image data in Python (using rasterio)\n- extract, mask, filter, and compute stats on craters located in planetary imagery\n- plot crater regions of interest\n\nCraterpy currently only supports simple cylindrical images and requires you to provide a table of crater locations and sizes (e.g. it isn\'t a crater detection program). See the example below!\n\n**Note:** *Craterpy is in alpha. We appreciate bug reports and feature requests on the [issues board](https://github.com/cjtu/craterpy/issues).*\n\n## Example\n\nCraterpy in action:\n\n```python\nimport pandas as pd\nfrom craterpy import dataset, stats\ndf = pd.DataFrame({\'Name\': ["Orientale", "Langrenus", "Compton"],\n                    \'Lat\': [-19.9, -8.86, 55.9],\n                    \'Lon\': [-94.7, 61.0, 104.0],\n                    \'Rad\': [147.0, 66.0, 82.3]})\nmoon = dataset.CraterpyDataset("moon.tif")\nstat_df = cs.ejecta_stats(df, moon, 4, [\'mean\', \'median\', \'std\'], plot=True)\n```\n\n![ejecta image](https://raw.githubusercontent.com/cjtu/craterpy/trunk/craterpy/data/_images/readme_crater_ejecta.png)\n\n```python\nstats_df.head()\n```\n\n![crater stats](https://raw.githubusercontent.com/cjtu/craterpy/trunk/craterpy/data/_images/readme_stat_df.png)\n\nNew users should start with the Jupyter notebook [tutorial](https://gist.github.com/cjtu/560f121049b342aa0b2bf70e038358b7) for typical usage with examples. See also [craterpy documentation](https://readthedocs.org/projects/craterpy/) on Read the Docs.\n\n**Note**: This package currently **only accepts image data in simple-cylindrical (Plate Caree) projection**. If your data is in another projection, please reproject it to simple-cylindrical before importing it with craterpy. If you would like add reprojection functionality to craterpy, consider [Contributing](https://github.com/cjtu/craterpy/blob/trunk/CONTRIBUTING.rst).\n\n## Installation\n\nWith pip:\n\n```bash\npip install craterpy\npython -c "import craterpy; print(craterpy.__version__)"\n```\n\nIn a new [conda environment](https://conda.io/docs/using/envs):\n\n```bash\n# Create and activate a new conda environment called "craterpy"\nconda create -n craterpy python=3.9\nconda activate craterpy\n\n# Install craterpy with pip\npip install craterpy\npython -c "import craterpy; print(craterpy.__version__)"\n```\n\nWith [git](https://git-scm.com) and [poetry](https://python-poetry.org/docs/) (for latest version & development):\n\n```bash\n# Clone this repository\n$ cd ~\n$ git clone https://github.com/cjtu/craterpy.git\n\n# Enter the repository\n$ cd craterpy\n\n# Configure poetry\npoetry config virtualenvs.create true --local\npoetry config virtualenvs.in-project true --local\n\n# Install craterpy with poetry\n$ poetry install\n\n# Check installation\npoetry version\n\n# Either open a Jupyter server\n$ poetry run jupyter notebook\n\n# Or activate the venv from your Python editor of choice\n# The venv is path is ~/craterpy/.venv/bin/python\n```\n\nTrouble installing craterpy? Let us know on the [issues](https://github.com/cjtu/craterpy/issues) board.\n\n## Dependencies\n\nCraterpy requires python >3.7 and is tested on Ubuntu and OS X. If you would like to use craterpy on Windows, we recommend getting the Windows Subsystem for Linux ([WSL](https://docs.microsoft.com/en-us/windows/wsl/install-win10)) and installing it from the bash shell.\n\nIt\'s core dependencies are:\n\n- rasterio\n- pandas\n- numpy\n- matplotlib\n\n## Documentation\n\nFull API documentation is available at [readthedocs](https://readthedocs.org/projects/craterpy/).\n\n## Contributing\n\nThere are two major ways you can help improve craterpy:\n\n### Bug Reporting and Feature Requests\n\nYou can report bugs or request new features on the [issues](https://github.com/cjtu/craterpy/issues) board.\n\n### Contributing Directly\n\nWant to fix a bug / implement a feature / fix some documentation? We welcome pull requests from all new contributors! You (yes you!) can help us make craterpy as good as it can be! See [CONTRIBUTING.rst](https://github.com/cjtu/craterpy/blob/trunk/CONTRIBUTING.rst) for details on how to get started - first time GitHub contributors welcome - and encouraged!\n\n## Citing craterpy\n\nCraterpy is [MIT Licenced](https://github.com/cjtu/craterpy/blob/master/LICENSE.txt) and is free to use with attribution. Citation information can be found [here](https://zenodo.org/badge/latestdoi/88457986).\n\n## Contact\n\nIf you have comments/question/concerns or just want to get in touch, you can email Christian at cj.taiudovicic@gmail.com or follow [@TaiUdovicic](https://twitter.com/TaiUdovicic) on Twitter.\n',
    'author': 'Christian J. Tai Udovicic',
    'author_email': 'cj.taiudovicic@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/cjtu/craterpy',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7.7,<4.0.0',
}


setup(**setup_kwargs)
