/// @file
/// @author David Pilger <dpilger26@gmail.com>
/// [GitHub Repository](https://github.com/dpilger26/NumCpp)
///
/// License
/// Copyright 2018-2022 David Pilger
///
/// Permission is hereby granted, free of charge, to any person obtaining a copy of this
/// software and associated documentation files(the "Software"), to deal in the Software
/// without restriction, including without limitation the rights to use, copy, modify,
/// merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
/// permit persons to whom the Software is furnished to do so, subject to the following
/// conditions :
///
/// The above copyright notice and this permission notice shall be included in all copies
/// or substantial portions of the Software.
///
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
/// INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
/// PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
/// FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
/// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
/// DEALINGS IN THE SOFTWARE.
///
/// Description
/// Special Functions
///
#pragma once

#include <cmath>

#if defined(__cpp_lib_math_special_functions) || !defined(NUMCPP_NO_USE_BOOST)

#include "NumCpp/Core/Internal/StaticAsserts.hpp"
#include "NumCpp/Core/Internal/StlAlgorithms.hpp"
#include "NumCpp/NdArray.hpp"

#ifndef __cpp_lib_math_special_functions
#include "boost/math/special_functions/zeta.hpp"
#endif

namespace nc
{
    namespace special
    {
        //============================================================================
        // Method Description:
        /// The Riemann Zeta function
        /// https://en.wikipedia.org/wiki/Riemann_zeta_function
        /// NOTE: Use of this function requires either using the Boost
        /// includes or a C++17 compliant compiler.
        ///
        /// @param inValue
        /// @return calculated-result-type
        ///
        template<typename dtype>
        auto riemann_zeta(dtype inValue)
        {
            STATIC_ASSERT_ARITHMETIC(dtype);

#ifdef __cpp_lib_math_special_functions
            return std::riemann_zeta(inValue);
#else
            return boost::math::zeta(inValue);
#endif
        }

        //============================================================================
        // Method Description:
        /// The Riemann Zeta function
        /// https://en.wikipedia.org/wiki/Riemann_zeta_function
        /// NOTE: Use of this function requires either using the Boost
        /// includes or a C++17 compliant compiler.
        ///
        /// @param inArray
        /// @return NdArray
        ///
        template<typename dtype>
        auto riemann_zeta(const NdArray<dtype>& inArray)
        {
            NdArray<decltype(riemann_zeta(dtype{ 0 }))> returnArray(inArray.shape());

            stl_algorithms::transform(
                inArray.cbegin(),
                inArray.cend(),
                returnArray.begin(),
                [](dtype inValue) -> auto{ return riemann_zeta(inValue); });

            return returnArray;
        }
    } // namespace special
} // namespace nc

#endif // #if defined(__cpp_lib_math_special_functions) || !defined(NUMCPP_NO_USE_BOOST)
