# Generated by Django 3.0.6 on 2020-05-27 11:25

import django.db.models.deletion
import django.utils.timezone
from django.conf import settings
from django.db import migrations, models

import weblate.lang.models
import weblate.trans.fields
import weblate.trans.mixins
import weblate.trans.validators
import weblate.utils.fields
import weblate.utils.render
import weblate.utils.validators
from weblate.formats.models import FILE_FORMATS
from weblate.utils.licenses import get_license_choices
from weblate.vcs.models import VCS_REGISTRY

FIELDS = (
    ("unit", "source"),
    ("unit", "target"),
    ("unit", "context"),
    ("unit", "note"),
    ("unit", "location"),
    ("suggestion", "target"),
    ("comment", "comment"),
    ("dictionary", "source"),
)

PG_TRGM = "CREATE INDEX {0}_{1}_fulltext ON trans_{0} USING GIN ({1} gin_trgm_ops)"
PG_DROP = "DROP INDEX {0}_{1}_fulltext"

MY_FTX = "CREATE FULLTEXT INDEX {0}_{1}_fulltext ON trans_{0}({1})"
MY_DROP = "ALTER TABLE trans_{0} DROP INDEX {0}_{1}_fulltext"


def create_index(apps, schema_editor):
    vendor = schema_editor.connection.vendor
    if vendor == "postgresql":
        # Install pg_trgm for trigram search and index
        cur = schema_editor.connection.cursor()
        cur.execute("SELECT * FROM pg_extension WHERE extname = 'pg_trgm'")
        if not cur.fetchone():
            schema_editor.execute("CREATE EXTENSION IF NOT EXISTS pg_trgm")

        # Create GIN trigram index on searched fields
        for table, field in FIELDS:
            schema_editor.execute(PG_TRGM.format(table, field))
    elif vendor == "mysql":
        for table, field in FIELDS:
            schema_editor.execute(MY_FTX.format(table, field))
        schema_editor.execute(
            "CREATE INDEX unit_source_index ON trans_unit(source(255))"
        )
        schema_editor.execute(
            "CREATE INDEX unit_context_index ON trans_unit(context(255))"
        )
    else:
        raise Exception("Unsupported database: {}".format(vendor))


def drop_index(apps, schema_editor):
    vendor = schema_editor.connection.vendor
    if vendor == "postgresql":
        for table, field in FIELDS:
            schema_editor.execute(PG_DROP.format(table, field))
    elif vendor == "mysql":
        for table, field in FIELDS:
            schema_editor.execute(MY_DROP.format(table, field))
        schema_editor.execute("ALTER TABLE trans_unit DROP INDEX unit_source_index")
        schema_editor.execute("ALTER TABLE trans_unit DROP INDEX unit_context_index")
    else:
        raise Exception("Unsupported database: {}".format(vendor))


class Migration(migrations.Migration):

    replaces = [
        ("trans", "0001_squashed_0143_auto_20180609_1655"),
        ("trans", "0002_delete_indexupdate"),
        ("trans", "0003_remove_translation_commit_message"),
        ("trans", "0004_project_use_shared_tm"),
        ("trans", "0005_auto_20180914_1135"),
        ("trans", "0006_suggestion_userdetails"),
        ("trans", "0007_auto_20181011_1634"),
        ("trans", "0008_auto_20181015_2104"),
        ("trans", "0009_auto_20181113_1219"),
        ("trans", "0010_auto_20181205_1505"),
        ("trans", "0011_component_merge_message"),
        ("trans", "0012_auto_20190129_1325"),
        ("trans", "0013_component_addon_message"),
        ("trans", "0014_auto_20190203_1923"),
        ("trans", "0015_linked_component_branch"),
        ("trans", "0016_fix_alert_occurrence"),
        ("trans", "0017_component_language_code_style"),
        ("trans", "0018_remove_unusednewbase_alert"),
        ("trans", "0019_auto_20190321_1003"),
        ("trans", "0020_auto_20190321_0921"),
        ("trans", "0021_auto_20190321_1004"),
        ("trans", "0022_auto_20190321_1443"),
        ("trans", "0023_auto_20190325_1037"),
        ("trans", "0024_resolve_auto_format"),
        ("trans", "0025_auto_20190426_0941"),
        ("trans", "0026_alert_change"),
        ("trans", "0027_auto_20190517_1125"),
        ("trans", "0028_vote_value"),
        ("trans", "0029_vote_value"),
        ("trans", "0030_vote_value"),
        ("trans", "0031_auto_20190721_1810"),
        ("trans", "0032_auto_20190724_1020"),
        ("trans", "0033_auto_20190802_1427"),
        ("trans", "0034_priority"),
        ("trans", "0035_remove_source_priority"),
        ("trans", "0036_auto_20190829_1309"),
        ("trans", "0037_auto_20190906_1526"),
        ("trans", "0038_contribute_tm"),
        ("trans", "0039_remove_component_save_history"),
        ("trans", "0040_auto_20190930_1528"),
        ("trans", "0041_auto_20191029_1144"),
        ("trans", "0042_original_state"),
        ("trans", "0043_auto_20191030_1244"),
        ("trans", "0044_translation_check_flags"),
        ("trans", "0045_auto_20191103_1029"),
        ("trans", "0046_source_strings"),
        ("trans", "0047_delete_source"),
        ("trans", "0048_auto_20191108_1349"),
        ("trans", "0049_change_comment"),
        ("trans", "0050_licenses"),
        ("trans", "0051_auto_20191210_1356"),
        ("trans", "0052_auto_20191212_1403"),
        ("trans", "0053_unitdata"),
        ("trans", "0054_auto_20191212_1441"),
        ("trans", "0055_auto_20191212_1508"),
        ("trans", "0056_fixup_source_translations"),
        ("trans", "0057_shapings"),
        ("trans", "0058_comment_resolved"),
        ("trans", "0059_auto_20200206_0937"),
        ("trans", "0060_component_enforced_checks"),
        ("trans", "0061_auto_20200218_1108"),
        ("trans", "0062_fix_num_words"),
        ("trans", "0063_auto_20200305_2202"),
        ("trans", "0064_fulltext_index"),
        ("trans", "0065_string_index"),
        ("trans", "0066_change_language"),
        ("trans", "0067_fill_change_language"),
        ("trans", "0068_auto_20200318_1014"),
        ("trans", "0069_auto_20200324_1121"),
        ("trans", "0070_auto_20200327_1225"),
        ("trans", "0071_component_intermediate"),
        ("trans", "0072_auto_20200403_1311"),
        ("trans", "0073_auto_20200403_1329"),
        ("trans", "0074_fix_broken_browser_alert"),
    ]

    initial = True

    dependencies = [
        migrations.swappable_dependency(settings.AUTH_USER_MODEL),
        ("lang", "0006_auto_20200309_1436"),
        ("lang", "0001_squashed_0011_auto_20180215_1158"),
    ]

    operations = [
        migrations.CreateModel(
            name="Project",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "name",
                    models.CharField(
                        help_text="Display name",
                        max_length=60,
                        unique=True,
                        verbose_name="Project name",
                    ),
                ),
                (
                    "slug",
                    models.SlugField(
                        help_text="Name used in URLs and filenames.",
                        max_length=60,
                        unique=True,
                        verbose_name="URL slug",
                    ),
                ),
                (
                    "web",
                    models.URLField(
                        help_text="Main website of translated project.",
                        verbose_name="Project website",
                    ),
                ),
                (
                    "mail",
                    models.EmailField(
                        blank=True,
                        help_text="Mailing list for translators.",
                        max_length=254,
                        verbose_name="Mailing list",
                    ),
                ),
                (
                    "instructions",
                    models.TextField(
                        blank=True,
                        help_text="You can use Markdown and mention users by @username.",
                        verbose_name="Translation instructions",
                    ),
                ),
                (
                    "set_language_team",
                    models.BooleanField(
                        default=True,
                        help_text='Lets Weblate update the "Language-Team" file header of your project.',
                        verbose_name='Set "Language-Team" header',
                    ),
                ),
                (
                    "enable_hooks",
                    models.BooleanField(
                        default=True,
                        help_text="Whether to allow updating this repository by remote hooks.",
                        verbose_name="Enable hooks",
                    ),
                ),
                (
                    "source_language",
                    models.ForeignKey(
                        default=weblate.lang.models.get_default_lang,
                        help_text="Language used for source strings in all components",
                        on_delete=django.db.models.deletion.CASCADE,
                        to="lang.Language",
                        verbose_name="Source language",
                    ),
                ),
                (
                    "access_control",
                    models.IntegerField(
                        choices=[
                            (0, "Public"),
                            (1, "Protected"),
                            (100, "Private"),
                            (200, "Custom"),
                        ],
                        default=settings.DEFAULT_ACCESS_CONTROL,
                        help_text="How to restrict access to this project is detailed in the documentation.",
                        verbose_name="Access control",
                    ),
                ),
                (
                    "translation_review",
                    models.BooleanField(
                        default=False,
                        help_text="Requires dedicated reviewers to approve translations.",
                        verbose_name="Enable reviews",
                    ),
                ),
                (
                    "source_review",
                    models.BooleanField(
                        default=False,
                        help_text="Requires dedicated reviewers to approve source strings.",
                        verbose_name="Enable source reviews",
                    ),
                ),
                (
                    "use_shared_tm",
                    models.BooleanField(
                        default=settings.DEFAULT_SHARED_TM,
                        help_text="Uses the pool of shared translations between projects.",
                        verbose_name="Use shared translation memory",
                    ),
                ),
                (
                    "contribute_shared_tm",
                    models.BooleanField(
                        default=settings.DEFAULT_SHARED_TM,
                        help_text="Contributes to the pool of shared translations between projects.",
                        verbose_name="Contribute to shared translation memory",
                    ),
                ),
            ],
            options={"verbose_name": "Project", "verbose_name_plural": "Projects"},
            bases=(
                models.Model,
                weblate.trans.mixins.URLMixin,
                weblate.trans.mixins.PathMixin,
            ),
        ),
        migrations.CreateModel(
            name="Component",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "name",
                    models.CharField(
                        help_text="Display name",
                        max_length=100,
                        verbose_name="Component name",
                    ),
                ),
                (
                    "slug",
                    models.SlugField(
                        help_text="Name used in URLs and filenames.",
                        max_length=100,
                        verbose_name="URL slug",
                    ),
                ),
                (
                    "repo",
                    models.CharField(
                        help_text="URL of a repository, use weblate://project/component to share it with other component.",
                        max_length=200,
                        verbose_name="Source code repository",
                    ),
                ),
                (
                    "push",
                    models.CharField(
                        blank=True,
                        help_text="URL of a push repository, pushing is turned off if empty.",
                        max_length=200,
                        verbose_name="Repository push URL",
                    ),
                ),
                (
                    "repoweb",
                    models.URLField(
                        blank=True,
                        help_text="Link to repository browser, use {{branch}} for branch, {{filename}} and {{line}} as filename and line placeholders.",
                        validators=[weblate.utils.render.validate_repoweb],
                        verbose_name="Repository browser",
                    ),
                ),
                (
                    "git_export",
                    models.CharField(
                        blank=True,
                        help_text="URL of repository where users can fetch changes from Weblate",
                        max_length=220,
                        verbose_name="Exported repository URL",
                    ),
                ),
                (
                    "report_source_bugs",
                    models.EmailField(
                        blank=True,
                        help_text="E-mail address for reports on errors in source strings. Leave empty for no e-mails.",
                        max_length=254,
                        verbose_name="Source string bug reporting address",
                    ),
                ),
                (
                    "branch",
                    models.CharField(
                        blank=True,
                        default="",
                        help_text="Repository branch to translate",
                        max_length=200,
                        verbose_name="Repository branch",
                    ),
                ),
                (
                    "filemask",
                    models.CharField(
                        help_text="Path of files to translate relative to repository root, use * instead of language code, for example: po/*.po or locale/*/LC_MESSAGES/django.po.",
                        max_length=400,
                        validators=[
                            weblate.trans.validators.validate_filemask,
                            weblate.utils.validators.validate_filename,
                        ],
                        verbose_name="Filemask",
                    ),
                ),
                (
                    "template",
                    models.CharField(
                        blank=True,
                        help_text="Filename of translation base file, containing all strings and their source; it is recommended for monolingual translation formats.",
                        max_length=400,
                        validators=[weblate.utils.validators.validate_filename],
                        verbose_name="Monolingual base language file",
                    ),
                ),
                (
                    "new_base",
                    models.CharField(
                        blank=True,
                        help_text="Filename of file used for creating new translations. For gettext choose .pot file.",
                        max_length=400,
                        validators=[weblate.utils.validators.validate_filename],
                        verbose_name="Template for new translations",
                    ),
                ),
                (
                    "intermediate",
                    models.CharField(
                        blank=True,
                        help_text="Filename of intermediate translation file. In most cases this is a translation file provided by developers and is used when creating actual source strings.",
                        max_length=400,
                        validators=[weblate.utils.validators.validate_filename],
                        verbose_name="Intermediate language file",
                    ),
                ),
                (
                    "file_format",
                    models.CharField(
                        choices=FILE_FORMATS.get_choices(),
                        max_length=50,
                        verbose_name="File format",
                    ),
                ),
                (
                    "locked",
                    models.BooleanField(
                        default=False,
                        help_text="Locked component will not get any translation updates.",
                        verbose_name="Locked",
                    ),
                ),
                (
                    "allow_translation_propagation",
                    models.BooleanField(
                        db_index=True,
                        default=settings.DEFAULT_TRANSLATION_PROPAGATION,
                        help_text="Whether translation updates in other components will cause automatic translation in this one",
                        verbose_name="Allow translation propagation",
                    ),
                ),
                (
                    "enable_suggestions",
                    models.BooleanField(
                        default=True,
                        help_text="Whether to allow translation suggestions at all.",
                        verbose_name="Turn on suggestions",
                    ),
                ),
                (
                    "suggestion_voting",
                    models.BooleanField(
                        default=False,
                        help_text="Whether users can vote for suggestions.",
                        verbose_name="Suggestion voting",
                    ),
                ),
                (
                    "suggestion_autoaccept",
                    models.PositiveSmallIntegerField(
                        default=0,
                        help_text="Automatically accept suggestions with this number of votes, use 0 to turn it off.",
                        validators=[weblate.trans.validators.validate_autoaccept],
                        verbose_name="Autoaccept suggestions",
                    ),
                ),
                (
                    "check_flags",
                    models.TextField(
                        blank=True,
                        default="",
                        help_text="Additional comma-separated flags to influence quality checks. Possible values can be found in the documentation.",
                        validators=[weblate.trans.validators.validate_check_flags],
                        verbose_name="Translation flags",
                    ),
                ),
                (
                    "project",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Project",
                        verbose_name="Project",
                    ),
                ),
                (
                    "commit_message",
                    models.TextField(
                        default=settings.DEFAULT_COMMIT_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_commit],
                        verbose_name="Commit message when translating",
                    ),
                ),
                (
                    "committer_email",
                    models.EmailField(
                        default=settings.DEFAULT_COMMITER_EMAIL,
                        max_length=254,
                        verbose_name="Committer e-mail",
                    ),
                ),
                (
                    "committer_name",
                    models.CharField(
                        default=settings.DEFAULT_COMMITER_NAME,
                        max_length=200,
                        verbose_name="Committer name",
                    ),
                ),
                (
                    "license",
                    models.CharField(
                        blank=True,
                        choices=get_license_choices(),
                        default="",
                        max_length=150,
                        verbose_name="Translation license",
                    ),
                ),
                (
                    "merge_style",
                    models.CharField(
                        choices=[("merge", "Merge"), ("rebase", "Rebase")],
                        default=settings.DEFAULT_MERGE_STYLE,
                        help_text="Define whether Weblate should merge the upstream repository or rebase changes onto it.",
                        max_length=10,
                        verbose_name="Merge style",
                    ),
                ),
                (
                    "new_lang",
                    models.CharField(
                        choices=[
                            ("contact", "Contact maintainers"),
                            ("url", "Point to translation instructions URL"),
                            ("add", "Create new language file"),
                            ("none", "Disable adding new translations"),
                        ],
                        default="add",
                        help_text="How to handle requests for creating new translations.",
                        max_length=10,
                        verbose_name="Adding new translation",
                    ),
                ),
                (
                    "vcs",
                    models.CharField(
                        choices=VCS_REGISTRY.get_choices(),
                        default=settings.DEFAULT_VCS,
                        help_text="Version control system to use to access your repository containing translations. You can also choose additional integration with third party providers to submit merge requests.",
                        max_length=20,
                        verbose_name="Version control system",
                    ),
                ),
                (
                    "edit_template",
                    models.BooleanField(
                        default=True,
                        help_text="Whether users will be able to edit the base file for monolingual translations.",
                        verbose_name="Edit base file",
                    ),
                ),
                (
                    "agreement",
                    models.TextField(
                        blank=True,
                        default="",
                        help_text="User agreement which needs to be approved before a user can translate this component.",
                        verbose_name="Contributor agreement",
                    ),
                ),
                (
                    "language_regex",
                    weblate.trans.fields.RegexField(
                        default="^[^.]+$",
                        help_text="Regular expression used to filter translation when scanning for filemask.",
                        max_length=500,
                        verbose_name="Language filter",
                    ),
                ),
                (
                    "add_message",
                    models.TextField(
                        default=settings.DEFAULT_ADD_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_commit],
                        verbose_name="Commit message when adding translation",
                    ),
                ),
                (
                    "delete_message",
                    models.TextField(
                        default=settings.DEFAULT_DELETE_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_commit],
                        verbose_name="Commit message when removing translation",
                    ),
                ),
                (
                    "priority",
                    models.IntegerField(
                        choices=[
                            (60, "Very high"),
                            (80, "High"),
                            (100, "Medium"),
                            (120, "Low"),
                            (140, "Very low"),
                        ],
                        default=100,
                        help_text="Components with higher priority are offered first to translators.",
                        verbose_name="Priority",
                    ),
                ),
                (
                    "commit_pending_age",
                    models.IntegerField(
                        default=settings.COMMIT_PENDING_HOURS,
                        help_text="Time in hours after which any pending changes will be committed to the VCS.",
                        verbose_name="Age of changes to commit",
                    ),
                ),
                (
                    "push_on_commit",
                    models.BooleanField(
                        default=settings.DEFAULT_PUSH_ON_COMMIT,
                        help_text="Whether the repository should be pushed upstream on every commit.",
                        verbose_name="Push on commit",
                    ),
                ),
                (
                    "linked_component",
                    models.ForeignKey(
                        editable=False,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Component",
                        verbose_name="Project",
                    ),
                ),
                (
                    "merge_message",
                    models.TextField(
                        default=settings.DEFAULT_MERGE_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_component],
                        verbose_name="Commit message when merging translation",
                    ),
                ),
                (
                    "addon_message",
                    models.TextField(
                        default=settings.DEFAULT_ADDON_MESSAGE,
                        help_text="You can use template language for various info, please consult the documentation for more details.",
                        validators=[weblate.utils.render.validate_render_addon],
                        verbose_name="Commit message when addon makes a change",
                    ),
                ),
                (
                    "language_code_style",
                    models.CharField(
                        blank=True,
                        choices=[
                            ("", "Default based on the file format"),
                            ("posix", "POSIX style using underscore as a separator"),
                            ("bcp", "BCP style using hyphen as a separator"),
                            ("android", "Android style"),
                            ("java", "Java style"),
                        ],
                        default="",
                        help_text="Customize language code used to generate the filename for translations created by Weblate.",
                        max_length=10,
                        verbose_name="Language code style",
                    ),
                ),
                (
                    "shaping_regex",
                    weblate.trans.fields.RegexField(
                        blank=True,
                        default="",
                        help_text="Regular expression used to determine shapings of a string.",
                        max_length=190,
                        validators=[weblate.utils.validators.validate_re_nonempty],
                        verbose_name="Shapings regular expression",
                    ),
                ),
                (
                    "enforced_checks",
                    weblate.utils.fields.JSONField(
                        blank=True,
                        default=[],
                        help_text="List of checks which can not be ignored.",
                        verbose_name="Enforced checks",
                    ),
                ),
            ],
            options={
                "verbose_name": "Component",
                "verbose_name_plural": "Components",
                "unique_together": {("project", "name"), ("project", "slug")},
            },
            bases=(
                models.Model,
                weblate.trans.mixins.URLMixin,
                weblate.trans.mixins.PathMixin,
            ),
        ),
        migrations.CreateModel(
            name="Translation",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("revision", models.CharField(blank=True, default="", max_length=200)),
                ("filename", models.CharField(max_length=400)),
                (
                    "language_code",
                    models.CharField(blank=True, default="", max_length=20),
                ),
                (
                    "language",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="lang.Language"
                    ),
                ),
                (
                    "component",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Component",
                    ),
                ),
                (
                    "plural",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="lang.Plural"
                    ),
                ),
                (
                    "check_flags",
                    models.TextField(
                        blank=True,
                        default="",
                        validators=[weblate.trans.validators.validate_check_flags],
                        verbose_name="Translation flags",
                    ),
                ),
            ],
            options={"unique_together": {("component", "language")}},
            bases=(models.Model, weblate.trans.mixins.URLMixin),
        ),
        migrations.CreateModel(
            name="Label",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("name", models.CharField(max_length=190, verbose_name="Label name")),
                (
                    "color",
                    models.CharField(
                        choices=[
                            ("navy", "Navy"),
                            ("blue", "Blue"),
                            ("aqua", "Aqua"),
                            ("teal", "Teal"),
                            ("olive", "Olive"),
                            ("green", "Green"),
                            ("lime", "Lime"),
                            ("yellow", "Yellow"),
                            ("orange", "Orange"),
                            ("red", "Red"),
                            ("maroon", "Maroon"),
                            ("fuchsia", "Fuchsia"),
                            ("purple", "Purple"),
                            ("black", "Black"),
                            ("gray", "Gray"),
                            ("silver", "Silver"),
                        ],
                        default=None,
                        max_length=30,
                        verbose_name="Color",
                    ),
                ),
                (
                    "project",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="trans.Project"
                    ),
                ),
            ],
            options={"unique_together": {("project", "name")}},
        ),
        migrations.CreateModel(
            name="Shaping",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("shaping_regex", weblate.trans.fields.RegexField(max_length=190)),
                ("key", models.CharField(db_index=True, max_length=190)),
                (
                    "component",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Component",
                    ),
                ),
            ],
            options={"unique_together": {("key", "component", "shaping_regex")}},
        ),
        migrations.CreateModel(
            name="Unit",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("location", models.TextField(blank=True, default="")),
                ("context", models.TextField(blank=True, default="")),
                ("note", models.TextField(blank=True, default="")),
                ("flags", models.TextField(blank=True, default="")),
                ("source", models.TextField()),
                ("previous_source", models.TextField(blank=True, default="")),
                ("target", models.TextField(blank=True, default="")),
                ("position", models.IntegerField()),
                ("has_suggestion", models.BooleanField(db_index=True, default=False)),
                ("has_comment", models.BooleanField(db_index=True, default=False)),
                (
                    "has_failing_check",
                    models.BooleanField(db_index=True, default=False),
                ),
                ("num_words", models.IntegerField(default=0)),
                ("priority", models.IntegerField(default=100)),
                (
                    "translation",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Translation",
                    ),
                ),
                ("content_hash", models.BigIntegerField(db_index=True)),
                ("id_hash", models.BigIntegerField()),
                ("pending", models.BooleanField(default=False)),
                (
                    "state",
                    models.IntegerField(
                        choices=[
                            (0, "Empty"),
                            (10, "Needs editing"),
                            (20, "Translated"),
                            (30, "Approved"),
                            (100, "Read only"),
                        ],
                        db_index=True,
                        default=0,
                    ),
                ),
                (
                    "original_state",
                    models.IntegerField(
                        choices=[
                            (0, "Empty"),
                            (10, "Needs editing"),
                            (20, "Translated"),
                            (30, "Approved"),
                            (100, "Read only"),
                        ],
                        default=0,
                    ),
                ),
                (
                    "extra_context",
                    models.TextField(
                        blank=True, default="", verbose_name="Additional context"
                    ),
                ),
                (
                    "extra_flags",
                    models.TextField(
                        blank=True,
                        default="",
                        help_text="Additional comma-separated flags to influence quality checks. Possible values can be found in the documentation.",
                        validators=[weblate.trans.validators.validate_check_flags],
                        verbose_name="Translation flags",
                    ),
                ),
                (
                    "timestamp",
                    models.DateTimeField(auto_now_add=True),
                ),
                (
                    "labels",
                    models.ManyToManyField(
                        blank=True, to="trans.Label", verbose_name="Labels"
                    ),
                ),
                (
                    "shaping",
                    models.ForeignKey(
                        blank=True,
                        default=None,
                        null=True,
                        on_delete=django.db.models.deletion.SET_NULL,
                        to="trans.Shaping",
                    ),
                ),
            ],
            options={
                "unique_together": {("translation", "id_hash")},
                "index_together": {
                    ("translation", "pending"),
                    ("priority", "position"),
                },
            },
        ),
        migrations.CreateModel(
            name="Suggestion",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("target", models.TextField()),
                (
                    "unit",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="trans.Unit"
                    ),
                ),
                (
                    "user",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
                (
                    "timestamp",
                    models.DateTimeField(auto_now_add=True),
                ),
                ("userdetails", weblate.utils.fields.JSONField(default={})),
            ],
        ),
        migrations.CreateModel(
            name="Vote",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "suggestion",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Suggestion",
                    ),
                ),
                (
                    "user",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
                ("value", models.SmallIntegerField(default=0)),
            ],
            options={"unique_together": {("suggestion", "user")}},
        ),
        migrations.AddField(
            model_name="suggestion",
            name="votes",
            field=models.ManyToManyField(
                related_name="user_votes",
                through="trans.Vote",
                to=settings.AUTH_USER_MODEL,
            ),
        ),
        migrations.CreateModel(
            name="Dictionary",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("source", models.CharField(db_index=True, max_length=190)),
                ("target", models.CharField(max_length=190)),
                (
                    "language",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="lang.Language"
                    ),
                ),
                (
                    "project",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="trans.Project"
                    ),
                ),
            ],
        ),
        migrations.CreateModel(
            name="ComponentList",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "name",
                    models.CharField(
                        help_text="Display name",
                        max_length=100,
                        unique=True,
                        verbose_name="Component list name",
                    ),
                ),
                (
                    "slug",
                    models.SlugField(
                        help_text="Name used in URLs and filenames.",
                        max_length=100,
                        unique=True,
                        verbose_name="URL slug",
                    ),
                ),
                (
                    "components",
                    models.ManyToManyField(blank=True, to="trans.Component"),
                ),
                (
                    "show_dashboard",
                    models.BooleanField(
                        db_index=True,
                        default=True,
                        help_text="When enabled this component list will be shown as a tab on the dashboard",
                        verbose_name="Show on dashboard",
                    ),
                ),
            ],
            options={
                "verbose_name": "Component list",
                "verbose_name_plural": "Component lists",
            },
        ),
        migrations.CreateModel(
            name="Alert",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("timestamp", models.DateTimeField(auto_now_add=True)),
                ("name", models.CharField(max_length=150)),
                ("details", weblate.utils.fields.JSONField(default={})),
                (
                    "component",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Component",
                    ),
                ),
            ],
            options={"unique_together": {("component", "name")}},
        ),
        migrations.CreateModel(
            name="Announcement",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("message", models.TextField(verbose_name="Message")),
                (
                    "language",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="lang.Language",
                        verbose_name="Language",
                    ),
                ),
                (
                    "project",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Project",
                        verbose_name="Project",
                    ),
                ),
                (
                    "component",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Component",
                        verbose_name="Component",
                    ),
                ),
                (
                    "category",
                    models.CharField(
                        choices=[
                            ("info", "Info (light blue)"),
                            ("warning", "Warning (yellow)"),
                            ("danger", "Danger (red)"),
                            ("success", "Success (green)"),
                        ],
                        default="info",
                        help_text="Category defines color used for the message.",
                        max_length=25,
                        verbose_name="Category",
                    ),
                ),
                (
                    "message_html",
                    models.BooleanField(
                        blank=True,
                        default=False,
                        help_text="When turned off, URLs will be converted to links and any markup will be escaped.",
                        verbose_name="Render as HTML",
                    ),
                ),
                (
                    "expiry",
                    models.DateField(
                        blank=True,
                        db_index=True,
                        help_text="The message will be not shown after this date. Use it to announce string freeze and translation deadline for next release.",
                        null=True,
                        verbose_name="Expiry date",
                    ),
                ),
            ],
            options={
                "verbose_name": "Announcement",
                "verbose_name_plural": "Announcements",
            },
        ),
        migrations.CreateModel(
            name="Comment",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("comment", models.TextField()),
                ("timestamp", models.DateTimeField(auto_now_add=True, db_index=True)),
                (
                    "user",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
                (
                    "unit",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="trans.Unit"
                    ),
                ),
                ("resolved", models.BooleanField(db_index=True, default=False)),
            ],
        ),
        migrations.CreateModel(
            name="ContributorAgreement",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("timestamp", models.DateTimeField(auto_now=True)),
                (
                    "component",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Component",
                    ),
                ),
                (
                    "user",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
            ],
            options={"unique_together": {("user", "component")}},
        ),
        migrations.CreateModel(
            name="AutoComponentList",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                (
                    "project_match",
                    weblate.trans.fields.RegexField(
                        default="^$",
                        help_text="Regular expression which is used to match project slug.",
                        max_length=200,
                        verbose_name="Project regular expression",
                    ),
                ),
                (
                    "component_match",
                    weblate.trans.fields.RegexField(
                        default="^$",
                        help_text="Regular expression which is used to match component slug.",
                        max_length=200,
                        verbose_name="Component regular expression",
                    ),
                ),
                (
                    "componentlist",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.ComponentList",
                        verbose_name="Component list to assign",
                    ),
                ),
            ],
            options={
                "verbose_name": "Automatic component list assignment",
                "verbose_name_plural": "Automatic component list assignments",
            },
        ),
        migrations.CreateModel(
            name="Change",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("timestamp", models.DateTimeField(auto_now_add=True)),
                (
                    "action",
                    models.IntegerField(
                        choices=[
                            (0, "Resource update"),
                            (1, "Translation completed"),
                            (2, "Translation changed"),
                            (5, "New translation"),
                            (3, "Comment added"),
                            (4, "Suggestion added"),
                            (6, "Automatic translation"),
                            (7, "Suggestion accepted"),
                            (8, "Translation reverted"),
                            (9, "Translation uploaded"),
                            (10, "Added to glossary"),
                            (11, "Glossary updated"),
                            (12, "Glossary uploaded"),
                            (13, "New source string"),
                            (14, "Component locked"),
                            (15, "Component unlocked"),
                            (16, "Found duplicated string"),
                            (17, "Committed changes"),
                            (18, "Pushed changes"),
                            (19, "Reset repository"),
                            (20, "Merged repository"),
                            (21, "Rebased repository"),
                            (22, "Failed merge on repository"),
                            (23, "Failed rebase on repository"),
                            (28, "Failed push on repository"),
                            (24, "Parse error"),
                            (25, "Removed translation"),
                            (26, "Suggestion removed"),
                            (27, "Search and replace"),
                            (29, "Suggestion removed during cleanup"),
                            (30, "Source string changed"),
                            (31, "New string added"),
                            (32, "Bulk status change"),
                            (33, "Changed visibility"),
                            (34, "Added user"),
                            (35, "Removed user"),
                            (36, "Translation approved"),
                            (37, "Marked for edit"),
                            (38, "Removed component"),
                            (39, "Removed project"),
                            (40, "Found duplicated language"),
                            (41, "Renamed project"),
                            (42, "Renamed component"),
                            (43, "Moved component"),
                            (44, "New string to translate"),
                            (45, "New contributor"),
                            (46, "New announcement"),
                            (47, "New alert"),
                            (48, "Added new language"),
                            (49, "Requested new language"),
                            (50, "Created project"),
                            (51, "Created component"),
                            (52, "Invited user"),
                            (53, "Received repository notification"),
                            (54, "Replaced file by upload"),
                        ],
                        db_index=True,
                        default=2,
                    ),
                ),
                ("target", models.TextField(blank=True, default="")),
                (
                    "author",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        related_name="author_set",
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
                (
                    "dictionary",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Dictionary",
                    ),
                ),
                (
                    "translation",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Translation",
                    ),
                ),
                (
                    "unit",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Unit",
                    ),
                ),
                (
                    "user",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to=settings.AUTH_USER_MODEL,
                    ),
                ),
                (
                    "component",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Component",
                    ),
                ),
                (
                    "language",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="lang.Language",
                    ),
                ),
                ("old", models.TextField(blank=True, default="")),
                ("details", weblate.utils.fields.JSONField(default={})),
                (
                    "project",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="trans.Project",
                    ),
                ),
                (
                    "comment",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.SET_NULL,
                        to="trans.Comment",
                    ),
                ),
                (
                    "suggestion",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.SET_NULL,
                        to="trans.Suggestion",
                    ),
                ),
                (
                    "alert",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.SET_NULL,
                        to="trans.Alert",
                    ),
                ),
                (
                    "announcement",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.SET_NULL,
                        to="trans.Announcement",
                    ),
                ),
            ],
            options={"index_together": {("timestamp", "translation")}},
        ),
        migrations.RunPython(
            code=create_index,
            reverse_code=drop_index,
            atomic=False,
        ),
    ]
