"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Queue = exports.FifoThroughputLimit = exports.DeduplicationScope = exports.QueueEncryption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue
 */
var QueueEncryption;
(function (QueueEncryption) {
    /**
     * Messages in the queue are not encrypted
     */
    QueueEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by SQS.
     */
    QueueEncryption["KMS_MANAGED"] = "MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    QueueEncryption["KMS"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * What kind of deduplication scope to apply
 */
var DeduplicationScope;
(function (DeduplicationScope) {
    /**
     * Deduplication occurs at the message group level
     */
    DeduplicationScope["MESSAGE_GROUP"] = "messageGroup";
    /**
     * Deduplication occurs at the message queue level
     */
    DeduplicationScope["QUEUE"] = "queue";
})(DeduplicationScope = exports.DeduplicationScope || (exports.DeduplicationScope = {}));
/**
 * Whether the FIFO queue throughput quota applies to the entire queue or per message group
 */
var FifoThroughputLimit;
(function (FifoThroughputLimit) {
    /**
     * Throughput quota applies per queue
     */
    FifoThroughputLimit["PER_QUEUE"] = "perQueue";
    /**
     * Throughput quota applies per message group id
     */
    FifoThroughputLimit["PER_MESSAGE_GROUP_ID"] = "perMessageGroupId";
})(FifoThroughputLimit = exports.FifoThroughputLimit || (exports.FifoThroughputLimit = {}));
/**
 * A new Amazon SQS queue
 */
class Queue extends queue_base_1.QueueBase {
    constructor(scope, id, props = {}) {
        var _b;
        super(scope, id, {
            physicalName: props.queueName,
        });
        this.autoCreatePolicy = true;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sqs_QueueProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount,
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: this.physicalName,
            ...fifoProps,
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(),
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(),
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(),
            visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds(),
        });
        queue.applyRemovalPolicy((_b = props.removalPolicy) !== null && _b !== void 0 ? _b : core_1.RemovalPolicy.DESTROY);
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        this.deadLetterQueue = props.deadLetterQueue;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.UNENCRYPTED;
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`,
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    /**
     * Import an existing SQS queue provided an ARN
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param queueArn queue ARN (i.e. arn:aws:sqs:us-east-2:444455556666:queue1)
     */
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * Import an existing queue
     */
    static fromQueueAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sqs_QueueAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromQueueAttributes);
            }
            throw error;
        }
        const stack = core_1.Stack.of(scope);
        const parsedArn = stack.splitArn(attrs.queueArn, core_1.ArnFormat.NO_RESOURCE_NAME);
        const queueName = attrs.queueName || parsedArn.resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${parsedArn.region}.${stack.urlSuffix}/${parsedArn.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = this.determineFifo();
                this.autoCreatePolicy = false;
            }
            /**
             * Determine fifo flag based on queueName and fifo attribute
             */
            determineFifo() {
                if (core_1.Token.isUnresolved(this.queueArn)) {
                    return attrs.fifo || false;
                }
                else {
                    if (typeof attrs.fifo !== 'undefined') {
                        if (attrs.fifo && !queueName.endsWith('.fifo')) {
                            throw new Error("FIFO queue names must end in '.fifo'");
                        }
                        if (!attrs.fifo && queueName.endsWith('.fifo')) {
                            throw new Error("Non-FIFO queue name may not end in '.fifo'");
                        }
                    }
                    return queueName.endsWith('.fifo') ? true : false;
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.deduplicationScope) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.fifoThroughputLimit) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        if (props.deduplicationScope && !fifoQueue) {
            throw new Error('Deduplication scope can only be defined for FIFO queues');
        }
        if (props.fifoThroughputLimit && !fifoQueue) {
            throw new Error('FIFO throughput limit can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            deduplicationScope: props.deduplicationScope,
            fifoThroughputLimit: props.fifoThroughputLimit,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
_a = JSII_RTTI_SYMBOL_1;
Queue[_a] = { fqn: "@aws-cdk/aws-sqs.Queue", version: "1.154.0" };
//# sourceMappingURL=data:application/json;base64,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