# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['auction_scraper',
 'auction_scraper.scrapers.catawiki',
 'auction_scraper.scrapers.ebay',
 'auction_scraper.scrapers.liveauctioneers']

package_data = \
{'': ['*']}

modules = \
['main']
install_requires = \
['babel>=2.8.0,<3.0.0',
 'bs4>=0.0.1,<0.0.2',
 'datetime>=4.3,<5.0',
 'pathlib>=1.0.1,<2.0.0',
 'python-dateutil>=2.8.1,<3.0.0',
 'requests>=2.24.0,<3.0.0',
 'slimit>=0.8.1,<0.9.0',
 'sqlalchemy>=1.3.19,<2.0.0',
 'sqlalchemy_utils>=0.36.8,<0.37.0',
 'termcolor>=1.1.0,<2.0.0',
 'typer>=0.3.2,<0.4.0',
 'validators>=0.18.0,<0.19.0']

entry_points = \
{'console_scripts': ['auction-scraper = main:main']}

setup_kwargs = {
    'name': 'auction-scraper',
    'version': '0.3.3',
    'description': 'Extensible auction house scraper for ebay, liveauctioneers, catawiki, and other platforms',
    'long_description': '# Auction Scraper\n\n>  Scrape auction data auction sites into a sqlite database\n\n> Currently supports: catawiki, ebay, liveauctioneers\n\n> Can be used as a CLI tool, or interfaced with directly\n\n## Installation\n\nYou can [install with pip](https://pypi.org/project/auction-scraper/):\n\n``` \npip install auction-scraper\n```\n\n## New backend support\nWant to scrape an auction house not listed above?  Fear not - through our partnership with [Dreaming Spires](dreamingspires.dev), you can request that we build additional backend scrapers to extend the functionality.  Email contact@dreamingspires.dev for more info.\n\nWe also accept PRs, so feel free to write your own backend and submit it, if you require.  Instructions for this can be found under the _Building new backends_ section.\n\n## Usage\n\n`auction-scraper` will scrape data from auctions, profiles, and searches on the specified auction site.  Resulting textual data is written to a `sqlite3` database, with images and backup web pages optionally being written to a _data directory_.\n\nThe tool is invoked as:\n\n```\nUsage: auction-scraper [OPTIONS] DB_PATH BACKEND:[ebay|liveauctioneers]\n                       COMMAND [ARGS]...\n\nOptions:\n  DB_PATH                         The path of the sqlite database file to be\n                                  written to  [required]\n\n  BACKEND:[ebay|liveauctioneers]  The auction scraping backend  [required]\n  --data-location TEXT            The path additional image and html data is\n                                  saved to\n\n  --save-images / --no-save-images\n                                  Save images to data-location.  Requires\n                                  --data-location  [default: False]\n\n  --save-pages / --no-save-pages  Save pages to data-location. Requires\n                                  --data-location  [default: False]\n\n  --verbose / --no-verbose        [default: False]\n  --base-uri TEXT                 Override the base url used to resolve the\n                                  auction site\n\n  --install-completion [bash|zsh|fish|powershell|pwsh]\n                                  Install completion for the specified shell.\n  --show-completion [bash|zsh|fish|powershell|pwsh]\n                                  Show completion for the specified shell, to\n                                  copy it or customize the installation.\n\n  --help                          Show this message and exit.\n\nCommands:\n  auction  Scrapes an auction site auction page.\n  profile  Scrapes an auction site profile page.\n  search   Performs a search, returning the top n_results results for each...\n```\n\n### Auction mode\nIn auction mode, an auction must be specified as either a unique _auction ID_ or as a URL.  The textual data is scraped into the `[BACKEND]_auctions` table of `DB_PATH`, the page is scraped into `[data-location]/[BACKEND]/auctions`, and the images into `[data-location]/[BACKEND]/images`.  The `--base-url` option determines the base URL from which to resolve _auction IDs_, _profile IDs_, and search _query strings_ if specified, otherwise defaulting to the default for the specified backend.\n\nExample usage:\n\n```bash\n# Scraping an auction by URL\nauction-scraper db.db liveauctioneers auction https://www.liveauctioneers.com/item/88566418_cameroon-power-or-reliquary-figure\n\n# Equivalently scraping from an auction ID\nauction-scraper db.db liveauctioneers auction 88566418\n\n# Scraping an auction, including all images and the page itself, into data-location\nauction-scraper --data-location=./data --save-images --save-pages db.db liveauctioneers auction 88566418\n```\n\n### Profile mode\nIn profile mode, a profile must be specified as either a unique _user ID_ or as a URL.  The textual data is scraped into the `[BACKEND]_profiles` table of `DB_PATH`, and the page is scraped into `[data-location]/[BACKEND]/profiles`.  The `--base-url` option determines the base URL from which to resolve _auction IDs_, _profile IDs_, and search _query strings_ if specified, otherwise defaulting to the default for the specified backend.\n\nExample usage:\n\n```bash\n# Scraping a profile by URL\nauction-scraper db.db liveauctioneers profile https://www.liveauctioneers.com/auctioneer/197/hindman/\n\n# Equivalently scraping from a profile ID\nauction-scraper db.db liveauctioneers auction 197\n\n# Scraping a profile, including the page itself, into data-location\nauction-scraper --data-location=./data --save-pages db.db liveauctioneers profile 197\n```\n\n\n### Search mode\nIn search mode, at least one `QUERY_STRING` must be provided alongside `N_RESULTS`.  It will scrape the auctions pertaining to the top `N_RESULTS` results from the `QUERY_STRING`.  The `--base-url` option determines the base URL from which to resolve the search if specified, otherwise defaulting to the default for the specified backend.\n\nExample usage:\n```bash\n# Search one result by a single search term\nauction-scraper db.db search 1 "mambila art"\n\n# Search ten results by two search terms, scraping images and pages into data-location\nauction-scraper --data-location=./data --save-images --save-pages db.db search 10 "mambila" "mambilla"\n```\n\n## Running continuously using systemd\n`auction-scraper@.service` and `auction-scraper@.timer`, once loaded by systemd, can be used to schedule the running of `auction-scraper` with user-given arguments according to a schedule.\n\n### Running as a systemd root service\n\nCopy `auction-scraper@.service` and `auction-scraper@.timer` to `/etc/systemd/system/`.\n\nModify `auction-scraper@.timer` to specify the schedule you require.\n\nReload the system daemons.  As root:\n```bash\nsystemctl daemon-reload\n```\n\nRun (start now) and enable (restart on boot) the systemd-timer, specifying the given arguments, within quotes, after the \'@\'.  For example, as root:\n```bash\nsystemctl enable --now auction-scraper@"db.db liveauctioneers search 10 mambila".timer\n```\n\nFind information about your running timers with:\n```bash\nsystemctl list-timers\n```\n\nStop your currently running timer with:\n```bash\nsystemctl stop auction-scraper@"db.db liveauctioneers search 10 mambila".timer\n```\n\nDisable your currently running timer with:\n```bash\nsystemctl disable auction-scraper@"db.db liveauctioneers search 10 mambila".timer\n```\n\nA new timer is created for each unique argument string, so the arguments must be specified when stopping or disabling the timer.\n\nSome modification may be required to run as a user service, including placing the service and timer files in `~/.local/share/systemd/user/`.\n\n## Building from source\n\nEnsure poetry is [installed](https://python-poetry.org/docs/#installation).  Then from this directory install dependencies into the poetry virtual environment and build:\n\n```bash\npoetry install\npoetry build\n```\n\nSource and wheel files are built into `auction_scraper/dist`.\n\nInstall it across your user with `pip`, outside the venv:\n```bash\ncd ./dist\npython3 -m pip install --user ./auction_scraper-0.0.1-py3-none-any.whl\n```\n\nor\n\n```bash\ncd ./dist\npip install ./auction_scraper-0.0.1-py3-none-any.whl\n```\n\nRun `auction-scraper` to invoke the utility.\n\n## Interfacing with the API\nEach backend of `auction-scraper` can also be invoked as a Python library to automate its operation.  The backends implement the abstract class `auction_scraper.abstract_scraper.AbstractAuctionScraper`, alongside the abstract SQLAlchemy models `auction_scraper.abstract_models.BaseAuction` and `auction_scraper.abstract_models.BaseProfile`.\nThe resulting scraper exposes methods to scrape auction, profile, and search pages into these SQLAlchemy model objects, according to the following interface:\n\n```\ndef scrape_auction(self, auction, save_page=False, save_images=False):\n    """\n    Scrapes an auction page, specified by either a unique auction ID\n    or a URI.  Returns an auction model containing the scraped data.\n    If specified by auction ID, constructs the URI using self.base_uri.\n    If self.page_save_path is set, writes out the downloaded pages to disk at\n    the given path according to the naming convention specified by\n    self.auction_save_name.\n    Returns a BaseAuction\n    """\n```\n\n```\ndef scrape_profile(self, profile, save_page=False):\n    """\n    Scrapes a profile page, specified by either a unique profile ID\n    or a URI.  Returns an profile model containing the scraped data.\n    If specified by profile ID, constructs the URI using self.base_uri.\n    If self.page_save_path is set, writes out the downloaded pages to disk at\n    the given path according to the naming convention specified by\n    self.profile_save_name.\n    Returns a BaseProfile\n    """\n```\n\n```\ndef scrape_search(self, query_string, n_results=None, save_page=False,\n        save_images=False):\n    """\n    Scrapes a search page, specified by either a query_string and n_results,\n    or by a unique URI.\n    If specified by query_string, de-paginates the results and returns up\n    to n_results results.  If n_results is None, returns all results.\n    If specified by a search_uri, returns just the results on the page.\n    Returns a dict {auction_id: SearchResult}\n    """\n```\n\n```\ndef scrape_auction_to_db(self, auction, save_page=False, save_images=False):\n    """\n    Scrape an auction page, writing the resulting page to the database.\n    Returns a BaseAuction\n    """\n```\n\n```\ndef scrape_profile_to_db(self, profile, save_page=False):\n    """\n    Scrape a profile page, writing the resulting profile to the database.\n    Returns a BaseProfile\n    """\n```\n\n```\ndef scrape_search_to_db(self, query_strings, n_results=None, \\\n        save_page=False, save_images=False):\n    """\n    Scrape a set of query_strings, writing the resulting auctions and profiles\n    to the database.\n    Returns a tuple ([BaseAuction], [BaseProfile])\n    """\n```\n\n## Building new backends\nAll backends live at `action_scraper/scrapers` in their own specific directory.  It should implement the abstract class `auction_scraper.abstract_scraper.AbstractAuctionScraper` in a file `scraper.py`, and the abstract SQLAlchemy models `auction_scraper.abstract_models.BaseAuction` and `auction_scraper.abstract_models.BaseProfile` in `models.py`.\n\nThe `AuctionScraper` class must extend `AbstractAuctionScraper` and implement the following methods:\n```python3\n# Given a uri, scrape the auction page into an auction object (of type BaseAuction)\ndef _scrape_auction_page(self, uri)\n\n# Given a uri, scrape the profile page into an profile object (of type BaseAuction)\ndef _scrape_profile_page(self, uri)\n\n# Given a uri, scrape the search page into a list of results (of type {auction_id: SearchResult})\ndef _scrape_search_page(self, uri)\n```\n\nIt must also supply defaults to the following variables:\n```python3\nauction_table\nprofile_table\nbase_uri\nauction_suffix\nprofile_suffix\nsearch_suffix\nbackend_name\n```\n\n## Authors\nEdd Salkield <edd@salkield.uk>  - Main codebase\n\nMark Todd                       - Liveauctioneers scraper\n\nJonathan Tanner                 - Catawiki scraper\n',
    'author': 'Edd Salkield',
    'author_email': 'edd@salkield.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/dreamingspires/auction-scraper',
    'packages': packages,
    'package_data': package_data,
    'py_modules': modules,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
