"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
const sqs = require("aws-cdk-lib/aws-sqs");
const kms = require("aws-cdk-lib/aws-kms");
// --------------------------------------------------------------
// Pattern deployment with default props
// --------------------------------------------------------------
test('Pattern deployment with default props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Creates an encryption key
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Testing with existing SQS Queue
// --------------------------------------------------------------
test('Pattern deployment with existing queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const props = {
        existingQueueObj: queue,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue-obj"
    });
});
// --------------------------------------------------------------
// Testing with passing queue and dead letter queue props
// --------------------------------------------------------------
test('Pattern deployment with queue and dead letter queue props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        deadLetterQueueProps: {
            queueName: 'dlq-name'
        },
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "queue-name",
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testiotsqsdeadLetterQueue66A04E81",
                    "Arn",
                ],
            },
            maxReceiveCount: 15
        }
    });
    // Creates a dead letter queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "dlq-name"
    });
});
// --------------------------------------------------------------
// Testing with dead letter queue turned off
// --------------------------------------------------------------
test('Pattern deployment with dead letter queue turned off', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        deployDeadLetterQueue: false,
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "queue-name"
    });
    // Does not create the default dead letter queue
    expect(stack).not.toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
// --------------------------------------------------------------
// Testing with custom maxReceiveCount
// --------------------------------------------------------------
test('Pattern deployment with custom maxReceiveCount', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        deadLetterQueueProps: {
            queueName: 'dlq-name'
        },
        deployDeadLetterQueue: true,
        maxReceiveCount: 1,
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "queue-name",
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testiotsqsdeadLetterQueue66A04E81",
                    "Arn",
                ],
            },
            maxReceiveCount: 1
        },
    });
});
// --------------------------------------------------------------
// Testing without creating a KMS key
// --------------------------------------------------------------
test('Pattern deployment without creating a KMS key', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        enableEncryptionWithCustomerManagedKey: false,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Does not create an encryption key
    expect(stack).not.toHaveResource("AWS::KMS::Key");
});
// --------------------------------------------------------------
// Testing with existing KMS key
// --------------------------------------------------------------
test('Pattern deployment with existing KMS key', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const kmsKey = new kms.Key(stack, 'existing-key', {
        enableKeyRotation: false,
        alias: 'existing-key-alias'
    });
    const props = {
        encryptionKey: kmsKey,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "existingkey205DFC01",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Uses the provided key
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
});
// --------------------------------------------------------------
// Testing with passing KMS key props
// --------------------------------------------------------------
test('Pattern deployment passing KMS key props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        encryptionKeyProps: {
            enableKeyRotation: false,
            alias: 'new-key-alias-from-props'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Uses the props to create the key
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
    expect(stack).toHaveResource("AWS::KMS::Alias", {
        AliasName: "alias/new-key-alias-from-props",
        TargetKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn",
            ]
        }
    });
});
// --------------------------------------------------------------
// Testing with passing a FIFO queue (not supported by IoT)
// --------------------------------------------------------------
test('Pattern deployment with passing a FIFO queue (not supported by IoT)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const queue = new sqs.Queue(stack, 'existing-fifo-queue-obj', {
        queueName: 'existing-queue.fifo',
        fifo: true
    });
    const props = {
        existingQueueObj: queue,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    expect.assertions(1);
    try {
        new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    }
    catch (err) {
        expect(err.message).toBe('The IoT SQS action doesn\'t support Amazon SQS FIFO (First-In-First-Out) queues');
    }
});
//# sourceMappingURL=data:application/json;base64,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