"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnauthenticatedAction = exports.ListenerAction = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * What to do when a client makes a request to a listener
 *
 * Some actions can be combined with other ones (specifically,
 * you can perform authentication before serving the request).
 *
 * Multiple actions form a linked chain; the chain must always terminate in a
 * *(weighted)forward*, *fixedResponse* or *redirect* action.
 *
 * If an action supports chaining, the next action can be indicated
 * by passing it in the `next` property.
 *
 * (Called `ListenerAction` instead of the more strictly correct
 * `ListenerAction` because this is the class most users interact
 * with, and we want to make it not too visually overwhelming).
 */
class ListenerAction {
    /**
     * Create an instance of ListenerAction
     *
     * The default class should be good enough for most cases and
     * should be created by using one of the static factory functions,
     * but allow overriding to make sure we allow flexibility for the future.
     */
    constructor(actionJson, next) {
        this.actionJson = actionJson;
        this.next = next;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_CfnListener_ActionProperty(actionJson);
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_ListenerAction(next);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ListenerAction);
            }
            throw error;
        }
    }
    /**
     * Authenticate using an identity provider (IdP) that is compliant with OpenID Connect (OIDC)
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-authenticate-users.html#oidc-requirements
     */
    static authenticateOidc(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_AuthenticateOidcOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.authenticateOidc);
            }
            throw error;
        }
        return new ListenerAction({
            type: 'authenticate-oidc',
            authenticateOidcConfig: {
                authorizationEndpoint: options.authorizationEndpoint,
                clientId: options.clientId,
                clientSecret: options.clientSecret.unsafeUnwrap(),
                issuer: options.issuer,
                tokenEndpoint: options.tokenEndpoint,
                userInfoEndpoint: options.userInfoEndpoint,
                authenticationRequestExtraParams: options.authenticationRequestExtraParams,
                onUnauthenticatedRequest: options.onUnauthenticatedRequest,
                scope: options.scope,
                sessionCookieName: options.sessionCookieName,
                sessionTimeout: options.sessionTimeout?.toSeconds().toString(),
            },
        }, options.next);
    }
    /**
     * Forward to one or more Target Groups
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     */
    static forward(targetGroups, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_ForwardOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.forward);
            }
            throw error;
        }
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a ListenerAction.forward()');
        }
        if (targetGroups.length === 1 && options.stickinessDuration === undefined) {
            // Render a "simple" action for backwards compatibility with old templates
            return new TargetGroupListenerAction(targetGroups, {
                type: 'forward',
                targetGroupArn: targetGroups[0].targetGroupArn,
            });
        }
        return new TargetGroupListenerAction(targetGroups, {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroupArn })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Forward to one or more Target Groups which are weighted differently
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     */
    static weightedForward(targetGroups, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_ForwardOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.weightedForward);
            }
            throw error;
        }
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a ListenerAction.weightedForward()');
        }
        return new TargetGroupListenerAction(targetGroups.map(g => g.targetGroup), {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroup.targetGroupArn, weight: g.weight })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Return a fixed response
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#fixed-response-actions
     */
    static fixedResponse(statusCode, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_FixedResponseOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fixedResponse);
            }
            throw error;
        }
        return new ListenerAction({
            type: 'fixed-response',
            fixedResponseConfig: {
                statusCode: core_1.Tokenization.stringifyNumber(statusCode),
                contentType: options.contentType,
                messageBody: options.messageBody,
            },
        });
    }
    /**
     * Redirect to a different URI
     *
     * A URI consists of the following components:
     * protocol://hostname:port/path?query. You must modify at least one of the
     * following components to avoid a redirect loop: protocol, hostname, port, or
     * path. Any components that you do not modify retain their original values.
     *
     * You can reuse URI components using the following reserved keywords:
     *
     * - `#{protocol}`
     * - `#{host}`
     * - `#{port}`
     * - `#{path}` (the leading "/" is removed)
     * - `#{query}`
     *
     * For example, you can change the path to "/new/#{path}", the hostname to
     * "example.#{host}", or the query to "#{query}&value=xyz".
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#redirect-actions
     */
    static redirect(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_RedirectOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.redirect);
            }
            throw error;
        }
        if ([options.host, options.path, options.port, options.protocol, options.query].findIndex(x => x !== undefined) === -1) {
            throw new Error('To prevent redirect loops, set at least one of \'protocol\', \'host\', \'port\', \'path\', or \'query\'.');
        }
        return new ListenerAction({
            type: 'redirect',
            redirectConfig: {
                statusCode: options.permanent ? 'HTTP_301' : 'HTTP_302',
                host: options.host,
                path: options.path,
                port: options.port,
                protocol: options.protocol,
                query: options.query,
            },
        });
    }
    /**
     * Render the actions in this chain
     */
    renderActions() {
        return this.renumber([this.actionJson, ...this.next?.renderActions() ?? []]);
    }
    /**
     * Called when the action is being used in a listener
     */
    bind(scope, listener, associatingConstruct) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_IApplicationListener(listener);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        // Empty on purpose
        Array.isArray(scope);
        Array.isArray(listener);
        Array.isArray(associatingConstruct);
    }
    /**
     * Renumber the "order" fields in the actions array.
     *
     * We don't number for 0 or 1 elements, but otherwise number them 1...#actions
     * so ELB knows about the right order.
     *
     * Do this in `ListenerAction` instead of in `Listener` so that we give
     * users the opportunity to override by subclassing and overriding `renderActions`.
     */
    renumber(actions) {
        if (actions.length < 2) {
            return actions;
        }
        return actions.map((action, i) => ({ ...action, order: i + 1 }));
    }
}
exports.ListenerAction = ListenerAction;
_a = JSII_RTTI_SYMBOL_1;
ListenerAction[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancingv2.ListenerAction", version: "1.168.0" };
/**
 * What to do with unauthenticated requests
 */
var UnauthenticatedAction;
(function (UnauthenticatedAction) {
    /**
     * Return an HTTP 401 Unauthorized error.
     */
    UnauthenticatedAction["DENY"] = "deny";
    /**
     * Allow the request to be forwarded to the target.
     */
    UnauthenticatedAction["ALLOW"] = "allow";
    /**
     * Redirect the request to the IdP authorization endpoint.
     */
    UnauthenticatedAction["AUTHENTICATE"] = "authenticate";
})(UnauthenticatedAction = exports.UnauthenticatedAction || (exports.UnauthenticatedAction = {}));
/**
 * Listener Action that calls "registerListener" on TargetGroups
 */
class TargetGroupListenerAction extends ListenerAction {
    constructor(targetGroups, actionJson) {
        super(actionJson);
        this.targetGroups = targetGroups;
    }
    bind(_scope, listener, associatingConstruct) {
        for (const tg of this.targetGroups) {
            tg.registerListener(listener, associatingConstruct);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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