"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
const sqs = require("@aws-cdk/aws-sqs");
const kms = require("@aws-cdk/aws-kms");
// --------------------------------------------------------------
// Pattern deployment with default props
// --------------------------------------------------------------
test('Pattern deployment with default props', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Creates an encryption key
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Testing with existing SQS Queue
// --------------------------------------------------------------
test('Pattern deployment with existing queue', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const props = {
        existingQueueObj: queue,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue-obj"
    });
});
// --------------------------------------------------------------
// Testing with passing queue and dead letter queue props
// --------------------------------------------------------------
test('Pattern deployment with queue and dead letter queue props', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deadLetterQueueProps: {
            queueName: 'dlq-name'
        },
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "queue-name",
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testiotsqsdeadLetterQueue66A04E81",
                    "Arn",
                ],
            },
            maxReceiveCount: 15
        }
    });
    // Creates a dead letter queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "dlq-name"
    });
});
// --------------------------------------------------------------
// Testing with dead letter queue turned off
// --------------------------------------------------------------
test('Pattern deployment with dead letter queue turned off', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deployDeadLetterQueue: false,
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "queue-name"
    });
    // Does not create the default dead letter queue
    expect(stack).not.toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
// --------------------------------------------------------------
// Testing with custom maxReceiveCount
// --------------------------------------------------------------
test('Pattern deployment with custom maxReceiveCount', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deadLetterQueueProps: {
            queueName: 'dlq-name'
        },
        deployDeadLetterQueue: true,
        maxReceiveCount: 1,
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a queue using the provided props
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "queue-name",
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testiotsqsdeadLetterQueue66A04E81",
                    "Arn",
                ],
            },
            maxReceiveCount: 1
        },
    });
});
// --------------------------------------------------------------
// Testing without creating a KMS key
// --------------------------------------------------------------
test('Pattern deployment without creating a KMS key', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        enableEncryptionWithCustomerManagedKey: false,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Does not create an encryption key
    expect(stack).not.toHaveResource("AWS::KMS::Key");
});
// --------------------------------------------------------------
// Testing with existing KMS key
// --------------------------------------------------------------
test('Pattern deployment with existing KMS key', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const kmsKey = new kms.Key(stack, 'existing-key', {
        enableKeyRotation: false,
        alias: 'existing-key-alias'
    });
    const props = {
        encryptionKey: kmsKey,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "existingkey205DFC01",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Uses the provided key
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
});
// --------------------------------------------------------------
// Testing with passing KMS key props
// --------------------------------------------------------------
test('Pattern deployment passing KMS key props', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        encryptionKeyProps: {
            enableKeyRotation: false,
            alias: 'new-key-alias-from-props'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    expect(stack).toHaveResource("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Uses the props to create the key
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
    expect(stack).toHaveResource("AWS::KMS::Alias", {
        AliasName: "alias/new-key-alias-from-props",
        TargetKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn",
            ]
        }
    });
});
// --------------------------------------------------------------
// Testing with passing a FIFO queue (not supported by IoT)
// --------------------------------------------------------------
test('Pattern deployment with passing a FIFO queue (not supported by IoT)', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'existing-fifo-queue-obj', {
        queueName: 'existing-queue.fifo',
        fifo: true
    });
    const props = {
        existingQueueObj: queue,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    expect.assertions(1);
    try {
        new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    }
    catch (err) {
        expect(err.message).toBe('The IoT SQS action doesn\'t support Amazon SQS FIFO (First-In-First-Out) queues');
    }
});
//# sourceMappingURL=data:application/json;base64,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