"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const http_origin_1 = require("./http-origin");
/**
 * An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 *
 * @stability stable
 */
class S3Origin {
    /**
     * @stability stable
     */
    constructor(bucket, props = {}) {
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    /**
     * The method called when a given Origin is added (for the first time) to a Distribution.
     *
     * @stability stable
     */
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
_a = JSII_RTTI_SYMBOL_1;
S3Origin[_a] = { fqn: "@aws-cdk/aws-cloudfront-origins.S3Origin", version: "1.90.1" };
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, { originAccessIdentity, ...props }) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
        if (originAccessIdentity) {
            this.originAccessIdentity = originAccessIdentity;
        }
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            // Using a bucket from another stack creates a cyclic reference with
            // the bucket taking a dependency on the generated S3CanonicalUserId for the grant principal,
            // and the distribution having a dependency on the bucket's domain name.
            // Fix this by parenting the OAI in the bucket's stack when cross-stack usage is detected.
            const bucketStack = cdk.Stack.of(this.bucket);
            const bucketInDifferentStack = bucketStack !== cdk.Stack.of(scope);
            const oaiScope = bucketInDifferentStack ? bucketStack : scope;
            const oaiId = bucketInDifferentStack ? `${cdk.Names.uniqueId(scope)}S3Origin` : 'S3Origin';
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(oaiScope, oaiId, {
                comment: `Identity for ${options.originId}`,
            });
            // Used rather than `grantRead` because `grantRead` will grant overly-permissive policies.
            // Only GetObject is needed to retrieve objects for the distribution.
            // This also excludes KMS permissions; currently, OAI only supports SSE-S3 for buckets.
            // Source: https://aws.amazon.com/blogs/networking-and-content-delivery/serving-sse-kms-encrypted-content-from-s3-using-cloudfront/
            this.bucket.addToResourcePolicy(new iam.PolicyStatement({
                resources: [this.bucket.arnForObjects('*')],
                actions: ['s3:GetObject'],
                principals: [this.originAccessIdentity.grantPrincipal],
            }));
        }
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,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