import numpy as np
from pathlib import Path
import cv2
import numpy.ma as ma
import hylite.io as io
from .sensor import Sensor

class Fenix(Sensor):
    """
    Implementation of sensor corrections for the Fenix sensor.
    """

    cal2 = None
    cal4 = None
    cal8 = None

    @classmethod
    def name(cls):
        """
        Returns this sensors name
        """
        return "FENIX"

    @classmethod
    def fov(cls):
        """
        Return the (vertical) sensor field of view .
        """
        return 32.3

    @classmethod
    def ypixels(cls):
        """
        Return the number of pixels in the y-dimension.
        """
        return 384  # n.b. sensor has 384 pixels but this is resized to 401 on lens correction

    @classmethod
    def xpixels(cls):
        """
        Return the number of pixels in the x-dimension (==1 for line scanners).
        """
        return 1  # the Fenix is a line-scanner

    @classmethod
    def pitch(cls):
        """
        Return the pitch of the each pixel in the y-dimension (though most pixels are square).
        """
        return 0.084

    @classmethod
    def correct_image(cls, image, verbose=True, **kwds):

        """
        Apply sensor corrections to an image.

        *Arguments*:
         - image = a hyImage instance of an image captured using this sensor.
         - verbose = true if updates/progress should be printed to the console. Default is False.
        *Keywords*:
         - rad = true if image should be converted to radiance by applying dark and white references. Default is True.
         - bpr = replace bad pixels (only for raw data). Default is True.
         - shift = shift bands to account for time-delay between their acquisitions.
                   Only use for near-field sensing (e.g. drill-core scans). Default is False.
         - flip = true if image should be flipped before applying lens correction (if camera mounted backwards in core
                  scanner). Default is False.
         - lens = apply GLTX lens correction to remove lens distortion. Default is True.
        """

        # get kwds
        rad = kwds.get("rad", True)
        bpr = kwds.get("bpr", True)
        shift = kwds.get("shift", False)
        lens = kwds.get("lens", True)

        if rad:
            if verbose: print("Converting to radiance... ", end="", flush="True")

            # convert from int to float
            image.data = image.data.astype(np.float32)

            # find saturation and store in mask
            r = image.get_band_index(975.)
            mask = np.ones(image.data.shape)
            mask[..., :r][image.data[..., :r] == 4095.] = np.nan
            mask[..., r:][image.data[..., r:] == 65535.] = np.nan

            # apply dark reference
            if cls.dark is None:
                print("Warning: dark calibration not found; no dark correction was applied! Something smells dodgy...")
            else:
                dref = np.nanmean(cls.dark.data, axis=1)  # calculate dark reference
                image.data[:, :, :] -= dref[:, None, :]  # apply dark calibration

            # apply laboratory calibration
            assert not image.header is None, "Error: image must be linked to a header file (.hdr) for FENIX correction."
            binning = int(image.header.get('binning', [2])[0])
            if image.xdim() == 384:
                if binning == 2:
                    if Fenix.cal2 is None:
                        Fenix.cal2 = io.load(
                            str(Path(__file__).parent / "calibration_data/fenix/Radiometric_2x2_1x1.hdr"))
                    cal = Fenix.cal2.data[:, 0, :]
                    calw = Fenix.cal2.get_wavelengths()
                elif binning == 4:
                    if Fenix.cal4 is None:
                        Fenix.cal4 = io.load(
                            str(Path(__file__).parent / "calibration_data/fenix/Radiometric_4x2_1x1.hdr"))
                    cal = Fenix.cal4.data[:, 0, :]
                    calw = Fenix.cal4.get_wavelengths()
                elif binning == 8:
                    if Fenix.cal8 is None:
                        Fenix.cal8 = io.load(
                            str(Path(__file__).parent / "calibration_data/fenix/Radiometric_8x2_1x1.hdr"))
                    cal = Fenix.cal8.data[:, 0, :]
                    calw = Fenix.cal8.get_wavelengths()
                else:
                    assert False, "Error: calibration data for binning=%d does not exist" % binning
            elif image.xdim() == 1024:
                if binning == 2:
                    if Fenix.cal2 is None:
                        Fenix.cal2 = io.load(
                            str(Path(__file__).parent / "calibration_data/fenix1k/Radiometric_2x2_1x1.hdr"))
                    cal = Fenix.cal2.data[:, 0, :]
                    calw = Fenix.cal2.get_wavelengths()
                elif binning == 4:
                    if Fenix.cal4 is None:
                        Fenix.cal4 = io.load(
                            str(Path(__file__).parent / "calibration_data/fenix1k/Radiometric_4x2_1x1.hdr"))
                    cal = Fenix.cal4.data[:, 0, :]
                    calw = Fenix.cal4.get_wavelengths()
                elif binning == 8:
                    if Fenix.cal8 is None:
                        Fenix.cal8 = io.load(
                            str(Path(__file__).parent / "calibration_data/fenix1k/Radiometric_8x2_1x1.hdr"))
                    cal = Fenix.cal8.data[:, 0, :]
                    calw = Fenix.cal8.get_wavelengths()
                else:
                    assert False, "Error: calibration data for binning=%d does not exist" % binning
            else:
                assert False, "Error: no known Fenix dimensions %d" % image.xdim()

            # convert from saturation to radiance based on laboratory calibration file
            cal = cal[:, [np.argmin(calw < w) for w in
                          image.get_wavelengths()]]  # match image wavelengths to calibration wavelengths
            image.data[:, :, :] *= cal[:, None, :]  # apply to image

            # apply white reference (if specified)
            if not cls.white is None:

                # calculate white reference radiance
                white = np.nanmean(cls.white.data.astype(np.float32),
                                   axis=1) - dref  # average each line and subtract dark reference
                white *= cal  # apply laboratory calibration to white reference

                # extract white (or grey) reference reflectance
                if cls.white_spectra is None:
                    refl = np.zeros(white.shape[1]) + 1.0  # assume pure white
                else:
                    # get known target spectra
                    refl = cls.white_spectra.get_reflectance()

                    # match bands with this image
                    idx = [np.argmin(cls.white_spectra.get_wavelengths() < w) for w in image.get_wavelengths()]
                    refl = refl[idx]

                # apply white reference
                cfac = refl[None, :] / white
                image.data[:, :, :] *= cfac[:, None, :]

            if verbose: print("DONE.")

        ##############################################################
        # replace bad pixels with an average of the surrounding ones
        ##############################################################
        if bpr:
            if verbose: print("Filtering bad pixels... ", end="", flush="True")
            invalids = np.argwhere(np.isnan(image.data) | np.isinf(image.data))  # search for bad pixels
            for px, py, band in invalids:
                n = 0
                sum = 0
                for xx in range(px - 1, px + 2):
                    for yy in range(py - 1, py + 2):
                        if xx == px and yy == py: continue  # skip invalid pixel
                        if xx < 0 or yy < 0 or xx >= image.data.shape[0] or yy >= image.data.shape[
                            1]: continue  # skip out of bounds pixels
                        if image.data[xx][yy][band] == np.nan or image.data[xx][yy][
                            band] == np.inf: continue  # maybe neighbour is nan also
                        n += 1
                        sum += image.data[xx][yy][band]
                if n > 0: sum /= n  # do averaging
                image.data[px][py][band] = sum
            if verbose: print("DONE.")

        ##############################################################
        # apply saturation flag
        ##############################################################
        if rad:
            image.data *= mask

        ######################################################################################
        # sensor alignment - identify tie points and apply rigid transform to second sensor
        ######################################################################################
        if shift:
            if verbose: print("Correcting sensor shift (SIFT)... ", end="", flush="True")

            # extract sift features from a band in each sensor
            m = 'sift'  # matching method
            xp1, des1 = image.get_keypoints(100, method=m)
            xp2, des2 = image.get_keypoints(200, method=m)

            # match features
            src_pts, dst_pts = io.HyImage.match_keypoints(xp1, xp2, des1, des2, method=m)

            # use RANSAC and a homography transform model to discard bad points
            H, status = cv2.findHomography(dst_pts, src_pts, cv2.RANSAC, 3.0)
            dst_mask = dst_pts[:, 0, :] * status
            src_mask = src_pts[:, 0, :] * status
            dst_mask = dst_mask[dst_mask.all(1)]
            src_mask = src_mask[src_mask.all(1)]
            dst_mask = np.expand_dims(dst_mask, axis=1)
            src_mask = np.expand_dims(src_mask, axis=1)

            # estimate translation matrix
            #M = cv2.estimateRigidTransform(dst_mask, src_mask, False)
            M = cv2.estimateAffinePartial2D(src_mask, dst_mask)[0]

            # transform bands from second sensor (>175)
            r = image.get_band_index(975.)
            for i in range(r, image.band_count()):
                image.data[:, :, i] = cv2.warpAffine(image.data[:, :, i], M, (image.data.shape[1], image.data.shape[0]))

            if verbose: print("DONE.")

        #############################
        # apply lens correction
        #############################
        if lens:
            if verbose: print("Applying lens correction... ", end="", flush="True")

            # load pixel offsets
            if image.xdim() == 1024:
                m = np.array([[0 , 1.239343002 , 2.48091711 , 3.718936015 , 4.95629651 , 6.190111768 , 7.426162682 , 8.658675201 , 9.890539938 , 11.11887611 , 12.34945242 , 13.5765069 , 14.80292409 , 16.02870581 , 17.25097875 , 18.47549659 , 19.69651227 , 20.91403203 , 22.13380071 , 23.35008003 , 24.56574256 , 25.78079011 , 26.99522447 , 28.20618537 , 29.41653952 , 30.62628871 , 31.83543472 , 33.04397929 , 34.24906926 , 35.45356415 , 36.6574657 , 37.85792492 , 39.06064644 , 40.25993188 , 41.45863372 , 42.65390846 , 43.85144962 , 45.04556985 , 46.23911607 , 47.43208996 , 48.62165484 , 49.81349051 , 51.00192325 , 52.1897931 , 53.37426872 , 54.56101914 , 55.74438132 , 56.92718991 , 58.10944657 , 59.28832655 , 60.46948557 , 61.64727381 , 62.82451926 , 63.99840238 , 65.17456843 , 66.34737795 , 67.51965372 , 68.69139735 , 69.86261042 , 71.03048103 , 72.1978268 , 73.36464932 , 74.53095015 , 75.69673087 , 76.85918581 , 78.02112627 , 79.18255379 , 80.34346993 , 81.50107394 , 82.66097319 , 83.81756584 , 84.97365571 , 86.12644692 , 87.28153705 , 88.43333396 , 89.5846366 , 90.73544646 , 91.88576507 , 93.03280366 , 94.18214544 , 95.32821255 , 96.47379675 , 97.61611397 , 98.76073795 , 99.90210022 , 101.0429878 , 102.1834022 , 103.320565 , 104.4600385 , 105.5962656 , 106.7320276 , 107.8673259 , 109.002162 , 110.1337643 , 111.2649095 , 112.395599 , 113.5258342 , 114.6556165 , 115.78218 , 116.911062 , 118.0367302 , 119.1619533 , 120.2839698 , 121.4083082 , 122.5294448 , 123.6501442 , 124.7704077 , 125.8902366 , 127.0096324 , 128.1258409 , 129.2416211 , 130.3569744 , 131.4719021 , 132.5864056 , 133.6977359 , 134.8086468 , 135.9191397 , 137.0292157 , 138.1388764 , 139.248123 , 140.3542127 , 141.4598931 , 142.5651654 , 143.670031 , 144.771751 , 145.8758081 , 146.9767242 , 148.0772409 , 149.1773593 , 150.2770809 , 151.3736727 , 152.4698722 , 153.5684128 , 154.6638304 , 155.7561293 , 156.8507722 , 157.9423009 , 159.0361741 , 160.1269375 , 161.2173207 , 162.3046004 , 163.3942279 , 164.4807562 , 165.569633 , 166.655415 , 167.7408262 , 168.8231488 , 169.907823 , 170.9894129 , 172.0706388 , 173.1515019 , 174.2320034 , 175.3121445 , 176.3892139 , 177.4686389 , 178.5449963 , 179.621 , 180.6966513 , 181.7719512 , 182.8441938 , 183.9160892 , 184.9903442 , 186.061548 , 187.1324083 , 188.2002233 , 189.2704009 , 190.3375373 , 191.4043366 , 192.4707999 , 193.5369284 , 194.6027234 , 195.6681859 , 196.730621 , 197.7927276 , 198.854507 , 199.9159603 , 200.9770886 , 202.0378931 , 203.0956837 , 204.1558449 , 205.212996 , 206.2698296 , 207.3263467 , 208.3798614 , 209.4357498 , 210.4886395 , 211.5412189 , 212.596173 , 213.6481343 , 214.6971062 , 215.7484553 , 216.7968189 , 217.8475605 , 218.8953202 , 219.94278 , 220.989941 , 222.0368044 , 223.0806949 , 224.126967 , 225.17027 , 226.2132811 , 227.2560016 , 228.2984324 , 229.3405747 , 230.3797585 , 231.4213278 , 232.4599422 , 233.4982738 , 234.5363238 , 235.5740932 , 236.611583 , 237.6461285 , 238.6830632 , 239.717057 , 240.750777 , 241.7842241 , 242.8173996 , 243.8503043 , 244.8829395 , 245.9126456 , 246.9447453 , 247.9739195 , 249.0028297 , 250.0314768 , 251.059862 , 252.0853298 , 253.1131948 , 254.1381459 , 255.1654947 , 256.1899332 , 257.2141171 , 258.2380477 , 259.2590741 , 260.2825015 , 261.3056785 , 262.3259564 , 263.3459872 , 264.365772 , 265.3853118 , 266.4046076 , 267.4236603 , 268.442471 , 269.4583954 , 270.4740811 , 271.4921731 , 272.5073836 , 273.5223584 , 274.5370983 , 275.5489629 , 276.5632369 , 277.5772792 , 278.5884507 , 279.5993937 , 280.610109 , 281.6205976 , 282.6308606 , 283.6408989 , 284.6507134 , 285.6576694 , 286.66704 , 287.6735552 , 288.6798517 , 289.6859304 , 290.6917924 , 291.6974387 , 292.7028701 , 293.7054564 , 294.7104616 , 295.7126247 , 296.7172075 , 297.7189513 , 298.7204871 , 299.7218158 , 300.7229385 , 301.723856 , 302.7219431 , 303.7224538 , 304.7201369 , 305.7176197 , 306.7175274 , 307.7146119 , 308.711499 , 309.7081895 , 310.7020623 , 311.698363 , 312.69447 , 313.6877635 , 314.6808662 , 315.6763986 , 316.6691218 , 317.6616569 , 318.6540048 , 319.6461665 , 320.6355257 , 321.6273181 , 322.618927 , 323.6077376 , 324.5963675 , 325.5874323 , 326.5757031 , 327.5637958 , 328.5517115 , 329.539451 , 330.5244029 , 331.5117932 , 332.4990101 , 333.4834435 , 334.4703166 , 335.4544089 , 336.4383323 , 337.4220876 , 338.4056758 , 339.3890977 , 340.3723542 , 341.3554462 , 342.3357677 , 343.3185339 , 344.2985321 , 345.2809761 , 346.2606547 , 347.2401749 , 348.2195378 , 349.2013482 , 350.1777949 , 351.1566908 , 352.1354328 , 353.1140218 , 354.0898567 , 355.0681427 , 356.0436772 , 357.0216637 , 357.9969012 , 358.9719917 , 359.9469362 , 360.9217353 , 361.8963901 , 362.8709015 , 363.8452702 , 364.8194971 , 365.7935831 , 366.7649321 , 367.7387392 , 368.7098118 , 369.6807477 , 370.6541433 , 371.6248079 , 372.5953384 , 373.5657357 , 374.5360004 , 375.5061336 , 376.4761361 , 377.4434156 , 378.4131595 , 379.3827752 , 380.3496716 , 381.3190338 , 382.2856791 , 383.2522002 , 384.2211888 , 385.1874639 , 386.1536175 , 387.1196503 , 388.0855632 , 389.051357 , 390.0170326 , 390.9825909 , 391.9454444 , 392.9107707 , 393.8759822 , 394.8384924 , 395.8034769 , 396.7657625 , 397.7279372 , 398.6925878 , 399.6545428 , 400.6163894 , 401.5781283 , 402.5397606 , 403.5012868 , 404.462708 , 405.4240249 , 406.3852383 , 407.3463491 , 408.3047749 , 409.2656832 , 410.2264913 , 411.1846177 , 412.1452283 , 413.1031593 , 414.0609939 , 415.0213145 , 415.9789586 , 416.9365089 , 417.8939662 , 418.8513312 , 419.8086047 , 420.7657877 , 421.7228808 , 422.6798849 , 423.6368009 , 424.5936295 , 425.5503716 , 426.5044494 , 427.4610211 , 428.4175086 , 429.3713351 , 430.3276571 , 431.28132 , 432.2374797 , 433.1909824 , 434.146983 , 435.1003287 , 436.0535972 , 437.0067893 , 437.9624817 , 438.9155232 , 439.8684907 , 440.821385 , 441.774207 , 442.7269574 , 443.679637 , 444.6322467 , 445.5847872 , 446.5346851 , 447.4870899 , 448.4394279 , 449.3916999 , 450.3413333 , 451.293476 , 452.2455551 , 453.1949986 , 454.1469531 , 455.0962739 , 456.0481071 , 456.9973085 , 457.9490235 , 458.8981086 , 459.8471368 , 460.7986806 , 461.7475974 , 462.6964596 , 463.6478394 , 464.5965949 , 465.5452983 , 466.4939503 , 467.4425519 , 468.3936742 , 469.342177 , 470.2906316 , 471.2390388 , 472.1873994 , 473.1357143 , 474.0839842 , 475.0322099 , 475.9803921 , 476.9259624 , 477.8740603 , 478.8221172 , 479.7701338 , 480.7181109 , 481.6660494 , 482.61395 , 483.5592448 , 484.507072 , 485.4548638 , 486.4000524 , 487.3477755 , 488.2954655 , 489.2431231 , 490.188181 , 491.1357763 , 492.0807737 , 493.0283098 , 493.9758175 , 494.9207298 , 495.8681831 , 496.8130427 , 497.7604446 , 498.707822 , 499.6526083 , 500.5999391 , 501.5446805 , 502.4919678 , 503.4366675 , 504.3839145 , 505.3285755 , 506.2757852 , 507.2204107 , 508.1675864 , 509.1121795 , 510.0593243 , 511.0038881 , 511.9484385 , 512.8955428 , 513.8400686 , 514.7871498 , 515.7316541 , 516.6787154 , 517.6232014 , 518.5702459 , 519.5147166 , 520.4617475 , 521.4062061 , 522.35066 , 523.2976761 , 524.2421226 , 525.1891329 , 526.1335751 , 527.0805827 , 528.0250237 , 528.9720317 , 529.9164747 , 530.8634863 , 531.8079344 , 532.7549527 , 533.699409 , 534.6464371 , 535.5909048 , 536.5379459 , 537.4849946 , 538.4294852 , 539.3765516 , 540.3210614 , 541.2681486 , 542.2126807 , 543.1597918 , 544.1069161 , 545.0514875 , 545.9986404 , 546.9458088 , 547.8929935 , 548.8376284 , 549.784848 , 550.7320863 , 551.6767769 , 552.6240548 , 553.5713537 , 554.5186744 , 555.4660177 , 556.4133844 , 557.3607753 , 558.3056235 , 559.2530649 , 560.2005329 , 561.1480282 , 562.0955516 , 563.0431038 , 563.9906858 , 564.9382982 , 565.8859418 , 566.8361857 , 567.7838943 , 568.7316365 , 569.679413 , 570.6272248 , 571.5776413 , 572.5255259 , 573.473448 , 574.4239776 , 575.3719773 , 576.320017 , 577.2706667 , 578.2187887 , 579.1695226 , 580.11773 , 581.0685512 , 582.0168473 , 582.967759 , 583.9187172 , 584.8671523 , 585.8182057 , 586.769308 , 587.72046 , 588.6690916 , 589.6203452 , 590.5716508 , 591.5230094 , 592.4744216 , 593.4258883 , 594.3774102 , 595.3289882 , 596.2831951 , 597.2348878 , 598.186639 , 599.1384494 , 600.0928926 , 601.0448241 , 601.9968172 , 602.951446 , 603.9035651 , 604.8583218 , 605.8105699 , 606.7654577 , 607.7204122 , 608.6728602 , 609.6279506 , 610.5831104 , 611.5383402 , 612.4936408 , 613.4490132 , 614.404458 , 615.3599761 , 616.3155684 , 617.2712355 , 618.2269784 , 619.1853743 , 620.1412713 , 621.0972464 , 622.0558776 , 623.0120118 , 623.970804 , 624.9271004 , 625.8860569 , 626.845097 , 627.8042213 , 628.7608522 , 629.7201474 , 630.6795293 , 631.6389989 , 632.5985568 , 633.558204 , 634.5205214 , 635.4803497 , 636.4402698 , 637.4002824 , 638.3629694 , 639.3231698 , 640.2860467 , 641.2490199 , 642.2095083 , 643.1726765 , 644.1359434 , 645.0993101 , 646.0627772 , 647.0263457 , 647.9900163 , 648.9537899 , 649.9176674 , 650.8816494 , 651.8483219 , 652.8125161 , 653.7794028 , 654.7438125 , 655.7109169 , 656.6781317 , 657.6428709 , 658.6103084 , 659.5778587 , 660.5455226 , 661.513301 , 662.4811948 , 663.4517931 , 664.4199203 , 665.3881654 , 666.3591186 , 667.3276023 , 668.2987965 , 669.2701125 , 670.2389606 , 671.2105227 , 672.1822093 , 673.1540212 , 674.1259593 , 675.0980244 , 676.0728101 , 677.0451323 , 678.017584 , 678.99276 , 679.9680681 , 680.9409146 , 681.9164889 , 682.8921979 , 683.8680425 , 684.8440234 , 685.8201415 , 686.7963978 , 687.7727931 , 688.7519256 , 689.7286019 , 690.7080178 , 691.6849786 , 692.6646815 , 693.6445295 , 694.6245233 , 695.6046639 , 696.5849522 , 697.565389 , 698.5459752 , 699.5293133 , 700.5102013 , 701.4938439 , 702.4750374 , 703.4589879 , 704.4430941 , 705.4273566 , 706.4117765 , 707.3963547 , 708.3810919 , 709.3685949 , 710.3536536 , 711.3414806 , 712.3268642 , 713.3150189 , 714.303339 , 715.2918254 , 716.280479 , 717.2693007 , 718.2582914 , 719.2500622 , 720.2393941 , 721.2315089 , 722.2211857 , 723.213648 , 724.2062858 , 725.1990999 , 726.1920913 , 727.1852608 , 728.1786094 , 729.1747528 , 730.1684629 , 731.1649706 , 732.161662 , 733.1585382 , 734.1555999 , 735.1528482 , 736.150284 , 737.1479083 , 738.1483411 , 739.1463455 , 740.1471613 , 741.1455496 , 742.1467523 , 743.14815 , 744.1497438 , 745.1541573 , 746.1561464 , 747.1583345 , 748.1633467 , 749.1685608 , 750.1739776 , 751.1769723 , 752.1854235 , 753.1914545 , 754.1976921 , 755.2041374 , 756.2134198 , 757.2229128 , 758.2299877 , 759.2399044 , 760.2500346 , 761.2630108 , 762.2735716 , 763.2843489 , 764.2979767 , 765.3091904 , 766.3232578 , 767.3375468 , 768.3520583 , 769.3667933 , 770.3843894 , 771.399575 , 772.4176249 , 773.4332652 , 774.4517728 , 775.4705112 , 776.4894812 , 777.5086839 , 778.5307617 , 779.5504335 , 780.5729835 , 781.5957715 , 782.6187984 , 783.6420653 , 784.6655733 , 785.6893233 , 786.7133163 , 787.7402003 , 788.7673307 , 789.7947084 , 790.8223346 , 791.8502103 , 792.8783364 , 793.9067141 , 794.9379957 , 795.9695323 , 797.0013248 , 798.0333744 , 799.065682 , 800.0982486 , 801.1310755 , 802.1668196 , 803.2028273 , 804.2364423 , 805.2729797 , 806.3124428 , 807.3495156 , 808.3868573 , 809.4271298 , 810.4676749 , 811.5084934 , 812.5495866 , 813.5909554 , 814.632601 , 815.6771896 , 816.7193927 , 817.7645423 , 818.8099745 , 819.8556903 , 820.9016909 , 821.9506467 , 822.9972207 , 824.0467537 , 825.0965772 , 826.1466925 , 827.1971006 , 828.2478027 , 829.3014744 , 830.3554438 , 831.4070358 , 832.4616032 , 833.5164716 , 834.5743206 , 835.6297952 , 836.6882543 , 837.7470205 , 838.8060949 , 839.8654788 , 840.9251731 , 841.9851792 , 843.0481828 , 844.111502 , 845.175138 , 846.2390918 , 847.3033647 , 848.3679577 , 849.4355617 , 850.5034899 , 851.5717434 , 852.6403235 , 853.7092312 , 854.7811615 , 855.8507289 , 856.923323 , 857.9962512 , 859.0695146 , 860.1431145 , 861.2197509 , 862.2940282 , 863.371346 , 864.4490069 , 865.5270119 , 866.6080654 , 867.6867633 , 868.7685139 , 869.8506153 , 870.9330687 , 872.0158754 , 873.0990365 , 874.1852627 , 875.2718475 , 876.3587924 , 877.4460985 , 878.533767 , 879.624513 , 880.7129112 , 881.8043912 , 882.8962404 , 883.9884603 , 885.0837703 , 886.1767361 , 887.2727964 , 888.3692342 , 889.4660509 , 890.5659707 , 891.6635498 , 892.7642364 , 893.8653089 , 894.9667686 , 896.0686168 , 897.1708546 , 898.2762134 , 899.3819664 , 900.488115 , 901.5946606 , 902.7043382 , 903.8116827 , 904.9221637 , 906.033049 , 907.1443398 , 908.2587762 , 909.3708832 , 910.4861405 , 911.6018108 , 912.7178954 , 913.8343957 , 914.9540578 , 916.0741404 , 917.1946448 , 918.3155724 , 919.4369247 , 920.5614529 , 921.6836595 , 922.8090469 , 923.9376197 , 925.0638741 , 926.1933187 , 927.3204457 , 928.4507678 , 929.5815314 , 930.7154974 , 931.8471492 , 932.9820083 , 934.1173168 , 935.2530759 , 936.3920523 , 937.5287183 , 938.6686066 , 939.8089536 , 940.9497608 , 942.0938002 , 943.238305 , 944.3805037 , 945.5287165 , 946.6746262 , 947.8210071 , 948.9706382 , 950.1207458 , 951.2713315 , 952.4223966 , 953.5767249 , 954.731538 , 955.8868375 , 957.042625 , 958.2016886 , 959.3584576 , 960.5185082 , 961.6790553 , 962.8401002 , 964.0044375 , 965.1692782 , 966.3346239 , 967.5004761 , 968.6668366 , 969.8365055 , 971.0066883 , 972.1773863 , 973.3486013 , 974.5231386 , 975.6953934 , 976.8709762 , 978.0498924 , 979.2265296 , 980.406506 , 981.5870157 , 982.7680601 , 983.9496409 , 985.1345751 , 986.3172351 , 987.5032542 , 988.6926381 , 989.8797514 , 991.0702354 , 992.2612726 , 993.4528646 , 994.6450133 , 995.8405472 , 997.0366438 , 998.2333046 , 999.4305315 , 1000.631158 , 1001.832357 , 1003.03413 , 1004.236479 , 1005.442243 , 1006.645749 , 1007.852676 , 1009.063031 , 1010.271133 , 1011.482667 , 1012.694796 , 1013.907519 , 1015.120839 , 1016.337608 , 1017.554981 , 1018.772959 , 1019.994398 , 1021.213592 , 1022.436255 , 1023.659533 , 1024.886288 , 1026.110804 , 1027.338803 , 1028.567428 , 1029.799546 , 1031.029429 , 1032.262812 , 1033.496831 , 1034.73436 , 1035.969657 , 1037.208472 , 1038.447933 , 1039.69092 , 1040.931681 , 1042.175976 , 1043.420927 , 1044.669422 , 1045.915695 , 1047.165519 , 1048.41601 , 1049.670062 , 1050.921897 , 1052.1773 , 1053.433381 , 1054.69304 , 1055.953386 , 1057.214419 , 1058.476143 , 1059.738559 , 1061.004574 , 1062.271288 , 1063.538704 , 1064.809733 , 1066.07856 , 1067.351007],
                              [72.08925996 , 71.73682008 , 71.38696088 , 71.03968216 , 70.68983899 , 70.34514838 , 69.99789326 , 69.65579027 , 69.31112269 , 68.96903477 , 68.6295263 , 68.29002514 , 67.95310313 , 67.61618825 , 67.27928045 , 66.94495143 , 66.61062932 , 66.27888571 , 65.94714884 , 65.61541867 , 65.28883811 , 64.95969255 , 64.63312486 , 64.30656354 , 63.98257983 , 63.65860231 , 63.33720214 , 63.01580801 , 62.69699097 , 62.37817981 , 62.05937451 , 61.74314597 , 61.42692314 , 61.11327682 , 60.79963606 , 60.48857157 , 60.17494178 , 59.86645879 , 59.55798106 , 59.24950857 , 58.94361182 , 58.63772016 , 58.33183357 , 58.02852244 , 57.72521623 , 57.42448526 , 57.12375908 , 56.82560793 , 56.52746144 , 56.22931959 , 55.93375248 , 55.63818989 , 55.34520184 , 55.05221818 , 54.75923888 , 54.46883386 , 54.17843308 , 53.89060639 , 53.60278382 , 53.31753514 , 53.02972072 , 52.74704979 , 52.46181307 , 52.17914994 , 51.89906027 , 51.61897437 , 51.33889221 , 51.06138327 , 50.78387797 , 50.50637629 , 50.23144761 , 49.95652243 , 49.68417009 , 49.41182115 , 49.1394756 , 48.86970267 , 48.59993302 , 48.33016665 , 48.06297267 , 47.79578187 , 47.53116332 , 47.26654785 , 47.00450447 , 46.73989506 , 46.48042664 , 46.21839216 , 45.95892953 , 45.69946976 , 45.44258171 , 45.18569642 , 44.92881389 , 44.67450288 , 44.42019454 , 44.16845759 , 43.91672322 , 43.66499143 , 43.41583085 , 43.16667275 , 42.91751714 , 42.67093256 , 42.42435038 , 42.17777059 , 41.93376167 , 41.68975505 , 41.44831918 , 41.20688555 , 40.96545414 , 40.72402493 , 40.48516628 , 40.2488781 , 40.01002369 , 39.77373967 , 39.54002598 , 39.30374604 , 39.07003635 , 38.83889684 , 38.60775925 , 38.37662357 , 38.14548978 , 37.91692599 , 37.68836404 , 37.46237199 , 37.23381365 , 37.01039318 , 36.7844064 , 36.56098936 , 36.33757401 , 36.1167283 , 35.89588422 , 35.67504175 , 35.45420089 , 35.23592953 , 35.01765971 , 34.80195929 , 34.58626036 , 34.37056292 , 34.15743477 , 33.94430804 , 33.73118274 , 33.51805886 , 33.30750412 , 33.09695074 , 32.88896642 , 32.68098341 , 32.47300171 , 32.26502131 , 32.05960984 , 31.85419962 , 31.65135824 , 31.44851807 , 31.24567909 , 31.0428413 , 30.84257224 , 30.64230432 , 30.44203754 , 30.24433939 , 30.04664233 , 29.84894635 , 29.65381892 , 29.45869252 , 29.26356717 , 29.07101026 , 28.87845435 , 28.68589944 , 28.4933455 , 28.30335991 , 28.11337526 , 27.92595889 , 27.73854343 , 27.55112887 , 27.3637152 , 27.17886972 , 26.99402509 , 26.80918132 , 26.62433839 , 26.44206355 , 26.26235676 , 26.08008353 , 25.90037831 , 25.72067386 , 25.54097018 , 25.36383445 , 25.18669945 , 25.00956518 , 24.83499878 , 24.65786595 , 24.48586809 , 24.31130378 , 24.13930726 , 23.9673114 , 23.7953162 , 23.62588872 , 23.45646188 , 23.28703565 , 23.11761005 , 22.9507521 , 22.78389474 , 22.61703796 , 22.45274878 , 22.28846017 , 22.12417211 , 21.9598846 , 21.79816462 , 21.63644516 , 21.4772932 , 21.31557478 , 21.15642382 , 20.99727336 , 20.84069032 , 20.68154083 , 20.52495874 , 20.37094403 , 20.21436286 , 20.06034904 , 19.90633566 , 19.75232272 , 19.60087708 , 19.44943186 , 19.29798705 , 19.14654266 , 18.99766551 , 18.84878875 , 18.69991238 , 18.55360321 , 18.40729441 , 18.26098598 , 18.11467791 , 17.97093699 , 17.82719642 , 17.68345619 , 17.5397163 , 17.39854351 , 17.25737105 , 17.11619891 , 16.97502708 , 16.83642232 , 16.69781785 , 16.55921369 , 16.42317655 , 16.28713969 , 16.15110311 , 16.01506681 , 15.87903079 , 15.74556174 , 15.61209295 , 15.47862442 , 15.34772285 , 15.21682151 , 15.08592042 , 14.95501956 , 14.82668562 , 14.6983519 , 14.57001841 , 14.44168514 , 14.31591875 , 14.18758591 , 14.06181994 , 13.93862083 , 13.81285526 , 13.68965654 , 13.566458 , 13.44582629 , 13.32262814 , 13.20199679 , 13.08136562 , 12.96073462 , 12.84267042 , 12.72460637 , 12.6065425 , 12.48847878 , 12.37041523 , 12.25491842 , 12.13942177 , 12.02392527 , 11.91099551 , 11.79806588 , 11.68513639 , 11.57220704 , 11.45927782 , 11.34891531 , 11.23855293 , 11.12819068 , 11.01782854 , 10.9100331 , 10.80223776 , 10.69444255 , 10.58664745 , 10.481419 , 10.37619067 , 10.27096244 , 10.16573432 , 10.0605063 , 9.957844918 , 9.855183632 , 9.752522443 , 9.652427875 , 9.549766873 , 9.449672485 , 9.349578185 , 9.252050489 , 9.151956359 , 9.054428827 , 8.956901375 , 8.859374001 , 8.764413212 , 8.666885991 , 8.571925349 , 8.47953128 , 8.384570779 , 8.292176845 , 8.197216481 , 8.104822679 , 8.014995432 , 7.922601756 , 7.83277463 , 7.742947563 , 7.653120553 , 7.563293601 , 7.476033187 , 7.388772826 , 7.301512517 , 7.214252259 , 7.126992052 , 7.04229837 , 6.957604734 , 6.872911145 , 6.790784071 , 6.706090571 , 6.623963583 , 6.541836637 , 6.459709731 , 6.377582866 , 6.298022503 , 6.218462177 , 6.138901887 , 6.059341633 , 5.982347873 , 5.902787689 , 5.825793995 , 5.751366788 , 5.674373156 , 5.597379556 , 5.522952438 , 5.448525348 , 5.374098287 , 5.302237702 , 5.227810694 , 5.15595016 , 5.084089651 , 5.012229166 , 4.942935149 , 4.873641155 , 4.801780739 , 4.732486788 , 4.6657593 , 4.59646539 , 4.529737941 , 4.46301051 , 4.396283098 , 4.329555704 , 4.265394764 , 4.198667405 , 4.134506499 , 4.070345608 , 4.008751168 , 3.944590308 , 3.882995896 , 3.821401498 , 3.759807113 , 3.698212741 , 3.639184814 , 3.580156898 , 3.521128994 , 3.462101102 , 3.40307322 , 3.346611779 , 3.287583918 , 3.231122497 , 3.177227512 , 3.120766109 , 3.064304715 , 3.010409757 , 2.956514806 , 2.902619864 , 2.851291355 , 2.797396427 , 2.746067932 , 2.694739443 , 2.643410961 , 2.592082486 , 2.54332044 , 2.4945584 , 2.445796365 , 2.397034335 , 2.348272311 , 2.299510292 , 2.2533147 , 2.207119112 , 2.160923528 , 2.114727949 , 2.071098795 , 2.027469644 , 1.983840497 , 1.940211353 , 1.896582213 , 1.852953076 , 1.811890362 , 1.770827651 , 1.729764942 , 1.688702236 , 1.650205952 , 1.609143251 , 1.570646972 , 1.532150694 , 1.493654419 , 1.457724564 , 1.419228292 , 1.38329844 , 1.34736859 , 1.311438742 , 1.275508895 , 1.242145467 , 1.208782041 , 1.172852198 , 1.142055191 , 1.108691768 , 1.075328347 , 1.044531343 , 1.01373434 , 0.982937338 , 0.952140337 , 0.923909754 , 0.893112754 , 0.864882172 , 0.83665159 , 0.808421009 , 0.780190429 , 0.754526266 , 0.728862103 , 0.70319794 , 0.677533778 , 0.651869616 , 0.628771871 , 0.60310771 , 0.580009966 , 0.556912221 , 0.533814477 , 0.51328315 , 0.490185406 , 0.469654079 , 0.449122752 , 0.428591425 , 0.410626514 , 0.390095187 , 0.372130276 , 0.354165366 , 0.336200455 , 0.318235545 , 0.30283705 , 0.28487214 , 0.269473646 , 0.254075151 , 0.241243073 , 0.225844578 , 0.210446084 , 0.197614006 , 0.184781927 , 0.171949849 , 0.161684186 , 0.148852108 , 0.138586445 , 0.128320783 , 0.11805512 , 0.107789457 , 0.097523795 , 0.089824548 , 0.082125301 , 0.074426054 , 0.066726807 , 0.05902756 , 0.053894729 , 0.048761897 , 0.043629066 , 0.038496235 , 0.033363403 , 0.028230572 , 0.025664156 , 0.023097741 , 0.020531325 , 0.01796491 , 0.015398494 , 0.015398494 , 0.015398494 , 0.015398494 , 0.015398494 , 0.015398494 , 0.015398494 , 0.01796491 , 0.020531325 , 0.023097741 , 0.025664156 , 0.030796988 , 0.033363403 , 0.038496235 , 0.043629066 , 0.048761897 , 0.053894729 , 0.061593976 , 0.069293222 , 0.074426054 , 0.082125301 , 0.092390963 , 0.10009021 , 0.110355873 , 0.11805512 , 0.128320783 , 0.141152861 , 0.151418524 , 0.161684186 , 0.174516265 , 0.187348343 , 0.200180422 , 0.2130125 , 0.228410994 , 0.241243073 , 0.256641567 , 0.272040061 , 0.287438556 , 0.305403466 , 0.320801961 , 0.338766871 , 0.356731782 , 0.374696692 , 0.392661603 , 0.41319293 , 0.433724256 , 0.451689168 , 0.472220495 , 0.495318238 , 0.515849566 , 0.538947309 , 0.559478637 , 0.582576382 , 0.608240542 , 0.631338287 , 0.654436033 , 0.680100194 , 0.705764356 , 0.731428519 , 0.757092682 , 0.785323262 , 0.810987426 , 0.839218007 , 0.867448588 , 0.895679171 , 0.92647617 , 0.954706754 , 0.985503755 , 1.016300757 , 1.04709776 , 1.080461181 , 1.111258186 , 1.144621609 , 1.177985033 , 1.211348458 , 1.244711885 , 1.28064173 , 1.31400516 , 1.349935008 , 1.385864858 , 1.42179471 , 1.460290982 , 1.496220837 , 1.534717113 , 1.57321339 , 1.61170967 , 1.652772371 , 1.691268655 , 1.732331361 , 1.77339407 , 1.814456782 , 1.855519496 , 1.899148633 , 1.940211353 , 1.983840497 , 2.027469644 , 2.073665215 , 2.11729437 , 2.163489949 , 2.209685533 , 2.255881121 , 2.302076714 , 2.348272311 , 2.397034335 , 2.445796365 , 2.4945584 , 2.54332044 , 2.592082486 , 2.643410961 , 2.694739443 , 2.746067932 , 2.797396427 , 2.848724929 , 2.902619864 , 2.956514806 , 3.010409757 , 3.064304715 , 3.118199682 , 3.174661085 , 3.231122497 , 3.287583918 , 3.34404535 , 3.400506791 , 3.459534672 , 3.515996134 , 3.575024037 , 3.634051951 , 3.695646309 , 3.754674248 , 3.816268632 , 3.877863029 , 3.939457439 , 4.003618299 , 4.065212738 , 4.129373627 , 4.193534532 , 4.257695453 , 4.32185639 , 4.388583782 , 4.455311192 , 4.522038621 , 4.588766068 , 4.655493534 , 4.724787462 , 4.79408141 , 4.86337538 , 4.932669371 , 5.001963384 , 5.073823866 , 5.145684371 , 5.217544902 , 5.289405457 , 5.361266038 , 5.435693094 , 5.510120179 , 5.584547292 , 5.658974434 , 5.735968059 , 5.81039526 , 5.887388947 , 5.964382667 , 6.043942879 , 6.120936665 , 6.200496947 , 6.280057265 , 6.359617619 , 6.439178011 , 6.521304907 , 6.603431843 , 6.685558821 , 6.76768584 , 6.849812901 , 6.934506478 , 7.019200101 , 7.10389377 , 7.188587487 , 7.27584773 , 7.360541544 , 7.447801888 , 7.537628768 , 7.624889219 , 7.714716211 , 7.801976772 , 7.891803877 , 7.984197533 , 8.074024759 , 8.166418539 , 8.258812385 , 8.351206295 , 8.443600272 , 8.538560817 , 8.633521433 , 8.728482122 , 8.823442883 , 8.918403718 , 9.015931139 , 9.11345864 , 9.21098622 , 9.308513882 , 9.408608147 , 9.508702498 , 9.608796938 , 9.708891467 , 9.808986085 , 9.911647327 , 10.01430867 , 10.1169701 , 10.22219818 , 10.32485981 , 10.4300881 , 10.53531648 , 10.64311153 , 10.74834014 , 10.85613541 , 10.96393081 , 11.07172631 , 11.18208851 , 11.28988426 , 11.4002467 , 11.51060927 , 11.62353854 , 11.73390138 , 11.84683093 , 11.95976062 , 12.07525703 , 12.18818701 , 12.30368372 , 12.41918059 , 12.53467761 , 12.65274139 , 12.77080533 , 12.88886943 , 13.00693371 , 13.12756477 , 13.24562938 , 13.36626079 , 13.48945902 , 13.6100908 , 13.7332894 , 13.85648819 , 13.97968718 , 14.10288636 , 14.2286524 , 14.35441864 , 14.4801851 , 14.60595177 , 14.73428532 , 14.8626191 , 14.99095311 , 15.12185403 , 15.25018849 , 15.38108989 , 15.51199153 , 15.64546013 , 15.77892898 , 15.90983137 , 16.04586746 , 16.1793371 , 16.31537373 , 16.45141065 , 16.58744784 , 16.72348532 , 16.86208984 , 17.00069466 , 17.13929978 , 17.28047198 , 17.41907772 , 17.56025055 , 17.70399049 , 17.84516397 , 17.98890459 , 18.13264554 , 18.27638685 , 18.42269532 , 18.56900416 , 18.71531336 , 18.86162294 , 19.01049973 , 19.15937691 , 19.30825449 , 19.45713246 , 19.6085777 , 19.76002336 , 19.91146943 , 20.06548283 , 20.21949666 , 20.37351093 , 20.52752566 , 20.68154083 , 20.83812339 , 20.99470642 , 21.15385687 , 21.31300782 , 21.47215927 , 21.63131122 , 21.79046368 , 21.95218363 , 22.11390412 , 22.27562514 , 22.43991372 , 22.60420285 , 22.76849255 , 22.93534985 , 23.0996407 , 23.26649918 , 23.43592533 , 23.60278503 , 23.77221241 , 23.94164043 , 24.11363619 , 24.28563261 , 24.4576297 , 24.62962745 , 24.80419302 , 24.97619214 , 25.15332627 , 25.32789395 , 25.50502954 , 25.68216587 , 25.85930294 , 26.03900798 , 26.2187138 , 26.3984204 , 26.58069505 , 26.76040324 , 26.94524679 , 27.1275239 , 27.31236915 , 27.49721526 , 27.68206223 , 27.86947742 , 28.05689352 , 28.24431052 , 28.43172845 , 28.62171468 , 28.81170187 , 29.00425744 , 29.19424657 , 29.38680412 , 29.58193012 , 29.77705718 , 29.96961779 , 30.16731443 , 30.36244464 , 30.56014345 , 30.75784336 , 30.95811194 , 31.15838167 , 31.35865255 , 31.56149219 , 31.76176542 , 31.96460746 , 32.17001835 , 32.3754305 , 32.58084391 , 32.7862586 , 32.99424227 , 33.20222726 , 33.41021358 , 33.62076899 , 33.83132578 , 34.04188396 , 34.25501134 , 34.46814015 , 34.68127042 , 34.89697 , 35.11010321 , 35.32837368 , 35.54407779 , 35.76235137 , 35.98319447 , 36.20147123 , 36.42231757 , 36.64316555 , 36.86658321 , 37.09000258 , 37.31342365 , 37.53941454 , 37.76540719 , 37.99140163 , 38.21996601 , 38.44853224 , 38.67710033 , 38.90823849 , 39.13937858 , 39.3705206 , 39.60423284 , 39.83794708 , 40.07166333 , 40.30794995 , 40.54423865 , 40.78052944 , 41.01939075 , 41.25825422 , 41.49968833 , 41.73855619 , 41.98256326 , 42.22400412 , 42.4680158 , 42.71202982 , 42.9586148 , 43.20520219 , 43.45179201 , 43.70095296 , 43.95011641 , 44.19928238 , 44.45101965 , 44.70275952 , 44.95707083 , 45.21138482 , 45.46570152 , 45.72002093 , 45.976912 , 46.23637482 , 46.49327154 , 46.75274011 , 47.01478062 , 47.27682412 , 47.53887062 , 47.80092013 , 48.06554183 , 48.33273583 , 48.59736381 , 48.86456417 , 49.13433704 , 49.40411321 , 49.67389271 , 49.94367554 , 50.21603115 , 50.49095963 , 50.76589165 , 51.0408272 , 51.31576632 , 51.5932786 , 51.87079455 , 52.15088384 , 52.4309769 , 52.71364348 , 52.9937442 , 53.27898833 , 53.56166665 , 53.84691877 , 54.13474484 , 54.42257504 , 54.71040942 , 54.99824798 , 55.28866079 , 55.58164799 , 55.8746396 , 56.16763564 , 56.46320633 , 56.75878158 , 57.05436141 , 57.35251616 , 57.65067563 , 57.95141022 , 58.25214966 , 58.55289396 , 58.85621368 , 59.16210897 , 59.46543876 , 59.77134426 , 60.07982563 , 60.38831227 , 60.69680422 , 61.00787234 , 61.31894591 , 61.63259588 , 61.94625145 , 62.26248367 , 62.57872163 , 62.89496537 , 63.21378608 , 63.53261272 , 63.85401658 , 64.17542653 , 64.4968426 , 64.82083622 , 65.14483612 , 65.47141385 , 65.80056957 , 66.12716026 , 66.45632913 , 66.78807635 , 67.11983036 , 67.45159119 , 67.78593074 , 68.12027728 , 68.45720283 , 68.79413555 , 69.13364757 , 69.47316695 , 69.81526591 , 70.15737242 , 70.4994865 , 70.84418058 , 71.18888242 , 71.53616455 , 71.88345465 , 72.23332536 , 72.58320422 , 72.93566401 , 73.28813217 , 73.64060872 , 73.99566664 , 74.35330613 , 74.7109544 , 75.06861149 , 75.4288506 , 75.78909875 , 76.15192927 , 76.51476904 , 76.88019153 , 77.24562349 , 77.61363853 , 77.98166327 , 78.35227145 , 78.72288956 , 79.09351764 , 79.46930356 , 79.84252586 , 80.21833246 , 80.59672364 , 80.97512546 , 81.35353797 , 81.73453558 , 82.11554412 , 82.49913814 , 82.88531794 , 83.27150913 , 83.65771177 , 84.04650071 , 84.43530137 , 84.82668875]
                              ]).T
            elif image.xdim() == 384:
                '''
                # old calib
                m = np.array([[0 , 1.122709249 , 2.244414581 , 3.363854021 , 4.482305091 , 5.598507977 , 6.71373788 , 7.828001339 , 8.938778591 , 10.04986879 , 11.15875316 , 12.26544134 , 13.37120175 , 14.47478292 , 15.57745108 , 16.67921254 , 17.77881881 , 18.87753294 , 19.97410841 , 21.06855442 , 22.16338157 , 23.25609353 , 24.34669935 , 25.43645583 , 26.52412217 , 27.61095316 , 28.69695478 , 29.78088901 , 30.86400767 , 31.94507455 , 33.02533952 , 34.10480846 , 35.1822478 , 36.25890459 , 37.33354701 , 38.40742022 , 39.47929412 , 40.55164704 , 41.62077968 , 42.69040273 , 43.75805442 , 44.82374294 , 45.88870716 , 46.95295268 , 48.01525596 , 49.07685329 , 50.1377502 , 51.19672553 , 52.25501303 , 53.31261819 , 54.36832213 , 55.42335616 , 56.4777257 , 57.53021411 , 58.58205031 , 59.63201914 , 60.68134793 , 61.73004201 , 62.77688833 , 63.82432956 , 64.86871815 , 65.91371214 , 66.95688369 , 67.99945488 , 69.04021683 , 70.08039016 , 71.11998002 , 72.15899155 , 73.1962185 , 74.23287868 , 75.26897718 , 76.30330966 , 77.33709189 , 78.3703289 , 79.40302574 , 80.43398051 , 81.46440636 , 82.49310261 , 83.52248616 , 84.55015124 , 85.57730844 , 86.60275941 , 87.62891623 , 88.65337778 , 89.67615069 , 90.6996435 , 91.72145851 , 92.74280212 , 93.76367916 , 94.78289564 , 95.80285453 , 96.82116353 , 97.837829 , 98.85525069 , 99.87103941 , 100.8863971 , 101.9013286 , 102.9158385 , 103.9287373 , 104.9424187 , 105.9544994 , 106.9649855 , 107.9762678 , 108.9859657 , 109.9952767 , 111.0042054 , 112.0127565 , 113.0209345 , 114.0275544 , 115.0338114 , 116.03971 , 117.0452548 , 118.0504503 , 119.0541136 , 120.0586247 , 121.0616135 , 122.064272 , 123.0654187 , 124.0674307 , 125.0679406 , 126.069324 , 127.0692151 , 128.0688038 , 129.0680943 , 130.0659079 , 131.0646162 , 132.061857 , 133.0588182 , 134.0566866 , 135.0531017 , 136.0480691 , 137.0439565 , 138.0395865 , 139.0337828 , 140.0289114 , 141.0226158 , 142.0160809 , 143.0093111 , 144.0023109 , 144.9950844 , 145.9864576 , 146.9787922 , 147.9697356 , 148.9616487 , 149.9521797 , 150.9425112 , 151.9326474 , 152.9225928 , 153.9123515 , 154.901928 , 155.8913265 , 156.8793752 , 157.868431 , 158.8573216 , 159.844876 , 160.8322743 , 161.820696 , 162.8077948 , 163.7947504 , 164.7827417 , 165.7694235 , 166.7559748 , 167.7424 , 168.7287033 , 169.7148889 , 170.7009612 , 171.6869244 , 172.6716092 , 173.657367 , 174.6430285 , 175.6285979 , 176.6129063 , 177.5983043 , 178.5836229 , 179.5688665 , 180.5528663 , 181.5379724 , 182.5230161 , 183.5068291 , 184.4917608 , 185.4754703 , 186.4603069 , 187.4451023 , 188.4286884 , 189.413414 , 190.3981112 , 191.3827841 , 192.3662647 , 193.3509017 , 194.3355272 , 195.3201453 , 196.303588 , 197.2882041 , 198.2728255 , 199.2574565 , 200.2421012 , 201.226764 , 202.2114491 , 203.1961607 , 204.180903 , 205.1656802 , 206.1516692 , 207.1365291 , 208.1214368 , 209.107569 , 210.0925848 , 211.0788338 , 212.0639747 , 213.0503575 , 214.0368136 , 215.0221741 , 216.0087895 , 216.995491 , 217.9822828 , 218.9703428 , 219.9573282 , 220.9444169 , 221.9327868 , 222.9200947 , 223.9086927 , 224.8974113 , 225.8850803 , 226.8740527 , 227.8631586 , 228.8524022 , 229.8429629 , 230.832495 , 231.8233533 , 232.8131912 , 233.8043643 , 234.7957014 , 235.7872068 , 236.7788848 , 237.7707399 , 238.7639531 , 239.7561754 , 240.7497651 , 241.7435497 , 242.7375335 , 243.7317209 , 244.7261163 , 245.720724 , 246.7167273 , 247.7129524 , 248.7082241 , 249.7049054 , 250.7030015 , 251.7001574 , 252.6987375 , 253.6963855 , 254.6954674 , 255.6948068 , 256.6944085 , 257.6954587 , 258.6955984 , 259.6971962 , 260.6990748 , 261.7012386 , 262.7048757 , 263.7076237 , 264.7118549 , 265.71639 , 266.7212336 , 267.7263903 , 268.7330505 , 269.7400336 , 270.7461576 , 271.754987 , 272.7629664 , 273.772475 , 274.7811419 , 275.7925368 , 276.8030994 , 277.8140226 , 278.826501 , 279.8393501 , 280.8525748 , 281.8673708 , 282.8813609 , 283.8969327 , 284.9128996 , 285.9304596 , 286.9472316 , 287.965607 , 288.9843975 , 290.004803 , 291.0244385 , 292.0456996 , 293.067396 , 294.0907295 , 295.1145091 , 296.1387394 , 297.1634255 , 298.188572 , 299.2153839 , 300.2426671 , 301.2716276 , 302.299869 , 303.3297987 , 304.3602206 , 305.3923431 , 306.4237651 , 307.4581032 , 308.4917509 , 309.5271227 , 310.5630192 , 311.5994454 , 312.6364065 , 313.6751156 , 314.714371 , 315.7553872 , 316.7969613 , 317.8390983 , 318.8818035 , 319.926294 , 320.9713644 , 322.0170199 , 323.0644798 , 324.1125366 , 325.1611957 , 326.2116784 , 327.2627753 , 328.3157095 , 329.3680516 , 330.422243 , 331.4782913 , 332.5337629 , 333.5911038 , 334.6503216 , 335.7102011 , 336.7707479 , 337.8319674 , 338.8950904 , 339.9588987 , 341.0246247 , 342.0910488 , 343.1581766 , 344.2272429 , 345.2970258 , 346.3675308 , 347.4399956 , 348.5131956 , 349.58837 , 350.6642929 , 351.74097 , 352.8196432 , 353.8990839 , 354.9792981 , 356.0615303 , 357.1445494 , 358.2296019 , 359.315455 , 360.4021147 , 361.4908302 , 362.5816103 , 363.6719736 , 364.7644155 , 365.858945 , 366.954323 , 368.0505556 , 369.1488991 , 370.2481115 , 371.3481991 , 372.4504212 , 373.5547869 , 374.6587956 , 375.7662188 , 376.8732976 , 377.9838098 , 379.0939904 , 380.2063633 , 381.3196768 , 382.4351997 , 383.5529419 , 384.670384 , 385.7913263 , 386.9119817 , 388.0348893 , 389.1600593 , 390.2862314 , 391.4134124 , 392.5428816 , 393.6746493 , 394.807451],
                              [11.63250977 , 11.45546127 , 11.27958685 , 11.10488646 , 10.93136004 , 10.75900754 , 10.58782891 , 10.4178241 , 10.25016549 , 10.08250816 , 9.916024497 , 9.751886869 , 9.587750405 , 9.425959869 , 9.264170427 , 9.104726811 , 8.945284221 , 8.788187362 , 8.631091464 , 8.476341204 , 8.321591844 , 8.169188035 , 8.017957395 , 7.866727567 , 7.717843168 , 7.568959527 , 7.422421236 , 7.275883652 , 7.131691346 , 6.987499697 , 6.845653257 , 6.703807427 , 6.563134467 , 6.424806618 , 6.286479315 , 6.149324803 , 6.013343058 , 5.878534057 , 5.744897775 , 5.61243419 , 5.481143281 , 5.351025024 , 5.222079401 , 5.093134163 , 4.966533749 , 4.839933688 , 4.715678407 , 4.591423448 , 4.468341017 , 4.346431094 , 4.225693665 , 4.106128714 , 3.987736224 , 3.870516181 , 3.753296374 , 3.638421185 , 3.523546209 , 3.409843629 , 3.297313434 , 3.18595561 , 3.075770147 , 2.966757032 , 2.857744073 , 2.749903445 , 2.644407316 , 2.538911317 , 2.43458762 , 2.330264039 , 2.228284918 , 2.126305899 , 2.026671321 , 1.927036831 , 1.828574596 , 1.730112439 , 1.633994691 , 1.537877009 , 1.444103723 , 1.350330492 , 1.256557314 , 1.165128513 , 1.074871918 , 0.984615362 , 0.895531004 , 0.80761884 , 0.720878865 , 0.634138914 , 0.548571145 , 0.464175555 , 0.38095214 , 0.298900898 , 0.216849664 , 0.137142758 , 0.057435855 , -0.022271046 , -0.099633627 , -0.17699621 , -0.253186638 , -0.328204912 , -0.402051033 , -0.475897164 , -0.548571145 , -0.620072978 , -0.690402663 , -0.759560202 , -0.828717757 , -0.896703167 , -0.964688595 , -1.030329717 , -1.095970859 , -1.160439858 , -1.223736714 , -1.287033591 , -1.349158327 , -1.41011092 , -1.469891371 , -1.529671844 , -1.587108009 , -1.645716363 , -1.701980408 , -1.758244476 , -1.8133364 , -1.867256179 , -1.920003812 , -1.972751469 , -2.024326979 , -2.075902513 , -2.125133727 , -2.174364965 , -2.222424052 , -2.270483162 , -2.316197947 , -2.361912754 , -2.407627582 , -2.450998081 , -2.4943686 , -2.53773914 , -2.578765346 , -2.619791572 , -2.659645638 , -2.698327543 , -2.737009467 , -2.774519228 , -2.810856826 , -2.84719444 , -2.882359889 , -2.916353172 , -2.950346469 , -2.981995415 , -3.013644373 , -3.045293346 , -3.075770147 , -3.105074775 , -3.133207229 , -3.161339694 , -3.188299983 , -3.214088096 , -3.239876218 , -3.264492162 , -3.287935926 , -3.310207509 , -3.3324791 , -3.353578508 , -3.374677923 , -3.394605154 , -3.413360201 , -3.430943062 , -3.448525928 , -3.464936607 , -3.48134729 , -3.495413593 , -3.510652092 , -3.523546209 , -3.536440328 , -3.548162257 , -3.558711995 , -3.569261735 , -3.578639283 , -3.586844639 , -3.595049996 , -3.602083159 , -3.60794413 , -3.613805101 , -3.618493878 , -3.622010461 , -3.62435485 , -3.626699239 , -3.629043628 , -3.629043628 , -3.629043628 , -3.627871433 , -3.626699239 , -3.62435485 , -3.620838267 , -3.616149489 , -3.611460712 , -3.605599741 , -3.599738771 , -3.592705608 , -3.584500251 , -3.575122703 , -3.565745155 , -3.555195416 , -3.544645678 , -3.531751557 , -3.518857438 , -3.505963322 , -3.490724825 , -3.476658523 , -3.460247841 , -3.443837163 , -3.425082108 , -3.407499248 , -3.387572013 , -3.367644784 , -3.34771756 , -3.325445965 , -3.303174377 , -3.279730608 , -3.256286846 , -3.231670906 , -3.205882787 , -3.17892249 , -3.151962204 , -3.123829743 , -3.094525107 , -3.065220483 , -3.034743687 , -3.003094719 , -2.971445765 , -2.938624641 , -2.904631348 , -2.869465889 , -2.834300446 , -2.797962838 , -2.760453066 , -2.722943311 , -2.684261394 , -2.644407316 , -2.604553257 , -2.563527039 , -2.521328663 , -2.477958131 , -2.43458762 , -2.390044953 , -2.344330133 , -2.298615335 , -2.251728385 , -2.203669284 , -2.154438032 , -2.105206805 , -2.054803428 , -2.003227904 , -1.951652403 , -1.897732586 , -1.844984963 , -1.789893025 , -1.733628943 , -1.677364885 , -1.619928684 , -1.562492507 , -1.50271202 , -1.442931557 , -1.381978951 , -1.321026368 , -1.257729479 , -1.194432611 , -1.131135765 , -1.065494612 , -0.99985348 , -0.931868041 , -0.865054784 , -0.79589722 , -0.726739673 , -0.655237819 , -0.584908141 , -0.512234153 , -0.438388017 , -0.364541891 , -0.289523613 , -0.213333183 , -0.137142758 , -0.058608015 , 0.019926725 , 0.099633627 , 0.179340531 , 0.261391761 , 0.343443 , 0.426666409 , 0.511061992 , 0.596629753 , 0.682197533 , 0.770109659 , 0.858021811 , 0.947106155 , 1.037362696 , 1.127619274 , 1.220220221 , 1.312821215 , 1.406594424 , 1.501539854 , 1.596485344 , 1.693775233 , 1.791065194 , 1.889527399 , 1.989161856 , 2.089968572 , 2.191947555 , 2.295098813 , 2.39825018 , 2.502573836 , 2.608069791 , 2.714738054 , 2.822578635 , 2.931591544 , 3.041776793 , 3.151962204 , 3.263319974 , 3.377022303 , 3.490724825 , 3.605599741 , 3.721647065 , 3.837694611 , 3.956086788 , 4.075651415 , 4.1952163 , 4.315953659 , 4.43903572 , 4.56211808 , 4.686372963 , 4.811800388 , 4.938400372 , 5.066172934 , 5.195118093 , 5.324063635 , 5.455354044 , 5.587817112 , 5.720280615 , 5.855089062 , 5.98989798 , 6.127051897 , 6.264206325 , 6.402533542 , 6.543205845 , 6.683878722 , 6.825724467 , 6.968743108 , 7.112934672 , 7.259471486 , 7.406008992 , 7.553719511 , 7.702603075 , 7.852659717 , 8.00388947 , 8.156292367 , 8.311040784 , 8.465790083 , 8.621712634 , 8.778808475 , 8.937077645 , 9.097692558 , 9.258308511 , 9.420097914 , 9.584233208 , 9.748369647 , 9.914852081 , 10.08133573 , 10.25016549 , 10.42016899 , 10.59017382 , 10.76252493 , 10.93604993 , 11.11074887 , 11.2866218 , 11.46366878 , 11.64188987 , 11.82245765 , 12.00302713 , 12.18594344 , 12.36886156 , 12.55412665 , 12.74056623 , 12.92818038 , 13.11696914 , 13.30810522]
                              ]).T
                '''
                m = np.array([[0 , 1.117310996 , 2.233628908 , 3.347690544 , 4.460774415 , 5.571619482 , 6.681501948 , 7.790428245 , 8.897140511 , 10.00291156 , 11.10774774 , 12.21039431 , 13.31086072 , 14.41167436 , 15.51032244 , 16.60681426 , 17.70367092 , 18.79838573 , 19.89096786 , 20.98267931 , 22.07352617 , 23.16351451 , 24.2514004 , 25.33719278 , 26.42339659 , 27.50752075 , 28.58957406 , 29.67205564 , 30.75248006 , 31.83085598 , 32.90967684 , 33.98646269 , 35.06122208 , 36.13520317 , 37.20841172 , 38.28085345 , 39.35129703 , 40.42098694 , 41.48992887 , 42.55689397 , 43.62312407 , 44.68739196 , 45.75216968 , 46.81499801 , 47.87588499 , 48.93729758 , 49.99678147 , 51.05434457 , 52.11244892 , 53.16864496 , 54.22416603 , 55.27779275 , 56.33075684 , 57.38306365 , 58.43471854 , 59.48450511 , 60.53365192 , 61.58216424 , 62.62882785 , 63.67486899 , 64.7202929 , 65.76388743 , 66.80809327 , 67.85048155 , 68.89105954 , 69.93226367 , 70.97166921 , 72.01049654 , 73.04753822 , 74.08522517 , 75.12113798 , 76.15649427 , 77.19008915 , 78.22313889 , 79.25564852 , 80.28762305 , 81.31906747 , 82.34877991 , 83.37797343 , 84.40665296 , 85.4336184 , 86.4600809 , 87.48604538 , 88.51151674 , 89.53649986 , 90.55979743 , 91.58261762 , 92.60496528 , 93.62564472 , 94.64706236 , 95.66682249 , 96.68613043 , 97.70499096 , 98.72221099 , 99.73899418 , 100.7553453 , 101.771269 , 102.7867701 , 103.8006579 , 104.8153284 , 105.828396 , 106.8398668 , 107.8521338 , 108.8628143 , 109.8742994 , 110.8842081 , 111.8937382 , 112.901703 , 113.9104901 , 114.9177219 , 115.9245944 , 116.931112 , 117.9372794 , 118.9419125 , 119.9473936 , 120.9513501 , 121.9549755 , 122.9582742 , 123.9600639 , 124.9627232 , 125.9638832 , 126.965921 , 127.9664693 , 128.9667187 , 129.965489 , 130.9651542 , 131.9633499 , 132.9624488 , 133.9600878 , 134.9574552 , 135.9545555 , 136.951393 , 137.9467903 , 138.9431159 , 139.9380107 , 140.9338421 , 141.928252 , 142.9224261 , 143.9163689 , 144.9089046 , 145.9023981 , 146.8956733 , 147.8875553 , 148.8792283 , 149.8718754 , 150.8631431 , 151.8542147 , 152.8450947 , 153.8357872 , 154.825119 , 155.81545 , 156.8044293 , 157.7944159 , 158.7830598 , 159.7727193 , 160.761045 , 161.7492182 , 162.7372432 , 163.7251241 , 164.7128653 , 165.7004711 , 166.6879457 , 167.6741181 , 168.6613433 , 169.6484501 , 170.6342679 , 171.6211509 , 172.6067537 , 173.5934299 , 174.5788346 , 175.5641467 , 176.5505446 , 177.5356841 , 178.5207437 , 179.5057276 , 180.4918139 , 181.4766591 , 182.4614414 , 183.4461649 , 184.4308339 , 185.4154526 , 186.4000253 , 187.3845562 , 188.3690494 , 189.3535093 , 190.33794 , 191.3223457 , 192.3067308 , 193.2910994 , 194.2754557 , 195.2598039 , 196.2441484 , 197.2284932 , 198.2128427 , 199.1972009 , 200.1815723 , 201.1659609 , 202.1503711 , 203.1359803 , 204.1204462 , 205.1049462 , 206.0894847 , 207.0752393 , 208.0598673 , 209.0445463 , 210.0304545 , 211.0152485 , 212.0012801 , 212.986206 , 213.9723782 , 214.958627 , 215.9437825 , 216.9301973 , 217.9167015 , 218.9032993 , 219.8899951 , 220.8767931 , 221.8636975 , 222.8518876 , 223.8390178 , 224.8262672 , 225.8148156 , 226.8023164 , 227.7911251 , 228.7800704 , 229.7691567 , 230.7583881 , 231.7477691 , 232.7373039 , 233.7269968 , 234.7180292 , 235.7080515 , 236.6994223 , 237.6909688 , 238.6826955 , 239.6746066 , 240.6667065 , 241.6589995 , 242.6526688 , 243.6453614 , 244.6394394 , 245.6337284 , 246.6282329 , 247.6229571 , 248.6179054 , 249.6130823 , 250.6096731 , 251.6065017 , 252.6035727 , 253.6008904 , 254.5984592 , 255.5962836 , 256.5955507 , 257.5950828 , 258.5948845 , 259.5949602 , 260.5953143 , 261.5971356 , 262.5980604 , 263.600462 , 264.6031608 , 265.6073468 , 266.6106536 , 267.6154574 , 268.6205772 , 269.6260173 , 270.6317824 , 271.6390651 , 272.6454943 , 273.6534509 , 274.6617516 , 275.6715906 , 276.6805936 , 277.6911449 , 278.7020596 , 279.7145337 , 280.7261894 , 281.7394147 , 282.7530228 , 283.7682116 , 284.7826 , 285.7985793 , 286.8149612 , 287.8317504 , 288.8501471 , 289.8689617 , 290.8881987 , 291.9090601 , 292.929157 , 293.9508888 , 294.9742619 , 295.9968846 , 297.0211592 , 298.0458921 , 299.0722889 , 300.0979535 , 301.1252929 , 302.1543136 , 303.1826162 , 304.2126113 , 305.2443053 , 306.2752955 , 307.3079959 , 308.3412073 , 309.3761411 , 310.4115972 , 311.4475806 , 312.4840962 , 313.5223582 , 314.5623736 , 315.6017288 , 316.6428489 , 317.6845293 , 318.7279875 , 319.7720177 , 320.8166252 , 321.8630292 , 322.9100223 , 323.9576096 , 325.0070127 , 326.057022 , 327.1088605 , 328.1613171 , 329.2143973 , 330.2693263 , 331.324891 , 332.3810968 , 333.4391713 , 334.4978992 , 335.5585097 , 336.619786 , 337.6817337 , 338.7455844 , 339.8113458 , 340.8765707 , 341.9437192 , 343.0127992 , 344.0825887 , 345.1530935 , 346.2255507 , 347.2987362 , 348.373889 , 349.4497832 , 350.5276595 , 351.6062904 , 352.6856818 , 353.7670771 , 354.8504848 , 355.9346741 , 357.0196508 , 358.1066621 , 359.1944746 , 360.2843373 , 361.375015 , 362.4677587 , 363.5613314 , 364.656986 , 365.7534836 , 366.8520793 , 367.9515323 , 369.0530996 , 370.1555386 , 371.2588555 , 372.3655643 , 373.4719129 , 374.5816721 , 375.6910837 , 376.8039249 , 377.9164314 , 379.0323866 , 380.1492812 , 381.2671218 , 382.387178 , 383.5081954 , 384.6314462 , 385.7569401 , 386.8834197 , 388.0121603 , 389.1419023 , 390.2739233 , 391.4069616 , 392.5422974 , 393.6799409 , 394.8186269],
                              [14.96003823 , 14.78396689 , 14.60789757 , 14.43300403 , 14.26045997 , 14.08791779 , 13.91772494 , 13.74753387 , 13.57969199 , 13.41302551 , 13.24636067 , 13.08204482 , 12.9189042 , 12.75693876 , 12.59614844 , 12.43653321 , 12.278093 , 12.12082777 , 11.96473747 , 11.80864846 , 11.65490791 , 11.50234214 , 11.35095113 , 11.20073484 , 11.05169321 , 10.90382621 , 10.75713381 , 10.61161595 , 10.46727261 , 10.32410374 , 10.18210932 , 10.0412893 , 9.900470156 , 9.761998849 , 9.624701845 , 9.487405645 , 9.352457159 , 9.217509427 , 9.084909336 , 8.952309951 , 8.8208847 , 8.690633554 , 8.56272991 , 8.434826886 , 8.308097887 , 8.181369478 , 8.056988457 , 7.933781386 , 7.811748241 , 7.68971561 , 7.56885687 , 7.450345377 , 7.331834346 , 7.214497142 , 7.098333742 , 6.983344128 , 6.868354922 , 6.755712824 , 6.643071106 , 6.532776452 , 6.422482149 , 6.31336153 , 6.205414579 , 6.098641279 , 5.991868287 , 5.887442246 , 5.783016489 , 5.67976433 , 5.577685755 , 5.476780749 , 5.3770493 , 5.277318087 , 5.178760411 , 5.08137626 , 4.985165621 , 4.890128483 , 4.796264834 , 4.702401373 , 4.609711385 , 4.519368148 , 4.427851788 , 4.33868216 , 4.249512684 , 4.162689917 , 4.075867289 , 3.990218073 , 3.904568987 , 3.821266572 , 3.737964275 , 3.655835362 , 3.574879826 , 3.493924394 , 3.415315593 , 3.336706886 , 3.259271532 , 3.181836264 , 3.1067476 , 3.031659014 , 2.957743761 , 2.885001836 , 2.812259979 , 2.741864695 , 2.671469472 , 2.601074307 , 2.533025702 , 2.464977149 , 2.398101897 , 2.33239994 , 2.267871276 , 2.203342654 , 2.139987319 , 2.077805268 , 2.015623252 , 1.954614516 , 1.894779056 , 1.836116869 , 1.778627953 , 1.721139063 , 1.664823441 , 1.608507841 , 1.554538746 , 1.500569672 , 1.446600616 , 1.394978059 , 1.343355518 , 1.292906232 , 1.243630199 , 1.19435418 , 1.146251411 , 1.099321892 , 1.053565622 , 1.007809361 , 0.963226347 , 0.918643341 , 0.876406816 , 0.834170299 , 0.791933788 , 0.752043755 , 0.712153728 , 0.673436942 , 0.63472016 , 0.598349854 , 0.561979551 , 0.525609252 , 0.491585426 , 0.457561603 , 0.423537782 , 0.391860434 , 0.360183087 , 0.329678977 , 0.299174868 , 0.269843995 , 0.241686358 , 0.213528722 , 0.187717557 , 0.161906391 , 0.136095227 , 0.111457297 , 0.087992603 , 0.065701143 , 0.043409684 , 0.022291459 , 0.002346469 , -0.01759852 , -0.036370276 , -0.053968796 , -0.071567317 , -0.087992603 , -0.103244654 , -0.118496706 , -0.131402288 , -0.145481105 , -0.157213452 , -0.1689458 , -0.179504913 , -0.190064026 , -0.199449905 , -0.207662548 , -0.215875192 , -0.221741366 , -0.228780775 , -0.233473714 , -0.238166654 , -0.241686358 , -0.245206063 , -0.247552533 , -0.248725767 , -0.248725767 , -0.248725767 , -0.247552533 , -0.246379298 , -0.242859593 , -0.240513123 , -0.235820184 , -0.231127245 , -0.22526107 , -0.218221662 , -0.211182253 , -0.202969609 , -0.194756965 , -0.184197852 , -0.173638739 , -0.163079626 , -0.151347279 , -0.138441696 , -0.124362879 , -0.110284063 , -0.095032011 , -0.078606725 , -0.062181439 , -0.044582919 , -0.025811163 , -0.007039408 , 0.014078816 , 0.034023806 , 0.056315266 , 0.078606725 , 0.10207142 , 0.125536114 , 0.150174044 , 0.175985209 , 0.202969609 , 0.22995401 , 0.258111646 , 0.286269284 , 0.316773392 , 0.347277502 , 0.377781613 , 0.410632195 , 0.44348278 , 0.476333367 , 0.511530427 , 0.54672749 , 0.583097791 , 0.619468096 , 0.658184876 , 0.69690166 , 0.735618449 , 0.776681716 , 0.817744988 , 0.858808266 , 0.902218025 , 0.945627791 , 0.990210802 , 1.034793821 , 1.081723325 , 1.12865284 , 1.176755604 , 1.224858381 , 1.274134408 , 1.324583689 , 1.376206224 , 1.427828775 , 1.480624584 , 1.534593651 , 1.588562739 , 1.64487833 , 1.701193944 , 1.757509583 , 1.816171732 , 1.874833909 , 1.934669359 , 1.995678085 , 2.056686844 , 2.118868882 , 2.182224205 , 2.246752813 , 2.311281463 , 2.376983406 , 2.443858643 , 2.51190718 , 2.579955769 , 2.650350916 , 2.720746122 , 2.791141387 , 2.863883225 , 2.936625131 , 3.010540363 , 3.085628928 , 3.16071757 , 3.238152814 , 3.315588144 , 3.394196827 , 3.473978869 , 3.553761008 , 3.634716517 , 3.716845401 , 3.800147669 , 3.884623327 , 3.969099109 , 4.05592157 , 4.142744166 , 4.229566901 , 4.318736343 , 4.409079222 , 4.49942226 , 4.590938751 , 4.683628703 , 4.777492127 , 4.871355738 , 4.967566136 , 5.063776737 , 5.16116085 , 5.259718486 , 5.359449657 , 5.459181064 , 5.561259342 , 5.663337875 , 5.766589991 , 5.871015704 , 5.976615027 , 6.083387976 , 6.190161231 , 6.299281475 , 6.40840205 , 6.518696308 , 6.630164266 , 6.742805941 , 6.856621352 , 6.970437156 , 7.086600089 , 7.202763446 , 7.32127398 , 7.439784969 , 7.559469804 , 7.680328506 , 7.802361097 , 7.9255676 , 8.049948038 , 8.175502436 , 8.301057404 , 8.42895979 , 8.556862785 , 8.687113262 , 8.81736439 , 8.948789623 , 9.081388989 , 9.215162517 , 9.351283697 , 9.487405645 , 9.624701845 , 9.763172331 , 9.902817134 , 10.04246279 , 10.18445632 , 10.32762428 , 10.47196669 , 10.61748359 , 10.76417502 , 10.91086748 , 11.05990808 , 11.21012332 , 11.36151326 , 11.51407793 , 11.66664378 , 11.82155803 , 11.97764715 , 12.13491117 , 12.29335014 , 12.45296412 , 12.61375315 , 12.77571729 , 12.93885657 , 13.10317106 , 13.2686608 , 13.43532586 , 13.60316628 , 13.77218212 , 13.94354719 , 14.11491406 , 14.2886303 , 14.46234845 , 14.63841611 , 14.81565958 , 14.99407893 , 15.17367422 , 15.35444553 , 15.53639291 , 15.71951645 , 15.9049901 , 16.09046618 , 16.27829254 , 16.46729539 , 16.65747479]
                              ]).T
            else:
                assert False, "Error: no known Fenix dimensions %d" % image.xdim()

            # flip x-axis correction if camera was mounted backwards
            if kwds.get("flip", False):
                m[:, 1] *= -1

            # convert to displacement vectors
            dmap = np.zeros((image.data.shape[0], image.data.shape[1], 2))
            dmap[:, :, 0] += -m[:, None, 1]  # displacements in x
            dmap[:, :, 0] -= np.min(-m[:, 1])  # avoid negative displacements
            dmap[:, :, 1] += (m[:, 0] - np.arange(image.data.shape[0]))[:, None]  # displacements in y

            # calculate width/height of corrected image
            width = int(image.data.shape[1] + np.max(m[:, 1]) - np.min(m[:, 1]))
            height = int(np.ceil(np.max(m[:, 0])))

            # resize displacement map to output dimensions
            dmap = cv2.resize(dmap, (width, height), cv2.INTER_LINEAR)

            # use displacement vectors to calculate mapping from output coordinates to original coordinates
            xx, yy = np.meshgrid(range(dmap.shape[1]), range(dmap.shape[0]))
            idx = np.dstack([xx, yy]).astype(np.float32)
            idx[:, :, 0] -= dmap[:, :, 0]
            idx[:, :, 1] -= dmap[:, :, 1]

            # apply remapping
            if image.data.shape[-1] < 512:  # open-cv cannot handle more than 512 bands at a time
                remap = cv2.remap(image.data, idx, None, cv2.INTER_LINEAR)
            else:  # we need to split into different stacks with < 512 bands and then recombine
                remap = []
                mn = 0
                mx = 500
                while mn < image.data.shape[-1]:
                    if mx > image.data.shape[-1]:
                        mx = image.data.shape[-1]

                    # apply mapping to slice of bands
                    remap.append(cv2.remap(image.data[:, :, mn:mx], idx, None, cv2.INTER_LINEAR))

                    # get next slice
                    mn = mx
                    mx += 500

                # stack
                remap = np.dstack(remap)

            image.data = remap
            if verbose: print("DONE.")

        # rotate image so that scanning direction is horizontal rather than vertical)
        image.data = np.rot90(image.data)  # np.transpose(remap, (1, 0, 2))
        image.data = np.flip(image.data, axis=1)

    @classmethod
    def correct_folder(cls, path, **kwds):

        """
        Many sensors use simple/common data structures to store data/headers/dark reference etc. Hence it is often easiest
        to pass an output folder to the sensor for correction.

        *Arguments*:
         - path = a path to the folder containing the sensor specific data.

        *Keywords*:
         - verbose = True if print outputs should be made to update progress. Default is True.
         - calib = Calibration spectra for any white references found.
         - other keywords are passed directly to correct_image.

        *Returns*:
         - a hyImage to which all sensor-specific corrections have been applied. Note that this will generally not include
           topographic or atmospheric corrections.

        """
        verbose = kwds.get("verbose", True)
        kwds["verbose"] = verbose

        imgs = [str(p) for p in Path(path).rglob("capture/*.hdr")]  # all image files [including data]
        dark = [str(p) for p in Path(path).rglob("capture/DARKREF*.hdr")]  # dark reference file
        white = [str(p) for p in Path(path).rglob("capture/WHITEREF*.hdr")]  # an white reference data (core scanner)
        refl = [str(p) for p in Path(path).rglob("capture/REFL*.hdr")]  # any processed reflectance data (SiSu Rock)
        for d in dark:
            del imgs[imgs.index(d)]
        for w in white:
            del imgs[imgs.index(w)]
        for r in refl:
            del imgs[imgs.index(r)]

        if len(imgs) > 1 or len(
            dark) > 1: assert False, "Error - multiple scenes found in folder. Double check file path..."
        if len(imgs) == 0 or len(
            dark) == 0: assert False, "Error - no image or dark calibration found in folder. Double check file path... %s" % path

        if verbose: print('\nLoading image %s' % imgs[0])

        # load image
        image = io.load(imgs[0])
        Fenix.set_dark_ref(dark[0])
        if len(white) > 0:  # white reference exists
            Fenix.set_white_ref(white[0])

        # correct
        Fenix.correct_image(image, **kwds)

        # return corrected image
        return image
