import binascii
import struct
from unittest import mock

import pytest

from vallox_websocket_api import PROFILE
from vallox_websocket_api.exceptions import ValloxInvalidInputException


@pytest.mark.parametrize(
    "profile, metric, value",
    [
        (PROFILE.HOME, "A_CYC_HOME_AIR_TEMP_TARGET", 19),
        (PROFILE.AWAY, "A_CYC_AWAY_AIR_TEMP_TARGET", 19),
        (PROFILE.BOOST, "A_CYC_BOOST_AIR_TEMP_TARGET", 19),
    ],
)
async def test_vallox_set_temperature(vallox, profile, metric, value):
    vallox.set_values = mock.AsyncMock()
    await vallox.set_temperature(profile, value)
    vallox.set_values.assert_called_once_with({metric: value})


async def test_vallox_set_temperature_not_supported(vallox):
    with pytest.raises(ValloxInvalidInputException):
        await vallox.set_temperature(PROFILE.FIREPLACE, 19)


async def test_vallox_get_temperature(vallox):
    vallox.fetch_metrics = mock.AsyncMock(
        return_value={"A_CYC_HOME_AIR_TEMP_TARGET": 19}
    )

    assert await vallox.get_temperature(PROFILE.HOME) == 19
    vallox.fetch_metrics.assert_called_once_with(
        list({"A_CYC_HOME_AIR_TEMP_TARGET": 19}.keys())
    )


async def test_vallox_fetch_metric(vallox, ws):
    """
    IoQueue.KItemTypeFetch = 3
    VlxDevConstants.WS_WEB_UI_COMMAND_READ_TABLES; = 246
    item.value = 0;
    checksum = 249
    Uint16Array(4) [3, 246, 0, 249]
    """
    ws.recv.return_value = binascii.unhexlify(
        "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"
    )

    assert await vallox.fetch_metric("A_CYC_HOME_AIR_TEMP_TARGET") == 20.0
    ws.send.assert_called_once_with(struct.pack("HHHH", 3, 246, 0, 249))
