"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assets = require("@aws-cdk/assets");
const fs = require("fs");
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *     Source.bucket(bucket, key)
 *     Source.asset('/local/path/to/directory')
 *     Source.asset('/local/path/to/a/file.zip')
 *
 */
class Source {
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     * @param bucket The S3 Bucket
     * @param zipObjectKey The S3 object key of the zip file with contents
     */
    static bucket(bucket, zipObjectKey) {
        return {
            bind: () => ({ bucket, zipObjectKey })
        };
    }
    /**
     * Uses a local asset as the deployment source.
     * @param path The path to a local .zip file or a directory
     */
    static asset(path) {
        return {
            bind(context) {
                const packaging = fs.lstatSync(path).isDirectory() ? assets.AssetPackaging.ZipDirectory : assets.AssetPackaging.File;
                const asset = new assets.Asset(context, 'Asset', { packaging, path });
                if (!asset.isZipArchive) {
                    throw new Error(`Asset path must be either a .zip file or a directory`);
                }
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey
                };
            }
        };
    }
    constructor() { }
}
exports.Source = Source;
//# sourceMappingURL=data:application/json;base64,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