"use strict";
const assert_1 = require("@aws-cdk/assert");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/cdk");
const path = require("path");
const s3deploy = require("../lib");
module.exports = {
    'deploy from local directory asset'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            source: s3deploy.Source.asset(path.join(__dirname, 'my-website')),
            destinationBucket: bucket,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536",
                    "Arn"
                ]
            },
            "SourceBucketName": {
                "Ref": "DeployAssetS3BucketB84349C9"
            },
            "SourceObjectKey": {
                "Fn::Join": [
                    "",
                    [
                        {
                            "Fn::Select": [
                                0,
                                {
                                    "Fn::Split": [
                                        "||",
                                        {
                                            "Ref": "DeployAssetS3VersionKeyB05C8986"
                                        }
                                    ]
                                }
                            ]
                        },
                        {
                            "Fn::Select": [
                                1,
                                {
                                    "Fn::Split": [
                                        "||",
                                        {
                                            "Ref": "DeployAssetS3VersionKeyB05C8986"
                                        }
                                    ]
                                }
                            ]
                        }
                    ]
                ]
            },
            "DestinationBucketName": {
                "Ref": "DestC383B82A"
            }
        }));
        test.done();
    },
    'fails if local asset is a non-zip file'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // THEN
        test.throws(() => new s3deploy.BucketDeployment(stack, 'Deploy', {
            source: s3deploy.Source.asset(path.join(__dirname, 'my-website', 'index.html')),
            destinationBucket: bucket,
        }), /Asset path must be either a \.zip file or a directory/);
        test.done();
    },
    'deploy from a local .zip file'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            source: s3deploy.Source.asset(path.join(__dirname, 'my-website.zip')),
            destinationBucket: bucket,
        });
        test.done();
    },
    'retainOnDelete can be used to retain files when resource is deleted'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            source: s3deploy.Source.asset(path.join(__dirname, 'my-website.zip')),
            destinationBucket: bucket,
            retainOnDelete: true,
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            RetainOnDelete: true
        }));
        test.done();
    },
    'lambda execution role gets permissions to read from the source bucket and read/write in destination'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const source = new s3.Bucket(stack, 'Source');
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            source: s3deploy.Source.bucket(source, 'file.zip'),
            destinationBucket: bucket,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "s3:GetObject*",
                            "s3:GetBucket*",
                            "s3:List*"
                        ],
                        "Effect": "Allow",
                        "Resource": [
                            {
                                "Fn::GetAtt": [
                                    "Source71E471F1",
                                    "Arn"
                                ]
                            },
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        {
                                            "Fn::GetAtt": [
                                                "Source71E471F1",
                                                "Arn"
                                            ]
                                        },
                                        "/*"
                                    ]
                                ]
                            }
                        ]
                    },
                    {
                        "Action": [
                            "s3:GetObject*",
                            "s3:GetBucket*",
                            "s3:List*",
                            "s3:DeleteObject*",
                            "s3:PutObject*",
                            "s3:Abort*"
                        ],
                        "Effect": "Allow",
                        "Resource": [
                            {
                                "Fn::GetAtt": [
                                    "DestC383B82A",
                                    "Arn"
                                ]
                            },
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        {
                                            "Fn::GetAtt": [
                                                "DestC383B82A",
                                                "Arn"
                                            ]
                                        },
                                        "/*"
                                    ]
                                ]
                            }
                        ]
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRoleDefaultPolicy88902FDF",
            "Roles": [
                {
                    "Ref": "CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRole89A01265"
                }
            ]
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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