"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * The record type.
 */
var RecordType;
(function (RecordType) {
    RecordType["A"] = "A";
    RecordType["AAAA"] = "AAAA";
    RecordType["CAA"] = "CAA";
    RecordType["CNAME"] = "CNAME";
    RecordType["MX"] = "MX";
    RecordType["NAPTR"] = "NAPTR";
    RecordType["NS"] = "NS";
    RecordType["PTR"] = "PTR";
    RecordType["SOA"] = "SOA";
    RecordType["SPF"] = "SPF";
    RecordType["SRV"] = "SRV";
    RecordType["TXT"] = "TXT";
})(RecordType = exports.RecordType || (exports.RecordType = {}));
/**
 * Type union for a record that accepts multiple types of target.
 */
class RecordTarget {
    constructor(values, aliasTarget) {
        this.values = values;
        this.aliasTarget = aliasTarget;
    }
    /**
     * Use string values as target.
     */
    static fromValues(...values) {
        return new RecordTarget(values);
    }
    /**
     * Use an alias as target.
     */
    static fromAlias(aliasTarget) {
        return new RecordTarget(undefined, aliasTarget);
    }
    /**
     * Use ip adresses as target.
     */
    static fromIpAddresses(...ipAddresses) {
        return RecordTarget.fromValues(...ipAddresses);
    }
}
exports.RecordTarget = RecordTarget;
/**
 * A record set.
 */
class RecordSet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const ttl = props.target.aliasTarget ? undefined : ((props.ttl && props.ttl.toSeconds()) || 1800).toString();
        const recordSet = new route53_generated_1.CfnRecordSet(this, 'Resource', {
            hostedZoneId: props.zone.hostedZoneId,
            name: util_1.determineFullyQualifiedDomainName(props.recordName || props.zone.zoneName, props.zone),
            type: props.recordType,
            resourceRecords: props.target.values,
            aliasTarget: props.target.aliasTarget && props.target.aliasTarget.bind(this),
            ttl,
            comment: props.comment
        });
        this.domainName = recordSet.ref;
    }
}
exports.RecordSet = RecordSet;
/**
 * @deprecated Use RecordTarget
 */
class AddressRecordTarget extends RecordTarget {
}
exports.AddressRecordTarget = AddressRecordTarget;
/**
 * A DNS A record
 *
 * @resource AWS::Route53::RecordSet
 */
class ARecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, Object.assign(Object.assign({}, props), { recordType: RecordType.A, target: props.target }));
    }
}
exports.ARecord = ARecord;
/**
 * A DNS AAAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class AaaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, Object.assign(Object.assign({}, props), { recordType: RecordType.AAAA, target: props.target }));
    }
}
exports.AaaaRecord = AaaaRecord;
/**
 * A DNS CNAME record
 *
 * @resource AWS::Route53::RecordSet
 */
class CnameRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, Object.assign(Object.assign({}, props), { recordType: RecordType.CNAME, target: RecordTarget.fromValues(props.domainName) }));
    }
}
exports.CnameRecord = CnameRecord;
/**
 * A DNS TXT record
 *
 * @resource AWS::Route53::RecordSet
 */
class TxtRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, Object.assign(Object.assign({}, props), { recordType: RecordType.TXT, target: RecordTarget.fromValues(...props.values.map(v => JSON.stringify(v))) }));
    }
}
exports.TxtRecord = TxtRecord;
/**
 * A DNS SRV record
 *
 * @resource AWS::Route53::RecordSet
 */
class SrvRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, Object.assign(Object.assign({}, props), { recordType: RecordType.SRV, target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.weight} ${v.port} ${v.hostName}`)) }));
    }
}
exports.SrvRecord = SrvRecord;
/**
 * The CAA tag.
 */
var CaaTag;
(function (CaaTag) {
    /**
     * Explicity authorizes a single certificate authority to issue a
     * certificate (any type) for the hostname.
     */
    CaaTag["ISSUE"] = "issue";
    /**
     * Explicity authorizes a single certificate authority to issue a
     * wildcard certificate (and only wildcard) for the hostname.
     */
    CaaTag["ISSUEWILD"] = "issuewild";
    /**
     * Specifies a URL to which a certificate authority may report policy
     * violations.
     */
    CaaTag["IODEF"] = "iodef";
})(CaaTag = exports.CaaTag || (exports.CaaTag = {}));
/**
 * A DNS CAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, Object.assign(Object.assign({}, props), { recordType: RecordType.CAA, target: RecordTarget.fromValues(...props.values.map(v => `${v.flag} ${v.tag} "${v.value}"`)) }));
    }
}
exports.CaaRecord = CaaRecord;
/**
 * A DNS Amazon CAA record.
 *
 * A CAA record to restrict certificate authorities allowed
 * to issue certificates for a domain to Amazon only.
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaAmazonRecord extends CaaRecord {
    constructor(scope, id, props) {
        super(scope, id, Object.assign(Object.assign({}, props), { values: [
                {
                    flag: 0,
                    tag: CaaTag.ISSUE,
                    value: 'amazon.com'
                }
            ], recordName: props.zone.zoneName }));
    }
}
exports.CaaAmazonRecord = CaaAmazonRecord;
/**
 * A DNS MX record
 *
 * @resource AWS::Route53::RecordSet
 */
class MxRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, Object.assign(Object.assign({}, props), { recordType: RecordType.MX, target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.hostName}`)) }));
    }
}
exports.MxRecord = MxRecord;
/**
 * A record to delegate further lookups to a different set of name servers.
 */
class ZoneDelegationRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, Object.assign(Object.assign({}, props), { recordType: RecordType.NS, target: RecordTarget.fromValues(...core_1.Token.isUnresolved(props.nameServers)
                ? props.nameServers // Can't map a string-array token!
                : props.nameServers.map(ns => (core_1.Token.isUnresolved(ns) || ns.endsWith('.')) ? ns : `${ns}.`)), ttl: props.ttl || core_1.Duration.days(2) }));
    }
}
exports.ZoneDelegationRecord = ZoneDelegationRecord;
//# sourceMappingURL=data:application/json;base64,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