import rich
import subprocess
import click
import click
from staircase.config import Config
from staircase.lib.click import async_cmd

from staircase.command_providers import (
    env_manager_provider,
    staircase_provider,
    config_provider,
    env_storage_provider,
    EnvironmentStorage,
)
from staircase.env_manager import EnvironmentManager

from staircase.lib.fzf import prompt
from staircase.lib.click import async_cmd
from staircase.staircase import Staircase
from staircase.lib.sdk.ci import clone, build, deploy
from staircase.user_prompts import select_product_git_repo

DEFAULT_CI_ENV = 'marketplace.staircaseapi.com'

@click.command(
    "ci", 
)
@click.argument("actions", type=click.Choice(["clone", "build", "deploy"]), nargs=-1)
@click.option(
    "--bundle-url",
    "-u",
    help="Url of bundle for any step you start your ci from. It will be passed correct to any ci step.",
)
@click.option(
    "--product",
    "-p",
    help="Product name in marketplace. If you dont specify clone, it will be used to get start bundle",
)
@click.option("--repo", "-r", help="Git repo name in Staircase")
@click.option("--ci-env", "-ci", help="Environment where ci products will be called. Ex. mydomain.staircaseapi.com")
@click.option("--branch", "-b", help="Branch name of git clone")
@click.option("--data", "-d", help="If this is data product", is_flag=True)
@click.option("--verbose", "-v", help="Verbose output", is_flag=True)
@click.option(
    "--target",
    "-t",
    help="Target domain of environment to be deployed. Ex. mydomain.staircaseapi.com",
)
@env_manager_provider()
@staircase_provider()
@config_provider()
@env_storage_provider()
@async_cmd
async def command(
    *,
    actions,
    bundle_url,
    target,
    product,
    branch,
    repo,
    data: bool,
    ci_env: str,
    verbose: bool,
    staircase: Staircase,
    env_manager: EnvironmentManager,
    env_storage: EnvironmentStorage,
    config: Config,
):
    """
    Setup your pipeline with clone, assess, test, deploy, build products.

    You can specify steps as arguments, order is not changable. Default order is [clone, build, assess, deploy, test]. If you dont specify any actions if will be [clone, build, deploy]
    """
    if not actions:
        actions = ["clone", "build", "deploy"]

    envs = env_storage.get_all_envs()
    env_names = [env.domain_name for env in envs]
    selected_env = prompt(env_names, header="Select ci-env to deploy, default = marketplace.staircaseapi.com...")
    if selected_env:
        ci_env = selected_env[0]
    else:
        ci_env = DEFAULT_CI_ENV

    ci_env = env_manager.get_staircase_env(ci_env)
    if not ci_env:
        raise click.ClickException("CI env is not saved to your envs.")
    

    # Getting args
    if not config.github_token:
        raise click.ClickException("Setup github_token in your config")

    if "clone" in actions:
        if not repo:
            repo = await select_product_git_repo(staircase=staircase, prompt_header="Select product repo to clone from...")
            if not repo:
                raise click.ClickException("You did not specify any repo")

        if not branch:
            pipe = subprocess.run(
                [
                    "git",
                    "ls-remote",
                    "--heads",
                    f"git@github.com:StaircaseAPI/{repo}.git",
                ],
                stdout=subprocess.PIPE,
                text=True
            )
            branches = []
            for line in pipe.stdout.splitlines():
                branch = line.split("refs/heads/")[1]
                branches.append(branch)

            branches_selected = prompt(branches, header="Select your branch")

            if not branches_selected:
                raise click.ClickException("You did not specify any branch")

    if "deploy" in actions and not target:
        envs = env_storage.get_all_envs()
        env_names = [env.domain_name for env in envs]
        selected_env = prompt(env_names, header="Select env to deploy...")
        if selected_env:
            target = selected_env[0]
        if not target:
            raise click.ClickException("Not target env for deploy specified")

    if ["build", "assess", "test"] in actions:
        if not "clone" in actions or not bundle_url:
            raise click.ClickException(
                "You did not specify clone action or bundle_url to start from"
            )

    if (
        "deploy" in actions
        and not bundle_url
        and not ["build", "assess", "test", "clone"]
    ):
        product_names = [
            product["name"] for product in await staircase.get_all_product_components()
        ]
        products = prompt(product_names)
        if not products:
            raise click.ClickException(
                "No product specified and no bundle_url and no previous action step like clone. Nothing to deploy"
            )
        product_data = staircase.retrieve_product_component(product)
        bundle_url = product_data.get("bundle_url")
        if not bundle_url:
            raise click.ClickException(
                "Selected product does not contain bundle_url in marketplace."
            )

    # Flow
    if "clone" in actions:
        if verbose:
            rich.print(f"Repo: StaircaseCLI/{repo}.git")
            rich.print(f"Repo: {branch}")
            rich.print(f"Cloning...")
        bundle_url = await clone(
            ci_env, repo=repo, branch=branch, github_token=config.github_token
        )
        if verbose:
            rich.print(f"[green]Cloned.")
            rich.print(bundle_url)

    if "build" in actions:
        if verbose:
            rich.print(f"Building...")
        bundle_url = await build(ci_env=ci_env, url=bundle_url)
        if verbose:
            rich.print(f"[green]Build is success.")
            rich.print(bundle_url)

    if "deploy" in actions:
        if verbose:
            if product:
                rich.print(f"Deploying bundle from marketplace for product: {product}")
            rich.print(f"Deploying to env: {target}")
            rich.print(f"Deploying...")

        to_env = env_manager.get_staircase_env(target)
        if not to_env:
            raise click.ClickException("You have no such api in your envs saved.")
        env_token = await staircase.get_env_token(to_env.api_key)

        await deploy(ci_env=ci_env, url=bundle_url, env_token=env_token)
        if verbose:
            rich.print(f"[green]Deploy is success:")
