import numpy as np
import scipy.interpolate
from typing import List


def reverse_arr(arr):
    return np.array(list(reversed(arr)))


def interpolate_using_wavelengths(wavelengths: np.ndarray, wlens: List[float],
                                  n: List[float], k: List[float]) -> np.ndarray:
    """Interpolate the refractive index from n,k versus wavelength data.

    Args:
        wavelengths: Numpy array of wavelengths to inerpolate.
        wlens: Wavelengths at which known data are specified.
        n: Known refractive indices.
        k: Known extinction coefficients.
    """
    n = scipy.interpolate.pchip_interpolate(wlens, n, wavelengths)
    k = scipy.interpolate.pchip_interpolate(wlens, k, wavelengths)
    return (n, k)


def interpolate_using_energy(wavelengths, n, k, En, Ek):
    """Interpolate the refractive index from photon energy data.

    Args:
        wavelengths: Numpy array of wavelengths to inerpolate (nm).
        n: Known refractive indices.
        k: Known extinction coefficients.
        En: Photon energy (eV)
        Ek: Photon energy (eV)
    """
    h = 4.135667516e-15  # Plank's constant (eV s)
    c = 299792458  # Speed of light (m/s)

    # Extract refractive index via interpolation.
    Eq = h * c / (wavelengths * 1e-9)  # Photon energy (query)
    n = scipy.interpolate.pchip_interpolate(reverse_arr(En), reverse_arr(n), Eq)
    k = scipy.interpolate.pchip_interpolate(reverse_arr(Ek), reverse_arr(k), Eq)
    return (n, k)


class Material:

    def refractive_index(self, wavelengths):
        """Returns the real and imaginary component of refractive index. """
        raise NotImplementedError('refractive_index not implemented.')

    def eps_real(self, wavelengths):
        """Returns real part of permittivity. """
        n, k = self.refractive_index(wavelengths)
        return n**2 - k**2

    def eps(self, wavelengths):
        """Returns complex permittivity. """
        n, k = self.refractive_index(wavelengths)
        return (n - 1j * k)**2


class CustomMaterial(Material):
    """Specifies material from user-inputted refractive index data."""

    def __init__(self, wlens: List[float], n: List[float], k: List[float]):
        """Constructor to initialize material properties.

        Args:
            wlens: Known wavelength data.
            n: Known refractive indices.
            k: Known extinction coefficients.
        """
        self.wlens = wlens
        self.n = n
        self.k = k

    def refractive_index(self, wavelengths: List[float]) -> List[float]:
        return interpolate_using_wavelengths(wavelengths, self.wlens, self.n,
                                             self.k)


class Air(Material):

    def refractive_index(self, wavelengths: List[float]) -> List[float]:
        return (np.ones_like(wavelengths), np.zeros_like(wavelengths))


class SiO2(Material):

    def refractive_index(self, wavelengths: List[float]) -> List[float]:
        #SiO2 refractive index with respect to wavelength (wl, nm)

        # SOURCE: Palik, Handbook of Optical Constants of Solids, 1995
        # Table data located at http://ftp.esrf.eu/pub/scisoft/xop2.3/DabaxFiles/nk_Palik.dat
        #
        # DATA: Al_x Ga_(1-x) As refractive index with respect to photon energy
        #     n   Refractive index
        #     k   Extinction coefficient
        #     Ez  Photon energy (eV)  where z = {n, k}
        En = [
            10000, 9772.4, 9549.9, 9332.5, 9120.1, 8912.5, 8709.6, 8511.4,
            8317.6, 8128.3, 7943.3, 7762.5, 7585.8, 7413.1, 7244.4, 7079.5,
            6918.3, 6760.8, 6606.9, 6456.5, 6309.6, 6166, 6025.6, 5888.4,
            5754.4, 5623.4, 5495.4, 5370.3, 5248.1, 5128.6, 5011.9, 4897.8,
            4786.3, 4677.4, 4570.9, 4466.8, 4365.2, 4265.8, 4168.7, 4073.8,
            3981.1, 3890.5, 3801.9, 3715.4, 3630.8, 3548.1, 3467.4, 3388.4,
            3311.3, 3235.9, 3162.3, 3090.3, 3020, 2951.2, 2884, 2818.4, 2754.2,
            2691.5, 2630.3, 2570.4, 2511.9, 2454.7, 2398.8, 2344.2, 2290.9,
            2238.7, 2187.8, 2138, 2089.3, 2041.7, 1995.3, 1949.8, 1905.5,
            1862.1, 1819.7, 1778.3, 1737.8, 1698.2, 1659.6, 1621.8, 1584.9,
            1548.8, 1513.6, 1479.1, 1445.4, 1412.5, 1380.4, 1349, 1318.3,
            1288.2, 1258.9, 1230.3, 1202.3, 1174.9, 1148.2, 1122, 1096.5,
            1071.5, 1047.1, 1023.3, 1000, 977.2, 955, 933.3, 912, 891.3, 871,
            851.1, 831.8, 812.8, 794.3, 776.3, 758.6, 741.3, 724.4, 708, 691.8,
            676.1, 660.7, 645.7, 631, 616.6, 602.6, 588.8, 575.4, 562.3, 549.5,
            537, 524.8, 512.9, 501.2, 489.8, 478.6, 467.7, 457.1, 446.7, 436.5,
            426.6, 416.9, 407.4, 398.1, 389, 380.2, 371.5, 363.1, 354.8, 346.7,
            338.8, 331.1, 323.6, 316.2, 309, 302, 295.1, 288.4, 281.8, 275.4,
            269.1, 263, 257, 251.2, 245.5, 239.9, 234.4, 229.1, 223.9, 218.8,
            213.8, 208.9, 204.2, 199.5, 195, 190.6, 186.2, 182, 177.8, 173.8,
            169.8, 166, 162.2, 158.5, 154.9, 151.4, 151.2, 147.9, 147.6, 144.5,
            144.2, 141.3, 140.9, 138, 137.8, 134.9, 134.8, 131.9, 131.8, 129.2,
            128.8, 126.5, 125.9, 124, 123, 121.6, 120.2, 119.2, 117.5, 117,
            114.8, 112.7, 112.2, 110.7, 109.7, 108.8, 107.2, 106.9, 105.1,
            104.7, 103.3, 102.3, 101.6, 100, 99.99, 98.4, 97.7, 96.86, 95.5,
            95.37, 93.93, 93.3, 92.53, 91.2, 91.17, 89.84, 87.1, 85.1, 83.2,
            82.66, 81.3, 79.4, 77.6, 77.49, 75.9, 74.1, 72.93, 72.4, 70.8, 69.2,
            68.88, 67.6, 66.1, 65.26, 64.6, 63.1, 61.99, 61.7, 60.3, 59.04,
            58.9, 57.5, 56.36, 56.2, 55, 53.91, 53.7, 52.5, 51.66, 51.3, 49.59,
            47.69, 45.92, 44.28, 42.75, 41.33, 40, 38.75, 37.57, 36.47, 35.42,
            34.44, 33.51, 32.63, 31.79, 31, 30.24, 29.52, 28.83, 28.18, 27.55,
            26.95, 26.38, 25.83, 25.3, 25, 24.8, 24.31, 24, 23.84, 23.39, 23,
            22.96, 22.54, 22.14, 22, 21.75, 21.38, 21.01, 21, 20.66, 20, 19.5,
            19, 18.5, 18, 17.75, 17.5, 17.25, 17, 16.75, 16.5, 16.25, 16, 15.75,
            15.5, 15.25, 15, 14.75, 14.5, 14.25, 14, 13.75, 13.5, 13.25, 13,
            12.75, 12.5, 12.25, 12, 11.8, 11.6, 11.4, 11.2, 11, 10.9, 10.8,
            10.7, 10.6, 10.5, 10.45, 10.4, 10.35, 10.3, 10.25, 10.2, 10.1, 10,
            9.9, 9.8, 9.7, 9.6, 9.5, 9.4, 9.3, 9.2, 9.1, 9, 8.9, 8.8, 8.7, 8.6,
            8.5, 8.4, 8.3, 8.2, 8.1, 8, 7.8, 7.6, 7.4, 7.2, 7, 6.75, 6.5, 6.25,
            6, 5.7976, 5.7819, 5.468, 5.3858, 5.2131, 5.1674, 4.9939, 4.6751,
            4.594, 4.504, 4.4226, 4.2848, 4.1784, 4.1034, 3.7542, 3.7105,
            3.6427, 3.577, 3.434, 3.3967, 3.064, 2.8448, 2.6503, 2.5504, 2.4379,
            2.2705, 2.1489, 2.1411, 2.1102, 2.1041, 1.9257, 1.8892, 1.8566,
            1.7549, 1.455, 1.3863, 1.2228, 1.1449, 1.0985, 0.91018, 0.88874,
            0.84372, 0.81061, 0.74663, 0.73225, 0.72543, 0.68384, 0.62934,
            0.60243, 0.57598, 0.53317, 0.50868, 0.38221, 0.37953, 0.37542,
            0.35354, 0.34863, 0.33449, 0.3224, 0.2976, 0.2728, 0.248, 0.2232,
            0.2108, 0.1984, 0.1922, 0.186, 0.1835, 0.181, 0.1785, 0.1761,
            0.1736, 0.1711, 0.1686, 0.1674, 0.1661, 0.1649, 0.1637, 0.1624,
            0.1612, 0.1599, 0.1587, 0.1575, 0.1562, 0.155, 0.1537, 0.1525,
            0.1513, 0.15, 0.1488, 0.1475, 0.1463, 0.1451, 0.1438, 0.1426,
            0.1413, 0.1401, 0.1389, 0.1376, 0.1364, 0.1351, 0.1333, 0.1302,
            0.1271, 0.124, 0.1209, 0.1178, 0.1147, 0.1116, 0.1091, 0.1066,
            0.1054, 0.1041, 0.1029, 0.1017, 0.1004, 0.09919, 0.09795, 0.09671,
            0.09547, 0.09423, 0.09299, 0.08989, 0.08679, 0.08369, 0.08059,
            0.07749, 0.07439, 0.07191, 0.06943, 0.06695, 0.06534, 0.06447,
            0.06323, 0.06199, 0.06075, 0.05951, 0.05889, 0.05827, 0.05703,
            0.05455, 0.05207, 0.04959, 0.04649, 0.04339, 0.0403, 0.0372, 0.0341,
            0.031, 0.0124, 0.009919, 0.007439, 0.004959, 0.00248
        ]
        n0 = [
            0.999995, 0.999994, 0.999994, 0.999994, 0.999993, 0.999993,
            0.999993, 0.999992, 0.999992, 0.999992, 0.999991, 0.999991, 0.99999,
            0.99999, 0.999989, 0.999989, 0.999988, 0.999988, 0.999987, 0.999987,
            0.999986, 0.999985, 0.999985, 0.999984, 0.999983, 0.999982,
            0.999981, 0.999981, 0.99998, 0.999979, 0.999978, 0.999977, 0.999976,
            0.999974, 0.999973, 0.999972, 0.99997, 0.999969, 0.999968, 0.999966,
            0.999965, 0.999963, 0.999961, 0.999959, 0.999957, 0.999955,
            0.999953, 0.999951, 0.999949, 0.999946, 0.999944, 0.999941,
            0.999938, 0.999936, 0.999933, 0.99993, 0.999926, 0.999923, 0.999919,
            0.999916, 0.999912, 0.999908, 0.999904, 0.9999, 0.999896, 0.999891,
            0.999887, 0.999882, 0.999878, 0.999873, 0.999869, 0.999865,
            0.999861, 0.999863, 0.999858, 0.999843, 0.999833, 0.999823,
            0.999813, 0.999802, 0.999792, 0.999781, 0.99977, 0.999758, 0.999746,
            0.999734, 0.999721, 0.999707, 0.999693, 0.999678, 0.999662,
            0.999646, 0.999629, 0.999611, 0.999593, 0.999573, 0.999553,
            0.999532, 0.99951, 0.999487, 0.999464, 0.999439, 0.999413, 0.999386,
            0.999358, 0.999329, 0.999299, 0.999268, 0.999235, 0.999202,
            0.999167, 0.999132, 0.999096, 0.999059, 0.999021, 0.99898, 0.99895,
            0.99891, 0.99887, 0.99883, 0.99879, 0.99876, 0.99874, 0.99872,
            0.99871, 0.99874, 0.9989, 0.99887, 0.99862, 0.99848, 0.99835,
            0.99824, 0.99812, 0.99801, 0.99789, 0.99778, 0.99765, 0.99753,
            0.9974, 0.99727, 0.99713, 0.99698, 0.99683, 0.99667, 0.99651,
            0.99634, 0.99616, 0.99598, 0.99579, 0.99559, 0.99539, 0.99518,
            0.99496, 0.99473, 0.9945, 0.99426, 0.99401, 0.99375, 0.99348,
            0.9932, 0.99291, 0.9926, 0.99229, 0.99197, 0.99165, 0.99131,
            0.99097, 0.99063, 0.99028, 0.9899, 0.9896, 0.9892, 0.9889, 0.9886,
            0.9884, 0.9882, 0.9881, 0.9879, 0.9878, 0.9875, 0.9872, 0.9868,
            0.9863, 0.9871, 0.9859, 0.9865, 0.9855, 0.986, 0.9853, 0.9855,
            0.9853, 0.9853, 0.9853, 0.9854, 0.9858, 0.9854, 0.9868, 0.9854,
            0.9872, 0.9854, 0.9874, 0.9855, 0.9865, 0.9853, 0.9851, 0.9849,
            0.9841, 0.98455, 0.9844, 0.9834, 0.9822, 0.9827, 0.9828, 0.9819,
            0.9867, 0.9858, 0.9813, 0.9839, 0.9809, 0.9823, 0.9821, 0.9813,
            0.9803, 0.9792, 0.9794, 0.9762, 0.9789, 0.9778, 0.974, 0.977,
            0.9721, 0.9761, 0.9747, 0.9683, 0.9665, 0.9646, 0.9634, 0.9627,
            0.9608, 0.9588, 0.9608, 0.9568, 0.9547, 0.9562, 0.9525, 0.9503,
            0.9481, 0.9509, 0.9457, 0.9434, 0.9458, 0.9409, 0.9384, 0.9416,
            0.9359, 0.9333, 0.9386, 0.9307, 0.9279, 0.9328, 0.9252, 0.9224,
            0.9271, 0.9196, 0.9168, 0.9222, 0.914, 0.9164, 0.9105, 0.9207,
            0.9175, 0.9137, 0.913, 0.907, 0.901, 0.895, 0.888, 0.882, 0.877,
            0.87, 0.866, 0.858, 0.851, 0.845, 0.839, 0.833, 0.827, 0.822, 0.817,
            0.813, 0.808, 0.804, 0.733, 0.803, 0.804, 0.753, 0.806, 0.811,
            0.774, 0.817, 0.822, 0.829, 0.797, 0.833, 0.843, 0.851, 0.827,
            0.862, 0.859, 0.879, 0.902, 0.927, 0.957, 0.975, 0.999, 1.03, 1.072,
            1.124, 1.137, 1.156, 1.172, 1.178, 1.172, 1.167, 1.168, 1.175,
            1.195, 1.225, 1.265, 1.32, 1.363, 1.371, 1.368, 1.372, 1.383, 1.41,
            1.475, 1.554, 1.635, 1.716, 1.766, 1.739, 1.687, 1.587, 1.513,
            1.492, 1.567, 1.645, 1.772, 1.919, 2.048, 2.152, 2.24, 2.332, 2.33,
            2.292, 2.243, 2.19, 2.14, 2.092, 2.047, 2.006, 1.969, 1.935, 1.904,
            1.876, 1.85, 1.825, 1.803, 1.783, 1.764, 1.747, 1.73, 1.716, 1.702,
            1.676, 1.653, 1.633, 1.616, 1.6, 1.582, 1.567, 1.554, 1.543,
            1.53429, 1.53371, 1.52276, 1.52009, 1.51474, 1.51338, 1.50841,
            1.50004, 1.49805, 1.49592, 1.49404, 1.49099, 1.48873, 1.48719,
            1.48053, 1.47976, 1.47858, 1.47746, 1.47512, 1.47453, 1.46961,
            1.46669, 1.46429, 1.46313, 1.46187, 1.46008, 1.45885, 1.45877,
            1.45847, 1.45841, 1.45671, 1.45637, 1.45608, 1.45515, 1.45248,
            1.45185, 1.45025, 1.44941, 1.44888, 1.44621, 1.44584, 1.44497,
            1.44427, 1.44267, 1.44226, 1.44205, 1.44069, 1.43851, 1.43722,
            1.43576, 1.43292, 1.43095, 1.41314, 1.41253, 1.41155, 1.40568,
            1.40418, 1.39936, 1.395, 1.383, 1.365, 1.342, 1.306, 1.278, 1.239,
            1.212, 1.175, 1.158, 1.135, 1.107, 1.084, 1.053, 1.014, 0.9702,
            0.9488, 0.9175, 0.8897, 0.86, 0.8213, 0.7719, 0.7037, 0.6232,
            0.5456, 0.4677, 0.4113, 0.3931, 0.402, 0.4329, 0.453, 0.46, 0.473,
            0.4746, 0.4656, 0.4563, 0.4309, 0.3915, 0.3563, 0.3705, 0.5846,
            1.043, 1.616, 2.25, 2.76, 2.839, 2.694, 2.448, 2.224, 2.038, 1.869,
            1.784, 1.69, 1.652, 1.619, 1.615, 1.658, 1.701, 12.5, 12.66, 12.82,
            12.99, 13.16, 13.33, 13.79, 14.29, 14.81, 15.38, 16, 16.67, 17.24,
            17.86, 18.52, 18.97, 19.23, 19.61, 20, 20.41, 20.83, 21.05, 21.28,
            21.74, 22.73, 23.81, 25, 26.67, 28.57, 30.77, 33.33, 36.36, 40, 100,
            125, 166.7, 250, 500
        ]
        Ek = [
            10000, 9772.4, 9549.9, 9332.5, 9120.1, 8912.5, 8709.6, 8511.4,
            8317.6, 8128.3, 7943.3, 7762.5, 7585.8, 7413.1, 7244.4, 7079.5,
            6918.3, 6760.8, 6606.9, 6456.5, 6309.6, 6166, 6025.6, 5888.4,
            5754.4, 5623.4, 5495.4, 5370.3, 5248.1, 5128.6, 5011.9, 4897.8,
            4786.3, 4677.4, 4570.9, 4466.8, 4365.2, 4265.8, 4168.7, 4073.8,
            3981.1, 3890.5, 3801.9, 3715.4, 3630.8, 3548.1, 3467.4, 3388.4,
            3311.3, 3235.9, 3162.3, 3090.3, 3020, 2951.2, 2884, 2818.4, 2754.2,
            2691.5, 2630.3, 2570.4, 2511.9, 2454.7, 2398.8, 2344.2, 2290.9,
            2238.7, 2187.8, 2138, 2089.3, 2041.7, 1995.3, 1949.8, 1905.5,
            1862.1, 1819.7, 1778.3, 1737.8, 1698.2, 1659.6, 1621.8, 1584.9,
            1548.8, 1513.6, 1479.1, 1445.4, 1412.5, 1380.4, 1349, 1318.3,
            1288.2, 1258.9, 1230.3, 1202.3, 1174.9, 1148.2, 1122, 1096.5,
            1071.5, 1047.1, 1023.3, 1000, 977.2, 955, 933.3, 912, 891.3, 871,
            851.1, 831.8, 812.8, 794.3, 776.3, 758.6, 741.3, 724.4, 708, 691.8,
            676.1, 660.7, 645.7, 631, 616.6, 602.6, 588.8, 575.4, 562.3, 549.5,
            537, 524.8, 512.9, 501.2, 489.8, 478.6, 467.7, 457.1, 446.7, 436.5,
            426.6, 416.9, 407.4, 398.1, 389, 380.2, 371.5, 363.1, 354.8, 346.7,
            338.8, 331.1, 323.6, 316.2, 309, 302, 295.1, 288.4, 281.8, 275.4,
            269.1, 263, 257, 251.2, 245.5, 239.9, 234.4, 229.1, 223.9, 218.8,
            213.8, 208.9, 204.2, 199.5, 195, 190.6, 186.2, 182, 177.8, 173.8,
            169.8, 166, 162.2, 158.5, 154.9, 151.4, 151.2, 147.9, 147.6, 144.5,
            144.2, 141.3, 140.9, 138, 137.8, 134.9, 134.8, 131.9, 131.8, 129.2,
            128.8, 126.5, 125.9, 124, 123, 121.6, 120.2, 119.2, 117.5, 117,
            114.8, 112.7, 112.2, 110.7, 109.7, 108.8, 107.2, 106.9, 105.1,
            104.7, 103.3, 102.3, 101.6, 100, 99.99, 98.4, 97.7, 96.86, 95.5,
            95.37, 93.93, 93.3, 92.53, 91.2, 91.17, 89.84, 87.1, 85.1, 83.2,
            82.66, 81.3, 79.4, 77.6, 77.49, 75.9, 74.1, 72.93, 72.4, 70.8, 69.2,
            68.88, 67.6, 66.1, 65.26, 64.6, 63.1, 61.99, 61.7, 60.3, 59.04,
            58.9, 57.5, 56.36, 56.2, 55, 53.91, 53.7, 52.5, 51.66, 51.3, 49.59,
            47.69, 45.92, 44.28, 42.75, 41.33, 40, 38.75, 37.57, 36.47, 35.42,
            34.44, 33.51, 32.63, 31.79, 31, 30.24, 29.52, 28.83, 28.18, 27.55,
            26.95, 26.38, 25.83, 25.3, 25, 24.8, 24.31, 24, 23.84, 23.39, 23,
            22.96, 22.54, 22.14, 22, 21.75, 21.38, 21.01, 21, 20.66, 20, 19.5,
            19, 18.5, 18, 17.75, 17.5, 17.25, 17, 16.75, 16.5, 16.25, 16, 15.75,
            15.5, 15.25, 15, 14.75, 14.5, 14.25, 14, 13.75, 13.5, 13.25, 13,
            12.75, 12.5, 12.25, 12, 11.8, 11.6, 11.4, 11.2, 11, 10.9, 10.8,
            10.7, 10.6, 10.5, 10.45, 10.4, 10.35, 10.3, 10.25, 10.2, 10.1, 10,
            9.9, 9.8, 9.7, 9.6, 9.5, 9.4, 9.3, 9.2, 9.1, 9, 8.9, 8.8, 8.7, 8.6,
            8.5, 8.4, 8.3, 8.2, 8.1, 8, 7.8, 0.1612, 0.1599, 0.1587, 0.1575,
            0.1562, 0.155, 0.1537, 0.1525, 0.1513, 0.15, 0.1488, 0.1475, 0.1463,
            0.1451, 0.1438, 0.1426, 0.1413, 0.1401, 0.1389, 0.1376, 0.1364,
            0.1351, 0.1333, 0.1302, 0.1271, 0.124, 0.1209, 0.1178, 0.1147,
            0.1116, 0.1091, 0.1066, 0.1054, 0.1041, 0.1029, 0.1017, 0.1004,
            0.09919, 0.09795, 0.09671, 0.09547, 0.09423, 0.09299, 0.08989,
            0.08679, 0.08369, 0.08059, 0.07749, 0.07439, 0.07191, 0.06943,
            0.06695, 0.06534, 0.06447, 0.06323, 0.06199, 0.06075, 0.05951,
            0.05889, 0.05827, 0.05703, 0.05455, 0.05207, 0.04959, 0.04649,
            0.04339, 0.0403, 0.0372, 0.0341, 0.031, 0.0124, 0.009919, 0.007439,
            0.004959, 0.00248
        ]
        k0 = [
            4.68e-08, 5.13e-08, 5.62e-08, 6.16e-08, 6.75e-08, 7.4e-08, 8.1e-08,
            8.88e-08, 9.73e-08, 1.07e-07, 1.17e-07, 1.28e-07, 1.4e-07, 1.53e-07,
            1.68e-07, 1.84e-07, 2.01e-07, 2.2e-07, 2.41e-07, 2.64e-07, 2.89e-07,
            3.16e-07, 3.46e-07, 3.79e-07, 4.14e-07, 4.53e-07, 4.96e-07,
            5.42e-07, 5.93e-07, 6.48e-07, 7.09e-07, 7.75e-07, 8.47e-07,
            9.26e-07, 1.01e-06, 1.11e-06, 1.21e-06, 1.32e-06, 1.44e-06,
            1.58e-06, 1.72e-06, 1.88e-06, 2.05e-06, 2.24e-06, 2.44e-06,
            2.67e-06, 2.91e-06, 3.17e-06, 3.46e-06, 3.77e-06, 4.11e-06,
            4.48e-06, 4.89e-06, 5.32e-06, 5.8e-06, 6.32e-06, 6.88e-06, 7.49e-06,
            8.15e-06, 8.86e-06, 9.64e-06, 1.05e-05, 1.14e-05, 1.24e-05,
            1.35e-05, 1.46e-05, 1.59e-05, 1.72e-05, 1.87e-05, 2.03e-05, 2.2e-05,
            2.38e-05, 2.58e-05, 2.8e-05, 9.04e-06, 9.87e-06, 1.08e-05, 1.18e-05,
            1.28e-05, 1.4e-05, 1.53e-05, 1.66e-05, 1.81e-05, 1.98e-05, 2.15e-05,
            2.35e-05, 2.55e-05, 2.78e-05, 3.03e-05, 3.3e-05, 3.59e-05, 3.91e-05,
            4.25e-05, 4.62e-05, 5.03e-05, 5.47e-05, 5.95e-05, 6.48e-05,
            7.05e-05, 7.66e-05, 8.29e-05, 9e-05, 9.8e-05, 0.000107, 0.000116,
            0.000125, 0.000136, 0.000147, 0.00016, 0.000174, 0.000189, 0.000206,
            0.000223, 0.000242, 0.000263, 0.000285, 0.000308, 0.000333,
            0.000362, 0.000393, 0.000426, 0.000461, 0.000499, 0.00054, 0.000585,
            0.000633, 0.000686, 0.000204, 0.000221, 0.000239, 0.000258,
            0.000279, 0.000302, 0.000326, 0.000352, 0.00038, 0.000411, 0.000444,
            0.00048, 0.000518, 0.00056, 0.000605, 0.000653, 0.000706, 0.000764,
            0.000826, 0.000893, 0.000965, 0.00104, 0.00113, 0.00122, 0.00132,
            0.00143, 0.00154, 0.00166, 0.00179, 0.00192, 0.00207, 0.00222,
            0.00239, 0.00257, 0.00277, 0.00298, 0.00321, 0.00346, 0.00372,
            0.00401, 0.00432, 0.00466, 0.00502, 0.00541, 0.00583, 0.00628,
            0.00678, 0.0073, 0.00776, 0.00821, 0.00861, 0.00892, 0.00918,
            0.00948, 0.00979, 0.0102, 0.0073, 0.0108, 0.0076, 0.0115, 0.0081,
            0.0123, 0.0089, 0.0129, 0.0099, 0.0134, 0.0107, 0.0114, 0.0138,
            0.0111, 0.0141, 0.0111, 0.0143, 0.0097, 0.0144, 0.0085, 0.0141,
            0.0083, 0.0139, 0.0096, 0.0118, 0.009, 0.0138, 0.0093, 0.0139,
            0.0159, 0.0142, 0.0106, 0.0075, 0.0145, 0.0065, 0.0148, 0.0068,
            0.0152, 0.007, 0.0073, 0.0109, 0.0076, 0.0116, 0.0076, 0.0083,
            0.0124, 0.0087, 0.0132, 0.0093, 0.0103, 0.0149, 0.0159, 0.0169,
            0.0199, 0.018, 0.0192, 0.0204, 0.0226, 0.0218, 0.0232, 0.0244,
            0.0246, 0.0262, 0.0279, 0.0281, 0.0297, 0.0316, 0.0335, 0.0337,
            0.0359, 0.0374, 0.0382, 0.0407, 0.0437, 0.0433, 0.0461, 0.0462,
            0.0491, 0.0523, 0.052, 0.0556, 0.0592, 0.0578, 0.0631, 0.065,
            0.0726, 0.068, 0.075, 0.082, 0.09, 0.092, 0.094, 0.098, 0.107,
            0.113, 0.12, 0.128, 0.137, 0.144, 0.156, 0.169, 0.18, 0.19, 0.202,
            0.218, 0.233, 0.25, 0.27, 0.282, 0.325, 0.3, 0.322, 0.375, 0.343,
            0.366, 0.434, 0.385, 0.408, 0.43, 0.48, 0.45, 0.47, 0.482, 0.53,
            0.497, 0.585, 0.613, 0.645, 0.677, 0.712, 0.731, 0.75, 0.763, 0.768,
            0.765, 0.755, 0.737, 0.717, 0.703, 0.696, 0.699, 0.711, 0.739,
            0.771, 0.799, 0.808, 0.795, 0.775, 0.755, 0.747, 0.766, 0.793,
            0.824, 0.861, 0.874, 0.859, 0.81, 0.718, 0.569, 0.565, 0.618, 0.725,
            0.914, 1.11, 1.136, 1.13, 1.045, 0.925, 0.81, 0.715, 0.46, 0.323,
            0.236, 0.168, 0.119, 0.077, 0.0561, 0.043, 0.0339, 0.0271, 0.0228,
            0.0189, 0.0156, 0.0132, 0.0109, 0.00838, 0.00557, 0.00317, 0.0014,
            0.000463, 0.000122, 3.2e-05, 4.7e-06, 0.0372, 0.0474, 0.0768, 0.132,
            0.216, 0.323, 0.446, 0.553, 0.635, 0.704, 0.771, 0.84, 0.903, 0.978,
            1.07, 1.17, 1.32, 1.53, 1.85, 2.27, 2.55, 2.63, 2.26, 1.65, 0.962,
            0.509, 0.231, 0.102, 0.046, 0.0506, 0.0775, 0.116, 0.152, 0.204,
            0.267, 0.323, 0.341, 1.753, 1.789, 1.811, 1.81, 1.779, 1.756, 1.698,
            1.643, 1.598, 1.555, 1.502, 1.45, 1.401, 1.337, 1.235, 1.161, 1.05,
            0.8857, 0.6616, 0.5777, 0.7517, 1.002, 1.484, 2.308, 2.936, 2.912,
            2.739, 2.537, 2.388, 2.284, 2.21, 2.147, 2.1, 1.967, 1.962, 1.959,
            1.957, 1.955
        ]
        return interpolate_using_energy(wavelengths, n0, k0, En, Ek)


class Si(Material):

    def refractive_index(self, wavelengths: List[float]) -> List[float]:
        #Si refractive index with respect to wavelength (wl, nm)

        # SOURCE: Palik, Handbook of Optical Constants of Solids, 1995
        # Table data located at http://ftp.esrf.eu/pub/scisoft/xop2.3/DabaxFiles/nk_Palik.dat
        #
        # DATA: Al_x Ga_(1-x) As refractive index with respect to photon energy
        #     n   Refractive index
        #     k   Extinction coefficient
        #     Ez  Photon energy (eV)  where z = {n, k}
        En = [
            2000, 1952, 1905, 1860, 1815, 1772, 1730, 1688, 1648, 1609, 1570,
            1533, 1496, 1460, 1426, 1392, 1358, 1326, 1294, 1263, 1233, 1204,
            1175, 1147, 1119, 1093, 1067, 1041, 1016, 992, 968, 945, 923, 901,
            879, 858, 818, 798, 779, 760, 742, 725, 707, 690, 674, 658, 642,
            627, 612, 597, 583, 569, 555, 542, 529, 516, 504, 492, 480, 469,
            458, 447, 436, 426, 415, 406, 396, 386, 377, 368, 359, 351, 342,
            334, 326, 318, 311, 303, 296, 289, 282, 275, 269, 262, 256, 250,
            244, 238, 232, 227, 221, 216, 211, 206, 201, 196, 191, 187, 182,
            178, 174, 170, 166, 162, 158, 154, 150, 147, 143, 140, 136, 133,
            130, 127, 124, 121, 118, 115, 112, 110, 107, 104, 102, 101.6, 100.8,
            100, 99.19, 98.4, 96.86, 95.37, 93.93, 91.17, 88.56, 82.66, 77.49,
            72.93, 68.88, 65.26, 61.99, 59.04, 56.36, 53.91, 51.66, 49.59,
            47.69, 45.92, 44.28, 42.75, 41.33, 40, 38.75, 37.57, 36.47, 35.42,
            34.44, 33.51, 32.63, 31.79, 31, 30.24, 29.52, 28.83, 28.18, 27.55,
            26.95, 26.38, 25.83, 25.3, 24.8, 24.31, 23.84, 23.39, 22.96, 22.54,
            22.14, 21.75, 21.38, 21.01, 20.66, 20.33, 20, 19.68, 19.5, 19.37,
            19.07, 19, 18.5, 18, 17.75, 17.5, 17.25, 17, 16.8, 16.7, 16.6, 16.5,
            16.25, 16, 15.75, 15.5, 15.25, 15, 14.75, 14.5, 14.25, 14, 13.75,
            13.5, 13.25, 13, 12.75, 12.5, 12.25, 12, 11.75, 11.5, 11.25, 11,
            10.75, 10.5, 10.25, 10, 9.75, 9.5, 9.25, 9, 8.75, 8.5, 8.25, 8,
            7.75, 7.5, 7.25, 7, 6.75, 6.5, 6.25, 6, 5.8, 5.6, 5.5, 5.4, 5.3,
            5.2, 5.1, 5, 1.107, 1.084, 1.033, 0.9037, 0.8856, 0.8093, 0.7749,
            0.731, 0.6888, 0.6199, 0.5087, 0.4959, 0.4568, 0.4133, 0.3753,
            0.3626, 0.3542, 0.31, 0.248, 0.2066, 0.1771, 0.155, 0.1378, 0.124,
            0.09184, 0.0855, 0.08265, 0.06199, 0.04959, 0.04092, 0.0248,
            0.01984, 0.01736, 0.01488, 0.01364, 0.0124, 0.009919, 0.00743,
            0.006199, 0.004959, 0.00372
        ]
        n0 = [
            0.999905, 0.999913, 0.999922, 0.999932, 0.999911, 0.999885,
            0.999874, 0.999863, 0.999854, 0.999844, 0.999834, 0.999824,
            0.999814, 0.999803, 0.999792, 0.999781, 0.999769, 0.999756,
            0.999743, 0.999729, 0.999715, 0.9997, 0.999684, 0.999667, 0.999649,
            0.999631, 0.999612, 0.999591, 0.99957, 0.999548, 0.999524, 0.9995,
            0.999474, 0.999447, 0.999418, 0.999388, 0.999325, 0.999289,
            0.999253, 0.999214, 0.999174, 0.999134, 0.999088, 0.999041, 0.99899,
            0.99894, 0.99889, 0.99883, 0.99878, 0.99871, 0.99865, 0.99858,
            0.99851, 0.99843, 0.99836, 0.99827, 0.99819, 0.9981, 0.998, 0.99791,
            0.99781, 0.9977, 0.99759, 0.99748, 0.99735, 0.99723, 0.9971,
            0.99695, 0.99682, 0.99667, 0.99651, 0.99636, 0.99619, 0.99602,
            0.99585, 0.99566, 0.99549, 0.99528, 0.99509, 0.99489, 0.99468,
            0.99446, 0.99427, 0.99403, 0.99382, 0.99361, 0.99339, 0.99318,
            0.99296, 0.99277, 0.99254, 0.99236, 0.99219, 0.99206, 0.99192,
            0.99179, 0.99169, 0.99167, 0.99164, 0.99164, 0.99167, 0.99176,
            0.99235, 0.99235, 0.99287, 0.9934, 0.99384, 0.99415, 0.9946,
            0.99496, 0.99549, 0.99594, 0.99645, 0.99705, 0.99773, 0.99856,
            0.99958, 1.0008, 1.0024, 1.0039, 1.007, 1.0132, 1.0241, 1.025, 1.03,
            1.032, 1.034, 1.034, 1.034, 1.03, 1.022, 1.008, 1, 0.993, 0.991,
            0.988, 0.985, 0.982, 0.978, 0.976, 0.972, 0.968, 0.964, 0.96, 0.956,
            0.952, 0.947, 0.942, 0.937, 0.93, 0.925, 0.918, 0.913, 0.906, 0.899,
            0.893, 0.885, 0.877, 0.869, 0.86, 0.853, 0.843, 0.834, 0.824, 0.814,
            0.803, 0.792, 0.778, 0.766, 0.752, 0.737, 0.722, 0.706, 0.691,
            0.675, 0.659, 0.644, 0.627, 0.61, 0.59, 0.568, 0.549, 0.542, 0.53,
            0.513, 0.514, 0.485, 0.455, 0.44, 0.426, 0.411, 0.397, 0.386, 0.379,
            0.374, 0.369, 0.357, 0.345, 0.333, 0.323, 0.313, 0.304, 0.296,
            0.288, 0.281, 0.275, 0.269, 0.265, 0.261, 0.258, 0.256, 0.255,
            0.256, 0.257, 0.259, 0.263, 0.267, 0.272, 0.278, 0.286, 0.295,
            0.306, 0.318, 0.332, 0.348, 0.367, 0.389, 0.414, 0.444, 0.478,
            0.517, 0.563, 0.618, 0.682, 0.756, 0.847, 0.968, 1.11, 1.24, 1.4,
            1.51, 1.64, 1.75, 1.78, 1.72, 1.68, 3.5361, 3.5295, 3.5193, 3.5007,
            3.4876, 3.4784, 3.471, 3.4644, 3.4578, 3.449, 3.4434, 3.4424,
            3.4393, 3.4361, 3.4335, 3.4327, 3.4321, 3.4294, 3.4261, 3.4242,
            3.4231, 3.4224, 3.4219, 3.4215, 3.4209, 3.4208, 3.4207, 3.4204,
            3.4201, 3.42, 3.4197, 3.4195, 3.4192, 3.419, 3.4188, 3.4185, 3.418,
            3.417, 3.4165, 3.416, 3.4155
        ]
        Ek = [
            2000, 1952, 1905, 1860, 1815, 1772, 1730, 1688, 1648, 1609, 1570,
            1533, 1496, 1460, 1426, 1392, 1358, 1326, 1294, 1263, 1233, 1204,
            1175, 1147, 1119, 1093, 1067, 1041, 1016, 992, 968, 945, 923, 901,
            879, 858, 818, 798, 779, 760, 742, 725, 707, 690, 674, 658, 642,
            627, 612, 597, 583, 569, 555, 542, 529, 516, 504, 492, 480, 469,
            458, 447, 436, 426, 415, 406, 396, 386, 377, 368, 359, 351, 342,
            334, 326, 318, 311, 303, 296, 289, 282, 275, 269, 262, 256, 250,
            244, 238, 232, 227, 221, 216, 211, 206, 201, 196, 191, 187, 182,
            178, 174, 170, 166, 162, 158, 154, 150, 147, 143, 140, 136, 133,
            130, 127, 124, 121, 118, 115, 112, 110, 107, 104, 102, 101.6, 100.8,
            100, 99.19, 98.4, 96.86, 95.37, 93.93, 91.17, 88.56, 82.66, 77.49,
            72.93, 68.88, 65.26, 61.99, 59.04, 56.36, 53.91, 51.66, 49.59,
            47.69, 45.92, 44.28, 42.75, 41.33, 40, 38.75, 37.57, 36.47, 35.42,
            34.44, 33.51, 32.63, 31.79, 31, 30.24, 29.52, 28.83, 28.18, 27.55,
            26.95, 26.38, 25.83, 25.3, 24.8, 24.31, 23.84, 23.39, 22.96, 22.54,
            22.14, 21.75, 21.38, 21.01, 20.66, 20.33, 20, 19.68, 19.5, 19.37,
            19.07, 19, 18.5, 18, 17.75, 17.5, 17.25, 17, 16.8, 16.7, 16.6, 16.5,
            16.25, 16, 15.75, 15.5, 15.25, 15, 14.75, 14.5, 14.25, 14, 13.75,
            13.5, 13.25, 13, 12.75, 12.5, 12.25, 12, 11.75, 11.5, 11.25, 11,
            10.75, 10.5, 10.25, 10, 9.75, 9.5, 9.25, 9, 8.75, 8.5, 8.25, 8,
            7.75, 7.5, 7.25, 7, 6.75, 6.5, 6.25, 6, 5.8, 5.6, 5.5, 5.4, 5.3,
            5.2, 5.1, 5, 2, 1.9, 1.8, 1.7, 1.6, 1.5, 1.4, 1.3, 1.2, 0.4568,
            0.3263, 0.248, 0.2418, 0.2356, 0.2294, 0.2232, 0.217, 0.2108,
            0.2046, 0.1984, 0.1922, 0.186, 0.1798, 0.1736, 0.1674, 0.1612,
            0.155, 0.1488, 0.1395, 0.1389, 0.1376, 0.1364, 0.1339, 0.1314,
            0.1289, 0.1265, 0.124, 0.1215, 0.1203, 0.119, 0.1178, 0.1165,
            0.1141, 0.1116, 0.1103, 0.1091, 0.1079, 0.1066, 0.1054, 0.1041,
            0.1029, 0.1017, 0.1004, 0.09919, 0.09795, 0.09671, 0.09547, 0.09423,
            0.09299, 0.09175, 0.09051, 0.08927, 0.08803, 0.08679, 0.08555,
            0.08431, 0.08307, 0.08183, 0.08059, 0.07935, 0.07811, 0.07687,
            0.07563, 0.07513, 0.07439, 0.07315, 0.07191, 0.07067, 0.06943,
            0.06695, 0.06447, 0.06199, 0.05951, 0.05703, 0.05455, 0.05207,
            0.04959, 0.04835, 0.04711, 0.04587, 0.04463, 0.04215, 0.01984,
            0.01736, 0.01488, 0.0124, 0.01116, 0.009919, 0.008679, 0.00743,
            0.006199, 0.004959
        ]
        k0 = [
            3.19e-05, 3.46e-05, 3.76e-05, 4.08e-05, 3.93e-06, 4.32e-06,
            4.74e-06, 5.2e-06, 5.7e-06, 6.24e-06, 6.85e-06, 7.49e-06, 8.22e-06,
            8.99e-06, 9.81e-06, 1.07e-05, 1.17e-05, 1.28e-05, 1.4e-05, 1.53e-05,
            1.67e-05, 1.83e-05, 2e-05, 2.18e-05, 2.39e-05, 2.6e-05, 2.84e-05,
            3.1e-05, 3.39e-05, 3.7e-05, 4.04e-05, 4.41e-05, 4.8e-05, 5.24e-05,
            5.73e-05, 6.25e-05, 7.42e-05, 8.11e-05, 8.84e-05, 9.66e-05,
            0.000105, 0.000114, 0.000125, 0.000136, 0.000148, 0.000161,
            0.000176, 0.000191, 0.000208, 0.000227, 0.000247, 0.000269,
            0.000294, 0.000319, 0.000347, 0.000379, 0.000411, 0.000447,
            0.000487, 0.000527, 0.000572, 0.000622, 0.000677, 0.000734,
            0.000801, 0.000863, 0.000939, 0.00102, 0.00111, 0.0012, 0.00131,
            0.00141, 0.00153, 0.00166, 0.0018, 0.00195, 0.0021, 0.00229,
            0.00247, 0.00267, 0.00288, 0.00313, 0.00336, 0.00365, 0.00393,
            0.00423, 0.00457, 0.00494, 0.00534, 0.0057, 0.00619, 0.00663,
            0.00712, 0.00763, 0.00818, 0.00879, 0.00946, 0.01, 0.0107, 0.0114,
            0.0121, 0.0128, 0.0145, 0.0145, 0.0152, 0.0157, 0.0162, 0.0166,
            0.0172, 0.0177, 0.0183, 0.0188, 0.0194, 0.02, 0.0206, 0.0212,
            0.0219, 0.0226, 0.0233, 0.0239, 0.0247, 0.0256, 0.0262, 0.024,
            0.005, 0.00154, 0.001, 0.00102, 0.00109, 0.00113, 0.0012, 0.0013,
            0.00143, 0.00178, 0.00215, 0.00254, 0.00297, 0.00343, 0.00393,
            0.00443, 0.005, 0.00553, 0.0061, 0.0067, 0.0073, 0.00785, 0.0084,
            0.009, 0.0095, 0.01, 0.0104, 0.01, 0.0113, 0.0117, 0.0121, 0.0124,
            0.0128, 0.0132, 0.0135, 0.0138, 0.0142, 0.0147, 0.0152, 0.0158,
            0.0168, 0.0178, 0.0192, 0.0205, 0.0223, 0.0243, 0.0264, 0.0292,
            0.0325, 0.0365, 0.0405, 0.0455, 0.051, 0.058, 0.065, 0.074, 0.10275,
            0.093, 0.14, 0.1, 0.113, 0.163, 0.189, 0.219, 0.237, 0.255, 0.275,
            0.296, 0.314, 0.323, 0.333, 0.342, 0.367, 0.394, 0.421, 0.45, 0.479,
            0.51, 0.541, 0.573, 0.607, 0.641, 0.677, 0.714, 0.752, 0.792, 0.833,
            0.875, 0.918, 0.963, 1.01, 1.06, 1.11, 1.16, 1.21, 1.26, 1.32, 1.38,
            1.45, 1.51, 1.58, 1.66, 1.73, 1.82, 1.9, 2, 2.1, 2.21, 2.32, 2.45,
            2.58, 2.73, 2.89, 3.05, 3.18, 3.33, 3.4, 3.44, 3.42, 3.36, 3.42,
            3.58, 0.0296, 0.024, 0.0225, 0.0197, 0.0142, 0.0125, 0.00775,
            0.00226, 0.00107, 2.5e-09, 1.3e-08, 1.99e-07, 2.82e-07, 2.97e-07,
            3.26e-07, 3.94e-07, 4.05e-07, 4.17e-07, 5.64e-07, 8.46e-07,
            1.74e-06, 2.66e-06, 2.45e-05, 1.775e-05, 1.62e-05, 2.655e-05,
            1.53e-05, 2.41e-05, 6.44e-05, 8.38e-05, 8.46e-05, 6.435e-05,
            4.72e-05, 3.75e-05, 3.67e-05, 4.84e-05, 6.76e-05, 0.000109,
            0.000122, 0.000124, 0.000122, 0.000127, 0.000166, 0.000202,
            0.000208, 0.000202, 0.000201, 0.000173, 0.000153, 0.000144,
            0.000177, 0.000206, 0.000197, 0.000177, 0.000207, 0.000214,
            0.000227, 0.000239, 0.000259, 0.000312, 0.000303, 0.000221,
            0.000157, 0.000102, 0.000112, 0.000125, 0.000124, 0.000141,
            0.000152, 0.000164, 0.000505, 0.000963, 0.00111, 0.00125, 0.000716,
            0.000418, 0.000365, 0.000415, 0.000384, 0.00034, 0.000318, 0.000286,
            0.000222, 0.000156, 0.000127, 0.000106, 9.15e-05, 9.39e-05,
            0.000105, 0.000112, 0.000108, 0.000108, 0.00017, 0.00019, 0.00023,
            0.00029, 0.00035, 0.00043, 0.00055, 0.00072, 0.001, 0.0014
        ]
        return interpolate_using_energy(wavelengths, n0, k0, En, Ek)


class Si3N4(Material):
    eV = [
        24, 23, 22, 21, 20, 19, 18, 17, 16, 15, 14, 13, 12, 11, 10.5, 10, 9.5,
        9, 8.5, 8, 7.75, 7.5, 7.25, 7, 6.75, 6.5, 6.25, 6, 5.75, 5.5, 5.25, 5,
        4.75, 4.5, 4.25, 4, 3.5, 3, 2.5, 2, 1.5, 1
    ]
    n = [
        0.655, 0.625, 0.611, 0.617, 0.635, 0.676, 0.735, 0.810, 0.902, 1.001,
        1.111, 1.247, 1.417, 1.657, 1.827, 2.000, 2.162, 2.326, 2.492, 2.651,
        2.711, 2.753, 2.766, 2.752, 2.724, 2.682, 2.620, 2.541, 2.464, 2.393,
        2.331, 2.278, 2.234, 2.198, 2.167, 2.141, 2.099, 2.066, 2.041, 2.022,
        2.008, 1.998
    ]
    k = [
        0.420, 0.481, 0.560, 0.647, 0.743, 0.841, 0.936, 1.03, 1.11, 1.18, 1.26,
        1.35, 1.43, 1.52, 1.53, 1.49, 1.44, 1.32, 1.16, 0.962, 0.866, 0.750,
        0.612, 0.493, 0.380, 0.273, 0.174, 0.102, 5.7e-2, 2.9e-2, 1.1e-2,
        4.9e-3, 1.2e-3, 2.3e-4, 0, 0, 0, 0, 0, 0, 0, 0
    ]

    def refractive_index(self, wavelengths: List[float]) -> List[float]:
        return interpolate_using_energy(wavelengths, Si3N4.n, Si3N4.k, Si3N4.eV,
                                        Si3N4.eV)
