# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['lido',
 'lido.constants',
 'lido.contracts',
 'lido.eth2deposit',
 'lido.eth2deposit.utils',
 'lido.multicall',
 'lido.utils']

package_data = \
{'': ['*'], 'lido.contracts': ['abi/goerli/*', 'abi/mainnet/*']}

install_requires = \
['py-ecc>=5.1.0,<6.0.0',
 'requests>=2.25.1,<3.0.0',
 'ssz>=0.2.4,<0.3.0',
 'web3>=5.15.0,<6.0.0']

setup_kwargs = {
    'name': 'lido',
    'version': '1.0.0',
    'description': 'Network helpers for Lido',
    'long_description': '# Lido\n\nThis library consolidates various functions to efficiently load network data for Lido, validate node operator keys and find key duplicates.\n\n## Installation\n\nThis library is available on PyPi:\n\n`pip install lido`\n\n## Quickstart\n\n```\nfrom web3 import Web3\nfrom lido import Lido\nw3 = Web3(...)\nlido = Lido(w3)\noperators = lido.fetch_and_validate()\n```\n\n## Main Features\n\n### Multicall Function Calls\n\nInstead of making network requests one-by-one, this library combines many requests into one RPC call. It uses [banteg/multicall.py](https://github.com/banteg/multicall.py), a Python wrapper for [makerdao/multicall](https://github.com/makerdao/multicall).\n\n### Multiprocess Signature Validations\n\nWhen using `validate_keys_multi()`, this library spreads processing of key signature validations to all system cores.\n\n### Automatic Testnet / Mainnet Switching\n\nDepending on which network is configured in web3 object, a set of contracts will be used. Even an appropriate ABI will be loaded for the chain automatically.\n\n## Helpers Provided\n\n- lido.get_operators_data() -> operator_data - load node operator data\n\n- lido.get_operators_keys(operator_data) -> operator_data - fetches and adds keys to operator_data\n- lido.validate_keys_mono(operator_data) -> operator_data - validates keys in single process and adds validation results to operator_data\n- lido.validate_keys_multi(operator_data) -> operator_data - validates keys in multiple processes and adds validation results to operator_data, requires a main function (see example)\n- lido.validate_key(chain_id, key, withdrawal_credentials) -> Boolean - low-level validation function, doesn\'t check for correct\n  chain_id and withdrawal_credentials for a Lido deployment. For most use-cases use validate_keys_multi or validate_key_list_multi instead\n- lido.find_duplicates(operator_data) -> operator_data - finds duplicate keys and adds results to operator_data\n\n- lido.fetch_and_validate() -> operator_data - combines fetching operator data and running all validations on it - useful when you would be running all validations on data anyway\n\n- lido.get_stats() -> stats - fetches various constants from Lido contract, but you can even pass a list of functions to fetch eg get_stats([isStopped])\n\nYou can mix and match these functions, but make sure to use get_operators_data() first.\n\n## Notes\n\n1. Signature validation will be skipped if its results are already present in operator_data. This way you can safely load validation results from cache and add `["valid_signature"] = Boolean` to already checked keys.\n\n## Running an example script\n\nThe example script uses web3.auto, so use a RPC provider url as an environment variable to run it:\n\n`WEB3_PROVIDER_URI=https://eth-mainnet.provider.xx example.py`\n\nSee `example.py` for a complete example, make sure to use a main function and a script entry point check when using validate_keys_multi() or fetch_and_validate().\n\n## Options\n\nIf you are testing a new deployment of Lido, you can override addresses and ABIs with constructor of Lido object. Also you can configure the maximum number of calls agregated to one multicall:\n\n```\nlido = Lido(\n    w3, \n    lido_address=LIDO_ADDRESS,\n    registry_address=REGISTRY_ADDRESS,\n    lido_abi_path=LIDO_ABI, # the file-path to the contract\'s ABI\n    registry_abi_path=REGISTRY_ABI, # the file-path to the contract\'s ABI\n    max_multicall=MAX_MULTICALL)\n```\n',
    'author': 'Lido',
    'author_email': 'info@lido.fi',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://lido.fi',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>3.7.1,<4',
}


setup(**setup_kwargs)
