"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Archiver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * Every week
 */
const DEFAULT_CRON_EXPRESSION = 'cron(0 0 ? * 1 *)';
class Archiver extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.kmsKey = this.createKey();
        this.logGroup = this.createLogGroup();
        this.topic = new aws_cdk_lib_1.aws_sns.Topic(this, 'notifications', {
            displayName: 'archiver-notifications',
            masterKey: this.kmsKey,
        });
        this.bucket = this.createArchiveBucket();
        this.createS3Notifications();
        this.createProjects();
        this.createCfnOutputs();
    }
    /**
     * Set up the S3-related event notifcations.
     *
     * @private
     * @memberof Archiver
     */
    createS3Notifications() {
        if (this.props.notificationEvents) {
            this.props.notificationEvents.forEach((event) => {
                this.bucket.addEventNotification(event, new aws_cdk_lib_1.aws_s3_notifications.SnsDestination(this.topic));
            });
        }
    }
    createCfnOutputs() {
        new aws_cdk_lib_1.CfnOutput(this, 's3-bucket-arn', {
            description: 'ARN of the S3 bucket storing the repositories.',
            value: this.bucket.bucketArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'log-group-arn', {
            description: 'ARN of the Cloudwatch Log group storing the code build logs.',
            value: this.logGroup.logGroupArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'kms-key', {
            description: 'ARN of the KMS key used to encrypt the Cloudwatch logs and the SNS topic.',
            value: this.kmsKey.keyArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'sns-topic-arn', {
            description: 'ARN of the SNS topic.',
            value: this.topic.topicArn,
        });
    }
    createLogGroup() {
        const loggroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'loggroup', {
            encryptionKey: this.kmsKey,
            retention: this.props.retentionDays
                ? this.props.retentionDays
                : aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        loggroup.node.addDependency(this.kmsKey);
        return loggroup;
    }
    /**
     *Create the S3 bucket that will later store the repositories.
     *
     * @private
     * @return {*}
     * @memberof Archiver
     */
    createArchiveBucket() {
        return new aws_cdk_lib_1.aws_s3.Bucket(this, 'destination', {
            blockPublicAccess: aws_cdk_lib_1.aws_s3.BlockPublicAccess.BLOCK_ALL,
            encryption: aws_cdk_lib_1.aws_s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
            lifecycleRules: [
                {
                    expiration: aws_cdk_lib_1.Duration.days(360),
                    transitions: [
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.GLACIER,
                            transitionAfter: aws_cdk_lib_1.Duration.days(90),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.DEEP_ARCHIVE,
                            transitionAfter: aws_cdk_lib_1.Duration.days(180),
                        },
                    ],
                },
            ],
            publicReadAccess: false,
            versioned: true,
        });
    }
    createKey() {
        const key = new aws_cdk_lib_1.aws_kms.Key(this, 'loggroupKey', {
            description: 'Azure DevOps git repository archiver',
            enableKeyRotation: true,
            pendingWindow: aws_cdk_lib_1.Duration.days(7),
            keyUsage: aws_cdk_lib_1.aws_kms.KeyUsage.ENCRYPT_DECRYPT,
            keySpec: aws_cdk_lib_1.aws_kms.KeySpec.SYMMETRIC_DEFAULT,
            alias: 'archiver-key',
        });
        key.grantEncryptDecrypt(new aws_cdk_lib_1.aws_iam.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`));
        if (this.props.notificationEvents) {
            key.grantEncryptDecrypt(new aws_cdk_lib_1.aws_iam.ServicePrincipal('s3.amazonaws.com'));
        }
        return key;
    }
    /**
     * Creates for each backup configuration a separate CodeBuild project.
     *
     * @private
     * @memberof Archiver
     */
    createProjects() {
        this.props.backupConfigurations.forEach((configuration) => {
            const project = this.createProject(configuration);
            this.bucket.grantReadWrite(project);
            this.createCloudwatchRule(project, configuration);
        });
    }
    createCloudwatchRule(project, element) {
        new aws_cdk_lib_1.aws_events.Rule(this, 'ScheduleRule-' + element.organizationName + '-' + element.projectName, {
            enabled: true,
            schedule: aws_cdk_lib_1.aws_events.Schedule.expression(DEFAULT_CRON_EXPRESSION),
            targets: [new aws_cdk_lib_1.aws_events_targets.CodeBuildProject(project)],
            description: 'Trigger for backing up Azure DevOps git repositories of organization ' +
                element.organizationName +
                ' and project ' +
                element.projectName +
                '.',
        });
    }
    /**
     * Create a CodeBuild project.
     *
     * @private
     * @param {BackupConfiguration} element
     * @return {*}
     * @memberof Archiver
     */
    createProject(element) {
        return new aws_cdk_lib_1.aws_codebuild.Project(this, 'archiver-' + element.organizationName + '-' + element.projectName, {
            timeout: aws_cdk_lib_1.Duration.hours(5),
            projectName: 'AzureDevOpsGitBackup' +
                '-' +
                element.organizationName +
                '-' +
                element.projectName,
            description: 'Backup Azure DevOps git repositories to an S3 bucket.',
            checkSecretsInPlainTextEnvVariables: true,
            concurrentBuildLimit: 90,
            environment: {
                environmentVariables: {
                    TOKEN: {
                        value: element.secretArn + ':pat',
                        type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    },
                    ORGANIZATION: { value: element.organizationName },
                    PROJECT: { value: element.projectName },
                },
                buildImage: aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.STANDARD_6_0,
            },
            logging: {
                cloudWatch: {
                    enabled: true,
                    logGroup: this.logGroup,
                },
            },
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject({
                version: 0.2,
                phases: {
                    build: {
                        commands: this.createCommands(element.repositoryNames),
                    },
                },
            }),
        });
    }
    createCommands(repositoryNames) {
        const output = [];
        repositoryNames.forEach((element) => {
            output.push('git clone --mirror "https://${TOKEN}@dev.azure.com/${ORGANIZATION}/${PROJECT}/_git/' + element + '"', 'tar czf ' + element + '.tgz ' + element + '.git', 'aws s3 cp ./' + element + '.tgz ' +
                this.bucket.s3UrlForObject() +
                '/${ORGANIZATION}/${PROJECT}/' + element + '.tgz', 'rm -f ' + element + '.tgz');
        });
        return output;
    }
}
exports.Archiver = Archiver;
_a = JSII_RTTI_SYMBOL_1;
Archiver[_a] = { fqn: "azure-devops-repository-archiver.Archiver", version: "1.0.1" };
//# sourceMappingURL=data:application/json;base64,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