# encoding: utf-8
#
# Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
# Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License.
# A copy of the License is located at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
# or in the "license" file accompanying this file. This file is distributed
# on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
# express or implied. See the License for the specific language governing
# permissions and limitations under the License.

from core.rest import *
from core.model import Gs2Constant
from datastore.request import *
from datastore.result import *


class Gs2DatastoreRestClient(AbstractGs2RestClient):

    def _describe_namespaces(
        self,
        request: DescribeNamespacesRequest,
        callback: Callable[[AsyncResult[DescribeNamespacesResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/"

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }
        if request.page_token is not None:
            query_strings["pageToken"] = request.page_token
        if request.limit is not None:
            query_strings["limit"] = request.limit

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='GET',
            result_type=DescribeNamespacesResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def describe_namespaces(
        self,
        request: DescribeNamespacesRequest,
    ) -> DescribeNamespacesResult:
        async_result = []
        with timeout(30):
            self._describe_namespaces(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def describe_namespaces_async(
        self,
        request: DescribeNamespacesRequest,
    ) -> DescribeNamespacesResult:
        async_result = []
        self._describe_namespaces(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _create_namespace(
        self,
        request: CreateNamespaceRequest,
        callback: Callable[[AsyncResult[CreateNamespaceResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/"

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.name is not None:
            body["name"] = request.name
        if request.description is not None:
            body["description"] = request.description
        if request.log_setting is not None:
            body["logSetting"] = request.log_setting.to_dict()
        if request.done_upload_script is not None:
            body["doneUploadScript"] = request.done_upload_script.to_dict()

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=CreateNamespaceResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def create_namespace(
        self,
        request: CreateNamespaceRequest,
    ) -> CreateNamespaceResult:
        async_result = []
        with timeout(30):
            self._create_namespace(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def create_namespace_async(
        self,
        request: CreateNamespaceRequest,
    ) -> CreateNamespaceResult:
        async_result = []
        self._create_namespace(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _get_namespace_status(
        self,
        request: GetNamespaceStatusRequest,
        callback: Callable[[AsyncResult[GetNamespaceStatusResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/status".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='GET',
            result_type=GetNamespaceStatusResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def get_namespace_status(
        self,
        request: GetNamespaceStatusRequest,
    ) -> GetNamespaceStatusResult:
        async_result = []
        with timeout(30):
            self._get_namespace_status(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def get_namespace_status_async(
        self,
        request: GetNamespaceStatusRequest,
    ) -> GetNamespaceStatusResult:
        async_result = []
        self._get_namespace_status(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _get_namespace(
        self,
        request: GetNamespaceRequest,
        callback: Callable[[AsyncResult[GetNamespaceResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='GET',
            result_type=GetNamespaceResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def get_namespace(
        self,
        request: GetNamespaceRequest,
    ) -> GetNamespaceResult:
        async_result = []
        with timeout(30):
            self._get_namespace(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def get_namespace_async(
        self,
        request: GetNamespaceRequest,
    ) -> GetNamespaceResult:
        async_result = []
        self._get_namespace(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _update_namespace(
        self,
        request: UpdateNamespaceRequest,
        callback: Callable[[AsyncResult[UpdateNamespaceResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.description is not None:
            body["description"] = request.description
        if request.log_setting is not None:
            body["logSetting"] = request.log_setting.to_dict()
        if request.done_upload_script is not None:
            body["doneUploadScript"] = request.done_upload_script.to_dict()

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='PUT',
            result_type=UpdateNamespaceResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def update_namespace(
        self,
        request: UpdateNamespaceRequest,
    ) -> UpdateNamespaceResult:
        async_result = []
        with timeout(30):
            self._update_namespace(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def update_namespace_async(
        self,
        request: UpdateNamespaceRequest,
    ) -> UpdateNamespaceResult:
        async_result = []
        self._update_namespace(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _delete_namespace(
        self,
        request: DeleteNamespaceRequest,
        callback: Callable[[AsyncResult[DeleteNamespaceResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='DELETE',
            result_type=DeleteNamespaceResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def delete_namespace(
        self,
        request: DeleteNamespaceRequest,
    ) -> DeleteNamespaceResult:
        async_result = []
        with timeout(30):
            self._delete_namespace(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def delete_namespace_async(
        self,
        request: DeleteNamespaceRequest,
    ) -> DeleteNamespaceResult:
        async_result = []
        self._delete_namespace(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _describe_data_objects(
        self,
        request: DescribeDataObjectsRequest,
        callback: Callable[[AsyncResult[DescribeDataObjectsResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/me/data".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }
        if request.status is not None:
            query_strings["status"] = request.status
        if request.page_token is not None:
            query_strings["pageToken"] = request.page_token
        if request.limit is not None:
            query_strings["limit"] = request.limit

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='GET',
            result_type=DescribeDataObjectsResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def describe_data_objects(
        self,
        request: DescribeDataObjectsRequest,
    ) -> DescribeDataObjectsResult:
        async_result = []
        with timeout(30):
            self._describe_data_objects(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def describe_data_objects_async(
        self,
        request: DescribeDataObjectsRequest,
    ) -> DescribeDataObjectsResult:
        async_result = []
        self._describe_data_objects(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _describe_data_objects_by_user_id(
        self,
        request: DescribeDataObjectsByUserIdRequest,
        callback: Callable[[AsyncResult[DescribeDataObjectsByUserIdResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/data".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
        )

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }
        if request.status is not None:
            query_strings["status"] = request.status
        if request.page_token is not None:
            query_strings["pageToken"] = request.page_token
        if request.limit is not None:
            query_strings["limit"] = request.limit

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='GET',
            result_type=DescribeDataObjectsByUserIdResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def describe_data_objects_by_user_id(
        self,
        request: DescribeDataObjectsByUserIdRequest,
    ) -> DescribeDataObjectsByUserIdResult:
        async_result = []
        with timeout(30):
            self._describe_data_objects_by_user_id(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def describe_data_objects_by_user_id_async(
        self,
        request: DescribeDataObjectsByUserIdRequest,
    ) -> DescribeDataObjectsByUserIdResult:
        async_result = []
        self._describe_data_objects_by_user_id(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_upload(
        self,
        request: PrepareUploadRequest,
        callback: Callable[[AsyncResult[PrepareUploadResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/me/data/file".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.name is not None:
            body["name"] = request.name
        if request.content_type is not None:
            body["contentType"] = request.content_type
        if request.scope is not None:
            body["scope"] = request.scope
        if request.allow_user_ids is not None:
            body["allowUserIds"] = [
                item
                for item in request.allow_user_ids
            ]
        if request.update_if_exists is not None:
            body["updateIfExists"] = request.update_if_exists

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareUploadResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_upload(
        self,
        request: PrepareUploadRequest,
    ) -> PrepareUploadResult:
        async_result = []
        with timeout(30):
            self._prepare_upload(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_upload_async(
        self,
        request: PrepareUploadRequest,
    ) -> PrepareUploadResult:
        async_result = []
        self._prepare_upload(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_upload_by_user_id(
        self,
        request: PrepareUploadByUserIdRequest,
        callback: Callable[[AsyncResult[PrepareUploadByUserIdResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/data/file".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.name is not None:
            body["name"] = request.name
        if request.content_type is not None:
            body["contentType"] = request.content_type
        if request.scope is not None:
            body["scope"] = request.scope
        if request.allow_user_ids is not None:
            body["allowUserIds"] = [
                item
                for item in request.allow_user_ids
            ]
        if request.update_if_exists is not None:
            body["updateIfExists"] = request.update_if_exists

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareUploadByUserIdResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_upload_by_user_id(
        self,
        request: PrepareUploadByUserIdRequest,
    ) -> PrepareUploadByUserIdResult:
        async_result = []
        with timeout(30):
            self._prepare_upload_by_user_id(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_upload_by_user_id_async(
        self,
        request: PrepareUploadByUserIdRequest,
    ) -> PrepareUploadByUserIdResult:
        async_result = []
        self._prepare_upload_by_user_id(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _update_data_object(
        self,
        request: UpdateDataObjectRequest,
        callback: Callable[[AsyncResult[UpdateDataObjectResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/me/data/{dataObjectName}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.scope is not None:
            body["scope"] = request.scope
        if request.allow_user_ids is not None:
            body["allowUserIds"] = [
                item
                for item in request.allow_user_ids
            ]

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=UpdateDataObjectResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def update_data_object(
        self,
        request: UpdateDataObjectRequest,
    ) -> UpdateDataObjectResult:
        async_result = []
        with timeout(30):
            self._update_data_object(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def update_data_object_async(
        self,
        request: UpdateDataObjectRequest,
    ) -> UpdateDataObjectResult:
        async_result = []
        self._update_data_object(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _update_data_object_by_user_id(
        self,
        request: UpdateDataObjectByUserIdRequest,
        callback: Callable[[AsyncResult[UpdateDataObjectByUserIdResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/data/{dataObjectName}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.scope is not None:
            body["scope"] = request.scope
        if request.allow_user_ids is not None:
            body["allowUserIds"] = [
                item
                for item in request.allow_user_ids
            ]

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=UpdateDataObjectByUserIdResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def update_data_object_by_user_id(
        self,
        request: UpdateDataObjectByUserIdRequest,
    ) -> UpdateDataObjectByUserIdResult:
        async_result = []
        with timeout(30):
            self._update_data_object_by_user_id(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def update_data_object_by_user_id_async(
        self,
        request: UpdateDataObjectByUserIdRequest,
    ) -> UpdateDataObjectByUserIdResult:
        async_result = []
        self._update_data_object_by_user_id(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_re_upload(
        self,
        request: PrepareReUploadRequest,
        callback: Callable[[AsyncResult[PrepareReUploadResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/me/data/{dataObjectName}/file/reUpload".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.content_type is not None:
            body["contentType"] = request.content_type

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareReUploadResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_re_upload(
        self,
        request: PrepareReUploadRequest,
    ) -> PrepareReUploadResult:
        async_result = []
        with timeout(30):
            self._prepare_re_upload(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_re_upload_async(
        self,
        request: PrepareReUploadRequest,
    ) -> PrepareReUploadResult:
        async_result = []
        self._prepare_re_upload(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_re_upload_by_user_id(
        self,
        request: PrepareReUploadByUserIdRequest,
        callback: Callable[[AsyncResult[PrepareReUploadByUserIdResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/data/{dataObjectName}/file/reUpload".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.content_type is not None:
            body["contentType"] = request.content_type

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareReUploadByUserIdResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_re_upload_by_user_id(
        self,
        request: PrepareReUploadByUserIdRequest,
    ) -> PrepareReUploadByUserIdResult:
        async_result = []
        with timeout(30):
            self._prepare_re_upload_by_user_id(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_re_upload_by_user_id_async(
        self,
        request: PrepareReUploadByUserIdRequest,
    ) -> PrepareReUploadByUserIdResult:
        async_result = []
        self._prepare_re_upload_by_user_id(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _done_upload(
        self,
        request: DoneUploadRequest,
        callback: Callable[[AsyncResult[DoneUploadResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/me/data/{dataObjectName}/done".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=DoneUploadResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def done_upload(
        self,
        request: DoneUploadRequest,
    ) -> DoneUploadResult:
        async_result = []
        with timeout(30):
            self._done_upload(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def done_upload_async(
        self,
        request: DoneUploadRequest,
    ) -> DoneUploadResult:
        async_result = []
        self._done_upload(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _done_upload_by_user_id(
        self,
        request: DoneUploadByUserIdRequest,
        callback: Callable[[AsyncResult[DoneUploadByUserIdResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/data/{dataObjectName}/done".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=DoneUploadByUserIdResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def done_upload_by_user_id(
        self,
        request: DoneUploadByUserIdRequest,
    ) -> DoneUploadByUserIdResult:
        async_result = []
        with timeout(30):
            self._done_upload_by_user_id(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def done_upload_by_user_id_async(
        self,
        request: DoneUploadByUserIdRequest,
    ) -> DoneUploadByUserIdResult:
        async_result = []
        self._done_upload_by_user_id(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _delete_data_object(
        self,
        request: DeleteDataObjectRequest,
        callback: Callable[[AsyncResult[DeleteDataObjectResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/me/data/{dataObjectName}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='DELETE',
            result_type=DeleteDataObjectResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def delete_data_object(
        self,
        request: DeleteDataObjectRequest,
    ) -> DeleteDataObjectResult:
        async_result = []
        with timeout(30):
            self._delete_data_object(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def delete_data_object_async(
        self,
        request: DeleteDataObjectRequest,
    ) -> DeleteDataObjectResult:
        async_result = []
        self._delete_data_object(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _delete_data_object_by_user_id(
        self,
        request: DeleteDataObjectByUserIdRequest,
        callback: Callable[[AsyncResult[DeleteDataObjectByUserIdResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/data/{dataObjectName}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='DELETE',
            result_type=DeleteDataObjectByUserIdResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def delete_data_object_by_user_id(
        self,
        request: DeleteDataObjectByUserIdRequest,
    ) -> DeleteDataObjectByUserIdResult:
        async_result = []
        with timeout(30):
            self._delete_data_object_by_user_id(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def delete_data_object_by_user_id_async(
        self,
        request: DeleteDataObjectByUserIdRequest,
    ) -> DeleteDataObjectByUserIdResult:
        async_result = []
        self._delete_data_object_by_user_id(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download(
        self,
        request: PrepareDownloadRequest,
        callback: Callable[[AsyncResult[PrepareDownloadResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/file".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.data_object_id is not None:
            body["dataObjectId"] = request.data_object_id

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareDownloadResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_download(
        self,
        request: PrepareDownloadRequest,
    ) -> PrepareDownloadResult:
        async_result = []
        with timeout(30):
            self._prepare_download(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_async(
        self,
        request: PrepareDownloadRequest,
    ) -> PrepareDownloadResult:
        async_result = []
        self._prepare_download(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_by_user_id(
        self,
        request: PrepareDownloadByUserIdRequest,
        callback: Callable[[AsyncResult[PrepareDownloadByUserIdResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/file".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.data_object_id is not None:
            body["dataObjectId"] = request.data_object_id

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareDownloadByUserIdResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_download_by_user_id(
        self,
        request: PrepareDownloadByUserIdRequest,
    ) -> PrepareDownloadByUserIdResult:
        async_result = []
        with timeout(30):
            self._prepare_download_by_user_id(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_by_user_id_async(
        self,
        request: PrepareDownloadByUserIdRequest,
    ) -> PrepareDownloadByUserIdResult:
        async_result = []
        self._prepare_download_by_user_id(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_by_generation(
        self,
        request: PrepareDownloadByGenerationRequest,
        callback: Callable[[AsyncResult[PrepareDownloadByGenerationResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/file/generation/{generation}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            generation=request.generation if request.generation is not None and request.generation != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.data_object_id is not None:
            body["dataObjectId"] = request.data_object_id

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareDownloadByGenerationResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_download_by_generation(
        self,
        request: PrepareDownloadByGenerationRequest,
    ) -> PrepareDownloadByGenerationResult:
        async_result = []
        with timeout(30):
            self._prepare_download_by_generation(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_by_generation_async(
        self,
        request: PrepareDownloadByGenerationRequest,
    ) -> PrepareDownloadByGenerationResult:
        async_result = []
        self._prepare_download_by_generation(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_by_generation_and_user_id(
        self,
        request: PrepareDownloadByGenerationAndUserIdRequest,
        callback: Callable[[AsyncResult[PrepareDownloadByGenerationAndUserIdResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/file/generation/{generation}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
            generation=request.generation if request.generation is not None and request.generation != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.data_object_id is not None:
            body["dataObjectId"] = request.data_object_id

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareDownloadByGenerationAndUserIdResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_download_by_generation_and_user_id(
        self,
        request: PrepareDownloadByGenerationAndUserIdRequest,
    ) -> PrepareDownloadByGenerationAndUserIdResult:
        async_result = []
        with timeout(30):
            self._prepare_download_by_generation_and_user_id(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_by_generation_and_user_id_async(
        self,
        request: PrepareDownloadByGenerationAndUserIdRequest,
    ) -> PrepareDownloadByGenerationAndUserIdResult:
        async_result = []
        self._prepare_download_by_generation_and_user_id(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_own_data(
        self,
        request: PrepareDownloadOwnDataRequest,
        callback: Callable[[AsyncResult[PrepareDownloadOwnDataResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/me/data/{dataObjectName}/file".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareDownloadOwnDataResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_download_own_data(
        self,
        request: PrepareDownloadOwnDataRequest,
    ) -> PrepareDownloadOwnDataResult:
        async_result = []
        with timeout(30):
            self._prepare_download_own_data(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_own_data_async(
        self,
        request: PrepareDownloadOwnDataRequest,
    ) -> PrepareDownloadOwnDataResult:
        async_result = []
        self._prepare_download_own_data(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_by_user_id_and_data_object_name(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameRequest,
        callback: Callable[[AsyncResult[PrepareDownloadByUserIdAndDataObjectNameResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/data/{dataObjectName}/file".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='GET',
            result_type=PrepareDownloadByUserIdAndDataObjectNameResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_download_by_user_id_and_data_object_name(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameRequest,
    ) -> PrepareDownloadByUserIdAndDataObjectNameResult:
        async_result = []
        with timeout(30):
            self._prepare_download_by_user_id_and_data_object_name(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_by_user_id_and_data_object_name_async(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameRequest,
    ) -> PrepareDownloadByUserIdAndDataObjectNameResult:
        async_result = []
        self._prepare_download_by_user_id_and_data_object_name(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_own_data_by_generation(
        self,
        request: PrepareDownloadOwnDataByGenerationRequest,
        callback: Callable[[AsyncResult[PrepareDownloadOwnDataByGenerationResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/me/data/{dataObjectName}/generation/{generation}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
            generation=request.generation if request.generation is not None and request.generation != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareDownloadOwnDataByGenerationResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_download_own_data_by_generation(
        self,
        request: PrepareDownloadOwnDataByGenerationRequest,
    ) -> PrepareDownloadOwnDataByGenerationResult:
        async_result = []
        with timeout(30):
            self._prepare_download_own_data_by_generation(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_own_data_by_generation_async(
        self,
        request: PrepareDownloadOwnDataByGenerationRequest,
    ) -> PrepareDownloadOwnDataByGenerationResult:
        async_result = []
        self._prepare_download_own_data_by_generation(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_by_user_id_and_data_object_name_and_generation(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameAndGenerationRequest,
        callback: Callable[[AsyncResult[PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/data/{dataObjectName}/generation/{generation}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
            generation=request.generation if request.generation is not None and request.generation != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def prepare_download_by_user_id_and_data_object_name_and_generation(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameAndGenerationRequest,
    ) -> PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult:
        async_result = []
        with timeout(30):
            self._prepare_download_by_user_id_and_data_object_name_and_generation(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_by_user_id_and_data_object_name_and_generation_async(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameAndGenerationRequest,
    ) -> PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult:
        async_result = []
        self._prepare_download_by_user_id_and_data_object_name_and_generation(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _restore_data_object(
        self,
        request: RestoreDataObjectRequest,
        callback: Callable[[AsyncResult[RestoreDataObjectResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/file/restore".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }
        if request.data_object_id is not None:
            body["dataObjectId"] = request.data_object_id

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=RestoreDataObjectResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def restore_data_object(
        self,
        request: RestoreDataObjectRequest,
    ) -> RestoreDataObjectResult:
        async_result = []
        with timeout(30):
            self._restore_data_object(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def restore_data_object_async(
        self,
        request: RestoreDataObjectRequest,
    ) -> RestoreDataObjectResult:
        async_result = []
        self._restore_data_object(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _describe_data_object_histories(
        self,
        request: DescribeDataObjectHistoriesRequest,
        callback: Callable[[AsyncResult[DescribeDataObjectHistoriesResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/me/data/{dataObjectName}/history".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }
        if request.page_token is not None:
            query_strings["pageToken"] = request.page_token
        if request.limit is not None:
            query_strings["limit"] = request.limit

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='GET',
            result_type=DescribeDataObjectHistoriesResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def describe_data_object_histories(
        self,
        request: DescribeDataObjectHistoriesRequest,
    ) -> DescribeDataObjectHistoriesResult:
        async_result = []
        with timeout(30):
            self._describe_data_object_histories(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def describe_data_object_histories_async(
        self,
        request: DescribeDataObjectHistoriesRequest,
    ) -> DescribeDataObjectHistoriesResult:
        async_result = []
        self._describe_data_object_histories(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _describe_data_object_histories_by_user_id(
        self,
        request: DescribeDataObjectHistoriesByUserIdRequest,
        callback: Callable[[AsyncResult[DescribeDataObjectHistoriesByUserIdResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/data/{dataObjectName}/history".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
        )

        headers = self._create_authorized_headers()
        query_strings = {
            'contextStack': request.context_stack,
        }
        if request.page_token is not None:
            query_strings["pageToken"] = request.page_token
        if request.limit is not None:
            query_strings["limit"] = request.limit

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='GET',
            result_type=DescribeDataObjectHistoriesByUserIdResult,
            callback=callback,
            headers=headers,
            query_strings=query_strings,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def describe_data_object_histories_by_user_id(
        self,
        request: DescribeDataObjectHistoriesByUserIdRequest,
    ) -> DescribeDataObjectHistoriesByUserIdResult:
        async_result = []
        with timeout(30):
            self._describe_data_object_histories_by_user_id(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def describe_data_object_histories_by_user_id_async(
        self,
        request: DescribeDataObjectHistoriesByUserIdRequest,
    ) -> DescribeDataObjectHistoriesByUserIdResult:
        async_result = []
        self._describe_data_object_histories_by_user_id(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _get_data_object_history(
        self,
        request: GetDataObjectHistoryRequest,
        callback: Callable[[AsyncResult[GetDataObjectHistoryResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/me/data/{dataObjectName}/history/{generation}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
            generation=request.generation if request.generation is not None and request.generation != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        if request.access_token:
            headers["X-GS2-ACCESS-TOKEN"] = request.access_token
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=GetDataObjectHistoryResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def get_data_object_history(
        self,
        request: GetDataObjectHistoryRequest,
    ) -> GetDataObjectHistoryResult:
        async_result = []
        with timeout(30):
            self._get_data_object_history(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def get_data_object_history_async(
        self,
        request: GetDataObjectHistoryRequest,
    ) -> GetDataObjectHistoryResult:
        async_result = []
        self._get_data_object_history(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _get_data_object_history_by_user_id(
        self,
        request: GetDataObjectHistoryByUserIdRequest,
        callback: Callable[[AsyncResult[GetDataObjectHistoryByUserIdResult]], None],
        is_blocking: bool,
    ):
        url = Gs2Constant.ENDPOINT_HOST.format(
            service='datastore',
            region=self.session.region,
        ) + "/{namespaceName}/user/{userId}/data/{dataObjectName}/history/{generation}".format(
            namespaceName=request.namespace_name if request.namespace_name is not None and request.namespace_name != '' else 'null',
            userId=request.user_id if request.user_id is not None and request.user_id != '' else 'null',
            dataObjectName=request.data_object_name if request.data_object_name is not None and request.data_object_name != '' else 'null',
            generation=request.generation if request.generation is not None and request.generation != '' else 'null',
        )

        headers = self._create_authorized_headers()
        body = {
            'contextStack': request.context_stack,
        }

        if request.request_id:
            headers["X-GS2-REQUEST-ID"] = request.request_id
        _job = NetworkJob(
            url=url,
            method='POST',
            result_type=GetDataObjectHistoryByUserIdResult,
            callback=callback,
            headers=headers,
            body=body,
        )

        self.session.send(
            job=_job,
            is_blocking=is_blocking,
        )

    def get_data_object_history_by_user_id(
        self,
        request: GetDataObjectHistoryByUserIdRequest,
    ) -> GetDataObjectHistoryByUserIdResult:
        async_result = []
        with timeout(30):
            self._get_data_object_history_by_user_id(
                request,
                lambda result: async_result.append(result),
                is_blocking=True,
            )

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def get_data_object_history_by_user_id_async(
        self,
        request: GetDataObjectHistoryByUserIdRequest,
    ) -> GetDataObjectHistoryByUserIdResult:
        async_result = []
        self._get_data_object_history_by_user_id(
            request,
            lambda result: async_result.append(result),
            is_blocking=False,
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result