# encoding: utf-8
#
# Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
# Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License.
# A copy of the License is located at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
# or in the "license" file accompanying this file. This file is distributed
# on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
# express or implied. See the License for the specific language governing
# permissions and limitations under the License.

import time
from core.web_socket import *
from core.model import Gs2Constant
from datastore.request import *
from datastore.result import *


class Gs2DatastoreWebSocketClient(AbstractGs2WebSocketClient):

    def _describe_namespaces(
        self,
        request: DescribeNamespacesRequest,
        callback: Callable[[AsyncResult[DescribeNamespacesResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='namespace',
            function='describeNamespaces',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.page_token is not None:
            body["pageToken"] = request.page_token
        if request.limit is not None:
            body["limit"] = request.limit

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=DescribeNamespacesResult,
                callback=callback,
                body=body,
            )
        )

    def describe_namespaces(
        self,
        request: DescribeNamespacesRequest,
    ) -> DescribeNamespacesResult:
        async_result = []
        with timeout(30):
            self._describe_namespaces(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def describe_namespaces_async(
        self,
        request: DescribeNamespacesRequest,
    ) -> DescribeNamespacesResult:
        async_result = []
        self._describe_namespaces(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _create_namespace(
        self,
        request: CreateNamespaceRequest,
        callback: Callable[[AsyncResult[CreateNamespaceResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='namespace',
            function='createNamespace',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.name is not None:
            body["name"] = request.name
        if request.description is not None:
            body["description"] = request.description
        if request.log_setting is not None:
            body["logSetting"] = request.log_setting.to_dict()
        if request.done_upload_script is not None:
            body["doneUploadScript"] = request.done_upload_script.to_dict()

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=CreateNamespaceResult,
                callback=callback,
                body=body,
            )
        )

    def create_namespace(
        self,
        request: CreateNamespaceRequest,
    ) -> CreateNamespaceResult:
        async_result = []
        with timeout(30):
            self._create_namespace(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def create_namespace_async(
        self,
        request: CreateNamespaceRequest,
    ) -> CreateNamespaceResult:
        async_result = []
        self._create_namespace(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _get_namespace_status(
        self,
        request: GetNamespaceStatusRequest,
        callback: Callable[[AsyncResult[GetNamespaceStatusResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='namespace',
            function='getNamespaceStatus',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=GetNamespaceStatusResult,
                callback=callback,
                body=body,
            )
        )

    def get_namespace_status(
        self,
        request: GetNamespaceStatusRequest,
    ) -> GetNamespaceStatusResult:
        async_result = []
        with timeout(30):
            self._get_namespace_status(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def get_namespace_status_async(
        self,
        request: GetNamespaceStatusRequest,
    ) -> GetNamespaceStatusResult:
        async_result = []
        self._get_namespace_status(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _get_namespace(
        self,
        request: GetNamespaceRequest,
        callback: Callable[[AsyncResult[GetNamespaceResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='namespace',
            function='getNamespace',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=GetNamespaceResult,
                callback=callback,
                body=body,
            )
        )

    def get_namespace(
        self,
        request: GetNamespaceRequest,
    ) -> GetNamespaceResult:
        async_result = []
        with timeout(30):
            self._get_namespace(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def get_namespace_async(
        self,
        request: GetNamespaceRequest,
    ) -> GetNamespaceResult:
        async_result = []
        self._get_namespace(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _update_namespace(
        self,
        request: UpdateNamespaceRequest,
        callback: Callable[[AsyncResult[UpdateNamespaceResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='namespace',
            function='updateNamespace',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.description is not None:
            body["description"] = request.description
        if request.log_setting is not None:
            body["logSetting"] = request.log_setting.to_dict()
        if request.done_upload_script is not None:
            body["doneUploadScript"] = request.done_upload_script.to_dict()

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=UpdateNamespaceResult,
                callback=callback,
                body=body,
            )
        )

    def update_namespace(
        self,
        request: UpdateNamespaceRequest,
    ) -> UpdateNamespaceResult:
        async_result = []
        with timeout(30):
            self._update_namespace(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def update_namespace_async(
        self,
        request: UpdateNamespaceRequest,
    ) -> UpdateNamespaceResult:
        async_result = []
        self._update_namespace(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _delete_namespace(
        self,
        request: DeleteNamespaceRequest,
        callback: Callable[[AsyncResult[DeleteNamespaceResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='namespace',
            function='deleteNamespace',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=DeleteNamespaceResult,
                callback=callback,
                body=body,
            )
        )

    def delete_namespace(
        self,
        request: DeleteNamespaceRequest,
    ) -> DeleteNamespaceResult:
        async_result = []
        with timeout(30):
            self._delete_namespace(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def delete_namespace_async(
        self,
        request: DeleteNamespaceRequest,
    ) -> DeleteNamespaceResult:
        async_result = []
        self._delete_namespace(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _describe_data_objects(
        self,
        request: DescribeDataObjectsRequest,
        callback: Callable[[AsyncResult[DescribeDataObjectsResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='describeDataObjects',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.status is not None:
            body["status"] = request.status
        if request.page_token is not None:
            body["pageToken"] = request.page_token
        if request.limit is not None:
            body["limit"] = request.limit

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=DescribeDataObjectsResult,
                callback=callback,
                body=body,
            )
        )

    def describe_data_objects(
        self,
        request: DescribeDataObjectsRequest,
    ) -> DescribeDataObjectsResult:
        async_result = []
        with timeout(30):
            self._describe_data_objects(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def describe_data_objects_async(
        self,
        request: DescribeDataObjectsRequest,
    ) -> DescribeDataObjectsResult:
        async_result = []
        self._describe_data_objects(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _describe_data_objects_by_user_id(
        self,
        request: DescribeDataObjectsByUserIdRequest,
        callback: Callable[[AsyncResult[DescribeDataObjectsByUserIdResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='describeDataObjectsByUserId',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.status is not None:
            body["status"] = request.status
        if request.page_token is not None:
            body["pageToken"] = request.page_token
        if request.limit is not None:
            body["limit"] = request.limit

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=DescribeDataObjectsByUserIdResult,
                callback=callback,
                body=body,
            )
        )

    def describe_data_objects_by_user_id(
        self,
        request: DescribeDataObjectsByUserIdRequest,
    ) -> DescribeDataObjectsByUserIdResult:
        async_result = []
        with timeout(30):
            self._describe_data_objects_by_user_id(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def describe_data_objects_by_user_id_async(
        self,
        request: DescribeDataObjectsByUserIdRequest,
    ) -> DescribeDataObjectsByUserIdResult:
        async_result = []
        self._describe_data_objects_by_user_id(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_upload(
        self,
        request: PrepareUploadRequest,
        callback: Callable[[AsyncResult[PrepareUploadResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareUpload',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.name is not None:
            body["name"] = request.name
        if request.content_type is not None:
            body["contentType"] = request.content_type
        if request.scope is not None:
            body["scope"] = request.scope
        if request.allow_user_ids is not None:
            body["allowUserIds"] = [
                item
                for item in request.allow_user_ids
            ]
        if request.update_if_exists is not None:
            body["updateIfExists"] = request.update_if_exists

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareUploadResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_upload(
        self,
        request: PrepareUploadRequest,
    ) -> PrepareUploadResult:
        async_result = []
        with timeout(30):
            self._prepare_upload(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_upload_async(
        self,
        request: PrepareUploadRequest,
    ) -> PrepareUploadResult:
        async_result = []
        self._prepare_upload(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_upload_by_user_id(
        self,
        request: PrepareUploadByUserIdRequest,
        callback: Callable[[AsyncResult[PrepareUploadByUserIdResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareUploadByUserId',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.name is not None:
            body["name"] = request.name
        if request.content_type is not None:
            body["contentType"] = request.content_type
        if request.scope is not None:
            body["scope"] = request.scope
        if request.allow_user_ids is not None:
            body["allowUserIds"] = [
                item
                for item in request.allow_user_ids
            ]
        if request.update_if_exists is not None:
            body["updateIfExists"] = request.update_if_exists

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareUploadByUserIdResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_upload_by_user_id(
        self,
        request: PrepareUploadByUserIdRequest,
    ) -> PrepareUploadByUserIdResult:
        async_result = []
        with timeout(30):
            self._prepare_upload_by_user_id(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_upload_by_user_id_async(
        self,
        request: PrepareUploadByUserIdRequest,
    ) -> PrepareUploadByUserIdResult:
        async_result = []
        self._prepare_upload_by_user_id(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _update_data_object(
        self,
        request: UpdateDataObjectRequest,
        callback: Callable[[AsyncResult[UpdateDataObjectResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='updateDataObject',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.scope is not None:
            body["scope"] = request.scope
        if request.allow_user_ids is not None:
            body["allowUserIds"] = [
                item
                for item in request.allow_user_ids
            ]

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=UpdateDataObjectResult,
                callback=callback,
                body=body,
            )
        )

    def update_data_object(
        self,
        request: UpdateDataObjectRequest,
    ) -> UpdateDataObjectResult:
        async_result = []
        with timeout(30):
            self._update_data_object(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def update_data_object_async(
        self,
        request: UpdateDataObjectRequest,
    ) -> UpdateDataObjectResult:
        async_result = []
        self._update_data_object(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _update_data_object_by_user_id(
        self,
        request: UpdateDataObjectByUserIdRequest,
        callback: Callable[[AsyncResult[UpdateDataObjectByUserIdResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='updateDataObjectByUserId',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.scope is not None:
            body["scope"] = request.scope
        if request.allow_user_ids is not None:
            body["allowUserIds"] = [
                item
                for item in request.allow_user_ids
            ]

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=UpdateDataObjectByUserIdResult,
                callback=callback,
                body=body,
            )
        )

    def update_data_object_by_user_id(
        self,
        request: UpdateDataObjectByUserIdRequest,
    ) -> UpdateDataObjectByUserIdResult:
        async_result = []
        with timeout(30):
            self._update_data_object_by_user_id(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def update_data_object_by_user_id_async(
        self,
        request: UpdateDataObjectByUserIdRequest,
    ) -> UpdateDataObjectByUserIdResult:
        async_result = []
        self._update_data_object_by_user_id(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_re_upload(
        self,
        request: PrepareReUploadRequest,
        callback: Callable[[AsyncResult[PrepareReUploadResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareReUpload',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.content_type is not None:
            body["contentType"] = request.content_type

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareReUploadResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_re_upload(
        self,
        request: PrepareReUploadRequest,
    ) -> PrepareReUploadResult:
        async_result = []
        with timeout(30):
            self._prepare_re_upload(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_re_upload_async(
        self,
        request: PrepareReUploadRequest,
    ) -> PrepareReUploadResult:
        async_result = []
        self._prepare_re_upload(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_re_upload_by_user_id(
        self,
        request: PrepareReUploadByUserIdRequest,
        callback: Callable[[AsyncResult[PrepareReUploadByUserIdResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareReUploadByUserId',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.content_type is not None:
            body["contentType"] = request.content_type

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareReUploadByUserIdResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_re_upload_by_user_id(
        self,
        request: PrepareReUploadByUserIdRequest,
    ) -> PrepareReUploadByUserIdResult:
        async_result = []
        with timeout(30):
            self._prepare_re_upload_by_user_id(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_re_upload_by_user_id_async(
        self,
        request: PrepareReUploadByUserIdRequest,
    ) -> PrepareReUploadByUserIdResult:
        async_result = []
        self._prepare_re_upload_by_user_id(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _done_upload(
        self,
        request: DoneUploadRequest,
        callback: Callable[[AsyncResult[DoneUploadResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='doneUpload',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=DoneUploadResult,
                callback=callback,
                body=body,
            )
        )

    def done_upload(
        self,
        request: DoneUploadRequest,
    ) -> DoneUploadResult:
        async_result = []
        with timeout(30):
            self._done_upload(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def done_upload_async(
        self,
        request: DoneUploadRequest,
    ) -> DoneUploadResult:
        async_result = []
        self._done_upload(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _done_upload_by_user_id(
        self,
        request: DoneUploadByUserIdRequest,
        callback: Callable[[AsyncResult[DoneUploadByUserIdResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='doneUploadByUserId',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.user_id is not None:
            body["userId"] = request.user_id

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=DoneUploadByUserIdResult,
                callback=callback,
                body=body,
            )
        )

    def done_upload_by_user_id(
        self,
        request: DoneUploadByUserIdRequest,
    ) -> DoneUploadByUserIdResult:
        async_result = []
        with timeout(30):
            self._done_upload_by_user_id(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def done_upload_by_user_id_async(
        self,
        request: DoneUploadByUserIdRequest,
    ) -> DoneUploadByUserIdResult:
        async_result = []
        self._done_upload_by_user_id(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _delete_data_object(
        self,
        request: DeleteDataObjectRequest,
        callback: Callable[[AsyncResult[DeleteDataObjectResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='deleteDataObject',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=DeleteDataObjectResult,
                callback=callback,
                body=body,
            )
        )

    def delete_data_object(
        self,
        request: DeleteDataObjectRequest,
    ) -> DeleteDataObjectResult:
        async_result = []
        with timeout(30):
            self._delete_data_object(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def delete_data_object_async(
        self,
        request: DeleteDataObjectRequest,
    ) -> DeleteDataObjectResult:
        async_result = []
        self._delete_data_object(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _delete_data_object_by_user_id(
        self,
        request: DeleteDataObjectByUserIdRequest,
        callback: Callable[[AsyncResult[DeleteDataObjectByUserIdResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='deleteDataObjectByUserId',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=DeleteDataObjectByUserIdResult,
                callback=callback,
                body=body,
            )
        )

    def delete_data_object_by_user_id(
        self,
        request: DeleteDataObjectByUserIdRequest,
    ) -> DeleteDataObjectByUserIdResult:
        async_result = []
        with timeout(30):
            self._delete_data_object_by_user_id(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def delete_data_object_by_user_id_async(
        self,
        request: DeleteDataObjectByUserIdRequest,
    ) -> DeleteDataObjectByUserIdResult:
        async_result = []
        self._delete_data_object_by_user_id(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download(
        self,
        request: PrepareDownloadRequest,
        callback: Callable[[AsyncResult[PrepareDownloadResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareDownload',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.data_object_id is not None:
            body["dataObjectId"] = request.data_object_id

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareDownloadResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_download(
        self,
        request: PrepareDownloadRequest,
    ) -> PrepareDownloadResult:
        async_result = []
        with timeout(30):
            self._prepare_download(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_async(
        self,
        request: PrepareDownloadRequest,
    ) -> PrepareDownloadResult:
        async_result = []
        self._prepare_download(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_by_user_id(
        self,
        request: PrepareDownloadByUserIdRequest,
        callback: Callable[[AsyncResult[PrepareDownloadByUserIdResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareDownloadByUserId',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.data_object_id is not None:
            body["dataObjectId"] = request.data_object_id

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareDownloadByUserIdResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_download_by_user_id(
        self,
        request: PrepareDownloadByUserIdRequest,
    ) -> PrepareDownloadByUserIdResult:
        async_result = []
        with timeout(30):
            self._prepare_download_by_user_id(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_by_user_id_async(
        self,
        request: PrepareDownloadByUserIdRequest,
    ) -> PrepareDownloadByUserIdResult:
        async_result = []
        self._prepare_download_by_user_id(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_by_generation(
        self,
        request: PrepareDownloadByGenerationRequest,
        callback: Callable[[AsyncResult[PrepareDownloadByGenerationResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareDownloadByGeneration',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.data_object_id is not None:
            body["dataObjectId"] = request.data_object_id
        if request.generation is not None:
            body["generation"] = request.generation

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareDownloadByGenerationResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_download_by_generation(
        self,
        request: PrepareDownloadByGenerationRequest,
    ) -> PrepareDownloadByGenerationResult:
        async_result = []
        with timeout(30):
            self._prepare_download_by_generation(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_by_generation_async(
        self,
        request: PrepareDownloadByGenerationRequest,
    ) -> PrepareDownloadByGenerationResult:
        async_result = []
        self._prepare_download_by_generation(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_by_generation_and_user_id(
        self,
        request: PrepareDownloadByGenerationAndUserIdRequest,
        callback: Callable[[AsyncResult[PrepareDownloadByGenerationAndUserIdResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareDownloadByGenerationAndUserId',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.data_object_id is not None:
            body["dataObjectId"] = request.data_object_id
        if request.generation is not None:
            body["generation"] = request.generation

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareDownloadByGenerationAndUserIdResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_download_by_generation_and_user_id(
        self,
        request: PrepareDownloadByGenerationAndUserIdRequest,
    ) -> PrepareDownloadByGenerationAndUserIdResult:
        async_result = []
        with timeout(30):
            self._prepare_download_by_generation_and_user_id(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_by_generation_and_user_id_async(
        self,
        request: PrepareDownloadByGenerationAndUserIdRequest,
    ) -> PrepareDownloadByGenerationAndUserIdResult:
        async_result = []
        self._prepare_download_by_generation_and_user_id(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_own_data(
        self,
        request: PrepareDownloadOwnDataRequest,
        callback: Callable[[AsyncResult[PrepareDownloadOwnDataResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareDownloadOwnData',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareDownloadOwnDataResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_download_own_data(
        self,
        request: PrepareDownloadOwnDataRequest,
    ) -> PrepareDownloadOwnDataResult:
        async_result = []
        with timeout(30):
            self._prepare_download_own_data(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_own_data_async(
        self,
        request: PrepareDownloadOwnDataRequest,
    ) -> PrepareDownloadOwnDataResult:
        async_result = []
        self._prepare_download_own_data(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_by_user_id_and_data_object_name(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameRequest,
        callback: Callable[[AsyncResult[PrepareDownloadByUserIdAndDataObjectNameResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareDownloadByUserIdAndDataObjectName',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareDownloadByUserIdAndDataObjectNameResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_download_by_user_id_and_data_object_name(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameRequest,
    ) -> PrepareDownloadByUserIdAndDataObjectNameResult:
        async_result = []
        with timeout(30):
            self._prepare_download_by_user_id_and_data_object_name(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_by_user_id_and_data_object_name_async(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameRequest,
    ) -> PrepareDownloadByUserIdAndDataObjectNameResult:
        async_result = []
        self._prepare_download_by_user_id_and_data_object_name(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_own_data_by_generation(
        self,
        request: PrepareDownloadOwnDataByGenerationRequest,
        callback: Callable[[AsyncResult[PrepareDownloadOwnDataByGenerationResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareDownloadOwnDataByGeneration',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.generation is not None:
            body["generation"] = request.generation

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareDownloadOwnDataByGenerationResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_download_own_data_by_generation(
        self,
        request: PrepareDownloadOwnDataByGenerationRequest,
    ) -> PrepareDownloadOwnDataByGenerationResult:
        async_result = []
        with timeout(30):
            self._prepare_download_own_data_by_generation(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_own_data_by_generation_async(
        self,
        request: PrepareDownloadOwnDataByGenerationRequest,
    ) -> PrepareDownloadOwnDataByGenerationResult:
        async_result = []
        self._prepare_download_own_data_by_generation(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _prepare_download_by_user_id_and_data_object_name_and_generation(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameAndGenerationRequest,
        callback: Callable[[AsyncResult[PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='prepareDownloadByUserIdAndDataObjectNameAndGeneration',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.generation is not None:
            body["generation"] = request.generation

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult,
                callback=callback,
                body=body,
            )
        )

    def prepare_download_by_user_id_and_data_object_name_and_generation(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameAndGenerationRequest,
    ) -> PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult:
        async_result = []
        with timeout(30):
            self._prepare_download_by_user_id_and_data_object_name_and_generation(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def prepare_download_by_user_id_and_data_object_name_and_generation_async(
        self,
        request: PrepareDownloadByUserIdAndDataObjectNameAndGenerationRequest,
    ) -> PrepareDownloadByUserIdAndDataObjectNameAndGenerationResult:
        async_result = []
        self._prepare_download_by_user_id_and_data_object_name_and_generation(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _restore_data_object(
        self,
        request: RestoreDataObjectRequest,
        callback: Callable[[AsyncResult[RestoreDataObjectResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObject',
            function='restoreDataObject',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.data_object_id is not None:
            body["dataObjectId"] = request.data_object_id

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=RestoreDataObjectResult,
                callback=callback,
                body=body,
            )
        )

    def restore_data_object(
        self,
        request: RestoreDataObjectRequest,
    ) -> RestoreDataObjectResult:
        async_result = []
        with timeout(30):
            self._restore_data_object(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def restore_data_object_async(
        self,
        request: RestoreDataObjectRequest,
    ) -> RestoreDataObjectResult:
        async_result = []
        self._restore_data_object(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _describe_data_object_histories(
        self,
        request: DescribeDataObjectHistoriesRequest,
        callback: Callable[[AsyncResult[DescribeDataObjectHistoriesResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObjectHistory',
            function='describeDataObjectHistories',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.page_token is not None:
            body["pageToken"] = request.page_token
        if request.limit is not None:
            body["limit"] = request.limit

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=DescribeDataObjectHistoriesResult,
                callback=callback,
                body=body,
            )
        )

    def describe_data_object_histories(
        self,
        request: DescribeDataObjectHistoriesRequest,
    ) -> DescribeDataObjectHistoriesResult:
        async_result = []
        with timeout(30):
            self._describe_data_object_histories(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def describe_data_object_histories_async(
        self,
        request: DescribeDataObjectHistoriesRequest,
    ) -> DescribeDataObjectHistoriesResult:
        async_result = []
        self._describe_data_object_histories(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _describe_data_object_histories_by_user_id(
        self,
        request: DescribeDataObjectHistoriesByUserIdRequest,
        callback: Callable[[AsyncResult[DescribeDataObjectHistoriesByUserIdResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObjectHistory',
            function='describeDataObjectHistoriesByUserId',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.page_token is not None:
            body["pageToken"] = request.page_token
        if request.limit is not None:
            body["limit"] = request.limit

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=DescribeDataObjectHistoriesByUserIdResult,
                callback=callback,
                body=body,
            )
        )

    def describe_data_object_histories_by_user_id(
        self,
        request: DescribeDataObjectHistoriesByUserIdRequest,
    ) -> DescribeDataObjectHistoriesByUserIdResult:
        async_result = []
        with timeout(30):
            self._describe_data_object_histories_by_user_id(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def describe_data_object_histories_by_user_id_async(
        self,
        request: DescribeDataObjectHistoriesByUserIdRequest,
    ) -> DescribeDataObjectHistoriesByUserIdResult:
        async_result = []
        self._describe_data_object_histories_by_user_id(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _get_data_object_history(
        self,
        request: GetDataObjectHistoryRequest,
        callback: Callable[[AsyncResult[GetDataObjectHistoryResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObjectHistory',
            function='getDataObjectHistory',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.access_token is not None:
            body["accessToken"] = request.access_token
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.generation is not None:
            body["generation"] = request.generation

        if request.request_id:
            body["xGs2RequestId"] = request.request_id
        if request.access_token:
            body["xGs2AccessToken"] = request.access_token

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=GetDataObjectHistoryResult,
                callback=callback,
                body=body,
            )
        )

    def get_data_object_history(
        self,
        request: GetDataObjectHistoryRequest,
    ) -> GetDataObjectHistoryResult:
        async_result = []
        with timeout(30):
            self._get_data_object_history(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def get_data_object_history_async(
        self,
        request: GetDataObjectHistoryRequest,
    ) -> GetDataObjectHistoryResult:
        async_result = []
        self._get_data_object_history(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result

    def _get_data_object_history_by_user_id(
        self,
        request: GetDataObjectHistoryByUserIdRequest,
        callback: Callable[[AsyncResult[GetDataObjectHistoryByUserIdResult]], None],
    ):
        import uuid

        request_id = str(uuid.uuid4())
        body = self._create_metadata(
            service="datastore",
            component='dataObjectHistory',
            function='getDataObjectHistoryByUserId',
            request_id=request_id,
        )

        if request.context_stack:
            body['contextStack'] = str(request.context_stack)
        if request.namespace_name is not None:
            body["namespaceName"] = request.namespace_name
        if request.user_id is not None:
            body["userId"] = request.user_id
        if request.data_object_name is not None:
            body["dataObjectName"] = request.data_object_name
        if request.generation is not None:
            body["generation"] = request.generation

        if request.request_id:
            body["xGs2RequestId"] = request.request_id

        self.session.send(
            NetworkJob(
                request_id=request_id,
                result_type=GetDataObjectHistoryByUserIdResult,
                callback=callback,
                body=body,
            )
        )

    def get_data_object_history_by_user_id(
        self,
        request: GetDataObjectHistoryByUserIdRequest,
    ) -> GetDataObjectHistoryByUserIdResult:
        async_result = []
        with timeout(30):
            self._get_data_object_history_by_user_id(
                request,
                lambda result: async_result.append(result),
            )

        with timeout(30):
            while not async_result:
                time.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result


    async def get_data_object_history_by_user_id_async(
        self,
        request: GetDataObjectHistoryByUserIdRequest,
    ) -> GetDataObjectHistoryByUserIdResult:
        async_result = []
        self._get_data_object_history_by_user_id(
            request,
            lambda result: async_result.append(result),
        )

        import asyncio
        with timeout(30):
            while not async_result:
                await asyncio.sleep(0.01)

        if async_result[0].error:
            raise async_result[0].error
        return async_result[0].result