# Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
# Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License.
# A copy of the License is located at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
# or in the "license" file accompanying this file. This file is distributed
# on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
# express or implied. See the License for the specific language governing
# permissions and limitations under the License.
from __future__ import annotations

from core import Gs2RestSession
from core.domain.access_token import AccessToken
from inventory import Gs2InventoryRestClient, request as request_, result as result_
from inventory.domain.iterator.namespaces import DescribeNamespacesIterator
from inventory.domain.iterator.inventory_model_masters import DescribeInventoryModelMastersIterator
from inventory.domain.iterator.inventory_models import DescribeInventoryModelsIterator
from inventory.domain.iterator.item_model_masters import DescribeItemModelMastersIterator
from inventory.domain.iterator.item_models import DescribeItemModelsIterator
from inventory.domain.iterator.inventories import DescribeInventoriesIterator
from inventory.domain.iterator.inventories_by_user_id import DescribeInventoriesByUserIdIterator
from inventory.domain.iterator.item_sets import DescribeItemSetsIterator
from inventory.domain.iterator.item_sets_by_user_id import DescribeItemSetsByUserIdIterator
from inventory.domain.iterator.reference_of import DescribeReferenceOfIterator
from inventory.domain.iterator.reference_of_by_user_id import DescribeReferenceOfByUserIdIterator
from inventory.domain.cache.namespace import NamespaceDomainCache
from inventory.domain.cache.inventory_model_master import InventoryModelMasterDomainCache
from inventory.domain.cache.inventory_model import InventoryModelDomainCache
from inventory.domain.cache.item_model_master import ItemModelMasterDomainCache
from inventory.domain.cache.item_model import ItemModelDomainCache
from inventory.domain.cache.inventory import InventoryDomainCache
from inventory.domain.cache.item_set import ItemSetDomainCache
from inventory.domain.current_item_model_master import CurrentItemModelMasterDomain
from inventory.domain.inventory_model import InventoryModelDomain
from inventory.domain.user import UserDomain
from inventory.domain.user_access_token import UserAccessTokenDomain
from inventory.domain.user_access_token import UserAccessTokenDomain
from inventory.domain.inventory_model_master import InventoryModelMasterDomain


class NamespaceDomain:
    _session: Gs2RestSession
    _client: Gs2InventoryRestClient
    _namespace_cache: NamespaceDomainCache
    _namespace_name: str
    _inventory_model_master_cache: InventoryModelMasterDomainCache
    _inventory_model_cache: InventoryModelDomainCache

    def __init__(
        self,
        session: Gs2RestSession,
        namespace_cache: NamespaceDomainCache,
        namespace_name: str,
    ):
        self._session = session
        self._client = Gs2InventoryRestClient(
            session,
        )
        self._namespace_cache = namespace_cache
        self._namespace_name = namespace_name
        self._inventory_model_master_cache = InventoryModelMasterDomainCache()
        self._inventory_model_cache = InventoryModelDomainCache()

    def get_status(
        self,
        request: request_.GetNamespaceStatusRequest,
    ) -> result_.GetNamespaceStatusResult:
        request.with_namespace_name(self._namespace_name)
        r = self._client.get_namespace_status(
            request,
        )
        return r

    def load(
        self,
        request: request_.GetNamespaceRequest,
    ) -> result_.GetNamespaceResult:
        request.with_namespace_name(self._namespace_name)
        r = self._client.get_namespace(
            request,
        )
        self._namespace_cache.update(r.item)
        return r

    def update(
        self,
        request: request_.UpdateNamespaceRequest,
    ) -> result_.UpdateNamespaceResult:
        request.with_namespace_name(self._namespace_name)
        r = self._client.update_namespace(
            request,
        )
        self._namespace_cache.update(r.item)
        return r

    def delete(
        self,
        request: request_.DeleteNamespaceRequest,
    ) -> result_.DeleteNamespaceResult:
        request.with_namespace_name(self._namespace_name)
        r = self._client.delete_namespace(
            request,
        )
        self._namespace_cache.delete(r.item)
        return r

    def create_inventory_model_master(
        self,
        request: request_.CreateInventoryModelMasterRequest,
    ) -> result_.CreateInventoryModelMasterResult:
        request.with_namespace_name(self._namespace_name)
        r = self._client.create_inventory_model_master(
            request,
        )
        return r

    def inventory_model_masters(
        self,
    ) -> DescribeInventoryModelMastersIterator:
        return DescribeInventoryModelMastersIterator(
            self._inventory_model_master_cache,
            self._client,
            self._namespace_name,
        )

    def inventory_models(
        self,
    ) -> DescribeInventoryModelsIterator:
        return DescribeInventoryModelsIterator(
            self._inventory_model_cache,
            self._client,
            self._namespace_name,
        )

    def current_item_model_master(
        self,
    ) -> CurrentItemModelMasterDomain:
        return CurrentItemModelMasterDomain(
            self._session,
            self._namespace_name,
        )

    def inventory_model(
        self,
        inventory_name: str,
    ) -> InventoryModelDomain:
        return InventoryModelDomain(
            self._session,
            self._inventory_model_cache,
            self._namespace_name,
            inventory_name,
        )

    def user(
        self,
        user_id: str,
    ) -> UserDomain:
        return UserDomain(
            self._session,
            self._namespace_name,
            user_id,
        )

    def access_token(
        self,
        access_token: AccessToken,
    ) -> UserAccessTokenDomain:
        return UserAccessTokenDomain(
            self._session,
            self._namespace_name,
            access_token,
        )

    def inventory_model_master(
        self,
        inventory_name: str,
    ) -> InventoryModelMasterDomain:
        return InventoryModelMasterDomain(
            self._session,
            self._inventory_model_master_cache,
            self._namespace_name,
            inventory_name,
        )
