# Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
# Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License.
# A copy of the License is located at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
# or in the "license" file accompanying this file. This file is distributed
# on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
# express or implied. See the License for the specific language governing
# permissions and limitations under the License.
from __future__ import annotations

from core import Gs2RestSession
from core.domain.access_token import AccessToken
from news import Gs2NewsRestClient, request as request_, result as result_
from news.domain.iterator.namespaces import DescribeNamespacesIterator
from news.domain.iterator.news import DescribeNewsIterator
from news.domain.iterator.news_by_user_id import DescribeNewsByUserIdIterator
from news.domain.cache.namespace import NamespaceDomainCache
from news.domain.cache.news import NewsDomainCache
from news.domain.current_news_master import CurrentNewsMasterDomain
from news.domain.user import UserDomain
from news.domain.user_access_token import UserAccessTokenDomain
from news.domain.user_access_token import UserAccessTokenDomain


class NamespaceDomain:
    _session: Gs2RestSession
    _client: Gs2NewsRestClient
    _namespace_cache: NamespaceDomainCache
    _namespace_name: str

    def __init__(
        self,
        session: Gs2RestSession,
        namespace_cache: NamespaceDomainCache,
        namespace_name: str,
    ):
        self._session = session
        self._client = Gs2NewsRestClient(
            session,
        )
        self._namespace_cache = namespace_cache
        self._namespace_name = namespace_name

    def get_status(
        self,
        request: request_.GetNamespaceStatusRequest,
    ) -> result_.GetNamespaceStatusResult:
        request.with_namespace_name(self._namespace_name)
        r = self._client.get_namespace_status(
            request,
        )
        return r

    def load(
        self,
        request: request_.GetNamespaceRequest,
    ) -> result_.GetNamespaceResult:
        request.with_namespace_name(self._namespace_name)
        r = self._client.get_namespace(
            request,
        )
        self._namespace_cache.update(r.item)
        return r

    def update(
        self,
        request: request_.UpdateNamespaceRequest,
    ) -> result_.UpdateNamespaceResult:
        request.with_namespace_name(self._namespace_name)
        r = self._client.update_namespace(
            request,
        )
        self._namespace_cache.update(r.item)
        return r

    def delete(
        self,
        request: request_.DeleteNamespaceRequest,
    ) -> result_.DeleteNamespaceResult:
        request.with_namespace_name(self._namespace_name)
        r = self._client.delete_namespace(
            request,
        )
        return r

    def current_news_master(
        self,
    ) -> CurrentNewsMasterDomain:
        return CurrentNewsMasterDomain(
            self._session,
            self._namespace_name,
        )

    def user(
        self,
        user_id: str,
    ) -> UserDomain:
        return UserDomain(
            self._session,
            self._namespace_name,
            user_id,
        )

    def access_token(
        self,
        access_token: AccessToken,
    ) -> UserAccessTokenDomain:
        return UserAccessTokenDomain(
            self._session,
            self._namespace_name,
            access_token,
        )
