# -*- coding: utf-8 -*-

# module constant

# Copyright (C) 2019 National Research Council Canada
# Author:  Harold Parks

# This file is part of MetroloPy.

# MetroloPy is free software: you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software 
# Foundation, either version 3 of the License, or (at your option) any later 
# version.

# MetroloPy is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more 
# details.

# You should have received a copy of the GNU General Public License along with 
# MetroloPy. If not, see <http://www.gnu.org/licenses/>.

"""
Constants used by both the siunits and the codata2018 modules.  Most constants
are from CODATA 2018
"""

import numpy as np
from warnings import warn
from .gummy import gummy
from .ummy import ummy,MFraction,_getfinfo,_iinfo


def _rounding_u(x):
    if not ummy.rounding_u:
        try:
            fi,_ = _getfinfo(x)
            if fi is _iinfo:
                raise ValueError()
            return ummy(x,x*fi.rel_u)
        except:
            warn('numpy.finfo cannot get the floating point accuracy for float64 math constants')
            return ummy(x)
    else:
        return ummy(x)
    
pi = _rounding_u(np.pi)
euler = _rounding_u(np.e)
sqrt2 = _rounding_u(np.sqrt(2))


# constants from CODATA 2018:
    
(
alph,
aral,
ryd,
are,
mpsme,
ae,
rd,
ghn,
arh,
sigmah,
mmu,
amu,
arn,
gnn,
ard,
gdn,
gp,
sigmapp,
mtu,
gtn
) = gummy.create([0.0072973525693,
           4.001506179127,
           10973731.56816,
           0.000548579909065,
           1836.15267343,
           0.00115965218128,
           2.12799e-15,
           -4.255250615,
           3.014932247175,
           5.996743000000001e-05,
           1.8835316269999999e-28,
           0.00116592089,
           1.00866491595,
           -3.82608545,
           2.013553212745,
           0.8574382338,
           5.5856946893,
           2.5689000000000004e-05,
           3.01550071621,
           5.957924931],
           u=[1.1e-12,
              6.3e-11,
              2.1e-05,
              1.6e-14,
              1.1e-07,
              1.8e-13,
              7.4e-19,
              5e-08,
              9.7e-11,
              1.0e-10,
              4.2e-36,
              6.3e-10,
              4.9e-10,
              9e-07,
              4e-11,
              2.2e-09,
              1.6e-09,
              1.1e-08,
              1.2e-10,
              1.2e-08],
           correlation_matrix=[[       1, 0.00003, 0.00207,-0.05927, 0.03103, 0.99492, 0.00320,-0.00017, 0.00147, 0.00000,-0.00013, 0.00055, 0.00003,-0.00001, 0.00002, 0.00005, 0.00001,-0.00019, 0.00117, 0.00000],
                               [ 0.00003,       1,-0.00000,-0.00050, 0.00024, 0.00003,-0.00000,-0.00000, 0.00000, 0.00000, 0.00000,-0.00000, 0.00000,-0.00000, 0.00000, 0.00000, 0.00000,-0.00000, 0.00000, 0.00000],
                               [ 0.00207,-0.00000,       1, 0.00704,-0.01206, 0.00206, 0.90366, 0.00006,-0.00578, 0.00000, 0.00017, 0.00001, 0.00044, 0.00000,-0.00093,-0.00002,-0.00000, 0.00007,-0.00463,-0.00000],
                               [-0.05927,-0.00050, 0.00704,       1,-0.48443,-0.05896, 0.00317, 0.00249,-0.00044, 0.00000,-0.00001, 0.00002,-0.00003, 0.00012,-0.00040,-0.00067,-0.00016, 0.00267,-0.00022,-0.00002],
                               [ 0.03103, 0.00024,-0.01206,-0.48443,       1, 0.03087,-0.01113,-0.00513, 0.47743, 0.00000, 0.00003,-0.00009,-0.03920,-0.00026, 0.00020, 0.00137, 0.00032,-0.00551, 0.38235, 0.00005],
                               [ 0.99492, 0.00003, 0.00206,-0.05896, 0.03087,       1, 0.00319,-0.00017, 0.00146, 0.00000,-0.00013, 0.00055, 0.00003,-0.00001, 0.00002, 0.00005, 0.00001,-0.00019, 0.00116, 0.00000],
                               [ 0.00320,-0.00000, 0.90366, 0.00317,-0.01113, 0.00319,       1, 0.00006,-0.00511, 0.00000, 0.00016, 0.00001, 0.00049, 0.00000, 0.00209,-0.00002,-0.00000, 0.00006,-0.00410,-0.00000],
                               [-0.00017,-0.00000, 0.00006, 0.00249,-0.00513,-0.00017, 0.00006,       1,-0.00245,-0.00854,-0.00000, 0.00000, 0.00020, 0.04291,-0.00000,-0.00001,-0.00000, 0.92869,-0.00196,-0.00000],
                               [ 0.00147, 0.00000,-0.00578,-0.00044, 0.47743, 0.00146,-0.00511,-0.00245,       1, 0.00000, 0.00001,-0.00005,-0.02445,-0.00012, 0.41560, 0.00066, 0.00015,-0.00263, 0.80098, 0.00002],
                               [ 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000,-0.00854, 0.00000,       1, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000],
                               [-0.00013, 0.00000, 0.00017,-0.00001, 0.00003,-0.00013, 0.00016,-0.00000, 0.00001, 0.00000,       1, 0.06971,-0.00000,-0.00000,-0.00000,-0.00002,-0.00016,-0.00000, 0.00001,-0.00002],
                               [ 0.00055,-0.00000, 0.00001, 0.00002,-0.00009, 0.00055, 0.00001, 0.00000,-0.00005, 0.00000, 0.06971,       1, 0.00000, 0.00000,-0.00000, 0.00006, 0.00054, 0.00002,-0.00004, 0.00008],
                               [ 0.00003, 0.00000, 0.00044,-0.00003,-0.03920, 0.00003, 0.00049, 0.00020,-0.02445, 0.00000,-0.00000, 0.00000,       1, 0.00001,-0.00000,-0.00005,-0.00001, 0.00022,-0.01959,-0.00000],
                               [-0.00001,-0.00000, 0.00000, 0.00012,-0.00026,-0.00001, 0.00000, 0.04291,-0.00012, 0.00000,-0.00000, 0.00000, 0.00001,       1,-0.00000,-0.00000,-0.00000, 0.04618,-0.00010,-0.00000],
                               [ 0.00002, 0.00000,-0.00093,-0.00040, 0.00020, 0.00002, 0.00209,-0.00000, 0.41560, 0.00000,-0.00000,-0.00000,-0.00000,-0.00000,       1, 0.00000, 0.00000,-0.00000, 0.33288, 0.00000],
                               [ 0.00005, 0.00000,-0.00002,-0.00067, 0.00137, 0.00005,-0.00002,-0.00001, 0.00066, 0.00000,-0.00002, 0.00006,-0.00005,-0.00000, 0.00000,       1, 0.10706, 0.00289, 0.00053, 0.01581],
                               [ 0.00001, 0.00000,-0.00000,-0.00016, 0.00032, 0.00001,-0.00000,-0.00000, 0.00015, 0.00000,-0.00016, 0.00054,-0.00001,-0.00000, 0.00000, 0.10706,       1, 0.02707, 0.00012, 0.14766],
                               [-0.00019,-0.00000, 0.00007, 0.00267,-0.00551,-0.00019, 0.00006, 0.92869,-0.00263, 0.00000,-0.00000, 0.00002, 0.00022, 0.04618,-0.00000, 0.00289, 0.02707,       1,-0.00211, 0.00400],
                               [ 0.00117, 0.00000,-0.00463,-0.00022, 0.38235, 0.00116,-0.00410,-0.00196, 0.80098, 0.00000, 0.00001,-0.00004,-0.01959,-0.00010, 0.33288, 0.00053, 0.00012,-0.00211,       1, 0.00002],
                               [ 0.00000, 0.00000,-0.00000,-0.00002, 0.00005, 0.00000,-0.00000,-0.00000, 0.00002, 0.00000,-0.00002, 0.00008,-0.00000,-0.00000, 0.00000, 0.01581, 0.14766, 0.00400, 0.00002,       1]]
)


c = 299792458 # speed of light in m/s
h = MFraction('6.62607015e-34') # planck constant in J s

    
e = MFraction('1.602176634e-19') # electron charge in C
k = MFraction('1.380649e-23') # Boltzmann constant in J/K

KJ = 2*e/h
RK = h/e**2

KJ90 = MFraction('483597.9e9') # Josephson constant Hz/V, 1990 conventional value
RK90 = MFraction('25812.807') # Von Klitzing constant in ohm, 1990 conventional value
    
G = ummy(6.67430e-11,0.00015e-11) # gravitational constant in m**3/kg s**2

dalton = 2*ryd*h/(are*c*alph**2)
me = 2*ryd*h/(c*alph**2) #electron mass in kg

a0 = alph/(4*pi*ryd) # bohr radius in m

mp = mpsme*me # in kg


# constants from IAU 2009:
earth_mass = ummy(3.986004418e14,8e5)/G # in kg
solar_mass = ummy(1.32712442099e20,1e10)/G # in kg
jupiter_mass = solar_mass/ummy(1.047348644e3,1.7e-5) # in kg

hbar = h/(2*pi)