"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcceleratorSecurityGroupPeer = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const custom_resources_1 = require("@aws-cdk/custom-resources");
/**
 * The security group used by a Global Accelerator to send traffic to resources in a VPC.
 */
class AcceleratorSecurityGroupPeer {
    constructor(securityGroupId) {
        this.securityGroupId = securityGroupId;
        this.canInlineRule = false;
        this.connections = new ec2.Connections({ peer: this });
        this.uniqueId = 'GlobalAcceleratorGroup';
    }
    /**
     * Lookup the Global Accelerator security group at CloudFormation deployment time.
     *
     * As of this writing, Global Accelerators (AGA) create a single security group per VPC. AGA security groups are shared
     * by all AGAs in an account. Additionally, there is no CloudFormation mechanism to reference the AGA security groups.
     *
     * This makes creating security group rules which allow traffic from an AGA complicated in CDK. This lookup will identify
     * the AGA security group for a given VPC at CloudFormation deployment time, and lets you create rules for traffic from AGA
     * to other resources created by CDK.
     */
    static fromVpc(scope, id, vpc, endpointGroup) {
        // The security group name is always 'GlobalAccelerator'
        const globalAcceleratorSGName = 'GlobalAccelerator';
        // How to reference the security group name in the response from EC2
        const ec2ResponseSGIdField = 'SecurityGroups.0.GroupId';
        // The AWS Custom Resource that make a call to EC2 to get the security group ID, for the given VPC
        const lookupAcceleratorSGCustomResource = new custom_resources_1.AwsCustomResource(scope, id + 'CustomResource', {
            onCreate: {
                service: 'EC2',
                action: 'describeSecurityGroups',
                parameters: {
                    Filters: [
                        {
                            Name: 'group-name',
                            Values: [
                                globalAcceleratorSGName,
                            ],
                        },
                        {
                            Name: 'vpc-id',
                            Values: [
                                vpc.vpcId,
                            ],
                        },
                    ],
                },
                // We get back a list of responses, but the list should be of length 0 or 1
                // Getting no response means no resources have been linked to the AGA
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse(ec2ResponseSGIdField),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // We add a dependency on the endpoint group, guaranteeing that CloudFormation won't
        // try and look up the SG before AGA creates it. The SG is created when a VPC resource
        // is associated with an AGA
        lookupAcceleratorSGCustomResource.node.addDependency(endpointGroup.node.defaultChild);
        // Look up the security group ID
        return new AcceleratorSecurityGroupPeer(lookupAcceleratorSGCustomResource.getResponseField(ec2ResponseSGIdField));
    }
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
exports.AcceleratorSecurityGroupPeer = AcceleratorSecurityGroupPeer;
//# sourceMappingURL=data:application/json;base64,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