# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pony-stubs']

package_data = \
{'': ['*'],
 'pony-stubs': ['flask/*',
                'orm/*',
                'orm/dbproviders/*',
                'orm/integration/*',
                'thirdparty/*',
                'utils/*']}

setup_kwargs = {
    'name': 'pony-stubs',
    'version': '0.5.2',
    'description': 'Type stubs for Pony ORM',
    'long_description': '# Pony stubs\n\nPython type hint stubs for [Pony ORM](https://github.com/ponyorm/pony)\n\n## Goals\n1. Provide type hints for Pony ORM that support both MyPy and Pyright on their strictest modes\n2. Integrate the contents of this package into the official Pony ORM repository (self-deprecation)\n3. Focus primarily on the aspects that users of Pony ORM most often run into (defining models, querying them)\n\n## Usage\nInstall the package with your preferred Python dependency manager:\n```sh\npip install pony-stubs\n\npipenv install pony-stubs\n\npoetry add -D pony-stubs\n```\n\nThen define your models:\n```python\n# We need this to avoid `TypeError: \'type\' object is not subscriptable`\n# later on when forward declaring relation types with strings\nfrom __future__ import annotations\n\nfrom pony.orm import (\n    Database,\n    Required,\n    Optional,\n    Set,\n    PrimaryKey,\n    select,\n    max,\n    count,\n    desc\n)\n\ndb = Database("sqlite", "store.sqlite", create_db=True)\n\n# Using `db.Entity` directly won\'t work, as both MyPy and Pyright won\'t\n# allow inheriting a class from a variable. For Pyright this declaration\n# is enough misdirection for it not to complain, but MyPy needs an extra\n# `type: ignore` comment above each model declaration to work.\nDbEntity = db.Entity\n\n\nclass Customer(DbEntity):  # type: ignore\n    # If we want the type checkers to know about the autogenerated ID\n    # field, we need to annotate it\n    id: int\n    # Otherwise, using `Required` allows `Customer.email` to be inferred\n    # as `str` later on\n    email = Required(str, unique=True)\n    password = Required(str)\n    country = Required(str)\n    # Using `Optional` marks the field attribute as `str |\xa0None`\n    address = Optional(str)\n    # When we forward declare a relation by using the class name as a\n    # string, we also need to manually annotate the field so that the\n    # type checkers can infer it correctly\n    orders: Set["Order"] = Set("Order")\n\nclass Order(DbEntity):  # type: ignore\n    # We can also declare the primary key with Pony constructors and\n    # infer the type that way\n    id = PrimaryKey(int, auto=True)\n    state = Required(str)\n    total_price = Required(Decimal)\n    # When declaring relationships after a class has been declared,\n    # there\'s no need for annotations\n    customer = Required(Customer)\n\nclass Product(DbEntity):  # type: ignore\n    id: int\n    name = Required(str)\n    price = Required(Decimal)\n```\n\nAnd use them in your code:\n```python\n# Here result infers as `QueryResult[Customer]` and all fields in the\n# generator expression inside `select` have proper types inferred\nresult = select(c for c in Customer if c.country != "USA")[:]\nresult = select(c for c in Customer if count(c.orders) > 1)[:]\n\n# Here result infers as `Decimal`\nresult = max(p.price for p in Product)\n\n# And here as `Customer |\xa0None` (as `.first()` might not find an object)\n# Here is also the first time we can\'t properly infer something:\n# `c.orders` is declared as `Set[Order]`, but Pony allows us to access\n# `c.orders.total_price` as though it was typed as a plain `Order`.\n# As Python doesn\'t yet support type intersections, we have yet to come\n# up with no choice other than to type each extra field of a `Set` as\n# `Any`\nresult = (\n    select(c for c in Customer)\n    .order_by(lambda c: desc(sum(c.orders.total_price)))\n    .first()\n)\n```\n\n## Limitations\n1. We need misdirection with `db.Entity` for `pyright`, and `# type: ignore` for `mypy`\n2. When forward declaring relations (relation to a model defined later in the file) we an additional manual annotation (`field: Set["RelatedObject"]`)\n3. "Attribute lifting" of related fields is typed as `Any`. Pony would allow us to access attributes of `Set[Order]` as though it\'s type was `Order`, but python doesn\'t yet support type intersections so statically typing this seems to be impossible without a plugin (which would only fix the issue for MyPy but not Pyright)\n4. `Query.where()` ([docs](https://docs.ponyorm.org/api_reference.html#Query.filter)) loses type information and it\'s results are typed as `Any`, as python doesn\'t keep track of generator expressions\' initial iterables: `(o.customer for o in Order)` is inferred as `Generator[Customer, None, None]`, without any native way of storing the `Order` type in a generic for inferring.\n\n## Contributing\nContributions are always most welcome! Please run the pre-commit hooks before setting up a pull request, and in case the Github actions fail, please try to fix those issues so the review itself can go as smoothly as possible\n\nThe development environment for this package requires `poetry` (https://python-poetry.org/docs/master/#installing-with-the-official-installer)\n\nUsing VSCode as the editor is recommended!\n\n### Setting up the repo\n1. Clone the repo\n    - `git clone git@github.com:Jonesus/pony-stubs.git`\n2. Install dependencies\n    - `poetry install`\n3. Install commit hooks\n    - `poetry run pre-commit install --install-hooks`\n4. Type ahead!\n\n## License\nThis project is licensed under the MIT license (see LICENSE.md)\n',
    'author': 'Joonas Palosuo',
    'author_email': 'joonas.palosuo@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Jonesus/pony-stubs',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
