# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/03_preprocessing.ipynb (unless otherwise specified).

__all__ = ['BaseProcessor', 'display_processor_info', 'CopyPreProcessor', 'FeatureSelectionPreProcessor',
           'TargetSelectionPreProcessor', 'ReduceMemoryProcessor', 'SyntheticDataGenerator',
           'BayesianGMMTargetProcessor', 'GroupStatsPreProcessor', 'TalibFeatureGenerator', 'KatsuFeatureGenerator',
           'EraQuantileProcessor', 'AwesomePreProcessor']

# Cell
import os
import sdv
import time
import warnings
import numpy as np
import pandas as pd
import datetime as dt
from pathlib import Path
from tqdm.auto import tqdm
from functools import wraps
from scipy.stats import rankdata
from typeguard import typechecked
from abc import ABC, abstractmethod
from rich import print as rich_print
from typing import Union, List, Tuple
from multiprocessing.pool import Pool
from sklearn.linear_model import Ridge
from sklearn.mixture import BayesianGaussianMixture
from sklearn.preprocessing import QuantileTransformer

from .download import NumeraiClassicDownloader
from .numerframe import NumerFrame, create_numerframe

# Cell
class BaseProcessor(ABC):
    """Common functionality for preprocessors and postprocessors."""

    def __init__(self):
        ...

    @abstractmethod
    def transform(
        self, dataf: Union[pd.DataFrame, NumerFrame], *args, **kwargs
    ) -> NumerFrame:
        ...

    def __call__(
        self, dataf: Union[pd.DataFrame, NumerFrame], *args, **kwargs
    ) -> NumerFrame:
        return self.transform(dataf=dataf, *args, **kwargs)

# Cell
def display_processor_info(func):
    """Fancy console output for data processing."""

    @wraps(func)
    def wrapper(*args, **kwargs):
        tic = dt.datetime.now()
        result = func(*args, **kwargs)
        time_taken = str(dt.datetime.now() - tic)
        class_name = func.__qualname__.split(".")[0]
        rich_print(
            f":white_check_mark: Finished step [bold]{class_name}[/bold]. Output shape={result.shape}. Time taken for step: [blue]{time_taken}[/blue]. :white_check_mark:"
        )
        return result

    return wrapper

# Cell
@typechecked
class CopyPreProcessor(BaseProcessor):
    """Copy DataFrame to avoid manipulation of original DataFrame."""

    def __init__(self):
        super().__init__()

    @display_processor_info
    def transform(self, dataf: Union[pd.DataFrame, NumerFrame]) -> NumerFrame:
        return NumerFrame(dataf.copy())

# Cell
@typechecked
class FeatureSelectionPreProcessor(BaseProcessor):
    """
    Keep only features given + all target, predictions and aux columns.
    """

    def __init__(self, feature_cols: Union[str, list]):
        super().__init__()
        self.feature_cols = feature_cols

    @display_processor_info
    def transform(self, dataf: NumerFrame) -> NumerFrame:
        keep_cols = (
            self.feature_cols
            + dataf.target_cols
            + dataf.prediction_cols
            + dataf.aux_cols
        )
        dataf = dataf.loc[:, keep_cols]
        return NumerFrame(dataf)

# Cell
@typechecked
class TargetSelectionPreProcessor(BaseProcessor):
    """
    Keep only features given + all target, predictions and aux columns.
    """

    def __init__(self, target_cols: Union[str, list]):
        super().__init__()
        self.target_cols = target_cols

    @display_processor_info
    def transform(self, dataf: NumerFrame) -> NumerFrame:
        keep_cols = (
            self.target_cols
            + dataf.feature_cols
            + dataf.prediction_cols
            + dataf.aux_cols
        )
        dataf = dataf.loc[:, keep_cols]
        return NumerFrame(dataf)

# Cell
class ReduceMemoryProcessor(BaseProcessor):
    """
    Reduce memory usage as much as possible.

    Credits to kainsama and others for writing about memory usage reduction for Numerai data:
    https://forum.numer.ai/t/reducing-memory/313

    :param deep_mem_inspect: Introspect the data deeply by interrogating object dtypes.
    Yields a more accurate representation of memory usage if you have complex object columns.
    """
    def __init__(self, deep_mem_inspect = False):
        super().__init__()
        self.deep_mem_inspect = deep_mem_inspect

    @display_processor_info
    def transform(self, dataf: Union[pd.DataFrame, NumerFrame]) -> NumerFrame:
        dataf = self._reduce_mem_usage(dataf)
        return NumerFrame(dataf)

    def _reduce_mem_usage(self, dataf: pd.DataFrame) -> pd.DataFrame:
        """
        Iterate through all columns and modify the numeric column types
        to reduce memory usage.
        """
        start_memory_usage = dataf.memory_usage(deep=self.deep_mem_inspect).sum() / 1024**2
        rich_print(f"Memory usage of DataFrame is [bold]{round(start_memory_usage, 2)} MB[/bold]")

        for col in dataf.columns:
            col_type = dataf[col].dtype.name

            if col_type not in ['object', 'category', 'datetime64[ns, UTC]','datetime64[ns]']:
                c_min = dataf[col].min()
                c_max = dataf[col].max()
                if str(col_type)[:3] == 'int':
                    if c_min > np.iinfo(np.int8).min and c_max < np.iinfo(np.int8).max:
                        dataf[col] = dataf[col].astype(np.int16)
                    elif c_min > np.iinfo(np.int16).min and c_max < np.iinfo(np.int16).max:
                        dataf[col] = dataf[col].astype(np.int16)
                    elif c_min > np.iinfo(np.int32).min and c_max < np.iinfo(np.int32).max:
                        dataf[col] = dataf[col].astype(np.int32)
                    elif c_min > np.iinfo(np.int64).min and c_max < np.iinfo(np.int64).max:
                        dataf[col] = dataf[col].astype(np.int64)
                else:
                    if c_min > np.finfo(np.float16).min and c_max < np.finfo(np.float16).max:
                        dataf[col] = dataf[col].astype(np.float16)
                    elif c_min > np.finfo(np.float32).min and c_max < np.finfo(np.float32).max:
                        dataf[col] = dataf[col].astype(np.float32)
                    else:
                        dataf[col] = dataf[col].astype(np.float64)

        end_memory_usage = dataf.memory_usage(deep=self.deep_mem_inspect).sum() / 1024**2
        rich_print(f"Memory usage after optimization is: [bold]{round(end_memory_usage, 2)} MB[/bold]")
        rich_print(f"[green] Usage decreased by [bold]{round(100 * (start_memory_usage - end_memory_usage) / start_memory_usage, 2)}%[/bold][/green]")
        return dataf

# Cell
class SyntheticDataGenerator(BaseProcessor):
    """
    Generate synthetic eras. Uses SDV (sdv.dev) under the hood.

    :param model_name: Exact class name of a model supported on sdv. \n
    :param model_path: Either: \n
    1. Path to trained model. \n
    2. Path to where you want to save the fitted model. \n
    If model_path does not point to a valid file, a new model will be initialized, fitted and saved.
    """
    SUPPORTED_MODELS = ["GaussianCopula", "CTGAN", "CopulaGAN", "TVAE"]
    def __init__(self, model_path: str,
                 model_name = "CTGAN",
                 rows_per_era: int = 5400,
                 eras_to_add: int = 1):
        super().__init__()
        self.model_name = model_name
        assert self.model_name in self.SUPPORTED_MODELS,\
            f"Only models '{self.SUPPORTED_MODELS}' are supported. Got '{self.model_name}'."
        self.model_path = Path(model_path)
        self.rows_per_era = rows_per_era
        self.eras_to_add = eras_to_add

    @display_processor_info
    def transform(self, dataf: Union[pd.DataFrame, NumerFrame]) -> NumerFrame:
        model = self.prepare_model(dataf=dataf)
        synth_datafs = []
        for era_n in tqdm(range(self.eras_to_add), desc="Generating synthetic eras"):
            synth_era_data = self.get_synthetic_batch(model=model)
            synth_era_data.loc[:, dataf.meta.era_col] = f"synth_{str(era_n).zfill(4)}"
            synth_datafs.append(synth_era_data)

        synth_dataf = pd.concat(synth_datafs)
        dataf = pd.concat([dataf, synth_dataf])
        return NumerFrame(dataf)

    def prepare_model(self, dataf: Union[pd.DataFrame, NumerFrame]):
        if self.model_path.is_file():
            rich_print(f"Loading '{self.model_name}' model from '{self.model_path}'.")
            model = getattr(sdv.tabular, self.model_name).load(self.model_path)
        else:
            rich_print(f":warning: Model path '{self.model_path}' does not point to a file. Initializing, fitting and saving new model. :warning:")
            model = getattr(sdv.tabular, self.model_name)()
            model.fit(dataf)
            model.save(self.model_path)
        return model

    def get_synthetic_batch(self, model) -> pd.DataFrame:
        synthetic_dataf = model.sample(num_rows=self.rows_per_era)
        return synthetic_dataf

# Cell
class BayesianGMMTargetProcessor(BaseProcessor):
    """
    Generate synthetic (fake) target using a Bayesian Gaussian Mixture model. \n
    Based on Michael Oliver's GitHub Gist implementation: \n
    https://gist.github.com/the-moliver/dcdd2862dc2c78dda600f1b449071c93

    :param target_col: Column from which to create fake target. \n
    :param n_components: Number of components for fitting Bayesian Gaussian Mixture Model.
    """
    def __init__(self, target_col: str = "target", n_components: int = 6):
        super().__init__()
        self.target_col = target_col
        self.n_components = n_components
        self.ridge = Ridge(fit_intercept=False)
        self.bins = [0, 0.05, 0.25, 0.75, 0.95, 1]

    @display_processor_info
    def transform(self, dataf: NumerFrame, *args, **kwargs) -> NumerFrame:
        all_eras = dataf[dataf.meta.era_col].unique()
        coefs = self._get_coefs(dataf=dataf, all_eras=all_eras)
        bgmm = self._fit_bgmm(coefs=coefs)
        fake_target = self._generate_target(dataf=dataf,
                                            bgmm=bgmm,
                                            all_eras=all_eras)
        dataf[f"fake_{self.target_col}"] = fake_target
        return NumerFrame(dataf)

    def _get_coefs(self, dataf: NumerFrame, all_eras: list) -> np.ndarray:
        """
        Generate coefficients for BGMM.
        Data should already be scaled between 0 and 1
        (Already done with Numerai Classic data)
        """
        coefs = []
        for era in all_eras:
            features, target = self.__get_features_target(dataf=dataf, era=era)
            self.ridge.fit(features, target)
            coefs.append(self.ridge.coef_)
        stacked_coefs = np.vstack(coefs)
        return stacked_coefs

    def _fit_bgmm(self, coefs: np.ndarray) -> BayesianGaussianMixture:
        """
        Fit Bayesian Gaussian Mixture model on coefficients and normalize.
        """
        bggm = BayesianGaussianMixture(n_components=self.n_components)
        bggm.fit(coefs)
        # make probability of sampling each component equal to better balance rare regimes
        bggm.weights_[:] = 1 / self.n_components
        return bggm

    def _generate_target(self, dataf: NumerFrame,
                         bgmm: BayesianGaussianMixture,
                         all_eras: list) -> np.ndarray:
        """ Generate fake target using Bayesian Gaussian Mixture model. """
        fake_target = []
        for era in tqdm(all_eras, desc="Generating fake target"):
            features, _ = self.__get_features_target(dataf=dataf, era=era)
            # Sample a set of weights from GMM
            beta, _ = bgmm.sample(1)
            # Create fake continuous target
            fake_targ = features @ beta[0]
            # Bin fake target like real target
            fake_targ = (rankdata(fake_targ) - .5) / len(fake_targ)
            fake_targ = (np.digitize(fake_targ, self.bins) - 1) / 4
            fake_target.append(fake_targ)
        return np.concatenate(fake_target)

    def __get_features_target(self, dataf: NumerFrame, era) -> tuple:
        """ Get features and target for one era and center data. """
        sub_df = dataf[dataf.era == era]
        features = sub_df.get_feature_data
        target = sub_df[self.target_col]
        features = features.values - .5
        target = target.values - .5
        return features, target


# Cell
class GroupStatsPreProcessor(BaseProcessor):
    """
    WARNING: Only supported for Version 1 (legacy) data. \n
    Calculate group statistics for all data groups. \n
    | :param groups: Groups to create features for. All groups by default.
    """

    def __init__(self, groups: list = None):
        super().__init__()
        self.all_groups = [
            "intelligence",
            "wisdom",
            "charisma",
            "dexterity",
            "strength",
            "constitution",
        ]
        self.group_names = groups if groups else self.all_groups

    @display_processor_info
    def transform(self, dataf: NumerFrame, *args, **kwargs) -> NumerFrame:
        """Check validity and add group features."""
        self._check_data_validity(dataf=dataf)
        dataf = dataf.pipe(self._add_group_features)
        return NumerFrame(dataf)

    def _add_group_features(self, dataf: pd.DataFrame) -> pd.DataFrame:
        """Mean, standard deviation and skew for each group."""
        for group in self.group_names:
            cols = [col for col in dataf.columns if group in col]
            dataf[f"feature_{group}_mean"] = dataf[cols].mean(axis=1)
            dataf[f"feature_{group}_std"] = dataf[cols].std(axis=1)
            dataf[f"feature_{group}_skew"] = dataf[cols].skew(axis=1)
        return dataf

    def _check_data_validity(self, dataf: NumerFrame):
        """Make sure this is only used for version 1 data."""
        assert hasattr(
            dataf.meta, "version"
        ), f"Version should be specified for '{self.__class__.__name__}' This Preprocessor will only work on version 1 data."
        assert (
            getattr(dataf.meta, "version") == 1
        ), f"'{self.__class__.__name__}' only works on version 1 data. Got version: '{getattr(dataf.meta, 'version')}'."

# Cell
class TalibFeatureGenerator(BaseProcessor):
    """
    Generate relevant features available in TA-Lib. \n
    More info: https://mrjbq7.github.io/ta-lib \n
    Input DataFrames for these functions should have the following columns defined:
    ['open', 'high', 'low', 'close', 'volume'] \n
    Make sure that all values are sorted in chronological order (by ticker). \n
    :param windows: List of ranges for window features.
    Windows will be applied for all features specified in self.window_features. \n
    :param ticker_col: Which column to groupby for feature generation.
    """

    def __init__(self, windows: List[int], ticker_col: str = "bloomberg_ticker"):
        self.__check_talib_import()
        super().__init__()

        self.windows = windows
        self.ticker_col = ticker_col
        self.window_features = [
            "NATR",
            "ADXR",
            "AROONOSC",
            "DX",
            "MFI",
            "MINUS_DI",
            "MINUS_DM",
            "MOM",
            "ROCP",
            "ROCR100",
            "PLUS_DI",
            "PLUS_DM",
            "BETA",
            "RSI",
            "ULTOSC",
            "TRIX",
            "ADXR",
            "CCI",
            "CMO",
            "WILLR",
        ]
        self.no_window_features = ["AD", "OBV", "APO", "MACD", "PPO"]
        self.hlocv_cols = ["open", "high", "low", "close", "volume"]

    def get_no_window_features(self, dataf: pd.DataFrame):
        for func in tqdm(self.no_window_features, desc="No window features"):
            dataf.loc[:, f"feature_{func}"] = (
                dataf.groupby(self.ticker_col)
                .apply(lambda x: pd.Series(self._no_window(x, func)).bfill())
                .values.astype(np.float32)
            )
        return dataf

    def get_window_features(self, dataf: pd.DataFrame):
        for win in tqdm(self.windows, position=0, desc="Window features"):
            for func in tqdm(self.window_features, position=1):
                dataf.loc[:, f"feature_{func}_{win}"] = (
                    dataf.groupby(self.ticker_col)
                    .apply(lambda x: pd.Series(self._window(x, func, win)).bfill())
                    .values.astype(np.float32)
                )
        return dataf

    def get_all_features(self, dataf: pd.DataFrame) -> pd.DataFrame:
        dataf = self.get_no_window_features(dataf)
        dataf = self.get_window_features(dataf)
        return dataf

    def transform(self, dataf: pd.DataFrame, *args, **kwargs) -> NumerFrame:
        return NumerFrame(self.get_all_features(dataf=dataf))

    def _no_window(self, dataf: pd.DataFrame, func) -> pd.Series:
        from talib import abstract as tab

        inputs = self.__get_inputs(dataf)
        if func in ["MACD"]:
            # MACD outputs tuple of 3 elements (value, signal and hist)
            return tab.Function(func)(inputs["close"])[0]
        else:
            return tab.Function(func)(inputs)

    def _window(self, dataf: pd.DataFrame, func, window: int) -> pd.Series:
        from talib import abstract as tab

        inputs = self.__get_inputs(dataf)
        if func in ["ULTOSC"]:
            # ULTOSC requires 3 timeperiods as input
            return tab.Function(func)(
                inputs["high"],
                inputs["low"],
                inputs["close"],
                timeperiod1=window,
                timeperiod2=window * 2,
                timeperiod3=window * 4,
            )
        else:
            return tab.Function(func)(inputs, timeperiod=window)

    def __get_inputs(self, dataf: pd.DataFrame) -> dict:
        return {col: dataf[col].values.astype(np.float64) for col in self.hlocv_cols}

    @staticmethod
    def __check_talib_import():
        try:
            from talib import abstract as tab
        except ImportError:
            raise ImportError(
                "TA-Lib is not installed for this environment. If you are using this class make sure to have TA-Lib installed. check https://mrjbq7.github.io/ta-lib/install.html for instructions on installation."
            )

# Cell
class KatsuFeatureGenerator(BaseProcessor):
    """
    Effective feature engineering setup based on Katsu's starter notebook.
    Based on source by Katsu1110: https://www.kaggle.com/code1110/numeraisignals-starter-for-beginners

    :param windows: Time interval to apply for window features: \n
    1. Percentage Rate of change \n
    2. Volatility \n
    3. Moving Average gap \n
    :param ticker_col: Columns with tickers to iterate over. \n
    :param close_col: Column name where you have closing price stored.
    """

    warnings.filterwarnings("ignore")

    def __init__(
        self,
        windows: list,
        ticker_col: str = "ticker",
        close_col: str = "close",
        num_cores: int = None,
    ):
        super().__init__()
        self.windows = windows
        self.ticker_col = ticker_col
        self.close_col = close_col
        self.num_cores = num_cores if num_cores else os.cpu_count()

    @display_processor_info
    def transform(self, dataf: Union[pd.DataFrame, NumerFrame]) -> NumerFrame:
        """Multiprocessing feature engineering."""
        tickers = dataf.loc[:, self.ticker_col].unique().tolist()
        rich_print(
            f"Feature engineering for {len(tickers)} tickers using {self.num_cores} CPU cores."
        )
        dataf_list = [
            x
            for _, x in tqdm(
                dataf.groupby(self.ticker_col), desc="Generating ticker DataFrames"
            )
        ]
        dataf = self._generate_features(dataf_list=dataf_list)
        return NumerFrame(dataf)

    def feature_engineering(self, dataf: pd.DataFrame) -> pd.DataFrame:
        """Feature engineering for single ticker."""
        close_series = dataf.loc[:, self.close_col]
        for x in self.windows:
            dataf.loc[
                :, f"feature_{self.close_col}_ROCP_{x}"
            ] = close_series.pct_change(x)

            dataf.loc[:, f"feature_{self.close_col}_VOL_{x}"] = (
                np.log1p(close_series).pct_change().rolling(x).std()
            )

            dataf.loc[:, f"feature_{self.close_col}_MA_gap_{x}"] = (
                close_series / close_series.rolling(x).mean()
            )

        dataf.loc[:, "feature_RSI"] = self._rsi(close_series)
        macd, macd_signal = self._macd(close_series)
        dataf.loc[:, "feature_MACD"] = macd
        dataf.loc[:, "feature_MACD_signal"] = macd_signal
        return dataf.bfill()

    def _generate_features(self, dataf_list: list) -> pd.DataFrame:
        """Add features for list of ticker DataFrames and concatenate."""
        with Pool(self.num_cores) as p:
            feature_datafs = list(
                tqdm(
                    p.imap(self.feature_engineering, dataf_list),
                    desc="Generating features",
                    total=len(dataf_list),
                )
            )
        return pd.concat(feature_datafs)

    @staticmethod
    def _rsi(close: pd.Series, period: int = 14) -> pd.Series:
        """
        See source https://github.com/peerchemist/finta
        and fix https://www.tradingview.com/wiki/Talk:Relative_Strength_Index_(RSI)
        """
        delta = close.diff()
        up, down = delta.copy(), delta.copy()
        up[up < 0] = 0
        down[down > 0] = 0

        gain = up.ewm(com=(period - 1), min_periods=period).mean()
        loss = down.abs().ewm(com=(period - 1), min_periods=period).mean()

        rs = gain / loss
        return pd.Series(100 - (100 / (1 + rs)))

    def _macd(
        self, close: pd.Series, span1=12, span2=26, span3=9
    ) -> Tuple[pd.Series, pd.Series]:
        """Compute MACD and MACD signal."""
        exp1 = self.__ema1(close, span1)
        exp2 = self.__ema1(close, span2)
        macd = 100 * (exp1 - exp2) / exp2
        signal = self.__ema1(macd, span3)
        return macd, signal

    @staticmethod
    def __ema1(series: pd.Series, span: int) -> pd.Series:
        """Exponential moving average"""
        a = 2 / (span + 1)
        return series.ewm(alpha=a).mean()

# Cell
class EraQuantileProcessor(BaseProcessor):
    """
    Transform features into quantiles on a per-era basis

    :param num_quantiles: Number of buckets to split data into: \n
    :param era_col: Era column name in the dataframe to perform each transformation \n
    """

    def __init__(
        self,
        num_quantiles: int = 50,
        era_col: str = "friday_date",
        features: list = None,
        num_cores: int = None,
    ):
        super().__init__()
        self.num_quantiles = num_quantiles
        self.era_col = era_col
        self.num_cores = num_cores if num_cores else os.cpu_count()
        self.features = features

    def _process_eras(self, groupby_object):
        quantizer = QuantileTransformer(n_quantiles=self.num_quantiles, random_state=0)
        qt = lambda x: quantizer.fit_transform(x.values.reshape(-1, 1)).ravel()

        column = groupby_object.transform(qt)
        return column

    @display_processor_info
    def transform(
        self,
        dataf: Union[pd.DataFrame, NumerFrame],
    ) -> NumerFrame:
        """Multiprocessing quantile transforms by era."""
        self.features = self.features if self.features else dataf.feature_cols
        rich_print(
            f"Quantiling for {len(self.features)} features using {self.num_cores} CPU cores."
        )

        date_groups = dataf.groupby(self.era_col)
        groupby_objects = [date_groups[feature] for feature in self.features]

        with Pool() as p:
            results = list(
                tqdm(
                    p.imap(self._process_eras, groupby_objects),
                    total=len(groupby_objects),
                )
            )

        quantiles = pd.concat(results, axis=1)
        dataf[[f"{feature}_quantile" for feature in self.features]] = quantiles
        return NumerFrame(dataf)

# Cell
class AwesomePreProcessor(BaseProcessor):
    """ TEMPLATE - Do some awesome preprocessing. """
    def __init__(self):
        super().__init__()

    @display_processor_info
    def transform(self, dataf: NumerFrame, *args, **kwargs) -> NumerFrame:
        # Do processing
        ...
        # Parse all contents of NumerFrame to the next pipeline step
        return NumerFrame(dataf)