import datetime
import logging
import os
from logging.handlers import RotatingFileHandler

from src.mb_cruise_migration.db.cruise_connection import BatchError
from src.mb_cruise_migration.framework.consts.log_level_consts import LogLevels
from src.mb_cruise_migration.logging.migration_logger import LoggerBuilder
from src.mb_cruise_migration.migration_properties import MigrationProperties
from src.mb_cruise_migration.models.cruise.cruise_files import CruiseFile
from src.mb_cruise_migration.models.intermediary.cruise_cargo import CruiseCargo, CruiseSurveyCrate
from src.mb_cruise_migration.models.intermediary.mb_cargo import MbFileCrate
from src.mb_cruise_migration.models.intermediary.migrating_survey import MigratingSurvey
from src.mb_cruise_migration.models.mb.mb_ngdcid_and_file import MbFile


class LogConsts(object):
    START = "START"
    END = "END"
    SKIP = "SKIP"
    BEGIN = "BEGIN"
    DONE = "DONE"
    REVIEW = "REVIEW"
    ERROR = "ERROR"


class MigrationLog(LoggerBuilder):
    logger: logging.Logger = None

    def __init__(self):
        self.__pre_check()

        keyword = "mig_log"
        level = MigrationProperties.log_config.level
        handler = self.__get_log_handler(keyword)
        formatter = self.__get_log_formatter()
        super(MigrationLog, self).__init__(keyword, level, handler, formatter)

        self.__finalize(level)

    @staticmethod
    def __pre_check():
        if not MigrationProperties.log_config:
            raise RuntimeError("Properties must be loaded prior to creating a logger")

    def __finalize(self, level):
        MigrationLog.logger = self.get_logger()
        self.__validate_level(level)

    @staticmethod
    def __get_log_handler(keyword):
        log_dir = MigrationProperties.log_config.log_path
        log_path = LoggerBuilder.create_log_file_path(os.getcwd(), log_dir, keyword)
        log_size = 2*1024*1024  # 2MB
        return logging.handlers.RotatingFileHandler(log_path, maxBytes=log_size, backupCount=10)

    @staticmethod
    def __get_log_formatter():
        return logging.Formatter('%(asctime)s %(levelname)-7s %(message)s', '%Y-%m-%d %H:%M:%S')
        # return logging.Formatter('%(asctime)s %(levelname)-7s %(name)-7s %('
        #                          'module)s:%(funcName)s:%(lineno)s -- %('
        #                          'message)s', '%Y-%m-%d %H:%M:%S')

    @staticmethod
    def __validate_level(level):
        if level == LogLevels.INFO:
            MigrationLog.logger.info(f"Logger initialized at level {LogLevels.INFO}.")
            return
        if level == LogLevels.WARNING:
            MigrationLog.logger.info(f"Logger initialized at level {LogLevels.WARNING}.")
            return
        if level == LogLevels.CRITICAL:
            MigrationLog.logger.info(f"Logger initialized at level {LogLevels.CRITICAL}.")
            return
        if level == LogLevels.DEBUG:
            MigrationLog.logger.info(f"Logger initialized at level {LogLevels.DEBUG}.")
            return
        MigrationLog.logger.info("Provided log level unrecognized. Logger defaulting to DEBUG level logging.")

    @staticmethod
    def log_start():
        migrate_list = MigrationProperties.manifest.use_list
        MigrationLog.logger.info(f"{LogConsts.START} -- Migration start at {datetime.datetime.now()}")
        MigrationLog.logger.info(f"")

    @staticmethod
    def log_end():
        MigrationLog.logger.info(f"{LogConsts.END} -- Migration end at {datetime.datetime.now()}")

    @staticmethod
    def log_skipped_file(file: MbFile):
        MigrationLog.logger.info(f"{LogConsts.SKIP} -- File: skipping migration of file {file.data_file} with ngdc_id {file.ngdc_id}")

    @staticmethod
    def log_invalidated_file(file: MbFileCrate, reason: str):
        filename = file.mb_file.data_file
        MigrationLog.logger.warning(f"{LogConsts.REVIEW} -- File migration skipped for {filename} due to: \r\t {reason}")

    @staticmethod
    def log_file_for_manual_review(mb_file_crate: MbFileCrate, instrument: str):
        file = mb_file_crate.mb_file.data_file

        MigrationLog.logger.info(f"{LogConsts.REVIEW} -- File Instrument: {instrument} derived from mb.survey.instrument for file {file}")

    @staticmethod
    def log_failed_file_migration(survey_name: str, file: CruiseFile, e: Exception):
        message = "Failed to identify issue with file migration. File migration was skipped due to thrown error."
        try:
            message = str(e)
        except UnicodeDecodeError:
            print("Failed to decode exception message.")
        MigrationLog.logger.error(f"{LogConsts.ERROR} -- File Insertion: file in survey {survey_name} with name {file.file_name} failed to migrate with error message: \r {message}")

    @staticmethod
    def log_dataset_start(cargo: CruiseCargo):
        dataset_name = cargo.dataset_crate.dataset.dataset_name
        num_files = len(cargo.related_file_crates)
        MigrationLog.logger.info(f"{LogConsts.BEGIN} -- Dataset Migration: start migration of dataset {dataset_name} and its {num_files} associated files")

    @staticmethod
    def log_migrated_dataset(cargo: CruiseCargo, actual_files, expected_files):
        dataset_name = cargo.dataset_crate.dataset.dataset_name
        survey_name = cargo.related_survey_crate.cruise_survey.survey_name

        if actual_files < expected_files:
            MigrationLog.logger.warning(f"{LogConsts.REVIEW} -- Dataset Migrated with problems: Survey: {survey_name}; {actual_files} files migrated out of {expected_files}")
        else:
            MigrationLog.logger.info(f"{LogConsts.DONE} -- Dataset Migrated: {dataset_name};  Survey: {survey_name};  Number of migrated files: {actual_files}")

    @staticmethod
    def log_failed_dataset(message):
        MigrationLog.logger.error(f"{LogConsts.ERROR} -- Dataset Migration Failed: {message}")

    @staticmethod
    def log_survey_start(survey_crate: CruiseSurveyCrate):
        survey_name = survey_crate.cruise_survey.survey_name
        MigrationLog.logger.info(f"{LogConsts.BEGIN} -- Survey Migration: {survey_name}")

    @staticmethod
    def log_migrated_survey(survey: MigratingSurvey, problem_flag: bool, problem_message: str):
        survey_name = survey.survey_name
        num_datasets = survey.migrated_datasets
        num_actual_files = survey.migrated_files
        num_expected_files = survey.expected_files
        if num_actual_files < num_expected_files:
            problem_flag = True
            problem_message = f"Only {num_actual_files} files of {num_expected_files} migrated successfully;" + problem_message
        if problem_flag:
            MigrationLog.log_problem_survey(survey_name, problem_message)
        else:
            MigrationLog.logger.info(f"{LogConsts.DONE} -- Survey Migration: {survey_name};  Number of migrated datasets: {num_datasets};  Number of migrated files: {num_actual_files}")

    @staticmethod
    def log_skipped_survey(survey_name: str):
        MigrationLog.logger.info(f"{LogConsts.SKIP} -- Survey: skipping migration of survey {survey_name}.")

    @staticmethod
    def log_problem_survey(survey_name: str, message: str):
        MigrationLog.logger.warning(f"{LogConsts.REVIEW} -- Problem Survey: problems identified with {survey_name} with message: {message}")

    @staticmethod
    def log_failed_survey(survey_name: str, e):
        message = "Failed to identify issue with survey. Survey migration was cancelled due to thrown error."
        try:
            message = str(e)
        except UnicodeDecodeError:
            print("Failed to decode exception message.")

        MigrationLog.logger.error(f"{LogConsts.ERROR} -- Survey: survey {survey_name} failed to migrate with message:\r {message}")

    @staticmethod
    def log_batch_insert_errors(errors: [BatchError], batch_data, context_message):
        for error in errors:
            try:
                data = batch_data[error.offset]
                MigrationLog.logger.error(f"{LogConsts.ERROR} -- Batch Insert: Error received during {context_message} batch insert of {data}: {error.error}")
            except IndexError:
                MigrationLog.logger.error(f"{LogConsts.ERROR} -- Batch Insert: Error received during {context_message} batch insert at offset {error.offset}: {error.error}")

    @staticmethod
    def log_database_error(message: str, exception: Exception):
        try:
            exception_message = str(exception)
            MigrationLog.logger.error(f"{LogConsts.ERROR} -- Database Error: {message} with exception: {exception_message}")
        except UnicodeDecodeError:
            print("Failed to decode exception message.")
            MigrationLog.logger.error(f"{LogConsts.ERROR} -- Database Error: {message}")

    @staticmethod
    def log_cruise_processor_unhandled_error(cruise_cargo: [CruiseCargo], error_message):
        datasets = []
        for cargo in cruise_cargo:
            datasets.append(cargo.dataset_crate.dataset.dataset_name)
        MigrationLog.logger.error(f"Unhandled error occurred during dataset insertion. Potentially affected datasets include {datasets}. ERROR: {error_message}")

    @staticmethod
    def log_mb_survey_query(query):
        MigrationLog.logger.info(f"Querying MB.SURVEY for next batch of surveys: \r\t\t{query}")
