"""
# /////////
# Atomic Type Library
# eg: `from dada_types import T`
# TODO: centralize all type concerns into single module
# (this should include response / request schema stuff... is there a way we can build that into our type dictionary?)
# /////////
"""
import logging
from typing import Dict, List, Any, Set, Optional, NewType

from marshmallow import fields

from dada_utils import dates

from dada_types.base import SerializableObject
from dada_types.dictionary import TYPES, TYPE_NAMES, NewVal


class DadaType(SerializableObject):
    """
    A bass class for a Dada Type
    """

    __abstract__ = True
    __name__ = None
    __dada_type__ = None

    def __init__(self, name: str, schema: dict):

        # set internal name/ type / schema
        self.name = name
        self.__name__ = name
        self.__dada_type__ = name
        self.__set_attribtues__(schema)
        self.__abstract__ = False

    def __set_attribtues__(self, type_schema: dict) -> None:

        # set attributes
        for attr_name, attr_value in type_schema.items():

            # ////////////////////
            # monkey patching other libraries
            # objects with __dada_type__ so we
            # can map back to our tpye library
            # in various contexts
            # ////////////////////
            if type(attr_value) == type(str):
                continue
            try:
                setattr(attr_value, "__dada_type__", self.__dada_type__)
            except AttributeError:
                pass

            # set attribute on the object
            setattr(self, attr_name, attr_value)

        # set default attributes
        # for this type
        # this exists to avoid repetitious
        # definitions in dictionary

        if not hasattr(self, "cat"):
            self.cat = ["text"]

        if not hasattr(self, "sql"):
            self.sql = "TEXT"

        if not hasattr(self, "load"):
            self.load = str

        if not hasattr(self, "val"):
            self.val = fields.Str(missing=None)

        # options for this type
        if not hasattr(self, "dump"):
            self.dump = str

        # options for this type
        if not hasattr(self, "opts"):
            self.opts = None

        # options for this type
        if not hasattr(self, "hum"):
            self.hum = lambda x: str(x)

        # make the cats frozen sets
        # for master comparisons
        self.cat_set = frozenset(self.cat)

        # make the cats frozen sets
        # for master comparisons
        self.opt_set = frozenset(self.opts or [])

    # name accessors

    @property
    def py_name(self) -> str:
        """
        The custom python type generated by `NewType`
        :return str
        """
        self.py.__name__

    # alternate version types
    @property
    def py_optional(self):
        """
        An optional version of this type
        """
        type_name = f"{self.__dada_type__}_optional"
        py_type = NewType(type_name, Optional[self.py])
        setattr(py_type, "__dada_type__", self.__dada_type__)
        return py_type

    # boolean helpers
    @property
    def has_validator(self) -> bool:
        """
        whether or not this type is an ID
        :return bool
        """
        return getattr(self, "val", None) is not None

    @property
    def has_options(self) -> bool:
        """
        whether or not this type is an ID
        :return bool
        """
        return self.opts is not None and len(self.opts) > 0

    @property
    def is_id(self) -> bool:
        """
        whether or not this type is an ID
        :return bool
        """
        return "id" in self.cat

    @property
    def is_enum(self) -> bool:
        """
        whether or not this type is an ``enum``
        :return bool
        """
        return "enum" in self.cat

    @property
    def is_pk(self) -> bool:
        """
        whether or not this type is an ID
        :return bool
        """
        return "pk" in self.cat

    @property
    def is_fk(self) -> bool:
        """
        whether or not this type is an ID
        :return bool
        """
        return "fk" in self.cat

    @property
    def is_int(self) -> bool:
        """
        whether or not this type is an ``int``
        :return bool
        """
        return "int" in self.cat

    @property
    def is_num(self) -> bool:
        """
        whether or not this type is a ``num``
        :return bool
        """
        return "num" in self.cat

    @property
    def is_stat(self) -> bool:
        """
        whether or not this type is a ``stat``
        :return bool
        """
        return "stat" in self.cat

    @property
    def is_text(self) -> bool:
        """
        whether or not this type is ``text``
        :return bool
        """
        return "text" in self.cat

    @property
    def is_date(self) -> bool:
        """
        whether or not this type is ``date``
        :return bool
        """
        return "date" in self.cat

    @property
    def is_time(self) -> bool:
        """
        whether or not this type is ``time``
        :return bool
        """
        return "time" in self.cat

    @property
    def is_json(self) -> bool:
        """
        whether or not this type is json
        :return bool
        """
        return "json" in self.cat

    @property
    def is_fields(self) -> bool:
        """
        whether or not this type represent an internally-typped fields object
        :return bool
        """
        return "fields" in self.cat

    @property
    def is_param(self) -> bool:
        """
        whether or not this type represent an internally-typped params object
        :return bool
        """
        return "param" in self.cat

    @property
    def is_theme(self) -> bool:
        """
        whether or not this type represents an internal theme object
        :return bool
        """
        return "theme" in self.cat

    @property
    def is_matrix(self):
        """
        whether or not this type represents an a matrix object
        """
        return "matrix" in self.cat

    @property
    def is_array(self):
        """
        whether or not this type represents an a array object
        """
        return "array" in self.cat

    # MEETHODS

    def is_opt_valid(self, val: Any) -> bool:
        """
        Check if a value is valid
        """
        if not self.has_options:
            return None
        return self.load(val) in self.opt_set

    def validate(self, val: Any) -> Any:
        """
        Validate this type
        """
        val = self.load(val)
        if self.has_validator and not self.is_date:  # dates are handling by `load`
            val = self.val.deserialize(val)
        if self.has_options:
            if not self.is_opt_valid(val):
                opt_string = ",".join(self.opts)
                raise ValueError(f"Invalid {self.name} type. choose from: {opt_string}")
        return val

    # render this type as a dictionary
    def to_dict(self) -> Dict[str, str]:
        """
        whether or not this type is
        :return bool
        """
        return {
            "name": self.__name__,
            "debug": self.info,
            "cat": self.cat,
            "sql": self.sql,
            "col": str(self.col),
            "py_name": self.py_name,
            "example": self.gen(),
        }


class DadaTypes(SerializableObject):
    """
    Our our internal type library
    """

    __abstract__ = True
    __dictionary__ = TYPES
    __dada_types__ = TYPE_NAMES
    __dada_type_set__ = frozenset(TYPE_NAMES)

    def __init__(self):
        self.names = self.__dada_types__
        self.name_set = frozenset(self.names)
        self._set_types()
        self.__abstract__ = False

    def _set_types(self) -> None:
        """
        Load the type dictionary as class attributes + methods
        :return None
        """
        for type_name, type_schema in self.__dictionary__.items():
            if type_name == "type":
                # include options for "type" type, which is all types
                type_schema["opts"] = self.names
            setattr(self, type_name, DadaType(name=type_name, schema=type_schema))

        # set the opts for the

    @classmethod
    def is_supported(cls, type: str) -> bool:
        """
        Whether or not a type name is supported
        """
        return type not in cls.__dada_type_set__

    def get(
        self,
        type: str,
        default: Optional[str] = None,
        partial: bool = False,
        partial_force_arrays: bool = False,
        partial_guess_arrays: bool = False,
        raise_on_unsupported: bool = False,
    ) -> Optional[DadaType]:
        """
        Look a type object by it's type name,
        """
        if not self.is_supported(type):
            if raise_on_unsupported:
                raise ValueError(f"Unsupported Type: {type}")
        typobj = getattr(self, type, None)

        # if not partial lookup, fall back on the default
        # if we havent found a type object
        if not partial:
            if not typobj and default is not None:
                typobj = getattr(self, default, None)
            return typobj

        # handle special cases

        if "vector" in type:
            return self.get("vector")

        if type.endswith("_id"):
            return self.get("fk")

        if dates.is_field_date(type):
            return self.get("date")

        if type.startswith("has_") or type.startswith("is_"):
            return self.get("bool")

        # partial lookup based on type name
        # eg `accepts_file_types` -> `T.file_type_array`
        for typname in self.names:

            # partial match of type name
            if type not in typname:
                continue

            # force array type
            if partial_force_arrays:
                if typname.endswith("_array"):
                    return self.get(typname)

            # guess array type based on 's' ending
            if partial_guess_arrays:
                if type.endswith("s"):
                    return self.get(typname)

            return self.get(typname)

        # fall back on default or return None
        if default is not None:
            return getattr(self, default, None)
        return None
