import { Resource, ResourceProps } from 'aws-cdk-lib';
import { CfnFlowLog, FlowLogResourceType, FlowLogTrafficType, IFlowLog } from 'aws-cdk-lib/aws-ec2';
import { IConstruct } from 'constructs';
import { FlowLogDestination } from './lib/flow-log-destination';
export declare enum FlowLogAggregationInterval {
    /**
       * Flow logs will be written at least every 60 seconds.
       */
    ONE_MINUTE = 60,
    /**
       * Flow logs will be written at least every ten minutes.
       */
    TEN_MINUTES = 600
}
export declare enum FlowLogDataType {
    /**
       * 32 bit signed int.
       */
    INT_32 = "INT_32",
    /**
       * 64 bit signed int.
       */
    INT_64 = "INT_64",
    /**
       * UTF-8 encoded character string.
       */
    STRING = "STRING"
}
export declare class FlowLogField {
    /**
       * The AWS account ID of the owner of the source network interface for
       * which traffic is recorded. If the network interface is created by an
       * AWS service, for example when creating a VPC endpoint or Network Load
       * Balancer, the record might display unknown for this field.
       */
    static readonly ACCOUNT_ID: FlowLogField;
    /**
       * The action that is associated with the traffic:
       *
       * ACCEPT: The recorded traffic was permitted by the security groups and
       * network ACLs.
       * REJECT: The recorded traffic was not permitted by the security groups
       * or network ACLs.
       */
    static readonly ACTION: FlowLogField;
    /**
       * The ID of the Availability Zone that contains the network interface for
       * which traffic is recorded. If the traffic is from a sublocation, the
       * record displays a '-' symbol for this field.
       */
    static readonly AZ_ID: FlowLogField;
    /**
       * The number of bytes transferred during the flow.
       */
    static readonly BYTES: FlowLogField;
    /**
       * The destination address for outgoing traffic, or the IPv4 or IPv6
       * address of the network interface for incoming traffic on the network
       * interface. The IPv4 address of the network interface is always its
       * private IPv4 address.
       *
       * See also:
       * {@link FlowLogField.PKT_DSTADDR | PKT_DSTADDR}
       */
    static readonly DSTADDR: FlowLogField;
    /**
       * The destination port of the traffic.
       */
    static readonly DSTPORT: FlowLogField;
    /**
       * The time, in Unix seconds, when the last packet of the flow was
       * received within the aggregation interval. This might be up to 60
       * seconds after the packet was transmitted or received on the network
       * interface.
       */
    static readonly END: FlowLogField;
    /**
       * The direction of the flow with respect to the interface where traffic
       * is captured. The possible values are: ingress | egress.
       */
    static readonly FLOW_DIRECTION: FlowLogField;
    /**
       * The ID of the instance that's associated with network interface for
       * which the traffic is recorded, if the instance is owned by you. Returns
       * a '-' symbol for a requester-managed network interface; for example,
       * the network interface for a NAT gateway.
       *
       * See also:
       * [Request-managed ENI](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/requester-managed-eni.html)
       */
    static readonly INSTANCE_ID: FlowLogField;
    /**
       * The ID of the network interface for which the traffic is recorded.
       */
    static readonly INTERFACE_ID: FlowLogField;
    /**
       * The logging status of the flow log:
       *
       * OK: Data is logging normally to the chosen destinations.
       * NODATA: There was no network traffic to or from the network interface
       * during the aggregation interval.
       * SKIPDATA — Some flow log records were skipped during the aggregation
       * interval. This might be because of an internal capacity constraint, or
       * an internal error.
       */
    static readonly LOG_STATUS: FlowLogField;
    /**
       * The number of packets transferred during the flow.
       */
    static readonly PACKETS: FlowLogField;
    /**
       * The name of the subset of IP address ranges for the pkt-dstaddr field,
       * if the destination IP address is for an AWS service. For a list of
       * possible values, see the {@link FlowLogField.PKT_SRC_AWS_SERVICE | PKT_SRC_AWS_SERVICE} field.
       */
    static readonly PKT_DST_AWS_SERVICE: FlowLogField;
    /**
       * The packet-level (original) destination IP address for the traffic. Use
       * this field with the dstaddr field to distinguish between the IP address
       * of an intermediate layer through which traffic flows, and the final
       * destination IP address of the traffic. For example, when traffic flows
       * through a network interface for a NAT gateway, or where the IP address
       * of a pod in Amazon EKS is different from the IP address of the network
       * interface of the instance node on which the pod is running (for
       * communication within a VPC).
       *
       * See also:
       * [Flow Log Example NAT](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs-records-examples.html#flow-log-example-nat)
       */
    static readonly PKT_DSTADDR: FlowLogField;
    /**
       * The name of the subset of IP address ranges for the pkt-srcaddr field,
       * if the source IP address is for an AWS service. The possible values
       * are: AMAZON | AMAZON_APPFLOW | AMAZON_CONNECT | API_GATEWAY |
       * CHIME_MEETINGS | CHIME_VOICECONNECTOR | CLOUD9 | CLOUDFRONT |
       * CODEBUILD | DYNAMODB | EBS | EC2 | EC2_INSTANCE_CONNECT |
       * GLOBALACCELERATOR | KINESIS_VIDEO_STREAMS | ROUTE53 |
       * ROUTE53_HEALTHCHECKS | ROUTE53_HEALTHCHECKS_PUBLISHING |
       * ROUTE53_RESOLVER | S3 | WORKSPACES_GATEWAYS.
       */
    static readonly PKT_SRC_AWS_SERVICE: FlowLogField;
    /**
       * The packet-level (original) source IP address of the traffic. Use this
       * field with the srcaddr field to distinguish between the IP address of
       * an intermediate layer through which traffic flows, and the original
       * source IP address of the traffic. For example, when traffic flows
       * through a network interface for a NAT gateway, or where the IP address
       * of a pod in Amazon EKS is different from the IP address of the network
       * interface of the instance node on which the pod is running (for
       * communication within a VPC).
       *
       * See also:
       * [Flow Log Example NAT](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs-records-examples.html#flow-log-example-nat)
       */
    static readonly PKT_SRCADDR: FlowLogField;
    /**
       * The IANA protocol number of the traffic.
       *
       * See also:
       * [Assigned Internet Protocol Numbers](http://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml).
       */
    static readonly PROTOCOL: FlowLogField;
    /**
       * The Region that contains the network interface for which traffic is
       * recorded.
       */
    static readonly REGION: FlowLogField;
    /**
       * The source address for incoming traffic, or the IPv4 or IPv6 address of
       * the network interface for outgoing traffic on the network interface.
       * The IPv4 address of the network interface is always its private IPv4
       * address.
       *
       * See also:
       * {@link FlowLogField.PKT_SRCADDR | PKT_SRCADDR}
       */
    static readonly SRCADDR: FlowLogField;
    /**
       * The source port of the traffic.
       */
    static readonly SRCPORT: FlowLogField;
    /**
       * The time, in Unix seconds, when the first packet of the flow was
       * received within the aggregation interval. This might be up to 60
       * seconds after the packet was transmitted or received on the network
       * interface.
       */
    static readonly START: FlowLogField;
    /**
       * The ID of the sublocation that contains the network interface for which
       * traffic is recorded. If the traffic is not from a sublocation, the
       * record displays a '-' symbol for this field.
       */
    static readonly SUBLOCATION_ID: FlowLogField;
    /**
       * The type of sublocation that's returned in the sublocation-id field.
       * The possible values are: wavelength | outpost | localzone. If the
       * traffic is not from a sublocation, the record displays a '-' symbol
       * for this field.
       *
       * See also:
       * [Wavelength](https://aws.amazon.com/wavelength/)
       * [Outposts](https://docs.aws.amazon.com/outposts/latest/userguide/)
       * [Local Zones](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html#concepts-local-zones)
       */
    static readonly SUBLOCATION_TYPE: FlowLogField;
    /**
       * The ID of the subnet that contains the network interface for which the
       * traffic is recorded.
       */
    static readonly SUBNET_ID: FlowLogField;
    /**
       * The bitmask value for the following TCP flags:
       *
       * FIN: 1
       * SYN: 2
       * RST: 4
       * PSH: 8
       * ACK: 16
       * SYN-ACK: 18
       * URG: 32
       *
       * When a flow log entry consists of only ACK packets, the flag value is
       * 0, not 16.
       *
       * TCP flags can be OR-ed during the aggregation interval. For short
       * connections, the flags might be set on the same line in the flow log
       * record, for example, 19 for SYN-ACK and FIN, and 3 for SYN and FIN.
       *
       * See also:
       * [TCP Segment Structure](https://en.wikipedia.org/wiki/Transmission_Control_Protocol#TCP_segment_structure)
       * [TCP Flag Sequence](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs-records-examples.html#flow-log-example-tcp-flag)
       */
    static readonly TCP_FLAGS: FlowLogField;
    /**
       * The path that egress traffic takes to the destination. To determine
       * whether the traffic is egress traffic, check the flow-direction field.
       * The possible values are as follows. If none of the values apply, the
       * field is set to -.
       *
       * 1: Through another resource in the same VPC
       * 2: Through an internet gateway or a gateway VPC endpoint
       * 3: Through a virtual private gateway
       * 4: Through an intra-region VPC peering connection
       * 5: Through an inter-region VPC peering connection
       * 6: Through a local gateway
       * 7: Through a gateway VPC endpoint (Nitro-based instances only)
       * 8: Through an internet gateway (Nitro-based instances only)
       */
    static readonly TRAFFIC_PATH: FlowLogField;
    /**
       * The type of traffic. The possible values are: IPv4 | IPv6 | EFA.
       *
       * See also:
       * [Elastic Fabric Adapter](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html)
       */
    static readonly TYPE: FlowLogField;
    /**
       * The VPC Flow Logs version. If you use the default format, the version
       * is 2. If you use a custom format, the version is the highest version
       * among the specified fields. For example, if you specify only fields
       * from version 2, the version is 2. If you specify a mixture of fields
       * from versions 2, 3, and 4, the version is 4.
       */
    static readonly VERSION: FlowLogField;
    /**
       * The ID of the VPC that contains the network interface for which the
       * traffic is recorded.
       */
    static readonly VPC_ID: FlowLogField;
    /**
       * The name of the Flow Log field, as it should be used when building a
       * format string.
       */
    readonly name: string;
    /**
       * The data type of the field as it would appear in Parquet. For
       * information on the type for various files, see documentation on the
       * [available fields](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-logs-fields).
       */
    readonly type: FlowLogDataType;
    /**
       * Creates a new instance of the FlowLogField class.
       *
       * @param name The name of the Flow Log field, as it should be used when
       * building a format string.
       * @param type The data type of the field as it would appear in Parquet.
       * For information on the type for various files, see documentation on the
       * [available fields](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-logs-fields).
       */
    constructor(name: string, type: FlowLogDataType);
}
export declare class FlowLogFormat {
    /**
       * The basic set of fields included in most flow logs. This is the default
       * format that is used when new flow logs are created without specifying a
       * custom format.
       */
    static readonly V2: FlowLogFormat;
    /**
       * Includes all the fields available in V2. Adds fields to help identify
       * AWS resources associated with traffic as well as fields that give
       * greater visibility into protocol specific details.
       */
    static readonly V3: FlowLogFormat;
    /**
       * Includes all the fields available in V3. Adds fields for identifying
       * the region and availabilty zone associated with flows, as well as
       * details related to extended zones such as Wavelength, Outputs, and
       * Local Zones.
       */
    static readonly V4: FlowLogFormat;
    /**
       * Includes all the fields available in V4. Adds fields to help identify
       * related AWS services and improve visibility into packet routing.
       */
    static readonly V5: FlowLogFormat;
    /**
       * The fields that make up the flow log format, in the order that they
       * should appear in the log entries.
       */
    readonly fields: FlowLogField[];
    /**
       * The rendered format string in the format expected by AWS when creating
       * a new Flow Log.
       */
    readonly template: string;
    /**
       * Creates a new instance of the FlowLogFormat class.
       *
       * @param fields The fields that should be included in the flow log output.
       */
    constructor(...fields: FlowLogField[]);
    /**
       * Adds a new field to the flow log output. New fields are added at the
       * end of a log entry after all the other fields that came before it.
       *
       * @param field The field to add to the FlowLogFormat.
       */
    addField(field: FlowLogField): void;
}
/**
 * Configuration for the FlowLog class.
 */
export interface FlowLogProps extends ResourceProps {
    /**
       * The location where flow logs should be delivered.
       *
       * @see [FlowLog LogDestination](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestination)
       * @see [FlowLog LogDestinationType](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestinationtype)
       *
       * @group Inputs
       */
    readonly destination?: FlowLogDestination;
    /**
       * The fields to include in the flow log record, in the order in which they
       * should appear. For a list of available fields, see {@link FlowLogField}.
       *
       * @see [FlowLog LogFormat](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logformat)
       *
       * @group Inputs
       */
    readonly format?: FlowLogFormat;
    /**
       * The maximum interval of time during which a flow of packets is captured
       * and aggregated into a flow log record.
       *
       * @see [FlowLog MaxAggregationInterval](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-maxaggregationinterval)
       *
       * @group Inputs
       */
    readonly maxAggregationInterval?: FlowLogAggregationInterval;
    /**
       * Details for the resource from which flow logs will be captured.
       *
       * @see [FlowLog ResourceId](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourceid)
       * @see [FlowLog ResourceType](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourcetype)
       *
       * @group Inputs
       */
    readonly resourceType: FlowLogResourceType;
    /**
       * The type of traffic to monitor (accepted traffic, rejected traffic, or
       * all traffic).
       *
       * @see [FlowLog TrafficType](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-traffictype)
       *
       * @group Inputs
       */
    readonly trafficType?: FlowLogTrafficType;
}
export declare class FlowLog extends Resource implements IFlowLog {
    /**
       * The location where flow logs should be delivered.
       *
       * @see [FlowLog LogDestination](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestination)
       * @see [FlowLog LogDestinationType](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestinationtype)
       *
       * @group Inputs
       */
    readonly destination: FlowLogDestination;
    /**
       * The fields to include in the flow log record, in the order in which they
       * should appear. For a list of available fields, see {@link FlowLogField}.
       *
       * @see [FlowLog LogFormat](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logformat)
       *
       * @group Inputs
       */
    readonly format: FlowLogFormat;
    /**
       * The maximum interval of time during which a flow of packets is captured
       * and aggregated into a flow log record.
       *
       * @see [FlowLog MaxAggregationInterval](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-maxaggregationinterval)
       *
       * @group Inputs
       */
    readonly maxAggregationInterval?: FlowLogAggregationInterval;
    /**
       * Details for the resource from which flow logs will be captured.
       *
       * @see [FlowLog ResourceId](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourceid)
       * @see [FlowLog ResourceType](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourcetype)
       *
       * @group Inputs
       */
    readonly resourceType: FlowLogResourceType;
    /**
       * The type of traffic to monitor (accepted traffic, rejected traffic, or
       * all traffic).
       *
       * @see [FlowLog TrafficType](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-traffictype)
       *
       * @group Inputs
       */
    readonly trafficType: FlowLogTrafficType;
    /**
       * The underlying FlowLog CloudFormation resource.
       *
       * @see [AWS::EC2::FlowLog](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html)
       *
       * @group Resources
       */
    readonly resource: CfnFlowLog;
    /**
       * The Amazon Resource Name (ARN) of the flow log.
       */
    readonly flowLogArn: string;
    /**
       * The ID of the flow log.
       */
    readonly flowLogId: string;
    /**
       * Creates a new instance of the FlowLog class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope: IConstruct, id: string, props: FlowLogProps);
}
